<?php

namespace ahura\inc\widgets;

// Block direct access to the main theme file.
defined('ABSPATH') or die('No script kiddies please!');

use ahura\app\mw_assets;
use Elementor\Controls_Manager;

class shop_carousel7 extends \ahura\app\elementor\Elementor_Widget_Base
{
    public function __construct($data = [], $args = [])
    {
        parent::__construct($data, $args);
        mw_assets::register_owlcarousel_assets();
        $this->register_elementor_script('shop_carousel7_js', mw_assets::get_js('elementor.shop_carousel7'));
    }

    public function __register_assets()
    {
        $this->register_elementor_style('shop_carousel7_css', mw_assets::get_css('elementor.shop_carousel7'));
    }

    public function get_script_depends()
    {
        return [$this->get_asset_handle_name('shop_carousel7_js')];
    }

    public function get_name()
    {
        return 'shop_carousel7';
    }

    public function get_title()
    {
        return esc_html__('Shop Carousel 7', 'ahura');
    }

    public function get_icon()
    {
        return 'aicon-svg-grid-products';
    }

    public function get_categories()
    {
        return ['ahuraelements', 'ahura_woocommerce'];
    }

    public function get_keywords()
    {
		return ['ahura', 'shop', 'product', 'products carousel', 'shop carousel 7', esc_html__( 'ahura' , 'ahura'), esc_html__( 'Products' , 'ahura'), esc_html__( 'Shop Carousel 7' , 'ahura')];
    }

    public function register_controls()
    {
        $this->start_controls_section(
            'display_settings',
            [
                'label' => esc_html__('Content', 'ahura'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT
            ]
        );

        $categories = get_terms( array(
            'taxonomy' => 'product_cat',
            'hide_empty' => false,
        ));
        $cats = [];
        if (\ahura\app\woocommerce::is_active()) {
            if($categories){
                foreach ( $categories as $category ) {
                    $cats[ $category->slug ] = $category->name;
                }
            }
        }

        $this->add_control(
            'catsid',
            [
                'label'    => __( 'Categories', 'ahura' ),
                'type'     => Controls_Manager::SELECT2,
                'options'  => array_merge(
                    [ 'allproducts'  => esc_html__( 'All Products', 'ahura' ) ],
                    $cats ),
                'label_block' => true,
                'multiple' => true,
                'default' => 'allproducts'
            ]
        );

        $this->add_control(
            'per_page',
            [
                'label' => esc_html__('Posts Count', 'ahura'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'min' => 1,
                'step' => 1,
                'default' => 12
            ]
        );

        $this->add_responsive_control(
            'layout_col',
            [
                'label' => esc_html__('Slides Per View', 'ahura'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 4,
                'options' => [
                    '1' => 1,
                    '2' => 2,
                    '3' => 3,
                    '4' => 4,
                    '5' => 5,
                    '6' => 6,
                    '7' => 7,
                ]
            ]
        );

        $this->add_control(
            'layout_spacing',
            [
                'label' => esc_html__('Spacing', 'ahura'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'min' => 1,
                'step' => 1,
                'default' => 25,
            ]
        );

        $alignment = array(
            'right' => [
                'title' => esc_html__('Right', 'ahura'),
                'icon' => 'eicon-text-align-right',
            ],
            'center' => [
                'title' => esc_html__('Center', 'ahura'),
                'icon' => 'eicon-text-align-center',
            ],
            'left' => [
                'title' => esc_html__('Left', 'ahura'),
                'icon' => 'eicon-text-align-left',
            ]
        );
        if (!is_rtl()) {
            $alignment = array_reverse($alignment);
        }

        $this->add_control(
            'box_text_alignment',
            [
                'label' => esc_html__('Text Alignment', 'ahura'),
                'type' => \Elementor\Controls_Manager::CHOOSE,
                'options' => $alignment,
                'default' => 'center',
                'toggle' => true,
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product' => 'text-align: {{VALUE}}',
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Image_Size::get_type(),
            [
                'name' => 'item_cover_size',
                'default' => 'shop_catalog',
            ]
        );

        $this->add_responsive_control(
            'object_fit',
            [
                'label' => esc_html__( 'Aspect ratio', 'ahura' ),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'cover',
                'options' => [
                    'fill' => esc_html__( 'Default', 'ahura' ),
                    'contain' => esc_html__( 'Contain', 'ahura' ),
                    'cover'  => esc_html__( 'Cover', 'ahura' ),
                ],
                'selectors' => [
                    '{{WRAPPER}} .product-cover img' => 'object-fit: {{VALUE}};',
                ],
            ]
        );

        $this->add_control('divider', ['type' => \Elementor\Controls_Manager::DIVIDER]);

        $stock_options = (function_exists('wc_get_product_stock_status_options')) ? wc_get_product_stock_status_options() : [];

        $this->add_control(
            'products_stock_status',
            [
                'label'   => esc_html__('Stock status of products', 'ahura'),
                'type'    => \Elementor\Controls_Manager::SELECT,
                'label_block' => true,
                'options' => array_merge(['none'  => esc_html__('None', 'ahura')], $stock_options),
                'default' => 'instock'
            ]
        );

        $this->add_control(
            'sale_price_product',
            [
                'label'   => __( 'Show only discounted products', 'ahura' ),
                'type'    => \Elementor\Controls_Manager::CHOOSE,
                'options' => [
                    'yes'  => [ 'title' => __( 'Yes', 'ahura' ), 'icon' => 'eicon-check' ],
                    'no' => [ 'title' => __( 'No', 'ahura' ), 'icon' => 'eicon-editor-close' ]
                ],
                'default' => 'no',
            ]
        );

        $this->add_control(
            'show_slider_btn',
            [
                'label' => esc_html__('Show Arrows', 'ahura'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'ahura'),
                'label_off' => esc_html__('Hide', 'ahura'),
                'return_value' => 'yes',
                'default' => 'yes',
                'selectors' => [
                    '{{WRAPPER}} .owl-carousel .owl-nav' => 'display:block;'
                ]
            ]
        );

        $this->add_control(
            'show_slider_pagination',
            [
                'label' => esc_html__('Show Pagination', 'ahura'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'ahura'),
                'label_off' => esc_html__('Hide', 'ahura'),
                'return_value' => 'yes',
                'default' => 'yes',
                'selectors' => [
                    '{{WRAPPER}} .owl-carousel .owl-dots' => 'display:block;'
                ]
            ]
        );
        $this->end_controls_section();
        /**
         *
         * More Settings
         *
         */
        $this->start_controls_section(
            'display_more_settings',
            [
                'label' => esc_html__('More Settings', 'ahura'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT
            ]
        );
        $this->add_control(
            'show_badge',
            [
                'label' => esc_html__('Badge', 'ahura'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'ahura'),
                'label_off' => esc_html__('Hide', 'ahura'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );
        $this->add_control(
            'show_title',
            [
                'label' => esc_html__('Title', 'ahura'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'ahura'),
                'label_off' => esc_html__('Hide', 'ahura'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'title_html_tag',
            [
                'label' => esc_html__('Title Html Tag', 'ahura'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'h2',
                'options' => [
                    'h1' => 'H1',
                    'h2' => 'H2',
                    'h3' => 'H3',
                    'h4' => 'H4',
                    'h5' => 'H5',
                    'h6' => 'H6',
                    'div' => 'DIV',
                    'span' => 'SPAN',
                    'P' => 'P',
                ],
                'condition' => [
                    'show_title' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'show_price',
            [
                'label' => esc_html__('Price', 'ahura'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'ahura'),
                'label_off' => esc_html__('Hide', 'ahura'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );
        $this->add_control(
            'show_cart_button',
            [
                'label' => esc_html__('Buy Button', 'ahura'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'ahura'),
                'label_off' => esc_html__('Hide', 'ahura'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'btn_text',
            [
                'label' => esc_html__('Button text', 'ahura'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Buy', 'ahura'),
                'condition' => ['show_cart_button' => 'yes']
            ]
        );

        $this->add_control(
            'btn_icon',
            [
                'label' => esc_html__( 'Button Icon', 'ahura' ),
                'type' => \Elementor\Controls_Manager::ICONS,
                'default' => [
                    'value' => 'fas fa-plus',
                    'library' => 'fa-solid',
                ],
                'condition' => ['show_cart_button' => 'yes']
            ]
        );
        $this->end_controls_section();


        /**
         *
         *
         * Start Item style
         *
         */
        $this->start_controls_section(
            'item_styles',
            [
                'label' => esc_html__('Box', 'ahura'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE
            ]
        );

        $this->start_controls_tabs('item_styles_tabs');

        $this->start_controls_tab(
            'item_styles_normal_tab',
            [
                'label' => esc_html__('Normal', 'ahura'),
            ]
        );

        $this->add_control(
            'item_bg_color',
            [
                'label' => esc_html__('Background Color', 'ahura'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#fff',
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'item_border',
                'label' => esc_html__('Border', 'ahura'),
                'selector' => '{{WRAPPER}} .products-list-carousel .product',
            ]
        );

        $this->add_responsive_control(
            'item_border_radius',
            [
                'label' => esc_html__('Border Radius', 'ahura'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product' => 'border-radius: {{SIZE}}{{UNIT}}',
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 5,
                ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
            ]
        );

        $this->add_responsive_control(
            'item_padding',
            [
                'label' => esc_html__('Padding', 'ahura'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'isLinked' => true,
                    'top' => 15,
                    'right' => 15,
                    'bottom' => 15,
                    'left' => 15,
                ]
            ]
        );

        $this->add_responsive_control(
            'item_margin',
            [
                'label' => esc_html__('Margin', 'ahura'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'isLinked' => true,
                    'top' => 100,
                    'right' => 0,
                    'bottom' => 0,
                    'left' => 0,
                ]
            ]
        );
        $this->end_controls_tab();
        $this->start_controls_tab(
            'item_styles_hover_tab',
            [
                'label' => esc_html__('Hover', 'ahura'),
            ]
        );

        $this->add_control(
            'item_bg_color_hover',
            [
                'label' => esc_html__('Background Color', 'ahura'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#fff',
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product:hover' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'item_border_hover',
                'label' => esc_html__('Border', 'ahura'),
                'selector' => '{{WRAPPER}} .products-list-carousel .product:hover',
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();
        $this->end_controls_section();

        /**
         *
         *
         * Start image style
         *
         *
         */
        $this->start_controls_section(
            'item_img_styles',
            [
                'label' => esc_html__('Image', 'ahura'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE
            ]
        );

        $this->start_controls_tabs('img_style_tabs');
        $this->start_controls_tab('img_style_normal_tab', ['label' => esc_html__( 'Normal', 'ahura' )]);

        $this->add_control(
            'item_cover_bg_color',
            [
                'label' => esc_html__('Background Color', 'ahura'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#fff',
                'selectors' => [
                    '{{WRAPPER}} .product .product-cover' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'item_img_border',
                'label' => esc_html__('Border', 'ahura'),
                'selector' => '{{WRAPPER}} .products-list-carousel .product .product-cover',
            ]
        );

        $this->add_responsive_control(
            'item_img_border_radius',
            [
                'label' => esc_html__('Border Radius', 'ahura'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product .product-cover' => 'border-radius: {{SIZE}}{{UNIT}}',
                    '{{WRAPPER}} .products-list-carousel .product .product-cover img' => 'border-radius: {{SIZE}}{{UNIT}}',
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 5,
                ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
            ]
        );

        $this->add_responsive_control(
            'item_img_height',
            [
                'label' => esc_html__('Height', 'ahura'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', 'rem'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                    ],
                    'em' => [
                        'min' => 0,
                        'max' => 1000,
                    ],
                    'rem' => [
                        'min' => 0,
                        'max' => 1000,
                    ],
                ],
                'devices' => ['desktop', 'tablet', 'mobile'],
                'default' => [
                    'size' => 275,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product .product-cover img' => 'height: {{SIZE}}{{UNIT}}'
                ]
            ]
        );

        $this->add_responsive_control(
            'item_img_padding',
            [
                'label' => esc_html__( 'Padding', 'ahura' ),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem'],
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product .product-cover' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'item_img_spacing',
            [
                'label' => esc_html__( 'Margin', 'ahura' ),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem'],
                'default' => [
                    'top' => -100,
                    'right' => 0,
                    'bottom' => 10,
                    'left' => 0,
                    'unit' => 'px',
                    'isLinked' => false,
                ],
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product .product-cover' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_tab();
        $this->start_controls_tab('img_style_hover_tab', ['label' => esc_html__( 'Hover', 'ahura' )]);

        $this->add_control(
            'item_cover_bg_color_hover',
            [
                'label' => esc_html__('Background Color', 'ahura'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#fff',
                'selectors' => [
                    '{{WRAPPER}} .product:hover .product-cover' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'item_img_border_hover',
                'label' => esc_html__('Border', 'ahura'),
                'selector' => '{{WRAPPER}} .products-list-carousel .product:hover .product-cover',
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();
        $this->end_controls_section();
        /**
         *
         *
         * Start title style
         *
         *
         */
        $this->start_controls_section(
            'item_title_styles',
            [
                'label' => esc_html__('Title', 'ahura'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_title' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'item_title_color',
            [
                'label' => esc_html__('Text Color', 'ahura'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#686868',
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product .title-wrap a *' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'item_title_color_hover',
            [
                'label' => esc_html__('Hover Text Color', 'ahura'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product:hover .title-wrap a *' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'label' => esc_html__('Typography', 'ahura'),
                'name' => 'item_title_typography',
                'selector' => '{{WRAPPER}} .products-list-carousel .product .title-wrap a *',
                'fields_options' => [
                    'typography' => ['default' => 'yes'],
                    'font_size' => [
                        'default' => [
                            'unit' => 'px',
                            'size' => '15',
                        ]
                    ]
                ]
            ]
        );

        $this->add_control(
            'item_title_spacing',
            [
                'label' => esc_html__('Margin', 'ahura'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product .title-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();
        /**
         *
         * Start price style
         *
         */
        $this->start_controls_section(
            'item_price_styles',
            [
                'label' => esc_html__('Price', 'ahura'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_price' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'item_price_color',
            [
                'label' => esc_html__('Text Color', 'ahura'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#35495c',
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product .price-wrap .price > span > bdi' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .products-list-carousel .product .price-wrap .price ins span bdi' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .products-list-carousel .product .price-wrap .price .woocommerce-Price-currencySymbol' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'item_price_color_hover',
            [
                'label' => esc_html__('Hover Text Color', 'ahura'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product:hover .price-wrap .price > span > bdi' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .products-list-carousel .product:hover .price-wrap .price ins span bdi' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .products-list-carousel .product:hover .price-wrap .price .woocommerce-Price-currencySymbol' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'item_price_margin',
            [
                'label' => esc_html__('Margin', 'ahura'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product .price-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'isLinked' => false,
                    'top' => 10,
                    'right' => 0,
                    'bottom' => 20,
                    'left' => 0,
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'label' => esc_html__('Typography', 'ahura'),
                'name' => 'item_price_typography',
                'selector' => '{{WRAPPER}} .products-list-carousel .product .price-wrap *',
                'fields_options' => [
                    'typography' => ['default' => 'yes'],
                    'font_weight' => ['default' => 'bold'],
                    'font_size' => [
                        'default' => [
                            'unit' => 'px',
                            'size' => '18',
                        ]
                    ],
                ]
            ]
        );

        $this->add_control(
            'divider_price',
            [
                'label' => esc_html__('Sale Price', 'ahura'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'item_price_dis_color',
            [
                'label' => esc_html__('Text Color', 'ahura'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#b7b7b7',
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product .price-wrap .price del' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .products-list-carousel .product .price-wrap .price del .woocommerce-Price-currencySymbol' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'item_price_dis_color_hover',
            [
                'label' => esc_html__('Hover Text Color', 'ahura'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product:hover .price-wrap .price del *' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .products-list-carousel .product:hover .price-wrap .price del .woocommerce-Price-currencySymbol *' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'item_price_dis_margin',
            [
                'label' => esc_html__('Margin', 'ahura'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product .price-wrap .price del' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'isLinked' => false,
                    'top' => 0,
                    'right' => 7,
                    'bottom' => 0,
                    'left' => 7,
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'label' => esc_html__('Typography', 'ahura'),
                'name' => 'item_price_dis_typography',
                'selector' => '{{WRAPPER}} .products-list-carousel .product .price-wrap .price del *',
                'fields_options' => [
                    'typography' => ['default' => 'yes'],
                    'font_weight' => ['default' => '300'],
                    'font_size' => [
                        'default' => [
                            'unit' => 'px',
                            'size' => '15',
                        ]
                    ],
                ]
            ]
        );

        $this->end_controls_section();
        /**
         *
         *
         * Start badge styles
         *
         */
        $this->start_controls_section(
            'item_badge_styles',
            [
                'label' => esc_html__('Badge', 'ahura'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_badge' => 'yes'
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'label' => esc_html__('Typography', 'ahura'),
                'name' => 'item_badge_typo',
                'selector' => '{{WRAPPER}} .products-list-carousel .product .product-label span',
                'fields_options' => [
                    'typography' => ['default' => 'yes'],
                    'font_size' => [
                        'default' => [
                            'unit' => 'px',
                            'size' => '14',
                        ]
                    ],
                ]
            ]
        );

        $this->add_control(
            'item_badge_padding',
            [
                'label' => esc_html__('padding', 'ahura'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem'],
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product .product-label span' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'isLinked' => false,
                    'top' => 3,
                    'right' => 13,
                    'bottom' => 3,
                    'left' => 13,
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'item_badge_border',
                'label' => esc_html__('Border', 'ahura'),
                'selector' => '{{WRAPPER}} .products-list-carousel .product .product-label span',
            ]
        );

        $this->add_responsive_control(
            'item_badge_border_radius',
            [
                'label' => esc_html__('Border Radius', 'ahura'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product .product-label span' => 'border-radius: {{SIZE}}{{UNIT}}',
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 7,
                ],
                'tablet_default' => [
                    'unit' => 'px',
                    'size' => 7,
                ],
                'mobile_default' => [
                    'unit' => 'px',
                    'size' => 7,
                ],

                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
            ]
        );

        $this->add_control(
            'divider_badge1',
            [
                'label' => esc_html__('New', 'ahura'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'item_badge_new_color',
            [
                'label' => esc_html__('Text Color', 'ahura'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#fff',
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product .product-label .new' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'item_badge_new_bg_color',
            [
                'label' => esc_html__('Background Color', 'ahura'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#00b0ff',
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product .product-label .new' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'divider_badge2',
            [
                'label' => esc_html__('Not In Stock', 'ahura'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'item_badge_soldout_color',
            [
                'label' => esc_html__('Text Color', 'ahura'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#fff',
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product .product-label .soldout' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'item_badge_soldout_bg_color',
            [
                'label' => esc_html__('Background Color', 'ahura'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#c00',
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product .product-label .soldout' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'divider_badge3',
            [
                'label' => esc_html__('On Sale', 'ahura'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'item_badge_onsale_color',
            [
                'label' => esc_html__('Text Color', 'ahura'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#fff',
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product .product-label .onsale' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'item_badge_onsale_bg_color',
            [
                'label' => esc_html__('Background Color', 'ahura'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ffb93a',
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product .product-label .onsale' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_section();
        /**
         *
         *
         * Item button styles
         *
         */
        $this->start_controls_section(
            'item_buttons_styles',
            [
                'label' => esc_html__('Button', 'ahura'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_cart_button' => 'yes'
                ]
            ]
        );

        $this->start_controls_tabs('item_buttons_styles_tabs');

        $this->start_controls_tab(
            'item_buttons_styles_normal_tab',
            [
                'label' => esc_html__('Normal', 'ahura'),
            ]
        );

        $this->add_control(
            'item_button_color',
            [
                'label' => esc_html__('Text Color', 'ahura'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#fff',
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product .product-foot a' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'item_button_bg_color',
            [
                'label' => esc_html__('Background Color', 'ahura'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#000',
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product .product-foot a' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'item_button_border',
                'label' => esc_html__('Border', 'ahura'),
                'selector' => '{{WRAPPER}} .products-list-carousel .product .product-foot a',
            ]
        );

        $this->add_responsive_control(
            'item_button_border_radius',
            [
                'label' => esc_html__('Border Radius', 'ahura'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', '%', 'em', 'rem'],
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product .product-foot a' => 'border-radius: {{SIZE}}{{UNIT}}',
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 5,
                ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
            ]
        );

        $this->add_responsive_control(
            'item_buttons_wrap_margin',
            [
                'label' => esc_html__('Margin', 'ahura'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem'],
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product .product-foot' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'devices' => ['desktop', 'tablet', 'mobile'],
                'default' => [
                    'isLinked' => false,
                    'top' => 0,
                    'right' => 0,
                    'bottom' => -32,
                    'left' => 0,
                ]
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'item_button_styles_hover_tab',
            [
                'label' => esc_html__('Hover', 'ahura'),
            ]
        );

        $this->add_control(
            'item_button_cart_color_hover',
            [
                'label' => esc_html__('Text Color', 'ahura'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#fff',
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product .product-foot a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'item_button_cart_bg_color_hover',
            [
                'label' => esc_html__('Background Color', 'ahura'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#000',
                'selectors' => [
                    '{{WRAPPER}} .products-list-carousel .product .product-foot a:hover' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'item_button_border_hover',
                'label' => esc_html__('Border', 'ahura'),
                'selector' => '{{WRAPPER}} .products-list-carousel .product .product-foot a:hover',
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();
        $this->end_controls_section();

        $this->start_controls_section(
            'box_paginate_style',
            [
                'label' => esc_html__('Pagination', 'ahura'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => ['show_slider_pagination' => 'yes']
            ]
        );

        $this->add_control(
            'paginate_color',
            [
                'label' => esc_html__('Normal Color', 'ahura'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#7e7e7e',
                'selectors' => [
                    '{{WRAPPER}} .owl-dots .owl-dot' => 'border-color: {{VALUE}};background-color: {{VALUE}};',
                ]
            ]
        );

        $this->add_control(
            'paginate_active_color',
            [
                'label' => esc_html__('Active Color', 'ahura'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#2289FF',
                'selectors' => [
                    '{{WRAPPER}} .owl-dots .owl-dot.active' => 'border-color: {{VALUE}};background-color: {{VALUE}};',
                ]
            ]
        );

        $this->end_controls_section();
    }

    /**
     *
     * Check is new product
     *
     * @return bool
     */
    public function is_new_product()
    {
        global $product;
        $days = 5;
        $created = strtotime($product->get_date_created());
        return ((time() - (60 * 60 * 24 * $days)) < $created);
    }

    /**
     *
     * Get products result
     *
     * @param array $params
     * @return false
     */
    public function get_products($params = [])
    {
        $settings = $this->get_settings_for_display();
        $cats_id = $settings['catsid'];
        $field_is_term = (is_array($cats_id) && isset($cats_id[0]) && is_numeric($cats_id[0])) || is_int($cats_id);

        if(!$cats_id || $cats_id == 'allproducts' || ($cats_id[0] == 'allproducts')) {
            $args = array(
                'post_type' => 'product',
                'posts_per_page' => $settings['per_page'],
                'post_status' => 'publish',
            );
        } else {
            $args = array(
                'post_type' => 'product',
                'posts_per_page' => $settings['per_page'],
                'post_status' => 'publish',
                'tax_query'		 => [
                    [
                        'taxonomy'   => 'product_cat',
                        'field'		 => $field_is_term ? 'term_id' : 'slug',
                        'terms'		 => $settings[ 'catsid' ],
                    ]
                ],
            );
        }

        $term = (array) get_queried_object();

        if ($cats_id && is_array($term) && isset($term['term_id']) && isset($term['slug'])) {
            $args['tax_query'] = array(
                array(
                    'taxonomy' => $term['taxonomy'],
                    'field'    => 'slug',
                    'terms'    => $term['slug'],
                ),
            );
        }

        $products_stock_status = $settings['products_stock_status'];

        if ($cats_id && $products_stock_status && $products_stock_status !== 'none') {
            $args['meta_query'] = array(array(
                'key' => '_stock_status',
                'value' => $products_stock_status,
                'compare' => '==',
            ));
        }

        if($settings['sale_price_product'] === 'yes'){
            $args = array_merge($args, [
                'order' 		 => $settings['product_order'],
                'meta_key' 		 => '_sale_price',
                'meta_value' 	 => '0',
                'meta_compare'   => '>='
            ]);
        }

        $products = new \WP_Query(array_merge($args, $params));
        return ($products->have_posts()) ? $products : false;
    }

    /**
     *
     * Render content for display
     *
     */
    public function render()
    {
        $settings = $this->get_settings_for_display();
        $args = [];

        if (!\ahura\app\woocommerce::is_active()) {
            if(is_admin()) {
                $this->renderElementorNotice(__('To use this element you must install woocommerce plugin.', 'ahura'));
            }
            return false;
        }

        $products = $this->get_products($args);

        $layout_col = $settings['layout_col'];
        $tablet_layout_col = isset($settings['layout_col_tablet']) ? $settings['layout_col_tablet'] : 3;
        $mobile_layout_col = isset($settings['layout_col_mobile']) ? $settings['layout_col_mobile'] : 1;

        $slider_data = [
            'slidesPerView' => $layout_col,
            'tabletSlidesPerView' => $tablet_layout_col,
            'mobileSlidesPerView' => $mobile_layout_col,
            'margin' => $settings['layout_spacing']
        ];

        if ($products) : ?>
            <div class="shop-carousel7">
                <div class="owl-carousel products-list-carousel no-item-bg" data-slider='<?php echo json_encode($slider_data) ?>'>
                    <?php
                    while ($products->have_posts()) :
                        $products->the_post();
                        global $product;
                        ?>
                        <div class="product-item-wrap">
                            <article <?php wc_product_class('', $product) ?>>
                                <div class="product-cover">
                                    <?php if ($settings['show_badge'] === 'yes') : ?>
                                        <div class="product-label">
                                            <?php
                                            if ($this->is_new_product()) :
                                                echo '<span class="new">' . esc_html__('New', 'ahura') . '</span>';
                                            endif;
                                            if (!$product->is_in_stock()) :
                                                echo apply_filters('woocommerce_product_is_in_stock', '<span class="soldout">' . esc_html__('Sold Out!', 'ahura') . '</span>', $product, $product);
                                            elseif ($product->is_on_sale()) :
                                                echo apply_filters('woocommerce_sale_flash', '<span class="sale">' . esc_html__('On Sale', 'ahura') . '</span>', $product, $product);
                                            endif;
                                            ?>
                                        </div>
                                    <?php endif; ?>
                                    <a href="<?php the_permalink(); ?>">
                                        <?php echo wp_get_attachment_image(get_post_thumbnail_id(), $settings['item_cover_size_size']); ?>
                                    </a>
                                </div>
                                <?php if ($settings['show_title'] === 'yes') : ?>
                                    <div class="title-wrap">
                                        <a href="<?php echo get_the_permalink() ?>">
                                            <<?php echo $settings['title_html_tag'] ?>>
                                                <?php the_title(); ?>
                                            </<?php echo $settings['title_html_tag'] ?>>
                                        </a>
                                    </div>
                                <?php endif; ?>
                                <?php if ($settings['show_price'] === 'yes') : ?>
                                    <div class="price-wrap">
                                        <?php woocommerce_template_single_price(); ?>
                                    </div>
                                <?php endif; ?>
                                <?php if ($settings['show_cart_button'] === 'yes') : ?>
                                    <div class="product-foot">
                                        <a href="<?php the_permalink() ?>?add-to-cart=<?php the_ID() ?>" class="mw_add_to_cart">
                                            <?php \Elementor\Icons_Manager::render_icon($settings['btn_icon'], ['aria-hidden' => 'true']); ?>
                                            <span><?php echo $settings['btn_text'] ?></span>
                                        </a>
                                    </div>
                                <?php endif; ?>
                            </article>
                        </div>
                    <?php
                    endwhile;
                    wp_reset_query();
                    wp_reset_postdata();
                    ?>
                </div>
            </div>
        <?php
        else:
            $this->renderElementorNotice(__('Sorry, no products were found for display.', 'ahura'));
        endif;
    }
}
