<?php

namespace ahura\inc\widgets;

// Block direct access to the main theme file.
defined('ABSPATH') or die('No script kiddies please!');

use ahura\app\mw_assets;
use ahura\app\traits\WoocommerceMethods;
use Elementor\Controls_Manager;

class shop_carousel11 extends \ahura\app\elementor\Elementor_Widget_Base
{
    use WoocommerceMethods;

    public function __construct($data = [], $args = null)
    {
        parent::__construct($data, $args);
        mw_assets::register_swiper_assets(true);
        mw_assets::register_script('shop_carousel11_js', mw_assets::get_js('elementor.shop_carousel11'));
    }

    public function __register_assets()
    {
        $this->register_elementor_style('shop_carousel11_css', mw_assets::get_css('elementor.shop_carousel11'));
    }

    public function get_script_depends()
    {
        return [mw_assets::get_handle_name('shop_carousel11_js')];
    }

    public function get_name()
    {
        return 'shop_carousel11';
    }

    public function get_title()
    {
        return esc_html__('Shop Carousel 11', 'ahura');
    }

    public function get_icon()
    {
        return 'aicon-svg-grid-products';
    }

    public function get_categories()
    {
        return ['ahuraelements', 'ahura_woocommerce'];
    }

    public function get_keywords()
    {
		return ['ahura', 'shop', 'product', 'products carousel', 'shop carousel 11', esc_html__( 'ahura' , 'ahura'), esc_html__( 'Products' , 'ahura'), esc_html__( 'Shop Carousel 11' , 'ahura')];
    }


    public function register_controls()
    {
        $this->start_controls_section(
            'display_settings',
            [
                'label' => esc_html__('Content', 'ahura'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT
            ]
        );

        $categories = get_terms( array(
            'taxonomy' => 'product_cat',
            'hide_empty' => false,
        ));
        $cats = [];
        if (\ahura\app\woocommerce::is_active()) {
            if($categories){
                foreach ( $categories as $category ) {
                    $cats[ $category->slug ] = $category->name;
                }
            }
        }

        $this->add_control(
            'catsid',
            [
                'label'    => __( 'Categories', 'ahura' ),
                'type'     => Controls_Manager::SELECT2,
                'options'  => array_merge(
                    [ 'allproducts'  => esc_html__( 'All Products', 'ahura' ) ],
                    $cats ),
                'label_block' => true,
                'multiple' => true,
                'default' => 'allproducts'
            ]
        );

        $this->add_control(
            'per_page',
            [
                'label' => esc_html__('Count', 'ahura'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'min' => 1,
                'default' => 12
            ]
        );

        $alignment = array(
            'right' => [
                'title' => esc_html__('Right', 'ahura'),
                'icon' => 'eicon-text-align-right',
            ],
            'center' => [
                'title' => esc_html__('Center', 'ahura'),
                'icon' => 'eicon-text-align-center',
            ],
            'left' => [
                'title' => esc_html__('Left', 'ahura'),
                'icon' => 'eicon-text-align-left',
            ]
        );
        if (!is_rtl()) {
            $alignment = array_reverse($alignment);
        }

        $this->add_control(
            'box_text_alignment',
            [
                'label' => esc_html__('Text Alignment', 'ahura'),
                'type' => \Elementor\Controls_Manager::CHOOSE,
                'options' => $alignment,
                'default' => 'center',
                'toggle' => true,
                'selectors' => [
                    '{{WRAPPER}} .products-list .product' => 'text-align: {{VALUE}}',
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Image_Size::get_type(),
            [
                'name' => 'item_cover_size',
                'default' => 'shop_catalog',
            ]
        );

        $this->add_responsive_control(
            'object_fit',
            [
                'label' => esc_html__( 'Aspect ratio', 'ahura' ),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'cover',
                'options' => [
                    'fill' => esc_html__( 'Default', 'ahura' ),
                    'contain' => esc_html__( 'Contain', 'ahura' ),
                    'cover'  => esc_html__( 'Cover', 'ahura' ),
                ],
                'selectors' => [
                    '{{WRAPPER}} .product-cover img' => 'object-fit: {{VALUE}};',
                ],
            ]
        );

        $this->add_control('divider', ['type' => \Elementor\Controls_Manager::DIVIDER]);

        $stock_options = (function_exists('wc_get_product_stock_status_options')) ? wc_get_product_stock_status_options() : [];

        $this->add_control(
            'products_stock_status',
            [
                'label'   => esc_html__('Stock status of products', 'ahura'),
                'type'    => \Elementor\Controls_Manager::SELECT,
                'label_block' => true,
                'options' => array_merge(['none'  => esc_html__('None', 'ahura')], $stock_options),
                'default' => 'instock'
            ]
        );

        $this->add_control(
            'sale_price_product',
            [
                'label'   => __( 'Show only discounted products', 'ahura' ),
                'type'    => \Elementor\Controls_Manager::CHOOSE,
                'options' => [
                    'yes'  => [ 'title' => __( 'Yes', 'ahura' ), 'icon' => 'eicon-check' ],
                    'no' => [ 'title' => __( 'No', 'ahura' ), 'icon' => 'eicon-editor-close' ]
                ],
                'default' => 'no',
            ]
        );

        $this->end_controls_section();
        /**
         *
         * More Settings
         *
         */
        $this->start_controls_section(
            'display_more_settings',
            [
                'label' => esc_html__('More Settings', 'ahura'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT
            ]
        );

        $this->add_control(
            'show_badge',
            [
                'label' => esc_html__('Badge', 'ahura'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'ahura'),
                'label_off' => esc_html__('Hide', 'ahura'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'show_title',
            [
                'label' => esc_html__('Title', 'ahura'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'ahura'),
                'label_off' => esc_html__('Hide', 'ahura'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'title_html_tag',
            [
                'label' => esc_html__('Title Html Tag', 'ahura'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'h2',
                'options' => [
                    'h1' => 'H1',
                    'h2' => 'H2',
                    'h3' => 'H3',
                    'h4' => 'H4',
                    'h5' => 'H5',
                    'h6' => 'H6',
                    'div' => 'DIV',
                    'span' => 'SPAN',
                    'P' => 'P',
                ],
                'condition' => [
                    'show_title' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'show_price',
            [
                'label' => esc_html__('Price', 'ahura'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'ahura'),
                'label_off' => esc_html__('Hide', 'ahura'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );
        $this->add_control(
            'show_cart_button',
            [
                'label' => esc_html__('Cart Button', 'ahura'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'ahura'),
                'label_off' => esc_html__('Hide', 'ahura'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'btn_text',
            [
                'label' => esc_html__('Button Text', 'ahura'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Buy', 'ahura'),
                'condition' => ['show_cart_button' => 'yes']
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'slider_section',
            [
                'label' => __( 'Slider', 'ahura' ),
                'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_responsive_control(
            'slides_per_view',
            [
                'label' => esc_html__('Slides per view', 'ahura'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    1 => '1',
                    2 => '2',
                    3 => '3',
                    4 => '4',
                ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'desktop_default' => 4,
                'tablet_default' => 3,
                'mobile_default' => 1,
            ]
        );

        $this->add_control(
            'double_row',
            [
                'label' => esc_html__('Double Row', 'ahura'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'ahura'),
                'label_off' => esc_html__('No', 'ahura'),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'show_arrows',
            [
                'label' => esc_html__( 'Show Arrows', 'ahura' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'ahura' ),
                'label_off' => esc_html__( 'Hide', 'ahura' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'autoplay',
            [
                'label' => esc_html__( 'Autoplay', 'ahura' ),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'ahura' ),
                'label_off' => esc_html__( 'Hide', 'ahura' ),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'autoplay_delay',
            [
                'label' => esc_html__( 'Autoplay delay', 'ahura' ),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'min' => 100,
                'max' => 10000,
                'step' => 1,
                'default' => 4500,
                'condition' => ['autoplay' => 'yes']
            ]
        );

        $this->add_control(
            'use_observer_slider',
            [
                'label' => esc_html__('Inner Slider', 'ahura'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'ahura'),
                'label_off' => esc_html__('No', 'ahura'),
                'return_value' => 'yes',
                'default' => 'yes',
                'description' => __('If you use the slider internally (tab, ...) and its display is not direct, be sure to activate this option.', 'ahura')
            ]
        );

        $this->end_controls_section();

        /**
         *
         *
         * Styles
         *
         *
         */
        $this->start_controls_section(
            'items_style_section',
            [
                'label' => __( 'Items', 'ahura' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'item_bg',
                'selector' => '{{WRAPPER}} .product',
                'exclude' => ['image'],
                'fields_options' => [
                    'background' => ['default' => 'classic'],
                    'color' => ['default' => '#fff']
                ],
            ]
        );

        $this->add_control(
            'item_title_color',
            [
                'label' => esc_html__('Title Color', 'ahura'),
                'type' => Controls_Manager::COLOR,
                'default' => '#000',
                'selectors' => [
                    '{{WRAPPER}} .p-title' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'label' => esc_html__('Title Typography', 'ahura'),
                'name' => 'item_title_typo',
                'selector' => '{{WRAPPER}} .p-title',
                'fields_options' => [
                    'typography' => ['default' => 'yes'],
                    'font_weight' => ['default' => 400],
                    'font_size' => [
                        'default' => [
                            'unit' => 'px',
                            'size' => '17',
                        ]
                    ]
                ],
            ]
        );

        $this->add_control(
            'price_color',
            [
                'label' => esc_html__( 'Price color', 'ahura' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .product .price-wrap *' => 'color: {{VALUE}}',
                ],
                'default' => '#333'
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'price_typography',
                'fields_options' => [
                    'typography' => ['default' => 'yes'],
                    'font_size' => [
                        'default' => [
                            'unit' => 'px',
                            'size' => '15',
                        ]
                    ]
                ],
                'selector' => '{{WRAPPER}} .product .price-wrap *',
            ]
        );

        $this->add_responsive_control(
            'item_border_radius',
            [
                'label' => esc_html__('Border Radius', 'ahura'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .product' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'isLinked' => true,
                    'top' => 10,
                    'right' => 10,
                    'bottom' => 10,
                    'left' => 10,
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'item_border',
                'label' => esc_html__('Border', 'ahura'),
                'selector' => '{{WRAPPER}} .product',
            ]
        );

        $this->add_responsive_control(
            'item_padding',
            [
                'label' => esc_html__('Padding', 'ahura'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .product' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'isLinked' => true,
                    'top' => 15,
                    'right' => 15,
                    'bottom' => 15,
                    'left' => 15,
                ]
            ]
        );

        $this->add_control(
            'btn_options',
            [
                'label' => esc_html__( 'Button', 'ahura' ),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'show_cart_button' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'item_btn_color',
            [
                'label' => esc_html__('Color', 'ahura'),
                'type' => Controls_Manager::COLOR,
                'default' => '#fff',
                'selectors' => [
                    '{{WRAPPER}} .product-foot a' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'show_cart_button' => 'yes'
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'item_btn_bg',
                'selector' => '{{WRAPPER}} .product-foot a',
                'exclude' => ['image'],
                'fields_options' => [
                    'background' => ['default' => 'classic'],
                    'color' => ['default' => '#FA722C']
                ],
                'condition' => [
                    'show_cart_button' => 'yes'
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'item_btn_typo',
                'selector' => '{{WRAPPER}} .product-foot a',
                'fields_options' => [
                    'typography' => ['default' => 'yes'],
                    'font_weight' => ['default' => 400],
                    'font_size' => [
                        'default' => [
                            'unit' => 'px',
                            'size' => '16',
                        ]
                    ]
                ],
                'condition' => [
                    'show_cart_button' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'item_btn_border_radius',
            [
                'label' => esc_html__('Button Border Radius', 'ahura'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .product-foot a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'isLinked' => true,
                    'top' => 5,
                    'right' => 5,
                    'bottom' => 5,
                    'left' => 5,
                ]
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'image_style_section',
            [
                'label' => __( 'Image', 'ahura' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'item_img_height',
            [
                'label' => esc_html__('Height', 'ahura'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', 'rem'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                    ],
                    'em' => [
                        'min' => 0,
                        'max' => 1000,
                    ],
                    'rem' => [
                        'min' => 0,
                        'max' => 1000,
                    ],
                ],
                'devices' => ['desktop', 'tablet', 'mobile'],
                'default' => [
                    'size' => 250,
                    'unit' => 'px',
                ],
                'tablet_default' => [
                    'size' => 200,
                    'unit' => 'px',
                ],
                'mobile_default' => [
                    'size' => 200,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .product-cover img' => 'height: {{SIZE}}{{UNIT}}'
                ]
            ]
        );

        $this->add_control(
            'img_border_radius',
            [
                'label' => esc_html__('Border Radius', 'ahura'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .product-cover img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'isLinked' => true,
                    'top' => 10,
                    'right' => 10,
                    'bottom' => 10,
                    'left' => 10,
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'img_border',
                'label' => esc_html__('Border', 'ahura'),
                'selector' => '{{WRAPPER}} .product-cover img',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'navigation_style_section',
            [
                'label' => __( 'Navigation', 'ahura' ),
                'tab'   => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_arrows' => 'yes'
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'slider_nav_btn_bg',
                'selector' => '{{WRAPPER}} .swiper-nav-button',
                'exclude' => ['image'],
                'fields_options' => [
                    'background' => ['default' => 'classic'],
                    'color' => ['default' => '#FA722C']
                ],
            ]
        );

        $this->add_control(
            'slider_btn_icon_color',
            [
                'label' => esc_html__('Icon Color', 'ahura'),
                'type' => Controls_Manager::COLOR,
                'default' => '#fff',
                'selectors' => [
                    '{{WRAPPER}} .swiper-nav-button' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'slider_btn_border_radius',
            [
                'label' => esc_html__('Border Radius', 'ahura'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .swiper-nav-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'isLinked' => true,
                    'top' => 5,
                    'right' => 5,
                    'bottom' => 5,
                    'left' => 5,
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'slider_btn_border',
                'label' => esc_html__('Border', 'ahura'),
                'selector' => '{{WRAPPER}} .swiper-nav-button',
            ]
        );

        $this->end_controls_section();
    }

    /**
     *
     * Get products result
     *
     * @param array $params
     * @return array
     */
    public function get_products($params = [])
    {
        $settings = $this->get_settings_for_display();
        $field_is_term = (is_array($settings['catsid']) && is_numeric($settings['catsid'][0])) || is_int($settings['catsid']);

        if( $settings[ 'catsid' ] == 'allproducts' || ( $settings[ 'catsid' ][ 0 ] == 'allproducts' ) ) {
            $args = array(
                'post_type' => 'product',
                'posts_per_page' => $settings['per_page'],
                'post_status' => 'publish',
            );

        } else {
            $args = array(
                'post_type' => 'product',
                'posts_per_page' => $settings['per_page'],
                'post_status' => 'publish',
                'tax_query'		 => [ [
                    'taxonomy'   => 'product_cat',
                    'field'		 => $field_is_term ? 'term_id' : 'slug',
                    'terms'		 => $settings[ 'catsid' ],
                ] ],
            );
        }

        $term = (array) get_queried_object();

        if (is_array($term) && isset($term['term_id']) && isset($term['slug'])) {
            $args['tax_query'] = array(
                array(
                    'taxonomy' => $term['taxonomy'],
                    'field'    => 'slug',
                    'terms'    => $term['slug'],
                ),
            );
        }

        $products_stock_status = $settings['products_stock_status'];

        if ($products_stock_status && $products_stock_status !== 'none') {
            $args['meta_query'] = array(array(
                'key' => '_stock_status',
                'value' => $products_stock_status,
                'compare' => '==',
            ));
        }

        if($settings['sale_price_product'] === 'yes'){
            $args = array_merge($args, [
                'order' 		 => $settings['product_order'],
                'meta_key' 		 => '_sale_price',
                'meta_value' 	 => '0',
                'meta_compare'   => '>='
            ]);
        }

        $products = new \WP_Query(array_merge($args, $params));
        $ids = [];
        $current_group = [];
        $counter = 0;

        if($products->have_posts()){
            while ($products->have_posts()) {
                $products->the_post();
                $current_group[] = get_post(get_the_ID());
                $counter++;

                if($settings['double_row'] == 'yes'){
                    if ($counter % 2 === 0) {
                        $ids[] = $current_group;
                        $current_group = [];
                    }
                } else {
                    $ids[] = $current_group;
                    $current_group = [];
                }
            }

            if (!empty($current_group)) {
                $ids[] = $current_group;
            }
            wp_reset_postdata();
        }

        return $ids;
    }

    /**
     *
     * Render content for display
     *
     */
    public function render()
    {
        $settings = $this->get_settings_for_display();
        $wid = $this->get_id();
        $show_cart_btn = $settings['show_cart_button'] === 'yes';
        $show_arrows = isset($settings['show_arrows']) && $settings['show_arrows'] == 'yes';
        $args = [];

        if (!\ahura\app\woocommerce::is_active()) {
            if(is_admin()) {
                ?>
                <div class="productcategorybox mw_elem_empty_box"><h3><?php _e('To use this element you must install woocommerce plugin.', 'ahura'); ?></h3></div>
                <?php
            }
            return false;
        }

        $slider_data = [
            'arrows' => $show_arrows,
            'autoPlay' => $settings['autoplay'] == 'yes',
            'autoplayTimeout' => $settings['autoplay_delay'] ? $settings['autoplay_delay'] : 4500,
            'observer' => $settings['use_observer_slider'] == 'yes',
            'desktopPerView' => (isset($settings['slides_per_view']) && intval($settings['slides_per_view'])) ? $settings['slides_per_view'] : 4,
            'tabletPerView' => (isset($settings['slides_per_view_tablet']) && intval($settings['slides_per_view_tablet'])) ? $settings['slides_per_view_tablet'] : 3,
            'mobilePerView' => (isset($settings['slides_per_view_mobile']) && intval($settings['slides_per_view_mobile'])) ? $settings['slides_per_view_mobile'] : 1,
        ];

        $products = $this->get_products($args);
        if ($products) : ?>
            <div class="shop-carousel11 <?php echo $settings['double_row'] == 'yes' ? 'is-double-row' : '' ?>">
                <div class="swiper" data-slider='<?php echo json_encode($slider_data) ?>'>
                    <div class="swiper-wrapper products-list">
                        <?php foreach ($products as $items): ?>
                            <div class="swiper-slide">
                                <?php
                                foreach($items as $post):
                                    $product = wc_get_product($post->ID);
                                    if (!is_object($product))
                                        continue;

                                    $sale_days_progress = $this->get_product_sale_progress_percent($product->get_id());
                                    ?>
                                    <article <?php wc_product_class('ah-has-bg', $product) ?>>
                                        <div class="product-cover">
                                            <?php if ($settings['show_badge'] === 'yes') : ?>
                                                <div class="product-label">
                                                    <?php
                                                    if (!$product->is_in_stock()) :
                                                        echo apply_filters('woocommerce_product_is_in_stock', '<span class="soldout">' . esc_html__('Sold Out!', 'ahura') . '</span>', $product, $product);
                                                    elseif ($product->is_on_sale()) :
                                                        echo apply_filters('woocommerce_sale_flash', '<span class="sale">' . esc_html__('On Sale', 'ahura') . '</span>', $product, $product);
                                                    endif;
                                                    ?>
                                                </div>
                                            <?php endif; ?>
                                            <a href="<?php echo get_permalink($product->get_id()); ?>">
                                                <?php echo wp_get_attachment_image(get_post_thumbnail_id($product->get_id()), $settings['item_cover_size_size']); ?>
                                            </a>
                                        </div>
                                        <?php if ($settings['show_title'] === 'yes') : ?>
                                            <div class="title-wrap ah-has-text">
                                                <a href="<?php echo get_the_permalink($post->ID) ?>">
                                                    <<?php echo $settings['title_html_tag'] ?> class="p-title">
                                                        <?php echo $product->get_title(); ?>
                                                    </<?php echo $settings['title_html_tag'] ?>>
                                                </a>
                                            </div>
                                        <?php endif; ?>
                                        <?php if ($settings['show_price'] === 'yes') : ?>
                                            <div class="price-wrap ah-has-text">
                                                <?php echo wc_price($product->get_price()); ?>
                                            </div>
                                        <?php endif; ?>
                                        <?php if ($show_cart_btn || $sale_days_progress) : ?>
                                            <div class="product-foot">
                                                <?php if ($show_cart_btn): ?>
                                                <a href="<?php echo get_permalink($product->get_id()); ?>?add-to-cart=<?php echo esc_attr($product->get_id()) ?>" class="mw_add_to_cart">
                                                    <?php echo $settings['btn_text'] ?>
                                                </a>
                                                <?php endif; ?>
                                                <?php if($sale_days_progress): ?>
                                                    <div class="sale-progress">
                                                        <div class="percent" style="width:<?php echo $sale_days_progress ?>%"></div>
                                                    </div>
                                                <?php endif; ?>
                                            </div>
                                        <?php endif; ?>
                                    </article>
                                <?php endforeach; ?>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php if ($show_arrows): ?>
                    <div class="swiper-nav-button swiper-button-next"></div>
                    <div class="swiper-nav-button swiper-button-prev"></div>
                <?php endif; ?>
            </div>
        <?php else : ?>
            <div class="mw_element_error">
                <?php echo esc_html__('Sorry, no products were found for display.', 'ahura'); ?>
            </div>
        <?php
        endif;
    }
}
