<?php

namespace ahura\inc\widgets;

// Block direct access to the main theme file.
defined('ABSPATH') or die('No script kiddies please!');

use Elementor\Controls_Manager;
use ahura\app\mw_assets;
class product_intro extends \ahura\app\elementor\Elementor_Widget_Base
{
    use \ahura\app\traits\WoocommerceMethods;
	function __construct($data = [], $args = null)
    {
        parent::__construct($data, $args);
        mw_assets::register_script('product_intro_js', mw_assets::get_js('elementor.product_intro'));
    }

    public function __register_assets()
    {
        $this->register_elementor_style('product_intro_css', mw_assets::get_css('elementor.product_intro'));
    }

	public function get_script_depends()
    {
        return [mw_assets::get_handle_name('product_intro_js')];
    }

    public function get_name()
    {
        return 'product_intro';
    }

    public function get_title()
    {
        return esc_html__('Product intro', 'ahura');
    }

	public function get_icon() {
		return 'eicon-product-description';
	}

    public function get_categories()
    {
        return ['ahuraelements','ahura_woocommerce'];
    }

    public function get_keywords()
    {
        return ['ahura', 'Product intro', 'product', esc_html__('ahura', 'ahura'), esc_html__('product', 'ahura'), esc_html__('Product intro', 'ahura')];
    }

    public function register_controls()
    {
        $this->start_controls_section(
            'content',
            [
                'label' => esc_html__('Content', 'ahura'),
                'tab' => Controls_Manager::TAB_CONTENT
            ]
        );

        $options = [];

        $products = $this->get_products_array();

        if($products){
            foreach($products as $product) {
                $options[$product['ID']] = $product['post_title'];
            }
        }

        $default = ($options) ? key($options) : 0;

        $this->add_control(
            'product',
            [
                'label' => esc_html__('Product', 'ahura'),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => $options,
                'default' => $default
            ]
        );

        $this->add_control(
			'show_price',
			[
				'label' => esc_html__( 'Show price', 'ahura' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'ahura' ),
				'label_off' => esc_html__( 'Hide', 'ahura' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

        $this->add_control(
			'show_description',
			[
				'label' => esc_html__( 'Show description', 'ahura' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'ahura' ),
				'label_off' => esc_html__( 'Hide', 'ahura' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

        $this->add_control(
			'show_image',
			[
				'label' => esc_html__( 'Show image', 'ahura' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'ahura' ),
				'label_off' => esc_html__( 'Hide', 'ahura' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

        $this->add_group_control(
            \Elementor\Group_Control_Image_Size::get_type(),
            [
                'name' => 'product_size',
                'default' => 'shop_catalog',
                'default' => 'full',
            ]
        );

        $this->add_control(
			'description_words',
			[
				'label' => esc_html__( 'Description words', 'ahura' ),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 5,
				'max' => 1000,
				'step' => 1,
				'default' => 44,
			]
		);

        $this->add_control(
			'custom_description',
			[
				'label' => esc_html__( 'Show custom description', 'ahura' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'ahura' ),
				'label_off' => esc_html__( 'Hide', 'ahura' ),
				'return_value' => 'yes',
				'default' => 'no',
                'condition' => ['show_description' => 'yes']
			]
		);

        $this->add_control(
			'content_description',
			[
				'label' => esc_html__( 'Description', 'ahura' ),
				'type' => \Elementor\Controls_Manager::TEXTAREA,
				'rows' => 10,
				'default' => esc_html__( 'Default description', 'ahura' ),
                'condition' => ['custom_description' => 'yes']
			]
		);

        $this->add_control(
			'show_tips',
			[
				'label' => esc_html__( 'Show tip', 'ahura' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'ahura' ),
				'label_off' => esc_html__( 'Hide', 'ahura' ),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$this->add_control(
			'tips',
			[
				'label' => esc_html__( 'Tip', 'ahura' ),
				'type' => \Elementor\Controls_Manager::TEXT,
                'condition' => ['show_tips' => 'yes']
			]
		);

        $this->add_control(
			'show_custom_details',
			[
				'label' => esc_html__( 'Show custom details', 'ahura' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'ahura' ),
				'label_off' => esc_html__( 'Hide', 'ahura' ),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$this->add_control(
			'custom_details_title',
			[
				'label' => esc_html__( 'Custom details title', 'ahura' ),
				'type' => \Elementor\Controls_Manager::TEXT,
                'condition' => ['show_custom_details' => 'yes']
			]
		);

		$this->add_control(
			'custom_details',
			[
				'label' => esc_html__( 'Custom details', 'ahura' ),
				'type' => \Elementor\Controls_Manager::WYSIWYG,
                'condition' => ['show_custom_details' => 'yes']
			]
		);

		$this->add_control(
			'unavailable_product_text',
			[
				'label' => esc_html__( 'Unavailable product text', 'ahura' ),
				'type' => \Elementor\Controls_Manager::TEXT,
			]
		);

        $this->end_controls_section();

        $this->start_controls_section(
            'product_style',
            [
                'label' => esc_html__('Product', 'ahura'),
                'tab' => Controls_Manager::TAB_STYLE
            ]
        );

        $content_alignment_default = 'row';
        is_rtl() ? $content_alignment_default = 'row' : $content_alignment_default = 'row-reverse';

        $this->add_responsive_control(
			'content_alignment',
			[
				'label' => esc_html__( 'Content alignment', 'ahura' ),
				'type' => \Elementor\Controls_Manager::CHOOSE,
				'options' => [
					'row-reverse' => [
						'title' => esc_html__( 'Right', 'ahura' ),
						'icon' => 'eicon-arrow-right',
					],
					'column-reverse' => [
						'title' => esc_html__( 'Top', 'ahura' ),
						'icon' => 'eicon-arrow-up',
					],
					'row' => [
						'title' => esc_html__( 'Left', 'ahura' ),
						'icon' => 'eicon-arrow-left',
					],
					'column' => [
						'title' => esc_html__( 'Down', 'ahura' ),
						'icon' => 'eicon-arrow-down',
					],
				],
				'default' => $content_alignment_default,
				'toggle' => true,
				'selectors' => [
					'{{WRAPPER}} .product-intro-content' => 'flex-direction: {{VALUE}};',
				],
			]
		);

        $this->add_responsive_control(
			'images_width',
			[
				'label' => esc_html__( 'Image Width', 'ahura' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => [ '%' ],
				'range' => [
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => '%',
					'size' => 50,
				],
				'selectors' => [
					'{{WRAPPER}} .images-wrapper' => 'width: {{SIZE}}%;',
				],
			]
		);

        $this->add_responsive_control(
			'content_width',
			[
				'label' => esc_html__( 'Content Width', 'ahura' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => [ '%' ],
				'range' => [
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => '%',
					'size' => 50,
				],
				'selectors' => [
					'{{WRAPPER}} .intro-content-container' => 'width: {{SIZE}}%;',
				],
                'condition' => ['show_image' => 'yes']
			]
		);

		$this->add_responsive_control(
			'content_padding',
			[
				'label' => esc_html__( 'Content padding', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem' ],
				'default' => [
					'top' => 10,
					'right' => 10,
					'bottom' => 10,
					'left' => 10,
					'unit' => 'px',
					'isLinked' => true,
				],
				'selectors' => [
					'{{WRAPPER}} .intro-content-container' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'content_margin',
			[
				'label' => esc_html__( 'Content margin', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem' ],
				'default' => [
					'top' => 0,
					'right' => is_rtl() ? 10 : 0,
					'bottom' => 0,
					'left' => is_rtl() ? 0 : 10,
					'unit' => 'px',
					'isLinked' => true,
				],
				'selectors' => [
					'{{WRAPPER}} .intro-content-container' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'images_padding',
			[
				'label' => esc_html__( 'Images padding', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem' ],
				'selectors' => [
					'{{WRAPPER}} .images-wrapper' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'images_margin',
			[
				'label' => esc_html__( 'Images margin', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem' ],
				'default' => [
					'top' => 0,
					'right' => is_rtl() ? 20 : 0,
					'bottom' => 0,
					'left' => is_rtl() ? 0 : 20,
					'unit' => 'px',
					'isLinked' => true,
				],
				'selectors' => [
					'{{WRAPPER}} .images-wrapper' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->end_controls_section();

        $this->start_controls_section(
            'breadcrumbs_style',
            [
                'label' => esc_html__('Breadcrumbs', 'ahura'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'bc_typography',
				'selector' => '{{WRAPPER}} .product-intro-breadcrumb a',
			]
		);

		$this->add_control(
			'bc_color',
			[
				'label' => esc_html__( 'Color', 'ahura' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .product-intro-breadcrumb a' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'bc_svg_margin',
			[
				'label' => esc_html__( 'SVG margin', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem' ],
				'default' => [
					'top' => 0,
					'right' => 5,
					'bottom' => 0,
					'left' => 5,
					'unit' => 'px',
					'isLinked' => true,
				],
				'selectors' => [
					'{{WRAPPER}} .product-intro-breadcrumb svg' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->end_controls_section();

        $this->start_controls_section(
            'thumbnails_style',
            [
                'label' => esc_html__('Thumbnails', 'ahura'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

		$this->add_control(
			'thumbnails_width',
			[
				'label' => esc_html__( 'Width', 'ahura' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'em', 'rem' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 100,
				],
				'selectors' => [
					'{{WRAPPER}} .images-wrapper .gallery-img' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'thumbnails_margin',
			[
				'label' => esc_html__( 'Margin', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem' ],
				'default' => [
					'top' => 0,
					'right' => is_rtl() ? 10 : 0,
					'bottom' => 0,
					'left' => is_rtl() ? 0 : 10,
					'unit' => 'px',
					'isLinked' => true,
				],
				'selectors' => [
					'{{WRAPPER}} .images-wrapper .product-gallery' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->end_controls_section();

        $this->start_controls_section(
            'image_style',
            [
                'label' => esc_html__('Image', 'ahura'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
			'image_border_radius',
			[
				'label' => esc_html__( 'Border radius', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'default' => [
					'top' => 0,
					'right' => 0,
					'bottom' => 0,
					'left' => 0,
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} .img-container img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->add_responsive_control(
			'image_padding',
			[
				'label' => esc_html__( 'Padding', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'default' => [
					'top' => 0,
					'right' => 0,
					'bottom' => 0,
					'left' => 0,
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} .img-container img' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->add_responsive_control(
			'image_margin',
			[
				'label' => esc_html__( 'Margin', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'default' => [
					'top' => 0,
					'right' => 0,
					'bottom' => 0,
					'left' => 0,
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} .img-container img' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->add_control(
			'show_image_box_shadow',
			[
				'label' => esc_html__( 'Show box shadow', 'ahura' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'ahura' ),
				'label_off' => esc_html__( 'Hide', 'ahura' ),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

        $this->add_control(
			'image_box_shadow',
			[
				'label' => esc_html__( 'Image Box shadow', 'ahura' ),
				'type' => \Elementor\Controls_Manager::BOX_SHADOW,
				'selectors' => [
					'{{SELECTOR}} .img-container img' => 'box-shadow: {{HORIZONTAL}}px {{VERTICAL}}px {{BLUR}}px {{SPREAD}}px {{COLOR}};',
				],
                'condition' => ['show_image_box_shadow' => 'yes']
			]
		);

        $this->end_controls_section();

        $this->start_controls_section(
            'title_style',
            [
                'label' => esc_html__('Title', 'ahura'),
                'tab' => Controls_Manager::TAB_STYLE
            ]
        );

        $title_alignment_default = 'right';
        is_rtl() ? $title_alignment_default = 'right' : $title_alignment_default = 'left';

        $this->add_responsive_control(
			'title_alignment',
			[
				'label' => esc_html__( 'Alignment', 'ahura' ),
				'type' => \Elementor\Controls_Manager::CHOOSE,
				'options' => [
					'right' => [
						'title' => esc_html__( 'Right', 'ahura' ),
						'icon' => 'eicon-text-align-right',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'ahura' ),
						'icon' => 'eicon-text-align-center',
					],
					'left' => [
						'title' => esc_html__( 'Left', 'ahura' ),
						'icon' => 'eicon-text-align-left',
					],
				],
				'default' => $title_alignment_default,
				'toggle' => true,
				'selectors' => [
					'{{WRAPPER}} .intro-title' => 'text-align: {{VALUE}};',
				],
			]
		);

        $this->add_control(
			'title_color',
			[
				'label' => esc_html__( 'Title color', 'ahura' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .intro-title' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'title_typography',
				'selector' => '{{WRAPPER}} .intro-title',
			]
		);

        $this->add_responsive_control(
			'title_padding',
			[
				'label' => esc_html__( 'Padding', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'default' => [
					'top' => 0,
					'right' => 0,
					'bottom' => 0,
					'left' => 0,
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} .intro-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->add_responsive_control(
			'title_margin',
			[
				'label' => esc_html__( 'Margin', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'default' => [
					'top' => 0,
					'right' => 0,
					'bottom' => 15,
					'left' => 0,
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} .intro-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->end_controls_section();

        $this->start_controls_section(
            'price_style',
            [
                'label' => esc_html__('Price', 'ahura'),
                'tab' => Controls_Manager::TAB_STYLE
            ]
        );

        $price_alignment_default = 'right';
        is_rtl() ? $price_alignment_default = 'right' : $price_alignment_default = 'left';

        $this->add_control(
			'price_alignment',
			[
				'label' => esc_html__( 'Alignment', 'ahura' ),
				'type' => \Elementor\Controls_Manager::CHOOSE,
				'options' => [
					'right' => [
						'title' => esc_html__( 'Right', 'ahura' ),
						'icon' => 'eicon-text-align-right',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'ahura' ),
						'icon' => 'eicon-text-align-center',
					],
					'left' => [
						'title' => esc_html__( 'Left', 'ahura' ),
						'icon' => 'eicon-text-align-left',
					],
				],
				'default' => $price_alignment_default,
				'toggle' => true,
				'selectors' => [
					'{{WRAPPER}} .intro-price' => 'text-align: {{VALUE}};',
				],
			]
		);

        $this->add_control(
			'price_color',
			[
				'label' => esc_html__( 'Title color', 'ahura' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .intro-price' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'price_typography',
				'selector' => '{{WRAPPER}} .intro-price',
                'fields_options' =>
				[
                    'typography' => [
                        'default' => 'yes'
                    ],
					'font_size' => [
						'default' => [
							'unit' => 'px',
							'size' => '17'
						]
                    ],
                    'font_weight' => [
                        'default' => 'bold'
                    ]
				]
			]
		);

        $this->add_responsive_control(
			'price_margin',
			[
				'label' => esc_html__( 'Margin', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'default' => [
					'top' => 0,
					'right' => 0,
					'bottom' => 20,
					'left' => 0,
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} .intro-price' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->end_controls_section();

        $this->start_controls_section(
            'rating_style',
            [
                'label' => esc_html__('Ratings', 'ahura'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

		$this->add_control(
			'rating_general_options',
			[
				'label' => esc_html__( 'General', 'ahura' ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

        $this->add_responsive_control(
			'rating_margin',
			[
				'label' => esc_html__( 'Margin', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'default' => [
					'top' => 0,
					'right' => 0,
					'bottom' => 20,
					'left' => 0,
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} .product-rating' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'rating_star_options',
			[
				'label' => esc_html__( 'Star', 'ahura' ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'star_color',
			[
				'label' => esc_html__( 'Color', 'ahura' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'default' => '#E0900E',
				'selectors' => [
					'{{WRAPPER}} .star-rating i' => 'color: {{VALUE}}',
				],
			]
		);

        $this->add_responsive_control(
			'star_margin',
			[
				'label' => esc_html__( 'Margin', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'default' => [
					'top' => 1,
					'right' => 1,
					'bottom' => 1,
					'left' => 1,
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} .star-rating i' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'rating_text_options',
			[
				'label' => esc_html__( 'Texts', 'ahura' ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'rating_text_typography',
				'selector' => '{{WRAPPER}} .rating-text',
			]
		);

		$this->add_control(
			'rating_text_color',
			[
				'label' => esc_html__( 'Color', 'ahura' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rating-text' => 'color: {{VALUE}}',
				],
			]
		);

        $this->add_responsive_control(
			'rating_text_margin',
			[
				'label' => esc_html__( 'Margin', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'default' => [
					'top' => 0,
					'right' => 8,
					'bottom' => 0,
					'left' => 8,
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} .rating-text' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'rating_dot_options',
			[
				'label' => esc_html__( 'Dot', 'ahura' ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_responsive_control(
			'dot_width',
			[
				'label' => esc_html__( 'Width', 'ahura' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'em', 'rem' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 5,
				],
				'selectors' => [
					'{{WRAPPER}} .dot' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
			]
		);

        $this->add_control(
			'dot_color',
			[
				'label' => esc_html__( 'Color', 'ahura' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'default' => '#333333',
				'selectors' => [
					'{{WRAPPER}} .dot' => 'background-color: {{VALUE}}',
				],
			]
		);

        $this->add_responsive_control(
			'dot_radius',
			[
				'label' => esc_html__( 'Border radius', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'default' => [
					'top' => 100,
					'right' => 100,
					'bottom' => 100,
					'left' => 100,
					'unit' => '%',
				],
				'selectors' => [
					'{{WRAPPER}} .dot' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->add_responsive_control(
			'dot_margin',
			[
				'label' => esc_html__( 'Margin', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'default' => [
					'top' => 0,
					'right' => is_rtl() ? 0 : 7,
					'bottom' => 0,
					'left' => is_rtl() ? 7 : 0,
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} .dot' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->end_controls_section();

        $this->start_controls_section(
            'description_style',
            [
                'label' => esc_html__('Description', 'ahura'),
                'tab' => Controls_Manager::TAB_STYLE
            ]
        );

        $description_alignment_default = 'right';
        is_rtl() ? $description_alignment_default = 'right' : $description_alignment_default = 'left';

        $this->add_control(
			'description_alignment',
			[
				'label' => esc_html__( 'Alignment', 'ahura' ),
				'type' => \Elementor\Controls_Manager::CHOOSE,
				'options' => [
					'right' => [
						'title' => esc_html__( 'Right', 'ahura' ),
						'icon' => 'eicon-text-align-right',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'ahura' ),
						'icon' => 'eicon-text-align-center',
					],
					'left' => [
						'title' => esc_html__( 'Left', 'ahura' ),
						'icon' => 'eicon-text-align-left',
					],
				],
				'default' => $description_alignment_default,
				'toggle' => true,
				'selectors' => [
					'{{WRAPPER}} .intro-description' => 'text-align: {{VALUE}};',
				],
			]
		);

        $this->add_control(
			'description_color',
			[
				'label' => esc_html__( 'Title color', 'ahura' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .intro-description' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'description_typography',
				'selector' => '{{WRAPPER}} .intro-description',
			]
		);

        $this->add_responsive_control(
			'description_padding',
			[
				'label' => esc_html__( 'Padding', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'default' => [
					'top' => 0,
					'right' => 0,
					'bottom' => 0,
					'left' => 0,
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} .intro-description' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->add_responsive_control(
			'description_margin',
			[
				'label' => esc_html__( 'Margin', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'default' => [
					'top' => 0,
					'right' => 0,
					'bottom' => 40,
					'left' => 0,
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} .intro-description' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->end_controls_section();

        $this->start_controls_section(
            'button_style',
            [
                'label' => esc_html__('Button', 'ahura'),
                'tab' => Controls_Manager::TAB_STYLE
            ]
        );

        $button_alignment_default = 'right';
        is_rtl() ? $button_alignment_default = 'right' : $button_alignment_default = 'left';

        $this->add_responsive_control(
			'button_alignment',
			[
				'label' => esc_html__( 'Alignment', 'ahura' ),
				'type' => \Elementor\Controls_Manager::CHOOSE,
				'options' => [
					'right' => [
						'title' => esc_html__( 'Right', 'ahura' ),
						'icon' => 'eicon-text-align-right',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'ahura' ),
						'icon' => 'eicon-text-align-center',
					],
					'left' => [
						'title' => esc_html__( 'Left', 'ahura' ),
						'icon' => 'eicon-text-align-left',
					],
				],
				'default' => $button_alignment_default,
				'toggle' => true,
				'selectors' => [
					'{{WRAPPER}} .add-to-cart-wrap' => 'text-align: {{VALUE}};',
				],
			]
		);

        $this->add_control(
			'button_color',
			[
				'label' => esc_html__( 'Button color', 'ahura' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} button[type="submit"]' => 'color: {{VALUE}}',
				],
                'default' => '#ffffff'
			]
		);

        $this->add_control(
			'button_backcolor',
			[
				'label' => esc_html__( 'Button background color', 'ahura' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} button[type="submit"]' => 'background-color: {{VALUE}}',
				],
                'default' => '#354ac4'
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'button_typography',
				'selector' => '{{WRAPPER}} button[type="submit"]',
			]
		);

        $this->add_responsive_control(
			'button_padding',
			[
				'label' => esc_html__( 'Padding', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'default' => [
					'top' => 10,
					'right' => 10,
					'bottom' => 10,
					'left' => 10,
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} button[type="submit"]' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->add_responsive_control(
			'button_margin',
			[
				'label' => esc_html__( 'Margin', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'default' => [
					'top' => 20,
					'right' => 0,
					'bottom' => 0,
					'left' => 0,
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} button[type="submit"]' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->add_control(
			'button_radius',
			[
				'label' => esc_html__( 'Border radius', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'default' => [
					'top' => 5,
					'right' => 5,
					'bottom' => 5,
					'left' => 5,
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} button[type="submit"]' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->end_controls_section();

        $this->start_controls_section(
            'tips_style',
            [
                'label' => esc_html__('Tip', 'ahura'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
			'tip_color',
			[
				'label' => esc_html__( 'Color', 'ahura' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .tip-text' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'tip_typography',
				'selector' => '{{WRAPPER}} .tip-text',
			]
		);

        $this->add_responsive_control(
			'tip_margin',
			[
				'label' => esc_html__( 'Margin', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'default' => [
					'top' => 20,
					'right' => 0,
					'bottom' => 0,
					'left' => 0,
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} .tip-text' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->end_controls_section();

        $this->start_controls_section(
            'details_style',
            [
                'label' => esc_html__('Details', 'ahura'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

		$this->add_control(
			'details_container_options',
			[
				'label' => esc_html__( 'Title', 'ahura' ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

        $this->add_control(
			'details_border_color',
			[
				'label' => esc_html__( 'Top border color', 'ahura' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'default' => '#E0900E',
				'selectors' => [
					'{{WRAPPER}} .product-details-container' => 'border-top: 1px solid {{VALUE}}',
				],
			]
		);

        $this->add_responsive_control(
			'details_padding',
			[
				'label' => esc_html__( 'Padding', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'default' => [
					'top' => 40,
					'right' => 0,
					'bottom' => 0,
					'left' => 0,
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} .product-details-container' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->add_responsive_control(
			'details_margin',
			[
				'label' => esc_html__( 'Margin', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'default' => [
					'top' => 30,
					'right' => 0,
					'bottom' => 0,
					'left' => 0,
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} .product-details-container' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'details_title_options',
			[
				'label' => esc_html__( 'Title', 'ahura' ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

        $this->add_control(
			'details_title_color',
			[
				'label' => esc_html__( 'Color', 'ahura' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .product-details-title span' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'details_title_typography',
				'selector' => '{{WRAPPER}} .product-details-title span',
			]
		);

		$this->add_control(
			'details_desc_options',
			[
				'label' => esc_html__( 'Description', 'ahura' ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

        $this->add_control(
			'details_desc_color',
			[
				'label' => esc_html__( 'Color', 'ahura' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .product-custom-details' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'details_desc_typography',
				'selector' => '{{WRAPPER}} .product-custom-details',
			]
		);

        $this->add_responsive_control(
			'details_desc_margin',
			[
				'label' => esc_html__( 'Margin', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'default' => [
					'top' => 20,
					'right' => 0,
					'bottom' => 0,
					'left' => 0,
					'unit' => 'px',
				],
				'selectors' => [
					'{{WRAPPER}} .product-custom-details' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->end_controls_section();

        $this->register_additional_controls();
    }

    /**
     *
     * Render content for display
     *
     */
    public function render() {
		$settings = $this->get_settings_for_display();
		$wid = $this->get_id();
		if (!\ahura\app\woocommerce::is_active()) return;
		$selected_product = $settings['product'] ? wc_get_product($settings['product']) : false;
		if (!$selected_product) return;
	
		$attributes = $selected_product->get_attributes();
		$available_variations = $selected_product->is_type('variable') ? $selected_product->get_available_variations() : [];
	
		$variations_data = [];
		foreach ($available_variations as $variation) {
			$variation_obj = wc_get_product($variation['variation_id']);
			$variation_data = [
				'variation_id' => $variation['variation_id'],
				'attributes' => array_values($variation['attributes']),
				'price_html' => $variation_obj->get_price_html(),
				'min_qty' => $variation_obj->get_min_purchase_quantity(),
				'max_qty' => $variation_obj->get_max_purchase_quantity()
			];
			$variations_data[] = $variation_data;
		}
	
		wp_localize_script(mw_assets::get_handle_name('product_intro_js'), 'productVariations_' . $wid, $variations_data);

		?>
		<div class="d-flex flex-column product-intro product-intro-<?php echo $wid ?>" data-widget-id="<?php echo $wid; ?>">
			<div class="d-flex align-items-center product-intro-breadcrumb">
				<a href="<?php echo esc_url(wc_get_page_permalink('shop')); ?>"><?php echo __('Shop', 'ahura'); ?></a>
				<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-chevron-left" viewBox="0 0 16 16">
					<path fill-rule="evenodd" d="M11.354 1.646a.5.5 0 0 1 0 .708L5.707 8l5.647 5.646a.5.5 0 0 1-.708.708l-6-6a.5.5 0 0 1 0-.708l6-6a.5.5 0 0 1 .708 0"/>
				</svg>
				<?php 
				$categories = wp_get_post_terms($selected_product->get_id(), 'product_cat');
				$category = !empty($categories) ? current($categories) : false;
				?>
				<a href="<?php echo esc_url(get_term_link($category)); ?>"><?php echo $category ? $category->name : __('Category', 'ahura'); ?></a>
				<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-chevron-left" viewBox="0 0 16 16">
					<path fill-rule="evenodd" d="M11.354 1.646a.5.5 0 0 1 0 .708L5.707 8l5.647 5.646a.5.5 0 0 1-.708.708l-6-6a.5.5 0 0 1 0-.708l6-6a.5.5 0 0 1 .708 0"/>
				</svg>
				<a href="<?php echo get_permalink($selected_product->get_id()) ? get_permalink($selected_product->get_id()) : '#'; ?>"><?php echo $selected_product->get_name(); ?></a>
			</div>
			<div class="d-flex align-items-start justify-content-center product-intro-content">
				<?php if ($settings['show_image'] == 'yes' && $selected_product->get_image() && $settings['product_size_size']): ?>
					<div class="d-flex images-wrapper">
						<?php
						$attachment_ids = $selected_product->get_gallery_image_ids();
						if ($attachment_ids && $selected_product->get_image_id()) {
							echo '<div class="product-gallery">';
							foreach ($attachment_ids as $attachment_id) {
								echo '<div class="gallery-img">';
								echo apply_filters('woocommerce_single_product_image_thumbnail_html', wc_get_gallery_image_html($attachment_id), $attachment_id);
								echo '</div>';
							}
							echo '</div>';
						}
						?>
						<div class="img-container">
							<?php $thumbnail_image_url = wp_get_attachment_url($selected_product->get_image_id()); ?>
							<a href="<?php echo $thumbnail_image_url; ?>" data-lightbox="product-gallery">
								<?php echo $selected_product->get_image($settings['product_size_size']); ?>
							</a>
						</div>
					</div>
				<?php endif; ?>
				<div class="intro-content-container">
					<div class="d-flex flex-column align-items-start intro-content-header">
						<?php if ($selected_product->get_name()): ?>
							<h2 class="intro-title w-100">
								<a href="<?php echo get_permalink($selected_product->get_id()) ? get_permalink($selected_product->get_id()) : '#'; ?>"><?php echo $selected_product->get_name(); ?></a>
							</h2>
						<?php endif; ?>
						<?php if ($settings['show_price'] == 'yes' && wc_price($selected_product->get_price())): ?>
							<?php if ($selected_product->is_type('variable')): ?>
								<p class="intro-price variation-price w-100">
									<?php echo sprintf(__('From %s to %s', 'ahura'), wc_price($selected_product->get_variation_price('min', true)), wc_price($selected_product->get_variation_price('max', true))); ?>
								</p>
							<?php else: ?>
								<p class="intro-price simple-price w-100">
									<?php echo wc_price($selected_product->get_price()) ?>
								</p>
							<?php endif; ?>
						<?php endif; ?>
						<?php if ($selected_product->get_average_rating()): ?>
							<div class="d-flex align-items-center product-rating w-100">
								<?php
								$average = $selected_product->get_average_rating();
								$rating_count = $selected_product->get_review_count();
								?>
								<div class="star-rating m-0" title="<?php echo sprintf(__('Rated %s out of 5', 'ahura'), $average); ?>">
									<span style="width:<?php echo (($average / 5) * 100); ?>%">
										<?php
										$full_stars = floor($average);
										$empty_stars = 5 - $full_stars;

										for ($i = 0; $i < $full_stars; $i++) {
											echo '<i class="fas fa-star"></i>';
										}
										
										for ($i = 0; $i < $empty_stars; $i++) {
											echo '<i class="far fa-star"></i>';
										}
										?>
									</span>
								</div>
								<span class="rating-text">
									(<?php echo number_format($average, 1) . esc_html__( ' stars', 'ahura' ); ?>)
								</span>
								<span class="d-flex justify-content-center align-items-center flex-row review-count">
									<span class="d-block dot"></span>
									<span class="rating-text"><?php echo $rating_count . esc_html__( ' reviews', 'ahura' ); ?></span>
								</span>
							</div>
						<?php endif; ?>
					</div>
					<?php if ($settings['show_description'] == 'yes'): ?>
						<p class="intro-description">
							<?php if ($settings['custom_description'] == 'yes'): ?>
								<?php echo $settings['content_description']; ?>
							<?php else: ?>
								<?php if ($selected_product->get_description()): ?>
									<?php echo wp_trim_words($selected_product->get_description(), $settings['description_words'], '...'); ?>
								<?php endif; ?>
							<?php endif; ?>
						</p>
					<?php endif; ?>
	
					<?php if ($selected_product && $selected_product->is_in_stock() && $selected_product->is_type('variable')):
						$attributes = $selected_product->get_variation_attributes();
						if (!empty($attributes)): ?>
							<div class="variations">
								<?php foreach ($attributes as $attribute_name => $options): ?>
									<div class="var-item var-item-<?php echo esc_attr(sanitize_title($attribute_name)); ?>">
										<div class="label"><label for="<?php echo esc_attr(sanitize_title($attribute_name)); ?>"><?php echo wc_attribute_label($attribute_name); ?></label></div>
										<div class="value">
											<?php
											wc_dropdown_variation_attribute_options(array(
												'options' => $options,
												'attribute' => $attribute_name,
												'product' => $selected_product,
											));
											?>
										</div>
									</div>
								<?php endforeach; ?>
							</div>
						<?php endif; ?>
						<div class="single_variation_wrap">
							<div class="woocommerce-variation-add-to-cart variations_button">
								<?php if ($selected_product->is_type('variable')): ?>
									<?php
									$variations = $selected_product->get_available_variations();
									if (!empty($variations)): ?>
										<form class="variations_form" method="post" enctype='multipart/form-data'>
											<button type="submit" class="single_add_to_cart_button button alt w-100"><?php echo esc_html($selected_product->single_add_to_cart_text()); ?></button>
											<input type="hidden" name="add-to-cart" value="<?php echo absint($selected_product->get_id()); ?>" />
											<input type="hidden" name="product_id" value="<?php echo absint($selected_product->get_id()); ?>" />
											<input type="hidden" name="variation_id" class="variation_id" value="<?php echo esc_attr($selected_product->get_id()); ?>" />
										</form>
									<?php else: ?>
										<p><?php echo __('Out of stock', 'ahura'); ?></p>
									<?php endif; ?>
								<?php endif; ?>
							</div>
						</div>
					<?php elseif($selected_product && $selected_product->is_in_stock() && $selected_product->is_type('simple')): ?>
						<!-- Simple product add to cart section -->
						<div class="simple_product_wrap">
							<div class="woocommerce-add-to-cart simple_product_button">
								<form class="cart" method="post" enctype='multipart/form-data'>
									<div class="quantity">
										<label class="screen-reader-text" for="quantity"><?php echo __('Quantity', 'ahura'); ?></label>
										<input type="number" id="quantity" class="input-text qty text" step="1" min="<?php echo esc_attr($selected_product->get_min_purchase_quantity()); ?>" max="<?php echo esc_attr($selected_product->get_max_purchase_quantity()); ?>" name="quantity" value="1" title="<?php echo __('Qty', 'ahura'); ?>" size="4" pattern="[0-9]*" inputmode="numeric" />
									</div>
									<button type="submit" class="single_add_to_cart_button button alt w-100"><?php echo esc_html($selected_product->single_add_to_cart_text()); ?></button>
									<input type="hidden" name="add-to-cart" value="<?php echo absint($selected_product->get_id()); ?>" />
								</form>
							</div>
						</div>
					<?php elseif($selected_product && !$selected_product->is_in_stock()): ?>
						<button type="submit" class="single_add_to_cart_button button alt w-100"><?php echo $settings['unavailable_product_text'] ? $settings['unavailable_product_text'] : esc_html__( 'Product is unavailable', 'ahura' ); ?></button>
					<?php endif; ?>
					<?php if($settings['show_tips'] === 'yes' && !empty($settings['tips'])): ?>
						<p class="tip-text"><?php echo $settings['tips']; ?></p>
					<?php endif; ?>
					<?php if(!empty($settings['custom_details']) && !empty($settings['custom_details_title'])): ?>
						<div class="product-details-container">
							<div class="d-flex justify-content-between align-items-center product-details-title">
								<span><?php echo $settings['custom_details_title']; ?></span>
								<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-chevron-down" viewBox="0 0 16 16">
									<path fill-rule="evenodd" d="M1.646 4.646a.5.5 0 0 1 .708 0L8 10.293l5.646-5.647a.5.5 0 0 1 .708.708l-6 6a.5.5 0 0 1-.708 0l-6-6a.5.5 0 0 1 0-.708"/>
								</svg>
							</div>
							<div class="product-custom-details">
								<?php echo $settings['custom_details']; ?>
							</div>
						</div>
					<?php endif; ?>
				</div>
			</div>
		</div>
		<?php
	}
	
	
}