<?php

namespace ahura\app;
class license
{
    const LICENSE_EXPIRATION_DATE_OPTION_NAME = 'ahura_license_expiration_date';
    const LICENSE_REMAINING_DAYS_TRANSIENT_OPTION_NAME = 'ahura_license_remaining_days';

    private static $_license_status_option_name = 'ahura_license_key_status';
    private static $_license_key_option_name = 'ahura_license_key';

    // private static $_base_api_server = 'https://mihanwp.com/api/v2/';

    public static function get_license_page_url()
    {
        return admin_url('admin.php?page=ahura-license');
    }

    public static function get_wizard_license_page_url()
    {
        return admin_url('admin.php?page=ahura-wizard&step=license');
    }

    private static function getBaseApiServer($version = 'v2')
    {
        $license = self::get_license_key();
        $server = mw_tools::getRemoteServerByLicenseKey($license);
        return $server . 'api/' . $version . '/';
    }

    private static function getBaseHost()
    {
        $url = self::getBaseApiServer();
        return parse_url($url, PHP_URL_HOST);
    }

    static function init()
    {
        if (!self::is_active()) {
            self::show_license_messages();
        }
        if (defined('AHURA_LICENSE_KEY')) {
            add_action('admin_init', [__CLASS__, 'handle_once_check_license']);
            add_action('switch_theme', [__CLASS__, 'afterDeactivateTheme']);
            add_action('after_switch_theme', [__CLASS__, 'afterActivateTheme']);
            return;
        }
        add_filter('merlin_ajax_activate_license', [__CLASS__, 'handle_license_in_merlin']);
    }

    public static function handle_once_check_license()
    {
        if (!self::is_active()) {
            self::afterActivateTheme();
        }
    }

    static function afterActivateTheme()
    {
        $result = self::activate_license_on_server();
        if ($result) {
            self::activate_license_in_local();
        }
    }

    static function afterDeactivateTheme()
    {
        $result = self::deactivate_license_on_server();
        if (is_array($result) && isset($result['status']) && $result['status']) {
            self::deactivate_license_in_local();
        }
    }

    static function handle_license_in_merlin($license_key)
    {
        $result['status'] = self::activate_license_on_server($license_key);
        if ($result['status']) {
            self::activate_license_in_local();
            $result['msg'] = esc_html__('Successfully activated.', 'ahura');
        } else {
            $result['msg'] = esc_html__('Has error in license activation', 'ahura');
        }
        return $result;
    }

    public static function handle_license_key_checker_from_db()
    {
        if(Studio::is_studio() && empty(self::get_license_key_from_db())){
            self::deactivate_license_on_server();
        }
    }

    public static function handle_activation_after_return_from_provider()
    {
        self::failedActivation();

        if (!isset($_GET['license_activation_status']) || $_GET['license_activation_status'] != 200)
            return false;

        $licenseKey = !empty($_GET['key']) ? sanitize_text_field($_GET['key']) : null;
        if($licenseKey)
        {
            self::save_license_key($licenseKey);
        }

        $serverLicenseStatus = self::check_license_status();
        $status = $serverLicenseStatus && isset($serverLicenseStatus['status']) && $serverLicenseStatus['status'];

        if ($status && $licenseKey) {
            self::getRemainingDays();
            self::activate_license_in_local();
            self::successActivation();
            // self::save_license_key($licenseKey);
            Post_Tag_Filter::getAllFilters();
            return true;
        } else {
            self::deactivate_license_in_local();
            self::failedActivation();
            return false;
        }
    }

    static function license_menu_c()
    {
        self::handle_activation_after_return_from_provider();
        $status = self::is_active();
        $host = self::getBaseHost();
        $isSuccessActivation = self::isSuccessActivation();
        ?>
        <div class="wrap">
            <div class="ah_license_menu_content_wrapper">
                <?php if ($isSuccessActivation): ?>
                    <div class="ah-license-notice success">
                        <?php _e('Ahura theme license activation was successful.', 'ahura') ?>
                    </div>
                <?php endif; ?>
                <div class="header-section">
                    <div class="onboarding-image">
                        <img src="<?php echo \ahura\app\mw_assets::get_img('license-page') ?>" alt="onboarding-image">
                    </div>
                    <div class="title-section"><?php _e('Welcome to license management', 'ahura') ?></div>
                </div>
                <div class="body-section">
                    <?php if ($status): ?>
                        <p class="description"><?php echo sprintf(__('Your site is connected to %s. If you want to disconnect, you can use the button below.', 'ahura'), $host) ?></p>
                        <div class="action-btn-wrapper">
                            <a target="_blank" href="#" class="ah-license-action-btn action-btn danger-mode"><?php _e('DeActivate license please!', 'ahura') ?></a>
                        </div>
                    <?php else: ?>
                        <p class="description"><?php echo sprintf(__('Could it get any easier than this? Just click the button below, and we’ll go to the %s site to activate the license on your website together.', 'ahura'), $host) ?></p>
                        <div class="action-btn-wrapper">
                            <a href="<?php echo esc_url(\ahura\app\license::get_smart_activation_url()); ?>" class="action-btn primary-mode"><?php _e('Let`s go for activation!', 'ahura') ?></a>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php
    }

    public static function render_license_notices()
    {
        $license = self::get_license_key();

        if (ahura_str_contains($license, 'ertano')) {
            $vals = '*.ertano.com,ertano.com';
        } else {
            $vals = '*.mihanwp.com,mihanwp.com';
        }

        if ($license && ahura_is_blocked_developer_http_requests($license)): ?>
            <div class="license-error"><?php echo sprintf(__('HTTP requests are blocked, the values (%s) must be added to (%s) in the wordpress config file.', 'ahura'), $vals, 'WP_ACCESSIBLE_HOSTS') ?></div>
        <?php endif;
    }

    static function setLicenseExpirationDate($value)
    {
        return update_option(self::LICENSE_EXPIRATION_DATE_OPTION_NAME, $value);
    }

    static function check_license_status()
    {
        do_action('ahura_start_check_license_status_schedule');
        $license = self::get_license_key();

        $apiUrl = self::getBaseApiServer('v3') . $license . '/license/check_license';
        $res = [
            'status' => false,
            'msg' => '',
        ];
        $args = [
            'body' => [
                'product_id' => mw_tools::getRemoteProductId($license),
                'domain' => get_site_url(),
            ],
            'timeout' => 120,
        ];
        $remote = wp_remote_get($apiUrl, $args);
        if (is_wp_error($remote)) {
            $res['msg'] = !empty($remote->get_error_message()) ? $remote->get_error_message() : esc_html__('An error occurred!', 'ahura');
            return $res;
        }

        $body = wp_remote_retrieve_body($remote);
        $remoteResponse = ahura_is_json($body) ? json_decode($body) : null;
        if (!is_object($remoteResponse))
            return false;

        if (isset($remoteResponse->code) && $remoteResponse->code != 200) {
            $res['msg'] = $remoteResponse->message;
            return $res;
        }

        $res['status'] = true;
        $res['msg'] = isset($remoteResponse->message) ? $remoteResponse->message : null;
        $expire_date = isset($remoteResponse->resid_expiration_date) ? $remoteResponse->resid_expiration_date : null;
        $res['expiration_date'] = $expire_date;
        if ($expire_date) {
            self::setLicenseExpirationDate($remoteResponse->resid_expiration_date);
        }
        return $res;
    }

    static function check_license()
    {
        if (!self::is_active()) {
            $res = self::check_license_status();
            return $res && isset($res['status']) && $res['status'];
        }
        self::activate_license_in_local();
        return true;
    }

    static function check_license_cron_job()
    {
        if(self::is_active())
        {
            try {
                $license_key = self::get_license_key();
                if(!$license_key)
                {
                    self::deactivate_license_in_local();
                    return false;
                }
                $from_server = self::check_license_status();
                if(!is_array($from_server) || !isset($from_server['status']) || !$from_server['status'])
                {
                    self::deactivate_license_in_local();
                    return false;
                }
            } catch (\Throwable $th){
                return false;
            }
        }
    }

    static function deleteLicenseCookieData()
    {
        if (isset($_COOKIE['ahura_dismiss_license_admin_notice'])){
            $domain = self::getHostFromSiteUrl();
            setcookie('ahura_dismiss_license_admin_notice', false, time() - 3600, '/', $domain, true, true);
            unset($_COOKIE['ahura_dismiss_license_admin_notice']);
        }
    }

    static function activate_license_in_local()
    {
        return update_option(self::$_license_status_option_name, 'valid');
    }

    static function deactivate_license_in_local()
    {
        self::deleteLicenseCookieData();
        return delete_option(self::$_license_status_option_name);
    }

    static function activate_license_on_server($license_key = null)
    {
        if (!$license_key) {
            $license_key = self::get_license_key();
        } else {
            self::update_license_key($license_key);
        }
        if (!$license_key) {
            return false;
        }
        $api = self::getBaseApiServer();
        $api .= $license_key . '/license/activate';
        $args = [
            'timeout' => 1000,
            'sslverify' => true,
            'body' => [
                'product_id' => MW_AHURA_UPDATER_ITEM_ID,
            ]
        ];
        $remote = wp_remote_get($api, $args);

        if (is_wp_error($remote) || wp_remote_retrieve_response_code($remote) != 200) {
            return false;
        }
        $body = wp_remote_retrieve_body($remote);
        $response = ahura_is_json($body) ? json_decode($body) : null;
        if (!is_object($response)) {
            return false;
        }
        return isset($response->message->activated) && isset($response->message->success) && $response->message->success;
    }

    static function deactivate_license_on_server()
    {
        $res = [
            'status' => false,
            'msg' => '',
        ];

        $apiUrl = self::get_smart_deactivation_url();
        $remote = wp_remote_get($apiUrl, ['timeout' => 900]);

        if (is_wp_error($remote)) {
            $res['msg'] = !empty($remote->get_error_message()) ? $remote->get_error_message() : esc_html__('An error occurred!', 'ahura');
            return $res;
        }

        $body = wp_remote_retrieve_body($remote);
        $remoteResponse = ahura_is_json($body) ? json_decode($body) : null;
        if (!is_object($remoteResponse))
            return false;

        if (isset($remoteResponse->code) && $remoteResponse->code != 200) {
            $res['msg'] = $remoteResponse->message ?? null;
            return $res;
        }

        $res['status'] = true;
        $res['msg'] = isset($remoteResponse->message) ? $remoteResponse->message : null;

        self::deactivate_license_in_local();

        return $res;
    }

    static function get_smart_deactivation_url()
    {
        $license = self::get_license_key();
        $baseUrl = self::getBaseApiServer('v3') . $license . '/license/deactivate';
        $args = [
            'product_id' => mw_tools::getRemoteProductId($license),
            'domain' => get_site_url(),
        ];
        return add_query_arg($args, $baseUrl);
    }

    public static function get_smart_activation_url($callbackUrl = null)
    {
        $license = self::get_license_key();
        $baseUrl = self::getBaseApiServer('v3') . $license . '/license/activate';
        $callbackUrl = !empty($callbackUrl) ? $callbackUrl : self::get_license_page_url();
        $args = [
            'product_id' => mw_tools::getRemoteProductId($license),
            'domain' => get_site_url(),
            'callback_url' => urlencode($callbackUrl),
        ];
        return add_query_arg($args, $baseUrl);
    }

    static function update_license_key($license_key)
    {
        return update_option(self::$_license_key_option_name, sanitize_key($license_key));
    }

    static function is_active()
    {
        return get_option(self::$_license_status_option_name) == 'valid';
    }

    private static function save_license_key($encrypted_license_key)
    {
        $license_key = base64_decode($encrypted_license_key);
        return update_option(self::$_license_key_option_name, $license_key);
    }

    static function get_license_key_from_db()
    {
        return defined('AHURA_LICENSE_KEY') ? AHURA_LICENSE_KEY : get_option(self::$_license_key_option_name);
    }

    static function get_license_key()
    {
        $license = self::get_license_key_from_db();
        return empty($license) ? 'aaa' : $license;
    }

    static function show_license_messages()
    {
        add_action('admin_notices', [__CLASS__, 'inactive_license_message_box']);
        add_action('customize_controls_enqueue_scripts', [__CLASS__, 'enqueue_customizer_notification']);
    }

    static function inactive_license_message_box()
    {
        $isDismiss = isset($_COOKIE['ahura_dismiss_license_admin_notice']);
        if ($isDismiss)
            return false;

        $host = self::getBaseHost();
        $isLicenseActive = self::check_license();
        $licenseRemainingDays = $isLicenseActive ? self::getRemainingDaysCachedData() : false;
        $message = '';
        $btnUrl = 'https://'. $host .'/mwpanel/?tab=3';
        $btnText = __('Renew License', 'ahura');
        $btnTargetMode = '_blank';

        $wrapperStyle = [
            'padding: 20px 20px 20px 30px',
            'border-radius: 27px',
            'font-size: 20px',
            'display: flex',
            'gap: 20px',
            'align-items: center',
            'color: black',
            'background: linear-gradient(45deg, #f3f5fd, #fdc39b96)',
            'border: 2px solid #fff',
            'position: relative',
        ];
        $linkStyle = [
            'display: block',
            'text-decoration: none',
            'color: black',
            'background-color: white',
            'padding: 15px 25px',
            'border-radius: 14px',
            'box-shadow: none',
            'background-color: #ffc33a',
            'color: black',
        ];
        $logoStyle = [
            'display: flex',
            'border: 4px solid #fff',
            'border-radius: 27px',
            'overflow: hidden',
            'background-color: #fff',
        ];
        $remindBtnStyle = [
            'box-shadow: none',
            'color: black',
            'text-decoration: none',
            'padding: 10px 20px',
            'cursor: pointer',
            'font-size: 16px',
        ];

        if ($isLicenseActive && $licenseRemainingDays > 30) {
            return null;
        } elseif (!$isLicenseActive) {
            $message = sprintf(__('Your license is inactive. Please connect your site to %s through the license management page.', 'ahura'), $host);
            $btnText = __('License Activation', 'ahura');
            $btnUrl = self::get_license_page_url();
            $btnTargetMode = '_self';
        } elseif ($licenseRemainingDays > 7) {
            $message = sprintf(__('Less than %s days remain until your license expires. You can easily renew it before it expires by clicking the renew button.', 'ahura'), 30);
        } elseif ($licenseRemainingDays > 1) {
            $message = sprintf(__('Less than %s days remain until your license expires. You can easily renew it before it expires by clicking the renew button.', 'ahura'), 7);
        } else {
            $message = __('Your license for Ahura has been expired. You can renew it by clicking the button on the right.', 'ahura');
        }

        $wrapperStyle = implode('; ', $wrapperStyle);
        $linkStyle = implode('; ', $linkStyle);
        $logoStyle = implode('; ', $logoStyle);
        $remindBtnStyle = implode('; ', $remindBtnStyle);
        ?>
        <div class="wrap">
            <div class="ahura_license_notice_wrapper" style="<?php echo esc_attr($wrapperStyle)?>">
                <span style="position: absolute;width: 5px;height: 75%;background-color: #f8802e;border-radius: 20px;right: 0"></span>
                <span style="<?php echo esc_attr($logoStyle)?>"><img src="<?php echo get_template_directory_uri() . '/img/ahura.svg'?>" style="max-width:100px;" alt="ah-logo"></span>
                <span style="flex: auto" class="text"><?php echo esc_html($message)?></span>
                <a target="<?php echo esc_attr($btnTargetMode)?>" style="<?php echo esc_attr($linkStyle)?>" href="<?php echo esc_url($btnUrl)?>" class="btn"><?php echo esc_html($btnText); ?></a>
                <a style="<?php echo esc_attr($remindBtnStyle)?>" class="ah-close-license-notice" href="#"><?php _e('Remind later', 'ahura')?></a>
            </div>
        </div>
        <?php
    }

    static function enqueue_customizer_notification()
    {
        $customizer_notification = get_template_directory_uri() . '/js/customizer_notification.js';
        wp_enqueue_script('ahura_customizer_notification', $customizer_notification, ['customize-controls'], 'version', true);
        wp_localize_script('ahura_customizer_notification', 'mw_license_data', ['msg' => __("Settings will appear after activate theme.", 'ahura')]);
    }

    static function getRemainingDays()
    {
        $license_key = self::get_license_key();

        $baseUrl = self::getBaseApiServer('v3') . $license_key . '/license/get_remaining_days';
        $args = [
            'body' => [
                'product_id' => mw_tools::getRemoteProductId($license_key),
                'domain' => get_site_url(),
            ],
        ];
        $remote = wp_remote_get($baseUrl, $args);

        if(is_wp_error($remote))
        {
            return false;
        }

        $body = wp_remote_retrieve_body($remote);
        $remoteResponse = ahura_is_json($body) ? json_decode($body) : null;
        if (!is_object($remoteResponse))
            return false;

        if(!isset($remoteResponse->status) || $remoteResponse->status != 200 || $remoteResponse->code != 200)
        {
            return false;
        }

        $days = $remoteResponse->days ?? null;
        if(!$days)
            return false;

        set_transient(self::LICENSE_REMAINING_DAYS_TRANSIENT_OPTION_NAME, $days, DAY_IN_SECONDS);
        return $days;
    }

    static function getRemainingDaysCachedData()
    {
        return get_transient(self::LICENSE_REMAINING_DAYS_TRANSIENT_OPTION_NAME) ?: self::getRemainingDays();
    }

    private static function successActivation()
    {
        set_transient('ahura_license_activation_status', true, MINUTE_IN_SECONDS);
    }

    public static function failedActivation()
    {
        delete_transient('ahura_license_activation_status');
    }

    public static function isSuccessActivation()
    {
        return get_transient('ahura_license_activation_status') ? true : false;
    }

    public static function getHostFromSiteUrl()
    {
        return parse_url(get_option('siteurl'), PHP_URL_HOST);
    }
}
