<?php

use WHMCS\Database\Capsule;
use WHMCS\Product\Product;
use WHMCS\User\Client;

if (!defined("WHMCS")) {
    die("This file cannot be accessed directly");
}

function prtgmonitoring_config()
{
    return [
        // Display name for your module
        'name' => 'prtgmonitoring',
        // Description displayed within the admin interface
        'description' => 'This module is created for Monitoring network by prtg',
        // Module author name
        'author' => '@Keoncloud',
        // Default language
        'language' => 'english',
        // Version number
        'version' => '1.5'
    ];
}

function prtgmonitoring_activate()
{
    // Create custom tables and schema required by your module
    try {
        // Create table for storing product groups to monitor
        Capsule::schema()->create('mod_prtg_groups', function ($table) {
            $table->increments('id');
            $table->integer('group_id')->unique();
            $table->timestamp('created_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
            $table->timestamp('updated_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
        });

        // Create table for traffic usage if it doesn't exist
        Capsule::schema()->create('mod_trafficusage', function ($table) {
            $table->increments('tuid');
            $table->integer('product_id');
            $table->decimal('usage', 10, 2)->default(0);
            $table->timestamp('lastupdate')->default(Capsule::raw('CURRENT_TIMESTAMP'));
        });

        // Create table for PRTG servers if it doesn't exist
        Capsule::schema()->create('mod_prtg_servers', function ($table) {
            $table->increments('id');
            $table->string('name', 255);
            $table->string('username', 255);
            $table->string('passhash', 255);
            $table->string('api_key', 255)->nullable();
            $table->string('url', 255);
            $table->string('traffic_type', 20)->default('total');
            $table->timestamp('created_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
            $table->timestamp('updated_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
        });

        // Create table for product-server mappings
        Capsule::schema()->create('mod_prtg_product_servers', function ($table) {
            $table->increments('id');
            $table->integer('product_id');
            $table->integer('server_id');
            $table->timestamp('created_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
            $table->unique(['product_id', 'server_id']);
        });
        Capsule::schema()->create('mod_prtg_credentials', function ($table) {
            $table->increments('id');
            $table->string('username', 255);
            $table->string('passhash', 255);
            $table->string('url', 255);
            $table->timestamp('created_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
            $table->timestamp('updated_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
        });
        Capsule::schema()->create('mod_prtg_license', function ($table) {
            $table->increments('id');
            $table->string('license_key', 255);
            $table->timestamp('created_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
            $table->timestamp('updated_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
        });
        Capsule::schema()->create('mod_prtg_license_cache', function ($table) {
            $table->increments('id');
            $table->string('status', 50);
            $table->text('message')->nullable();
            $table->integer('expiry');
            $table->timestamp('created_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
            $table->timestamp('updated_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
        });
        Capsule::schema()->create('mod_prtg_version_cache', function ($table) {
            $table->increments('id');
            $table->string('latest_version', 50);
            $table->string('download_url', 255)->nullable();
            $table->timestamp('checked_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
        });


        return [
            'status' => 'success',
            'description' => 'PRTG Monitoring tables created successfully',
        ];
    } catch (\Exception $e) {
        return [
            'status' => "error",
            'description' => 'Unable to create database tables: ' . $e->getMessage(),
        ];
    }
}

function prtgmonitoring_deactivate()
{
    // Undo any database and schema modifications made by your module here
    try {
        // Drop the product groups table
        $tables = [
            'mod_prtg_groups',
            'mod_prtg_servers',
            'mod_prtg_product_servers',
            'mod_prtg_credentials',
            'mod_prtg_license',
            'mod_prtg_license_cache',
            'mod_prtg_version_cache',
            'mod_trafficusage'
        ];

        foreach ($tables as $table) {
            if (Capsule::schema()->hasTable($table)) {
                Capsule::schema()->drop($table);
            }
        }
        // Note: We're not dropping the traffic usage table to preserve historical data

        return [
            'status' => 'success',
            'description' => 'PRTG Monitoring module deactivated successfully. Traffic usage data has been preserved.',
        ];
    } catch (\Exception $e) {
        return [
            "status" => "error",
            "description" => "Unable to drop tables: {$e->getMessage()}",
        ];
    }
}

function prtgmonitoring_upgrade($vars)
{

    $currentVersion = $vars['version'];
    if (!Capsule::schema()->hasTable('mod_prtg_servers')) {
        Capsule::schema()->create('mod_prtg_servers', function ($table) {
            $table->increments('id');
            $table->string('name', 255);
            $table->string('username', 255);
            $table->string('passhash', 255);
            $table->string('api_key', 255)->nullable();
            $table->string('url', 255);
            $table->timestamp('created_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
            $table->timestamp('updated_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
        });
    }

    // Add API key column if it doesn't exist (for upgrades)
    if (!Capsule::schema()->hasColumn('mod_prtg_servers', 'api_key')) {
        try {
            Capsule::schema()->table('mod_prtg_servers', function ($table) {
                $table->string('api_key', 255)->nullable()->after('passhash');
            });
            logActivity("PRTG Monitoring: API key column added successfully");
        } catch (Exception $e) {
            logActivity("PRTG Monitoring error: " . $e->getMessage());
        }
    }
    if (!Capsule::schema()->hasTable('mod_prtg_product_servers')) {
        Capsule::schema()->create('mod_prtg_product_servers', function ($table) {
            $table->increments('id');
            $table->integer('product_id');
            $table->integer('server_id');
            $table->timestamp('created_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
            $table->unique(['product_id', 'server_id']);
        });
    }
    // If upgrading from a version before 1.1
    if (version_compare($currentVersion, '1.1', '<')) {
        try {
            // Create the product groups table if it doesn't exist
            if (!Capsule::schema()->hasTable('mod_prtg_groups')) {
                Capsule::schema()->create('mod_prtg_groups', function ($table) {
                    $table->increments('id');
                    $table->integer('group_id')->unique();
                    $table->timestamp('created_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
                    $table->timestamp('updated_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
                });

                // Add default group ID (26) for backward compatibility
                Capsule::table('mod_prtg_groups')->insert([
                    'group_id' => 26,
                    'created_at' => Capsule::raw('NOW()'),
                    'updated_at' => Capsule::raw('NOW()')
                ]);
            }
        } catch (\Exception $e) {
            // Log error
            logActivity("PRTG Monitoring upgrade error: " . $e->getMessage());
        }
    }

    // If upgrading from a version before 1.2
    if (version_compare($currentVersion, '1.2', '<')) {
        try {
            // Create table for PRTG credentials if it doesn't exist
            if (!Capsule::schema()->hasTable('mod_prtg_credentials')) {
                Capsule::schema()->create('mod_prtg_credentials', function ($table) {
                    $table->increments('id');
                    $table->string('username', 255);
                    $table->string('passhash', 255);
                    $table->string('url', 255);
                    $table->timestamp('created_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
                    $table->timestamp('updated_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
                });
            }
        } catch (\Exception $e) {
            // Log error
            logActivity("PRTG Monitoring upgrade error: " . $e->getMessage());
        }
    }
    if (!Capsule::schema()->hasColumn('mod_prtg_servers', 'traffic_type')) {
        // Add traffic_type column
        Capsule::schema()->table('mod_prtg_servers', function ($table) {
            $table->string('traffic_type', 20)->default('total')->after('url');
        });

        echo "Traffic type column added successfully!";
    }
}

function prtgmonitoring_output($vars)
{
    // Include license system
    require_once(dirname(__FILE__) . '/license.php');
    require_once(dirname(__FILE__) . '/license_form.php');

    // Check license status
    $licenseValid = checkLicense();

    // Get common module parameters
    $modulelink = $vars['modulelink']; // eg. prtgmonitorings.php?module=prtgmonitoring
    $version = $vars['version']; // eg. 1.0

    // Check for module updates from license API
    $versionInfo = getVersionInfo();
    if ($versionInfo && version_compare($version, $versionInfo['latest_version'], '<')) {
        echo '<div class="alert alert-warning">
            <strong>Module Update Available!</strong> 
            Version ' . $versionInfo['latest_version'] . ' is now available. 
            You are currently using version ' . $version . '.
            <a href="' . $versionInfo['download_url'] . '" class="btn btn-warning btn-sm pull-right" target="_blank">
                Download Update
            </a>
        </div>';
    }

    $_lang = $vars['_lang']; // an array of the currently loaded language variables
    require_once(dirname(__FILE__) . '/version_check.php');
    // Check for module updates
    $versionCheck = checkModuleVersion($version);
    $updateAvailable = ($versionCheck['status'] === 'update_available');

    // Display update notification if needed
    if ($updateAvailable) {
        echo '<div class="alert alert-warning">
            <strong>Module Update Available!</strong> 
            Version ' . $versionCheck['latest_version'] . ' is now available. 
            You are currently using version ' . $version . '.
            <a href="' . $versionCheck['download_url'] . '" class="btn btn-warning btn-sm pull-right" target="_blank">
                Download Update
            </a>
        </div>';
    }

    // Get the current action
    $action = isset($_REQUEST['action']) ? $_REQUEST['action'] : 'services';

    // Handle license form submission
    if (isset($_POST['save_license'])) {
        $licenseKey = isset($_POST['license_key']) ? trim($_POST['license_key']) : '';
        if (!empty($licenseKey)) {
            saveLicenseKey($licenseKey);
            $licenseValid = checkLicense();
        }
    }

    // Handle form submission for adding/removing product groups
    if (isset($_POST['save_groups'])) {
        // Process group IDs
        if (isset($_POST['group_id']) && !empty($_POST['group_id'])) {
            $groupId = (int) $_POST['group_id'];

            // Check if group already exists
            $existingGroup = Capsule::table('mod_prtg_groups')
                ->where('group_id', $groupId)
                ->first();

            if (!$existingGroup) {
                // Add new group
                Capsule::table('mod_prtg_groups')->insert([
                    'group_id' => $groupId,
                    'created_at' => Capsule::raw('NOW()'),
                    'updated_at' => Capsule::raw('NOW()')
                ]);
            }
        }

        // Handle deletion
        if (isset($_POST['delete_group']) && !empty($_POST['delete_group'])) {
            $deleteId = (int) $_POST['delete_group'];
            Capsule::table('mod_prtg_groups')
                ->where('id', $deleteId)
                ->delete();
        }
    }

    // Handle PRTG server form submission
    if (isset($_POST['save_server'])) {
        $name = isset($_POST['server_name']) ? trim($_POST['server_name']) : '';
        $username = isset($_POST['prtg_username']) ? trim($_POST['prtg_username']) : '';
        $passhash = isset($_POST['prtg_passhash']) ? trim($_POST['prtg_passhash']) : '';
        $apikey = isset($_POST['prtg_apikey']) ? trim($_POST['prtg_apikey']) : '';
        $url = isset($_POST['prtg_url']) ? trim($_POST['prtg_url']) : '';
        $serverId = isset($_POST['server_id']) ? (int) $_POST['server_id'] : 0;

        $trafficType = isset($_POST['traffic_type']) ? trim($_POST['traffic_type']) : 'total';

        // Validate inputs
        if (!empty($name) && !empty($username) && !empty($passhash) && !empty($url)) {
            if ($serverId > 0) {
                // Update existing server
                Capsule::table('mod_prtg_servers')
                    ->where('id', $serverId)
                    ->update([
                        'name' => $name,
                        'username' => $username,
                        'passhash' => $passhash,
                        'api_key' => $apikey,
                        'url' => $url,
                        'traffic_type' => $trafficType,
                        'updated_at' => Capsule::raw('NOW()')
                    ]);
            } else {
                // Insert new server
                Capsule::table('mod_prtg_servers')->insert([
                    'name' => $name,
                    'username' => $username,
                    'passhash' => $passhash,
                    'api_key' => $apikey,
                    'url' => $url,
                    'traffic_type' => $trafficType,
                    'created_at' => Capsule::raw('NOW()'),
                    'updated_at' => Capsule::raw('NOW()')
                ]);
            }
        }
    }

    // Handle server deletion
    if (isset($_POST['delete_server']) && !empty($_POST['delete_server'])) {
        $deleteId = (int) $_POST['delete_server'];
        // Delete server and its mappings
        Capsule::table('mod_prtg_servers')
            ->where('id', $deleteId)
            ->delete();
        Capsule::table('mod_prtg_product_servers')
            ->where('server_id', $deleteId)
            ->delete();
    }

    // Handle product-server mapping
    if (isset($_POST['save_mapping'])) {
        $productId = isset($_POST['product_id']) ? (int) $_POST['product_id'] : 0;
        $serverId = isset($_POST['server_id']) ? (int) $_POST['server_id'] : 0;

        if ($productId > 0 && $serverId > 0) {
            // Check if mapping already exists
            $existingMapping = Capsule::table('mod_prtg_product_servers')
                ->where('product_id', $productId)
                ->first();

            if ($existingMapping) {
                // Update existing mapping
                Capsule::table('mod_prtg_product_servers')
                    ->where('product_id', $productId)
                    ->update([
                        'server_id' => $serverId
                    ]);
            } else {
                // Insert new mapping
                Capsule::table('mod_prtg_product_servers')->insert([
                    'product_id' => $productId,
                    'server_id' => $serverId,
                    'created_at' => Capsule::raw('NOW()')
                ]);
            }
        }
    }

    // Handle mapping deletion
    if (isset($_POST['delete_mapping']) && !empty($_POST['delete_mapping'])) {
        $deleteId = (int) $_POST['delete_mapping'];
        Capsule::table('mod_prtg_product_servers')
            ->where('id', $deleteId)
            ->delete();
    }

    // Get all monitored groups
    $monitoredGroups = Capsule::table('mod_prtg_groups')->get();

    // Build group IDs for query
    $groupIds = [];
    foreach ($monitoredGroups as $group) {
        $groupIds[] = $group->group_id;
    }

    // Get all PRTG servers
    $prtgServers = Capsule::table('mod_prtg_servers')->get();

    // Create the main navigation menu
    $menu = '
    <div class="row">
        <div class="col-md-12">
            <ul class="nav nav-tabs">
                <li class="' . ($action == 'services' ? 'active' : '') . '">
                    <a href="' . $modulelink . '&action=services">Services</a>
                </li>
                <li class="' . ($action == 'servers' ? 'active' : '') . '">
                    <a href="' . $modulelink . '&action=servers">PRTG Servers</a>
                </li>
                <li class="' . ($action == 'mappings' ? 'active' : '') . '">
                    <a href="' . $modulelink . '&action=mappings">Server Mappings</a>
                </li>
                <li class="' . ($action == 'settings' ? 'active' : '') . '">
                    <a href="' . $modulelink . '&action=settings">Settings</a>
                </li>
                <li class="' . ($action == 'graphs' ? 'active' : '') . '">
                    <a href="' . $modulelink . '&action=graphs">Graphs</a>
                </li>
                <li class="' . ($action == 'usersgraphs' ? 'active' : '') . '">
                    <a href="' . $modulelink . '&action=usersgraphs">Users & Graphs</a>
                </li>
                <li class="' . ($action == 'license' ? 'active' : '') . '">
                    <a href="' . $modulelink . '&action=license">License</a>
                </li>
            </ul>
            <br>
        </div>
    </div>';

    // Display license warning if not valid
    if (!$licenseValid) {
        $menu .= '
        <div class="alert alert-danger">
            <strong>License Invalid!</strong> Please enter a valid license key to continue using this module.
            <a href="' . $modulelink . '&action=license" class="btn btn-danger btn-sm pull-right">Enter License</a>
        </div>';
    }

    // Display PRTG servers form
    $serversForm = '
    <div class="panel panel-default">
        <div class="panel-heading">
            <h3 class="panel-title">Manage PRTG Servers</h3>
        </div>
        <div class="panel-body">
            <form method="post" action="' . $modulelink . '&action=servers">
                <input type="hidden" name="server_id" id="server_id" value="">
                <div class="row">
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="server_name">Server Name:</label>
                            <input type="text" name="server_name" id="server_name" class="form-control" placeholder="Enter server name">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="prtg_username">PRTG Username:</label>
                            <input type="text" name="prtg_username" id="prtg_username" class="form-control" placeholder="Enter PRTG username">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="prtg_passhash">PRTG Passhash:</label>
                            <input type="text" name="prtg_passhash" id="prtg_passhash" class="form-control" placeholder="Enter PRTG passhash">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="prtg_apikey">PRTG API Key:</label>
                            <input type="text" name="prtg_apikey" id="prtg_apikey" class="form-control" placeholder="Enter PRTG API key">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="prtg_url">PRTG Server URL:</label>
                            <input type="text" name="prtg_url" id="prtg_url" class="form-control" placeholder="Enter PRTG server URL">
                        </div>
                    </div>
                </div>
                <div class="row">
                    <div class="col-md-12">
                        <div class="form-group">
                            <label for="traffic_type">Traffic Type to Monitor:</label>
                            <select name="traffic_type" id="traffic_type" class="form-control">
                                <option value="total">Total Traffic (Download + Upload)</option>
                                <option value="download">Download Traffic Only</option>
                                <option value="upload">Upload Traffic Only</option>
                            </select>
                        </div>
                    </div>
                </div>
                <div class="row">
                    <div class="col-md-12">
                        <button type="submit" name="save_server" class="btn btn-primary">Add Server</button>
                    </div>
                </div>
            </form>
            
            <h4>Current PRTG Servers</h4>
            <table class="table table-bordered">
                <thead>
                    <tr>
                        <th>Name</th>
                        <th>Username</th>
                        <th>URL</th>
                        <th>Added Date</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>';

    foreach ($prtgServers as $server) {
        $serversForm .= '
                    <tr>
                        <td>' . htmlspecialchars($server->name) . '</td>
                        <td>' . htmlspecialchars($server->username) . '</td>
                        <td>' . htmlspecialchars($server->url) . '</td>
                        <td>' . $server->created_at . '</td>
                        <td>
                            <form method="post" action="' . $modulelink . '&action=servers" style="display:inline;">
                                <input type="hidden" name="delete_server" value="' . $server->id . '">
                                <button type="submit" class="btn btn-danger btn-sm">Delete</button>
                            </form>
                            <button type="button" class="btn btn-primary btn-sm edit-server" 
                                data-id="' . $server->id . '" 
                                data-name="' . htmlspecialchars($server->name) . '" 
                                data-username="' . htmlspecialchars($server->username) . '" 
                                data-passhash="' . htmlspecialchars($server->passhash) . '" 
                                data-url="' . htmlspecialchars($server->url) . '" 
                                data-traffic-type="' . (isset($server->traffic_type) ? $server->traffic_type : 'total') . '">
                                Edit
                            </button>
                        </td>
                    </tr>';
    }

    $serversForm .= '
                </tbody>
            </table>
        </div>
    </div>
    <script>
    $(document).ready(function() {
        $(".edit-server").click(function() {
            $("#server_id").val($(this).data("id"));
            $("#server_name").val($(this).data("name"));
            $("#prtg_username").val($(this).data("username"));
            $("#prtg_passhash").val($(this).data("passhash"));
            $("#prtg_apikey").val($(this).data("apikey"));
            $("#prtg_url").val($(this).data("url"));
            
            // Set traffic type if available
            var trafficType = $(this).data("traffic-type");
            if (trafficType) {
                $("#traffic_type").val(trafficType);
            } else {
                $("#traffic_type").val("total"); // Default
            }
        });
    });
    </script>';

    // Get all products from monitored groups
    $products = [];
    if (!empty($groupIds)) {
        $products = Capsule::table('tblproducts')
            ->whereIn('gid', $groupIds)
            ->select('id', 'name')
            ->get();
    }

    // Get existing mappings
    $mappings = Capsule::table('mod_prtg_product_servers')
        ->get();

    $mappingsByProduct = [];
    foreach ($mappings as $mapping) {
        $mappingsByProduct[$mapping->product_id] = $mapping;
    }

    // Create mapping form
    $mappingForm = '
    <div class="panel panel-default">
        <div class="panel-heading">
            <h3 class="panel-title">Map Products to PRTG Servers</h3>
        </div>
        <div class="panel-body">
            <form method="post" action="' . $modulelink . '&action=mappings">
                <div class="row">
                    <div class="col-md-5">
                        <div class="form-group">
                            <label for="product_id">Select Product:</label>
                            <select name="product_id" id="product_id" class="form-control">
                                <option value="">-- Select Product --</option>';

    foreach ($products as $product) {
        $mappingForm .= '<option value="' . $product->id . '">' . htmlspecialchars($product->name) . '</option>';
    }

    $mappingForm .= '
                            </select>
                        </div>
                    </div>
                    <div class="col-md-5">
                        <div class="form-group">
                            <label for="server_id">Select PRTG Server:</label>
                            <select name="server_id" id="server_id" class="form-control">
                                <option value="">-- Select Server --</option>';

    foreach ($prtgServers as $server) {
        $mappingForm .= '<option value="' . $server->id . '">' . htmlspecialchars($server->name) . '</option>';
    }

    $mappingForm .= '
                            </select>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="form-group">
                            <label>&nbsp;</label>
                            <button type="submit" name="save_mapping" class="btn btn-primary form-control">Add Mapping</button>
                        </div>
                    </div>
                </div>
            </form>
            
            <h4>Current Product-Server Mappings</h4>
            <table class="table table-bordered">
                <thead>
                    <tr>
                        <th>Product</th>
                        <th>PRTG Server</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>';

    // Get product and server details for mappings
    $productDetails = [];
    foreach ($products as $product) {
        $productDetails[$product->id] = $product->name;
    }

    $serverDetails = [];
    foreach ($prtgServers as $server) {
        $serverDetails[$server->id] = $server->name;
    }

    foreach ($mappings as $mapping) {
        $productName = isset($productDetails[$mapping->product_id]) ? $productDetails[$mapping->product_id] : 'Unknown Product';
        $serverName = isset($serverDetails[$mapping->server_id]) ? $serverDetails[$mapping->server_id] : 'Unknown Server';

        $mappingForm .= '
                    <tr>
                        <td>' . htmlspecialchars($productName) . '</td>
                        <td>' . htmlspecialchars($serverName) . '</td>
                        <td>
                            <form method="post" action="' . $modulelink . '&action=mappings">
                                <input type="hidden" name="delete_mapping" value="' . $mapping->id . '">
                                <button type="submit" class="btn btn-danger btn-sm">Delete</button>
                            </form>
                        </td>
                    </tr>';
    }

    $mappingForm .= '
                </tbody>
            </table>
        </div>
    </div>';

    // Get all product groups from WHMCS
    $productGroups = [];
    $result = full_query("SELECT id, name FROM tblproductgroups ORDER BY name");
    while ($group = mysql_fetch_array($result)) {
        $productGroups[$group['id']] = $group['name'];
    }

    // Display group management form
    $groupForm = '
    <div class="panel panel-default">
        <div class="panel-heading">
            <h3 class="panel-title">Manage Product Groups to Monitor</h3>
        </div>
        <div class="panel-body">
            <form method="post" action="' . $modulelink . '&action=settings">
                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="group_id">Add Product Group:</label>
                            <select name="group_id" id="group_id" class="form-control">
                                <option value="">-- Select Product Group --</option>';

    // Add options for each product group
    foreach ($productGroups as $id => $name) {
        // Skip groups that are already being monitored
        if (in_array($id, $groupIds)) {
            continue;
        }
        $groupForm .= '<option value="' . $id . '">' . htmlspecialchars($name) . ' (ID: ' . $id . ')</option>';
    }

    $groupForm .= '
                            </select>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="form-group">
                            <label>&nbsp;</label>
                            <button type="submit" name="save_groups" class="btn btn-primary form-control">Add Group</button>
                        </div>
                    </div>
                </div>
            </form>
            
            <h4>Currently Monitored Groups</h4>
            <table class="table table-bordered">
                <thead>
                    <tr>
                        <th>Group ID</th>
                        <th>Added Date</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>';

    foreach ($monitoredGroups as $group) {
        $groupName = isset($productGroups[$group->group_id]) ? htmlspecialchars($productGroups[$group->group_id]) : 'Unknown Group';
        $groupForm .= '
                    <tr>
                        <td>' . $group->group_id . ' - ' . $groupName . '</td>
                        <td>' . $group->created_at . '</td>
                        <td>
                            <form method="post" action="' . $modulelink . '&action=settings">
                                <input type="hidden" name="delete_group" value="' . $group->id . '">
                                <button type="submit" name="save_groups" class="btn btn-danger btn-sm">Delete</button>
                            </form>
                        </td>
                    </tr>';
    }

    $groupForm .= '
                </tbody>
            </table>
        </div>
    </div>';

    // Build the query for monitored services
    $whereClause = "";
    if (!empty($groupIds)) {
        $groupIdsStr = implode(',', $groupIds);
        $whereClause = "and `gid` IN ($groupIdsStr)";
    } else {
        // If no groups are defined, use a default (for backward compatibility)
        $whereClause = "and `gid` = 26";
    }

    $q = full_query("SELECT *,tblclients.id as userid,firstname,lastname,dedicatedip,tblproducts.id as productid,tblhosting.id as hostingid 
                    FROM `tblproducts`,tblhosting,tblclients 
                    WHERE userid=tblclients.id $whereClause and packageid=tblproducts.id and domainstatus='Active'");

    $graphs = "";
    $out = '
<table class="table table-hover" >
    <thead>
      <tr>
        <th>User</th>
        <th>IP</th>
        <th>Group ID</th>
        <th>PRTG Server</th>
        <th>Next Due Date</th>
        <th>Last Renew</th>
        <th>Monitoring ID</th>
        <th>Usage</th>
        <th>Limit</th>
        <th>Last Update</th>
      </tr>
    </thead>
    <tbody>
    ';
    $adminurl = "/admin";
    if (isset($customadminpath) && $customadminpath != null)
        $adminurl = $customadminpath;
    while ($data = mysql_fetch_array($q)) {
        $date = date_create($data["nextduedate"]);
        date_modify($date, '-1 month');
        $sql = full_query("SELECT * from tblcustomfields,tblcustomfieldsvalues
WHERE tblcustomfields.relid=" . $data["productid"] . " and tblcustomfieldsvalues.relid = " . $data["hostingid"] . " 
and tblcustomfieldsvalues.fieldid = tblcustomfields.id and    tblcustomfields.fieldname='monitoringID'");
        $s = mysql_fetch_array($sql);

        $sql = full_query("SELECT * from tblcustomfields,tblcustomfieldsvalues
WHERE tblcustomfields.relid=" . $data["productid"] . " and tblcustomfieldsvalues.relid = " . $data["hostingid"] . " 
and tblcustomfieldsvalues.fieldid = tblcustomfields.id and    tblcustomfields.fieldname='TrafficLimit'");
        $m = mysql_fetch_array($sql);

        $nsql = full_query("SELECT * from mod_trafficusage WHERE product_id=" . $data["hostingid"]);
        $ns = mysql_fetch_array($nsql);
        $nextduedate = $data["nextduedate"];
        $date = date_create($nextduedate);
        $ex = date_modify($date, '-1 month');
        $start = $ex->format("Y-m-d-00-00-00");
        $end = date("Y-m-d-h-i-s");

        // Get server for this product
        $serverMapping = Capsule::table('mod_prtg_product_servers')
            ->where('product_id', $data["productid"])
            ->first();

        $serverName = 'Default';
        $serverUsername = 'admin';
        $serverPasshash = '1314677760';
        $serverUrl = '';

        if ($serverMapping) {
            $server = Capsule::table('mod_prtg_servers')
                ->where('id', $serverMapping->server_id)
                ->first();

            if ($server) {
                $serverName = htmlspecialchars($server->name);
                $serverUsername = $server->username;
                $serverPasshash = $server->passhash;
                $serverUrl = $server->url;
            }
        }

        $out .= "<tr>
        <td><a href='" . $adminurl . "/clientssummary.php?userid=" . $data["userid"] . "'>" . $data["firstname"] . " " . $data["lastname"] . "</a></td>
        <td><a href='" . $adminurl . "/clientsservices.php?userid=" . $data["userid"] . "&id=" . $data["hostingid"] . "'>" . $data["dedicatedip"] . "</a></td>
        <td>" . $data["gid"] . " - " . (isset($productGroups[$data["gid"]]) ? htmlspecialchars($productGroups[$data["gid"]]) : 'Unknown Group') . "</td>
        <td>" . $serverName . "</td>
        <td>" . $data["nextduedate"] . "</td>
        <td>" . date_format($date, 'Y-m-d') . "</td>
        <td>" . $s["value"] . "</td>
        <td>" . ($ns["usage"] . str_replace(",", "")) . " GB - (" . (($ns["usage"] . str_replace(",", "")) / $m["value"] * 100) . "%)</td>
        <td>" . $m["value"] . " GB  </td>
        <td>" . $ns["lastupdate"] . "</td>
      </tr>";
        $graphs .= "<h2><a href='" . $adminurl . "/clientssummary.php?userid=" . $data["userid"] . "'>" . $data["firstname"] . " " . $data["lastname"] . "</a>
                </h2>
                <h3><a href='" . $adminurl . "/clientsservices.php?userid=" . $data["userid"] . "&id=" . $data["hostingid"] . "'>" . $data["dedicatedip"] . "</a></h3>
                 <img src='" . rtrim($serverUrl, '/') . "/chart.png?username=" . $serverUsername . "&passhash=" . $serverPasshash . "&id=" . $s["value"] . "&avg=15&sdate=" . $start . "&edate=" . $end . "&width=850&height=470&graphstyling=baseFontSize=%2712%27%20showLegend=%271%27%20graphid=-1' />";
    }
    $out .= '  
    </tbody>
  </table>
';

    // Generate the page content based on the action
    $content = '';

    switch ($action) {
        case 'servers':
            $content = $licenseValid ? $serversForm : getLicenseForm($modulelink);
            break;

        case 'mappings':
            $content = $licenseValid ? $mappingForm : getLicenseForm($modulelink);
            break;

        case 'settings':
            $content = $licenseValid ? $groupForm : getLicenseForm($modulelink);
            break;

        case 'graphs':
            $content = $licenseValid ?
                '<div class="panel panel-default"><div class="panel-body">' . $graphs . '</div></div>' :
                getLicenseForm($modulelink);
            break;

        case 'usersgraphs':
            if ($licenseValid) {
                // Include the helper file
                require_once(dirname(__FILE__) . '/usersgraphs.php');
                // Get users with their graphs
                $content = getUsersTable($whereClause, $adminurl) . getUsersWithGraphs($whereClause, $adminurl);
            } else {
                $content = getLicenseForm($modulelink);
            }
            break;

        case 'license':
            $content = getLicenseForm($modulelink);
            break;

        case 'services':
        default:
            $content = $licenseValid ?
                '<div class="panel panel-default"><div class="panel-body">' . $out . '</div></div>' :
                getLicenseForm($modulelink);
            break;
    }

    // Output the menu and content
    echo $menu . $content;
}


/**
 * Generate license form HTML
 * 
 * @param string $modulelink Module link for form action
 * @return string HTML form
 */
function getLicenseForm($modulelink)
{
    $licenseKey = getLicenseKey();
    $licenseCache = getLicenseCache();

    $status = isset($licenseCache['status']) ? $licenseCache['status'] : 'unknown';
    $message = isset($licenseCache['message']) ? $licenseCache['message'] : '';

    $statusClass = 'danger';
    if ($status === 'active') {
        $statusClass = 'success';
    } elseif ($status === 'pending') {
        $statusClass = 'warning';
    }

    return '
    <div class="panel panel-default">
        <div class="panel-heading">
            <h3 class="panel-title">License Management</h3>
        </div>
        <div class="panel-body">
            <form method="post" action="' . $modulelink . '&action=license">
                <div class="row">
                    <div class="col-md-8">
                        <div class="form-group">
                            <label for="license_key">License Key:</label>
                            <input type="text" name="license_key" id="license_key" class="form-control" value="' . htmlspecialchars($licenseKey) . '" placeholder="Enter your license key">
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="form-group">
                            <label>&nbsp;</label>
                            <button type="submit" name="save_license" class="btn btn-primary form-control">Save License</button>
                        </div>
                    </div>
                </div>
            </form>
            
            <div class="alert alert-' . $statusClass . '">
                <strong>License Status:</strong> ' . ucfirst($status) . '
                ' . ($message ? '<br><small>' . htmlspecialchars($message) . '</small>' : '') . '
            </div>
            
            <p>
                <strong>Need a license?</strong> Visit <a href="https://keloncloud.com" target="_blank">our website</a> to purchase a license for this module.
            </p>
            
            <p>
                <strong>Having issues?</strong> Contact our support team at <a href="mailto:support@keloncloud.com">support@keloncloud.com</a>.
            </p>
        </div>
    </div>';
}

function prtgmonitoring_clientarea($vars)
{
    // Client area functionality can be implemented here
    return [];
}