<?php
/**
 * PRTG Monitoring Module License System
 */

if (!defined("WHMCS")) {
    die("This file cannot be accessed directly");
}

use WHMCS\Database\Capsule;

/**
 * Check license status with the license server
 * 
 * @return bool True if license is valid, false otherwise
 */
function checkLicense()
{
    // Check if license cache exists and is valid
    // $cache = getLicenseCache();
    // if ($cache && $cache['status'] === 'active') {
    //     return true;
    // }
    
    // Get license key from database
    $licenseKey = getLicenseKey();
    
    // if (empty($licenseKey)) {
    //     return false;
    // }
    
    // Check license with license server
    $result = callLicenseServer($licenseKey);
    
    // Cache result to avoid frequent API calls
    setLicenseCache($result);
    
    return $result['status'] === 'active';
}

/**
 * Get license key from database
 * 
 * @return string|null License key or null if not found
 */
function getLicenseKey()
{
    // Create license table if it doesn't exist
    if (!Capsule::schema()->hasTable('mod_prtg_license')) {
        try {
            Capsule::schema()->create('mod_prtg_license', function ($table) {
                $table->increments('id');
                $table->string('license_key', 255);
                $table->timestamp('created_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
                $table->timestamp('updated_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
            });
        } catch (Exception $e) {
            return null;
        }
    }
    
    // Try to get from database
    try {
        $result = Capsule::table('mod_prtg_license')->first();
        
        if ($result && !empty($result->license_key)) {
            return $result->license_key;
        }
    } catch (Exception $e) {
        // Handle error silently
    }
    
    return null;
}

/**
 * Call license server to verify license
 * 
 * @param string $licenseKey License key to verify
 * @return array Response from license server
 */
function callLicenseServer($licenseKey)
{
    // License server URL
    $licenseServerUrl = 'https://cl.keloncloud.com/licensing/checklicense.php';
    
    // Prepare data for API call
    $postData = [
        'licensekey' => $licenseKey,
        'domain' => $_SERVER['SERVER_NAME'],
        'version' => '1.5' // Current module version
    ];
    
    // Initialize cURL
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $licenseServerUrl);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($postData));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    // Execute request
    $response = curl_exec($ch);
    
    // Check for errors
    if (curl_errno($ch)) {
        curl_close($ch);
        return ['status' => 'error', 'message' => 'Connection error'];
    }
    
    curl_close($ch);
    
    // Parse response
    $result = json_decode($response, true);
    
    if (!$result || !isset($result['status'])) {
        return ['status' => 'error', 'message' => 'Invalid response from license server'];
    }
    
    // Check for version information
    if (isset($result['latest_version']) && version_compare($postData['version'], $result['latest_version'], '<')) {
        // Store version information in cache
        storeVersionInfo($result);
    }
    
    return $result;
}

/**
 * Store version information in cache
 * 
 * @param array $versionInfo Version information from license server
 */
function storeVersionInfo($versionInfo)
{
    try {
        // Create version cache table if it doesn't exist
        if (!Capsule::schema()->hasTable('mod_prtg_version_cache')) {
            Capsule::schema()->create('mod_prtg_version_cache', function ($table) {
                $table->increments('id');
                $table->string('latest_version', 50);
                $table->string('download_url', 255)->nullable();
                $table->timestamp('checked_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
            });
        }
        
        // Check if cache exists
        $existing = Capsule::table('mod_prtg_version_cache')->first();
        
        if ($existing) {
            // Update existing cache
            Capsule::table('mod_prtg_version_cache')
                ->update([
                    'latest_version' => $versionInfo['latest_version'],
                    'download_url' => isset($versionInfo['download_url']) ? $versionInfo['download_url'] : '',
                    'checked_at' => date('Y-m-d H:i:s')
                ]);
        } else {
            // Create new cache
            Capsule::table('mod_prtg_version_cache')
                ->insert([
                    'latest_version' => $versionInfo['latest_version'],
                    'download_url' => isset($versionInfo['download_url']) ? $versionInfo['download_url'] : '',
                    'checked_at' => date('Y-m-d H:i:s')
                ]);
        }
    } catch (Exception $e) {
        // Handle error silently
    }
}

/**
 * Get version information from cache
 * 
 * @return array|null Version information or null if not available
 */
function getVersionInfo()
{
    if (!Capsule::schema()->hasTable('mod_prtg_version_cache')) {
        return null;
    }
    
    try {
        $cache = Capsule::table('mod_prtg_version_cache')->first();
        
        if (!$cache) {
            return null;
        }
        
        return [
            'latest_version' => $cache->latest_version,
            'download_url' => $cache->download_url
        ];
    } catch (Exception $e) {
        return null;
    }
}

/**
 * Cache license check result
 * 
 * @param array $result License check result
 */
function setLicenseCache($result)
{
    // Create cache table if it doesn't exist
    if (!Capsule::schema()->hasTable('mod_prtg_license_cache')) {
        try {
            Capsule::schema()->create('mod_prtg_license_cache', function ($table) {
                $table->increments('id');
                $table->string('status', 50);
                $table->text('message')->nullable();
                $table->integer('expiry');
                $table->timestamp('created_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
                $table->timestamp('updated_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
            });
        } catch (Exception $e) {
            return;
        }
    }
    
    // Store in database with expiry time (24 hours)
    $expiry = time() + 86400;
    
    try {
        // Check if cache exists
        $existing = Capsule::table('mod_prtg_license_cache')->first();
        
        if ($existing) {
            // Update existing cache
            Capsule::table('mod_prtg_license_cache')
                ->update([
                    'status' => $result['status'],
                    'message' => isset($result['message']) ? $result['message'] : '',
                    'expiry' => $expiry,
                    'updated_at' => date('Y-m-d H:i:s')
                ]);
        } else {
            // Create new cache
            Capsule::table('mod_prtg_license_cache')
                ->insert([
                    'status' => $result['status'],
                    'message' => isset($result['message']) ? $result['message'] : '',
                    'expiry' => $expiry,
                    'created_at' => date('Y-m-d H:i:s'),
                    'updated_at' => date('Y-m-d H:i:s')
                ]);
        }
    } catch (Exception $e) {
        // Handle error silently
    }
}

/**
 * Get cached license status
 * 
 * @return array|null Cached license status or null if cache expired
 */
function getLicenseCache()
{
    if (!Capsule::schema()->hasTable('mod_prtg_license_cache')) {
        return null;
    }
    
    try {
        $cache = Capsule::table('mod_prtg_license_cache')->first();
        
        if (!$cache || $cache->expiry < time()) {
            return null;
        }
        
        return [
            'status' => $cache->status,
            'message' => $cache->message
        ];
    } catch (Exception $e) {
        return null;
    }
}

/**
 * Save license key to database
 * 
 * @param string $licenseKey License key to save
 * @return bool True if saved successfully, false otherwise
 */
function saveLicenseKey($licenseKey)
{
    try {
        // Check if license table exists
        if (!Capsule::schema()->hasTable('mod_prtg_license')) {
            Capsule::schema()->create('mod_prtg_license', function ($table) {
                $table->increments('id');
                $table->string('license_key', 255);
                $table->timestamp('created_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
                $table->timestamp('updated_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
            });
        }
        
        // Check if license key already exists
        $existing = Capsule::table('mod_prtg_license')->first();
        
        if ($existing) {
            // Update existing license
            Capsule::table('mod_prtg_license')
                ->update([
                    'license_key' => $licenseKey,
                    'updated_at' => date('Y-m-d H:i:s')
                ]);
        } else {
            // Insert new license
            Capsule::table('mod_prtg_license')
                ->insert([
                    'license_key' => $licenseKey,
                    'created_at' => date('Y-m-d H:i:s'),
                    'updated_at' => date('Y-m-d H:i:s')
                ]);
        }
        
        // Clear license cache
        if (Capsule::schema()->hasTable('mod_prtg_license_cache')) {
            Capsule::table('mod_prtg_license_cache')->delete();
        }
        
        return true;
    } catch (Exception $e) {
        logActivity("License save error: " . $e->getMessage());
        return false;
    }
}