<?php

namespace RSThemes\Extensions;

use RSThemes\Template\Template;
use RSThemes\Extensions\ExtensionInterface;
use RSThemes\Models\Settings;
use RSThemes\Models\Configuration;
use RSThemes\View\ViewHelper;
use ReflectionClass;

if (!defined('DS')) {
    define('DS', DIRECTORY_SEPARATOR);
}

abstract class Extension implements ExtensionInterface
{
    /** @var Template */
    public $template = null;
    public $extension = null;
    public $name = null;
    public $description = null;
    public $version = null;
    public $linkName = null;
    public $key = null;
    public $className = null;

    public final function __construct(Template $template)
    {
        $this->template = $template;
        $class = new ReflectionClass($this);
        $this->className = str_replace("Extension", "", $class->getShortName());
        $this->key = sprintf("%s_%s_", $this->template->getMainName(), strtolower($this->className));
        $this->has();
    }

    public function updateRequired()
    {
        if (!$this->isActive()) {
            return false;
        }

        $storedVersion = Settings::getValue($this->template->getMainName() . "_extension_" . $this->className . "_version");

        if ($storedVersion != $this->getVersion()) {
            if (!$storedVersion && method_exists($this, "updater")) {
                return "unknown";
            }
            return $storedVersion;
        }

        return false;
    }

    public function getPreview()
    {
        // Placeholder method
    }

    public function isActive()
    {
        if (isset($GLOBALS["_SESSION"]["activatedDemoExtensions"]) && is_array($GLOBALS["_SESSION"]["activatedDemoExtensions"])) {
            if (in_array($this->className, $GLOBALS["_SESSION"]["activatedDemoExtensions"])) {
                return true;
            }
        }

        return (new Configuration())->getConfig($this->getKey("exstatus")) == 1;
    }

    public final function getName()
    {
        return $this->name;
    }

    public final function getDescription()
    {
        return $this->description;
    }

    public final function getLinkName()
    {
        return $this->className;
    }

    public final function getVersion()
    {
        return $this->version;
    }

    public final function has()
    {
        $class = new ReflectionClass($this);
        $this->className = str_replace("Extension", "", $class->getShortName());
        $path = $this->template->getFullPath() . DS . "core" . DS . "extensions" . DS . $this->className . DS . $this->className . ".php";

        if (file_exists($path)) {
            include_once $path;
            $className = $this->getClassNameFromFile($path);
            $templateExtension = new $className();

            if ($templateExtension instanceof \RSThemes\Extensions\TemplateExtensionInterface) {
                $this->extension = $templateExtension;
                return true;
            }
        }

        return false;
    }

    private function getClassNameFromFile($path_to_file)
    {
        $contents = file_get_contents($path_to_file);
        $namespace = $class = "";
        $getting_namespace = $getting_class = false;

        foreach (token_get_all($contents) as $token) {
            if (is_array($token) && $token[0] === T_NAMESPACE) {
                $getting_namespace = true;
            }
            if (is_array($token) && $token[0] === T_CLASS) {
                $getting_class = true;
            }
            if ($getting_namespace === true) {
                if (is_array($token) && in_array($token[0], [T_STRING, T_NS_SEPARATOR])) {
                    $namespace .= $token[1];
                } elseif ($token === ";") {
                    $getting_namespace = false;
                }
            }
            if ($getting_class === true) {
                if (is_array($token) && $token[0] === T_STRING) {
                    $class = $token[1];
                    break;
                }
            }
        }

        return ($namespace ? $namespace . "\\" . $class : $class);
    }

    public function render()
    {
        // Placeholder method
    }

    public function getKey($name)
    {
        return sprintf("%s%s", $this->key, $name);
    }

    public function getLink($action)
    {
        return (new ViewHelper())->url("Template@extension", [
            "templateName" => $this->template->getMainName(),
            "extension" => $this->getLinkName(),
            "exaction" => $action
        ]);
    }

    public function removeConfig()
    {
        if (isset($_GET["delete"])) {
            $extensionVersion = Settings::where("setting", $this->template->getMainName() . "_extension_" . $this->getLinkName() . "_version")->first();
            if ($extensionVersion) {
                $extensionVersion->delete();
            }
        }

        (new Configuration())->removeConfig($this->getKey("exstatus"));
        (new Configuration())->removeConfig($this->getKey("exstatus") . "_unlicensed");
    }

    public function loadConfig()
    {
        if (!Settings::getValue($this->template->getMainName() . "_extension_" . $this->className . "_version")) {
            Settings::updateOrCreate(
                ["setting" => $this->template->getMainName() . "_extension_" . $this->getLinkName() . "_version"],
                ["value" => $this->getVersion()]
            );
        }

        (new Configuration())->saveConfig($this->getKey("exstatus"), 1);
        (new Configuration())->removeConfig($this->getKey("exstatus") . "_unlicensed");
    }

    public function licenseRemoveConfig()
    {
        (new Configuration())->saveConfig($this->getKey("exstatus") . "_unlicensed", 1);
        (new Configuration())->removeConfig($this->getKey("exstatus"));
    }

    public function isLicenseDisabled()
    {
        return (new Configuration())->getConfig($this->getKey("exstatus") . "_unlicensed") == 1;
    }

    public function getPreviewLink()
    {
        return (new ViewHelper())->url("Template@extension", [
            "templateName" => $this->template->getMainName(),
            "extension" => $this->getLinkName(),
            "exaction" => "preview"
        ]);
    }

    public function getPreviewTplPath()
    {
        return "extensions/emailstyle/preview";
    }
}
