<?php
namespace ahura\inc\widgets;

// Block direct access to the main theme file.
defined('ABSPATH') or die('No script kiddies please!');

use Elementor\Controls_Manager;
use ahura\app\mw_assets;
use ahura\app\Ahura_Alert;

class video_player extends \ahura\app\elementor\Elementor_Widget_Base
{
    /**
     * @param array $data
     * @param null $args
     */
    public function __construct($data = [], $args = null)
    {
        parent::__construct($data, $args);
        mw_assets::register_script('video_player_js', mw_assets::get_js('elementor.video_player'), null, false);
        mw_assets::localize('video_player_js', 'element_avp', [
            'restart' => esc_html__('Restart', 'ahura'),
            'rewind' => sprintf(esc_html__('Rewind %s seconds', 'ahura'), '{seektime}'),
            'play' => esc_html__('Play', 'ahura'),
            'pause' => esc_html__('Pause', 'ahura'),
            'fastForward' => sprintf(esc_html__('Forward %s seconds', 'ahura'), '{seektime}'),
            'seekLabel' => sprintf(esc_html__('%s of %s', 'ahura'), '{currentTime}', '{duration}'),
            'played' => esc_html__('Played', 'ahura'),
            'currentTime' => esc_html__('Current time', 'ahura'),
            'duration' => esc_html__('Duration', 'ahura'),
            'volume' => esc_html__('Volume', 'ahura'),
            'mute' => esc_html__('Mute', 'ahura'),
            'unmute' => esc_html__('Unmute', 'ahura'),
            'enableCaptions' => esc_html__('Enable captions', 'ahura'),
            'disableCaptions' => esc_html__('Disable captions', 'ahura'),
            'download' => esc_html__('Download', 'ahura'),
            'enterFullscreen' => esc_html__('Enter fullscreen', 'ahura'),
            'exitFullscreen' => esc_html__('Exit fullscreen', 'ahura'),
            'frameTitle' => sprintf(esc_html__('Player for %s', 'ahura'), '{title}'),
            'captions' => esc_html__('Captions', 'ahura'),
            'settings' => esc_html__('Settings', 'ahura'),
            'pip' => esc_html__('PIP', 'ahura'),
            'menuBack' => esc_html__('Go back to previous menu', 'ahura'),
            'speed' => esc_html__('Speed', 'ahura'),
            'normal' => esc_html__('Normal', 'ahura'),
            'quality' => esc_html__('Quality', 'ahura'),
            'loop' => esc_html__('Loop', 'ahura'),
            'start' => esc_html__('Start', 'ahura'),
            'end' => esc_html__('End', 'ahura'),
            'all' => esc_html__('All', 'ahura'),
            'reset' => esc_html__('Reset', 'ahura'),
            'disabled' => esc_html__('Disabled', 'ahura'),
            'enabled' => esc_html__('Enabled', 'ahura'),
        ]);
        mw_assets::register_script('plyr_js', mw_assets::get_js('plyr'), null, false);
    }

    public function __register_assets()
    {
        $this->register_elementor_style('video_player_css', mw_assets::get_css('elementor.video_player'));
        $this->register_elementor_style('plyr_css', mw_assets::get_css('plyr'));
    }

    public function get_script_depends()
    {
        return [mw_assets::get_handle_name('plyr_js'), mw_assets::get_handle_name('video_player_js')];
    }

    public function get_name()
    {
        return 'ahura_video_player';
    }

    public function get_title()
    {
        return esc_html__('Video Player', 'ahura');
    }

    public function get_icon()
    {
        return 'eicon-play-o';
    }

    public function get_categories()
    {
        return ['ahuraelements'];
    }

    public function get_keywords()
    {
        return ['ahura', 'video', 'video player', esc_html__('Video', 'ahura'), esc_html__('Video Player', 'ahura')];
    }

    public function register_controls()
    {
        $this->start_controls_section(
            'content_section',
            [
                'label' => esc_html__('Content', 'ahura'),
                'tab' => Controls_Manager::TAB_CONTENT
            ]
        );

        $this->add_control(
            'type',
            [
                'label' => esc_html__( 'Type', 'ahura' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'direct',
                'options' => [
                    'direct' => esc_html__( 'Direct Link', 'ahura' ),
                    'aparat' => esc_html__( 'Aparat', 'ahura' ),
                    'youtube' => esc_html__( 'Youtube', 'ahura' ),
                    'vimeo' => esc_html__( 'Vimeo', 'ahura' ),
                ],
            ]
        );

        $this->add_control(
            'video_cover',
            [
                'label' => esc_html__('Cover', 'ahura'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
                'condition' => [
                    'type' => 'direct'
                ]
            ]
        );

        $this->add_responsive_control(
            'object_fit',
            [
                'label' => esc_html__( 'Aspect ratio', 'ahura' ),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'cover',
                'options' => [
                    'fill' => esc_html__( 'Default', 'ahura' ),
                    'contain' => esc_html__( 'Contain', 'ahura' ),
                    'cover'  => esc_html__( 'Cover', 'ahura' ),
                ],
                'selectors' => [
                    '{{WRAPPER}} .plyr__poster' => 'background-size: {{VALUE}};',
                ],
                'condition' => ['video_cover[url]!' => '']
            ]
        );

        $video_repeater = new \Elementor\Repeater();

        $video_repeater->add_control(
            'video',
            [
                'label' => esc_html__('Video', 'ahura'),
                'type' => Controls_Manager::MEDIA,
                'media_types' => ['video'],
                'default' => [
                    'url' => 'undefined',
                ],
            ]
        );

        $video_repeater->add_control(
            'quality',
            [
                'label' => esc_html__( 'Quality', 'ahura' ),
                'type' => Controls_Manager::SELECT,
                'default' => 0,
                'options' => [
                    0 => esc_html__( 'Default', 'ahura' ),
                    '480' => '480P',
                    '576' => '576P',
                    '720' => '720P (HD)',
                    '1080' => '1080P (Full HD)',
                    '1440' => '1440P (QHD)',
                    '2160' => '4K',
                ],
            ]
        );

        $this->add_control(
            'video_list',
            [
                'label' => esc_html__('Video Qualities', 'ahura'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $video_repeater->get_controls(),
                'condition' => [
                    'type' => 'direct'
                ]
            ]
        );

        $this->add_control(
            'use_caption',
            [
                'label' => esc_html__('Use Caption', 'ahura'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'ahura'),
                'label_off' => esc_html__('No', 'ahura'),
                'return_value' => 'yes',
                'default' => 'no',
                'condition' => [
                    'type' => 'direct',
                ]
            ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'caption_title',
            [
                'label' => esc_html__('Title', 'ahura'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('English', 'ahura'),
            ]
        );

        $repeater->add_control(
            'caption_code',
            [
                'label' => esc_html__('Caption Code', 'ahura'),
                'type' => Controls_Manager::TEXT,
                'default' => 'EN',
            ]
        );

        $repeater->add_control(
            'caption_file',
            [
                'label' => esc_html__('Caption File', 'ahura'),
                'type' => Controls_Manager::URL,
                'dynamic' => ['active' => true],
                'description' => esc_html__('Support .vtt format.', 'ahura')
            ]
        );

        $repeater->add_control(
            'default_caption',
            [
                'label' => esc_html__('Default', 'ahura'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'ahura'),
                'label_off' => esc_html__('No', 'ahura'),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'captions',
            [
                'label' => esc_html__('Captions', 'ahura'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'condition' => [
                    'type' => 'direct',
                    'use_caption' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'iframe_video_id',
            [
                'label' => esc_html__( 'Video ID', 'ahura' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'condition' => [
                        'type!' => 'direct'
                ]
            ]
        );

        $this->add_control(
			'show_post_content',
			[
				'label' => esc_html__( 'Post content', 'ahura' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'ahura' ),
				'label_off' => esc_html__( 'Hide', 'ahura' ),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

        $posts = get_posts();

        $post_options = array();

        foreach ($posts as $post) {
            $post_options[$post->ID] = $post->post_title;
        }

        $this->add_control(
            'posts_list',
            [
                'label' => __('Posts', 'ahura'),
                'type' => \Elementor\Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => $post_options,
                'multiple' => false,
                'condition' => [ 'show_post_content' => 'yes' ]
            ]
        );

        $this->add_control(
			'show_meta',
			[
				'label' => esc_html__( 'Show meta', 'ahura' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'ahura' ),
				'label_off' => esc_html__( 'Hide', 'ahura' ),
				'return_value' => 'yes',
				'default' => 'no',
                'condition' => [ 'show_post_content' => 'yes' ]
			]
		);

        $this->add_control(
			'show_addto_wishlist',
			[
				'label' => esc_html__( 'Add to wishlist', 'ahura' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'ahura' ),
				'label_off' => esc_html__( 'Hide', 'ahura' ),
				'return_value' => 'yes',
				'default' => 'no',
                'condition' => [ 'show_meta' => 'yes', 'show_post_content' => 'yes' ]
			]
		);

        $this->add_control(
			'show_share',
			[
				'label' => esc_html__( 'Sharing controls', 'ahura' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'ahura' ),
				'label_off' => esc_html__( 'Hide', 'ahura' ),
				'return_value' => 'yes',
				'default' => 'no',
                'condition' => [ 'show_meta' => 'yes', 'show_post_content' => 'yes' ]
			]
		);
        
        $this->add_control(
			'sharer_url',
			[
				'label' => esc_html__( 'Sharer URL', 'ahura' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'https://telegram.me/share/url?url=', 'ahura' ),
                'condition' => [ 'show_share' => 'yes', 'show_post_content' => 'yes' ]
			]
		);

        $this->add_control(
			'show_like',
			[
				'label' => esc_html__( 'Like button', 'ahura' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'ahura' ),
				'label_off' => esc_html__( 'Hide', 'ahura' ),
				'return_value' => 'yes',
				'default' => 'no',
                'condition' => [ 'show_meta' => 'yes', 'show_post_content' => 'yes' ]
			]
		);

        $this->add_control(
			'show_readmore_btn',
			[
				'label' => esc_html__( 'Read more button', 'ahura' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'ahura' ),
				'label_off' => esc_html__( 'Hide', 'ahura' ),
				'return_value' => 'yes',
				'default' => 'no',
                'condition' => [ 'show_meta' => 'yes', 'show_post_content' => 'yes' ]
			]
		);

        $this->end_controls_section();

        $this->start_controls_section(
            'settings_section',
            [
                'label' => esc_html__('Settings', 'ahura'),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'type' => 'direct'
                ]
            ]
        );
        
        $this->add_control(
			'show_controls_onhover',
			[
				'label' => esc_html__( 'Show controls on hover', 'ahura' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'ahura' ),
				'label_off' => esc_html__( 'Hide', 'ahura' ),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

        $this->add_control(
            'player_controls',
            [
                'label' => esc_html__( 'Controls', 'ahura' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'multiple' => true,
                'options' => [
                    'play-large' => esc_html__('Play Large Button', 'ahura'),
                    'mute' => esc_html__('Mute', 'ahura'),
                    'pip' => esc_html__('PIP', 'ahura'),
                    'download' => esc_html__('Download', 'ahura'),
                    'fullscreen' => esc_html__('Fullscreen', 'ahura'),
                    'settings' => esc_html__('Settings', 'ahura'),
                ],
                'default' => ['play-large', 'download', 'fullscreen', 'pip', 'mute', 'settings'],
            ]
        );

        $this->end_controls_section();
        /**
         *
         *
         * Styles
         *
         *
         */
        $this->start_controls_section(
            'player_style_section',
            [
                'label' => esc_html__('Player', 'ahura'),
                'tab' => Controls_Manager::TAB_STYLE
            ]
        );

        $this->add_control(
            'pri_color',
            [
                'label' => esc_html__('Primary Color', 'ahura'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .plyr' => '--plyr-color-main: {{VALUE}};'
                ],
                'default' => '#00b3ff',
            ]
        );

        $this->add_control(
            'sec_color',
            [
                'label' => esc_html__('Secondary Color', 'ahura'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .plyr' => '--plyr-video-control-color-hover: {{VALUE}};',
                    '{{WRAPPER}} .plyr button.plyr__control--overlaid' => 'color: {{VALUE}};'
                ],
                'default' => '#ffffff',
            ]
        );

        $this->add_responsive_control(
			'player_height',
			[
				'label' => esc_html__( 'Height', 'ahura' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'em', 'rem' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 1,
					],
				],
                'devices' => ['desktop', 'tablet', 'mobile'],
                'mobile_default' => [
                    'size' => 200,
                    'unit' => 'px',
                ],
				'selectors' => [
					'{{WRAPPER}} .ahura-video-player-element' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);

        $this->end_controls_section();
        $this->start_controls_section(
            'box_style_section',
            [
                'label' => esc_html__('Box', 'ahura'),
                'tab' => Controls_Manager::TAB_STYLE
            ]
        );

        $this->add_control(
			'video_box_options',
			[
				'label' => esc_html__( 'Video box', 'ahura' ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'box_border',
                'label' => esc_html__('Video Border', 'ahura'),
                'selector' => '{{WRAPPER}} .plyr, {{WRAPPER}} iframe',
            ]
        );

        $this->add_control(
            'box_border_radius',
            [
                'label' => esc_html__( 'Video border radius', 'ahura' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px', '%' ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 10,
                ],
                'selectors' => [
                    '{{WRAPPER}} .plyr, {{WRAPPER}} iframe' => 'border-radius: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .video-content-area' => 'border-radius: 0 0 {{SIZE}}{{UNIT}} {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'box_shadow',
                'label' => esc_html__('Video box Shadow', 'ahura'),
                'selector' => '{{WRAPPER}} .plyr, {{WRAPPER}} iframe',
            ]
        );

        $this->add_control(
			'content_area_options',
			[
				'label' => esc_html__( 'Content area', 'ahura' ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);
        
        $this->add_control(
			'content_area_backcolor',
			[
				'label' => esc_html__( 'Content background color', 'ahura' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .video-content-area' => 'background-color: {{VALUE}}',
				],
			]
		);

        $this->add_responsive_control(
			'content_padding',
			[
				'label' => esc_html__( 'Content padding', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem' ],
				'default' => [
					'top' => 50,
					'right' => 40,
					'bottom' => 30,
					'left' => 40,
					'unit' => 'px',
					'isLinked' => true,
				],
				'selectors' => [
					'{{WRAPPER}} .video-content-header' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->add_responsive_control(
			'content_margin',
			[
				'label' => esc_html__( 'Content margin', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem' ],
				'default' => [
					'top' => -20,
					'right' => 0,
					'bottom' => 0,
					'left' => 0,
					'unit' => 'px',
					'isLinked' => true,
				],
				'selectors' => [
					'{{WRAPPER}} .video-content-header' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
        
        $this->add_control(
			'title_color',
			[
				'label' => esc_html__( 'Title color', 'ahura' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .video-content-area h2' => 'color: {{VALUE}}',
				],
			]
		);

        $this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'title_typography',
				'selector' => '{{WRAPPER}} .video-content-area h2',
			]
		);
        
        $this->add_control(
			'description_color',
			[
				'label' => esc_html__( 'Description color', 'ahura' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .video-content-area p' => 'color: {{VALUE}}',
				],
			]
		);

        $this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'description_typography',
				'selector' => '{{WRAPPER}} .video-content-area p',
			]
		);

        $this->add_responsive_control(
			'description_padding',
			[
				'label' => esc_html__( 'Description padding', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem' ],
				'default' => [
					'top' => 20,
					'right' => 0,
					'bottom' => 0,
					'left' => 0,
					'unit' => 'px',
					'isLinked' => true,
				],
				'selectors' => [
					'{{WRAPPER}} .video-content-area p' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->add_control(
			'meta_options',
			[
				'label' => esc_html__( 'Meta', 'ahura' ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
                'condition' => [ 'show_meta' => 'yes' ]
			]
		);
        
        $this->add_control(
			'meta_color',
			[
				'label' => esc_html__( 'Meta color', 'ahura' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .sharing-buttons span' => 'color: {{VALUE}}',
					'{{WRAPPER}} .like-post span.btn-title' => 'color: {{VALUE}}',
				],
			]
		);

        $this->add_responsive_control(
			'meta_padding',
			[
				'label' => esc_html__( 'Meta padding', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem' ],
				'default' => [
					'top' => 20,
					'right' => 20,
					'bottom' => 20,
					'left' => 20,
					'unit' => 'px',
					'isLinked' => true,
				],
				'selectors' => [
					'{{WRAPPER}} .video-content-meta' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
                'condition' => [ 'show_meta' => 'yes' ]
			]
		);
        
        $this->add_control(
			'button_color',
			[
				'label' => esc_html__( 'Button color', 'ahura' ),
				'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#54565e',
				'selectors' => [
					'{{WRAPPER}} .read-more-btn' => 'background-color: {{VALUE}}',
				],
                'condition' => [ 'show_readmore_btn' => 'yes' ]
			]
		);

        $this->add_responsive_control(
			'button_padding',
			[
				'label' => esc_html__( 'Button padding', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem' ],
				'default' => [
					'top' => 10,
					'right' => 10,
					'bottom' => 10,
					'left' => 10,
					'unit' => 'px',
					'isLinked' => true,
				],
				'selectors' => [
					'{{WRAPPER}} .read-more-btn' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
                'condition' => [ 'show_readmore_btn' => 'yes' ]
			]
		);

        $this->add_control(
			'button_radius',
			[
				'label' => esc_html__( 'Button border radius', 'ahura' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem' ],
				'default' => [
					'top' => 5,
					'right' => 5,
					'bottom' => 5,
					'left' => 5,
					'unit' => 'px',
					'isLinked' => true,
				],
				'selectors' => [
					'{{WRAPPER}} .read-more-btn' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
                'condition' => [ 'show_readmore_btn' => 'yes' ]
			]
		);

        $this->end_controls_section();
    }

    /**
     *
     * Render content for display
     *
     */
    public function render()
    {
        $settings = $this->get_settings_for_display();
        $wid = $this->get_id();

        $type = $settings['type'];
        $video_qualities = $settings['video_list'];
        $use_caption = $settings['use_caption'] === 'yes';
        $captions = $settings['captions'];
        $iframe_video_id = $settings['iframe_video_id'];
        $controls = $settings['player_controls'];

        $controlsName = [
            is_array($controls) && in_array('play-large', $controls) ? 'play-large' : null,
            'play',
            'progress',
            'current-time',
            'duration',
            is_array($controls) && in_array('mute', $controls) ? 'mute' : null,
            'volume',
            $use_caption ? 'captions' : null,
            is_array($controls) && in_array('settings', $controls) ? 'settings' : null,
            is_array($controls) && in_array('download', $controls) ? 'download' : null,
            is_array($controls) && in_array('pip', $controls) ? 'pip' : null,
            is_array($controls) && in_array('fullscreen', $controls) ? 'fullscreen' : null,
        ];        

        $controlsName = array_filter($controlsName);

        $data = [
            'type' => $type,
            'tooltips' => [
                'controls' => true,
                'seek' => true
            ],
            'controls' => $controlsName
        ];
        ?>
            <div data-json='<?php echo json_encode($data) ?>' data-id="<?php echo esc_attr($wid) ?>" class="ahura-video-player-element video-player-<?php echo $wid; ?><?php echo $settings['show_controls_onhover'] === 'yes' ? ' controls-onhover' : ''; ?>">
                <?php if ($type == 'direct'): ?>
                    <video id="ahura_player_<?php echo $wid ?>" playsinline preload="none" data-poster="<?php echo $settings['video_cover']['url'] ?>">
                        <?php
                        if (!empty($video_qualities)):
                            foreach ($video_qualities as $item):
                                $quality = $item['quality'] != 0 ? sprintf('size="%s"', $item['quality']) : '';
                                ?>
                            <source src="<?php echo $item['video']['url'] ?>" type="video/mp4" <?php echo $quality ?>>
                            <?php endforeach;
                            endif; ?>
                        <?php
                        if ($use_caption && !empty($captions)):
                            foreach ($captions as $item):
                                $is_default = $item['default_caption'] === 'yes' ? 'default' : '';
                                ?>
                                <track kind="captions" label="<?php echo $item['caption_title'] ?>" src="<?php echo $item['caption_file']['url'] ?>" srclang="<?php echo strtolower($item['caption_code']) ?>" <?php echo $is_default ?>/>
                            <?php endforeach;
                        endif; ?>
                    </video>
                <?php else:
                    if($type === 'vimeo') {
                        $src = 'https://player.vimeo.com/video/' . $iframe_video_id;
                    } elseif($type === 'youtube') {
                        $src = 'https://www.youtube.com/embed/' . $iframe_video_id;
                    } else {
                        $src = 'https://www.aparat.com/video/video/embed/videohash/'.$iframe_video_id.'/vt/frame';
                    }
                    ?>
                    <div class="plyr__video-embed" id="ahura_player_<?php echo $wid ?>">
                        <iframe src="<?php echo $src ?>" allowfullscreen allow="accelerometer;autoplay;clipboard-write;encrypted-media;gyroscope;picture-in-picture;web-share"></iframe>
                    </div>
                <?php endif; ?>
                <?php $selected_post = get_post($settings['posts_list']);
                if ($selected_post && $settings['show_post_content'] === 'yes'): ?>
                    <div class="video-content-area">
                        <div class="video-content-header">
                            <a href="<?php echo get_permalink($selected_post); ?>">
                                <h2><?php echo $selected_post->post_title; ?></h2>
                            </a>
                            <p><?php echo wp_trim_words($selected_post->post_content, 30, '...'); ?></p>
                        </div>
                        <?php if($settings['show_meta'] === 'yes'): ?>
                            <div class="d-flex justify-content-between align-items-center video-content-meta">
                                <div class="d-flex justify-content-start align-items-center right-meta-content">
                                    <?php if($settings['show_addto_wishlist'] === 'yes'): ?>
                                        <div class="d-flex align-items-center addto-wishlist <?php echo is_rtl() ? 'ml-5' : ''; ?>">
                                            <?php if (in_array('yith-woocommerce-wishlist/init.php', apply_filters('active_plugins', get_option('active_plugins')))): ?>
                                                <div class="fav-btns <?php echo is_rtl() ? 'pr-1' : 'pl-1'; ?>"><?php echo do_shortcode('[yith_wcwl_add_to_wishlist]'); ?></div>
                                            <?php endif; ?>
                                        </div>
                                    <?php endif; ?>
                                    <?php if($settings['show_share'] === 'yes'): ?>
                                        <div class="d-flex align-items-center sharing-buttons <?php echo is_rtl() ? 'ml-5' : 'mr-5'; ?>">
                                            <svg width="31" height="30" viewBox="0 0 31 30" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                <path fill-rule="evenodd" clip-rule="evenodd" d="M25.7501 14.9998L17.0001 7.49976V12.541C12.2739 12.541 7.00012 14.3748 7.00012 22.4998C9.46137 18.5148 13.2501 17.4998 17.0001 17.4998V22.4998L25.7501 14.9998Z" stroke="white" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"/>
                                            </svg>
                                            <a href="<?php echo $settings['sharer_url'] . get_permalink($selected_post); ?>">
                                                <span class="<?php echo is_rtl() ? 'pr-1' : 'pl-1'; ?>"><?php echo esc_html__('Sharing', 'ahura'); ?></span>
                                            </a>
                                        </div>
                                    <?php endif; ?>
                                    <?php if($settings['show_like'] === 'yes'): ?>
                                        <div class="d-flex align-items-center like-post <?php echo is_rtl() ? '' : 'mr-5'; ?>">
                                            <div class="ahura-post-like">
                                                <div class="ahura-post-like-msg ahura-post-like-msg-<?php echo $selected_post->ID ?>"></div>
                                                <div class="post-like-buttons">
                                                    <a href="#" class="btn-post-like-action btn-post-like" data-like="1" data-post-id="<?php echo $selected_post->ID ?>">
                                                        <svg class="icon-post-like" width="31" height="30" viewBox="0 0 31 30" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <rect x="6.12506" y="14.3748" width="3.75" height="10" stroke="white" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"/>
                                                            <path d="M12.3751 24.3748H21.0001C22.1716 24.3746 23.1858 23.5609 23.4401 22.4173L24.8288 16.1673C24.9935 15.4262 24.8126 14.6505 24.3371 14.0587C23.8616 13.467 23.1429 13.1234 22.3838 13.1248H17.3751V8.12476C17.3751 6.74404 16.2558 5.62476 14.8751 5.62476L12.3751 13.1248" stroke="white" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"/>
                                                        </svg>
                                                        <span class="btn-title"><?php echo esc_html__('Like it', 'ahura'); ?></span>
                                                    </a>
                                                </div>
                                            </div>
                                        </div>
                                    <?php endif; ?>
                                </div>
                                <?php if($settings['show_readmore_btn'] === 'yes'): ?>
                                    <div class="d-flex justify-content-end align-items-center left-meta-content">
                                        <a href="<?php echo get_permalink($selected_post); ?>" class="d-flex justify-content-center align-items-center read-more-btn">
                                            <svg width="28" height="24" viewBox="0 0 28 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                <circle cx="14" cy="12.0001" r="2" stroke="white" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                                                <circle cx="4.99997" cy="12.0001" r="2" stroke="white" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                                                <circle cx="23" cy="12.0001" r="2" stroke="white" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                                            </svg>
                                        </a>
                                    </div>
                                <?php endif; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>
        <?php
    }
}