<?php
namespace ahura\inc\widgets;

// Block direct access to the main theme file.
defined('ABSPATH') or die('No script kiddies please!');

use Elementor\Controls_Manager;
use ahura\app\mw_assets;

class post_carousel11 extends \ahura\app\elementor\Elementor_Widget_Base
{
    public function __construct($data=[], $args=[])
    {
        parent::__construct($data, $args);
        mw_assets::register_script('post_carousel11_js', mw_assets::get_js('elementor.post_carousel11'));
    }

    public function __register_assets()
    {
        $this->register_elementor_style('post_carousel11_css', mw_assets::get_css('elementor.post_carousel11'));
        if(!is_rtl()){
            $this->register_elementor_style('post_carousel11_ltr_css', mw_assets::get_css('elementor.ltr.post_carousel11_ltr'));
        }
    }

    public function get_script_depends()
    {
        return [mw_assets::get_handle_name('post_carousel11_js')];
    }

    public function get_name()
    {
        return 'post_carousel11';
    }

    public function get_title()
    {
        return esc_html__('Post Carousel 11', 'ahura');
    }

    public function get_icon() {
        return 'eicon-posts-grid';
    }

    public function get_categories()
    {
        return ['ahuraelements', 'ahura_posts'];
    }

    public function get_keywords()
    {
        return ['ahura', 'post', 'post carousel 11', esc_html__('ahura', 'ahura'), esc_html__( 'Post Carousel 11' , 'ahura')];
    }

    private function add_taxonomy_control($taxonomy, $control_id, $label, $post_type, $condition) { 
        $terms = get_terms(['taxonomy' => $taxonomy]);
        
        $term_list = [];
        if (!is_wp_error($terms)) {
            foreach ($terms as $term) {
                $term_list[$term->term_id] = $term->name;
            }
        }
    
        $this->add_control(
            $control_id,
            [
                'label' => esc_html__($label, 'ahura'),
                'type' => Controls_Manager::SELECT2,
                'options' => $term_list,
                'label_block' => true,
                'multiple' => true,
                'condition' => array_merge(['post_type' => $post_type], $condition)
            ]
        );
    }

    public function register_controls()
    {
        $this->start_controls_section(
            'display_settings',
            [
                'label' => esc_html__('Layout', 'ahura'),
                'tab' => Controls_Manager::TAB_CONTENT
            ]
        );

        $this->add_control(
            'post_type',
            [
                'label' => esc_html__('Post Type', 'ahura'),
                'type' => Controls_Manager::SELECT,
                'default' => 'post',
                'options' => [
                    'post' => __('Posts', 'ahura'),
                    'product' => __('Products', 'ahura'),
                ],
            ]
        );
        
        $this->add_control(
            'filter_by_tax',
            [
                'label' => esc_html__('Filter by', 'ahura'),
                'type' => Controls_Manager::SELECT,
                'default' => 'cat',
                'options' => [
                    'tag' => __('Tag', 'ahura'),
                    'cat' => __('Category', 'ahura'),
                ],
            ]
        );
        
        $this->add_taxonomy_control('post_tag', 'post_tax_name', 'Taxonomy', 'post', ['filter_by_tax' => 'tag']);
        $this->add_taxonomy_control('product_tag', 'product_tax_name', 'Taxonomy', 'product', ['filter_by_tax' => 'tag']);
        $this->add_taxonomy_control('category', 'post_cat_id', 'Categories', 'post', ['filter_by_tax' => 'cat']);
        $this->add_taxonomy_control('product_cat', 'product_cat_id', 'Categories', 'product', ['filter_by_tax' => 'cat']);        

        $this->add_control(
            'per_page',
            [
                'label' => esc_html__('Count', 'ahura'),
                'type' => Controls_Manager::NUMBER,
                'min' => 1,
                'default' => 4
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'display_more_settings',
            [
                'label' => esc_html__('More Settings', 'ahura'),
                'tab' => Controls_Manager::TAB_CONTENT
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Image_Size::get_type(),
            [
                'name' => 'item_cover',
                'default' => 'full',
            ]
        );

        $this->add_responsive_control(
            'object_fit',
            [
                'label' => esc_html__( 'Aspect ratio', 'ahura' ),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'cover',
                'options' => [
                    'fill' => esc_html__( 'Default', 'ahura' ),
                    'contain' => esc_html__( 'Contain', 'ahura' ),
                    'cover'  => esc_html__( 'Cover', 'ahura' ),
                ],
                'selectors' => [
                    '{{WRAPPER}} .post-cover img' => 'object-fit: {{VALUE}};',
                ],
            ]
        );

        $this->add_control('divider', ['type' => Controls_Manager::DIVIDER]);

        $this->add_control(
            'show_like_btn',
            [
                'label' => esc_html__('Like Button', 'ahura'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'ahura'),
                'label_off' => esc_html__('Hide', 'ahura'),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'item_excerpt_show',
            [
                'label' => esc_html__('Excerpt', 'ahura'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'ahura'),
                'label_off' => esc_html__('Hide', 'ahura'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'excerpt_chars_count',
            [
                'label'   => __( 'Excerpt Characters', 'ahura' ),
                'type'    => Controls_Manager::NUMBER,
                'default' => 30,
                'condition' => [
                    'item_excerpt_show' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'show_cats',
            [
                'label' => esc_html__('Show categories', 'ahura'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'ahura'),
                'label_off' => esc_html__('Hide', 'ahura'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'item_category_icon',
            [
                'label' => esc_html__( 'Category Icon', 'ahura' ),
                'type' => Controls_Manager::ICONS,
                'skin' => 'inline',
                'default' => [
                    'value' => 'fas fa-angle-up',
                    'library' => 'fa-solid',
                ],
                'condition' => [
                    'show_cats' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'item_btn_show',
            [
                'label' => esc_html__('Button', 'ahura'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'ahura'),
                'label_off' => esc_html__('Hide', 'ahura'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'post_box_button_text',
            [
                'label' => esc_html__('Button Text', 'ahura'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('More...', 'ahura'),
                'condition' => [
                    'item_btn_show' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'post_btn_icon',
            [
                'label' => esc_html__( 'Post button icon', 'ahura' ),
                'type' => Controls_Manager::ICONS,
                'skin' => 'inline',
                'default' => [
                    'value' => 'fas fa-angle-left',
                    'library' => 'fa-solid',
                ],
                'condition' => [
                    'item_btn_show' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'item_sec_btn_show',
            [
                'label' => esc_html__('Secondary Button', 'ahura'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'ahura'),
                'label_off' => esc_html__('Hide', 'ahura'),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'item_sec_btn_text',
            [
                'label' => esc_html__('Button Text', 'ahura'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Event', 'ahura'),
                'condition' => [
                    'item_sec_btn_show' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'item_sec_btn_icon',
            [
                'label' => esc_html__( 'Button Icon', 'ahura' ),
                'type' => Controls_Manager::ICONS,
                'skin' => 'inline',
                'default' => [
                    'value' => 'fas fa-angle-left',
                    'library' => 'fa-solid',
                ],
                'condition' => [
                    'item_sec_btn_show' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'item_sec_btn_link',
            [
                'label' => esc_html__( 'Button Link', 'ahura' ),
                'type' => \Elementor\Controls_Manager::URL,
                'label_block' => true,
                'options' => [ 'url', 'is_external', 'nofollow' ],
                'dynamic' => ['active' => true],
                'default' => [
                    'url' => '#',
                    'is_external' => false,
                    'nofollow' => true,
                ],
                'condition' => [
                    'item_sec_btn_show' => 'yes'
                ]
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'slider_settings',
            [
                'label' => esc_html__('Slider', 'ahura'),
                'tab' => Controls_Manager::TAB_CONTENT
            ]
        );

        $this->add_control(
            'autoplay',
            [
                'label' => esc_html__('Autoplay', 'ahura'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'ahura'),
                'label_off' => esc_html__('No', 'ahura'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'transition_duration',
            [
                'label' => esc_html__('Transition Duration', 'ahura'),
                'type' => Controls_Manager::NUMBER,
                'default' => 4500,
                'condition' => [
                    'autoplay' => 'yes'
                ]
            ]
        );

        $this->add_responsive_control(
            'slides_per_view',
            [
                'label' => esc_html__('Slides per view', 'ahura'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    1 => '1',
                    2 => '2',
                    3 => '3',
                    4 => '4',
                    5 => '5',
                    6 => '6',
                ],
                'default' => 3,
            ]
        );

        $this->add_control(
            'show_pagination',
            [
                'label' => esc_html__('Pagination', 'ahura'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'ahura'),
                'label_off' => esc_html__('Hide', 'ahura'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'infinite_loop',
            [
                'label' => esc_html__('Infinite Loop', 'ahura'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'ahura'),
                'label_off' => esc_html__('No', 'ahura'),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'use_observer_slider',
            [
                'label' => esc_html__('Inner Slider', 'ahura'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'ahura'),
                'label_off' => esc_html__('No', 'ahura'),
                'return_value' => 'yes',
                'default' => 'yes',
                'description' => __('If you use the slider internally (tab, ...) and its display is not direct, be sure to activate this option.', 'ahura')
            ]
        );

        $this->end_controls_section();
        /**
         *
         *
         * Start styles
         *
         *
         */
        $this->start_controls_section(
            'items_cover_style_section',
            [
                'label' => __( 'Cover', 'ahura' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'item_cover_height',
            [
                'label' => esc_html__('Height', 'ahura'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'rem'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000
                    ],
                    'rem' => [
                        'min' => 0,
                        'max' => 500
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 200
                ],
                'selectors' => [
                    '{{WRAPPER}} .post-cover img' => 'height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'item_cover_border_radius',
            [
                'label' => esc_html__('Border Radius', 'ahura'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .post-cover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'isLinked' => true,
                    'top' => 10,
                    'right' => 10,
                    'bottom' => 10,
                    'left' => 10,
                ]
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'items_style_section',
            [
                'label' => __( 'Items', 'ahura' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->start_controls_tabs('item_style_tabs');
        $this->start_controls_tab('item_style_normal_tab', ['label' => esc_html__( 'Normal', 'ahura' )]);

        $alignment = array(
            'right' => [
                'title' => esc_html__('Right', 'ahura'),
                'icon' => 'eicon-text-align-right',
            ],
            'center' => [
                'title' => esc_html__('Center', 'ahura'),
                'icon' => 'eicon-text-align-center',
            ],
            'left' => [
                'title' => esc_html__('Left', 'ahura'),
                'icon' => 'eicon-text-align-left',
            ]
        );

        $this->add_responsive_control(
            'item_text_alignment',
            [
                'label' => esc_html__('Text Alignment', 'ahura'),
                'type' => Controls_Manager::CHOOSE,
                'options' => (is_rtl()) ? $alignment : array_reverse($alignment),
                'default' => (is_rtl()) ? 'right' : 'left',
                'toggle' => true,
                'selectors' => [
                    '{{WRAPPER}} .post-item-content' => 'text-align: {{VALUE}}',
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'item_bg',
                'selector' => '{{WRAPPER}} .post-item-content',
                'fields_options' => [
                    'background' => ['default' => 'classic'],
                    'color' => ['default' => '#303030']
                ]
            ]
        );

        $this->add_control(
            'category_color',
            [
                'label' => esc_html__('Category Color', 'ahura'),
                'type' => Controls_Manager::COLOR,
                'default' => '#fff',
                'selectors' => [
                    '{{WRAPPER}} .post-cats .item-cat' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'show_cats' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'category_icon_color',
            [
                'label' => esc_html__('Category Icon Color', 'ahura'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .post-cats i' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .post-cats :is(svg, path)' => 'fill: {{VALUE}}',
                ],
                'condition' => [
                    'show_cats' => 'yes'
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'label' => esc_html__('Category Typography', 'ahura'),
                'name' => 'item_category_typo',
                'selector' => '{{WRAPPER}} .post-cats .item-cat',
                'fields_options' => [
                    'typography' => ['default' => 'yes'],
                    'font_weight' => ['default' => 400],
                    'font_size' => [
                        'default' => [
                            'unit' => 'px',
                            'size' => '15',
                        ]
                    ]
                ],
                'condition' => [
                    'show_cats' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'item_title_color',
            [
                'label' => esc_html__('Title Color', 'ahura'),
                'type' => Controls_Manager::COLOR,
                'default' => '#fff',
                'selectors' => [
                    '{{WRAPPER}} .post-title' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'label' => esc_html__('Title Typography', 'ahura'),
                'name' => 'item_title_typo',
                'selector' => '{{WRAPPER}} .post-title',
                'fields_options' => [
                    'typography' => ['default' => 'yes'],
                    'font_weight' => ['default' => 600],
                    'font_size' => [
                        'default' => [
                            'unit' => 'px',
                            'size' => '17',
                        ]
                    ]
                ],
            ]
        );

        $this->add_control(
            'item_excerpt_color',
            [
                'label' => esc_html__('Excerpt Color', 'ahura'),
                'type' => Controls_Manager::COLOR,
                'default' => '#fff',
                'selectors' => [
                    '{{WRAPPER}} .post-excerpt p' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'item_excerpt_show' => 'yes'
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'label' => esc_html__('Excerpt Typography', 'ahura'),
                'name' => 'item_excerpt_typo',
                'selector' => '{{WRAPPER}} .post-excerpt p',
                'fields_options' => [
                    'typography' => ['default' => 'yes'],
                    'font_weight' => ['default' => 300],
                    'font_size' => [
                        'default' => [
                            'unit' => 'px',
                            'size' => '15',
                        ]
                    ]
                ],
                'condition' => [
                    'item_excerpt_show' => 'yes'
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'item_border',
                'selector' => '{{WRAPPER}} .post-item-content',
            ]
        );

        $this->add_control(
            'item_border_radius',
            [
                'label' => esc_html__('Border Radius', 'ahura'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .post-item-content' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'isLinked' => true,
                    'top' => 10,
                    'right' => 10,
                    'bottom' => 10,
                    'left' => 10,
                ]
            ]
        );

        $this->add_responsive_control(
            'item_padding',
            [
                'label' => esc_html__('Box padding', 'ahura'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .post-item-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'isLinked' => true,
                    'top' => 15,
                    'right' => 15,
                    'bottom' => 15,
                    'left' => 15,
                ]
            ]
        );

        $this->add_responsive_control(
            'title_padding',
            [
                'label' => esc_html__('Title padding', 'ahura'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .post-item-content h2' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'desc_padding',
            [
                'label' => esc_html__('Description padding', 'ahura'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .post-item-content .post-excerpt' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_tab();
        $this->start_controls_tab('item_style_hover_tab', ['label' => esc_html__( 'Hover', 'ahura' )]);

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'item_bg_hover',
                'selector' => '{{WRAPPER}} .post-item-content:hover',
            ]
        );

        $this->add_control(
            'category_color_hover',
            [
                'label' => esc_html__('Category Color', 'ahura'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .post-item-content:hover .post-cats .item-cat' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'show_cats' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'category_icon_color_hover',
            [
                'label' => esc_html__('Category Icon Color', 'ahura'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .post-item-content:hover .post-cats i' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .post-item-content:hover .post-cats :is(svg, path)' => 'fill: {{VALUE}}',
                ],
                'condition' => [
                    'show_cats' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'item_title_color_hover',
            [
                'label' => esc_html__('Title Color', 'ahura'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .post-item-content:hover .post-title' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'item_excerpt_color_hover',
            [
                'label' => esc_html__('Excerpt Color', 'ahura'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .post-item-content:hover .post-excerpt p' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'item_excerpt_show' => 'yes'
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'item_border_hover',
                'selector' => '{{WRAPPER}} .post-item-content:hover',
            ]
        );

        $this->add_control(
            'item_border_radius_hover',
            [
                'label' => esc_html__('Border Radius', 'ahura'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .post-item-content:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();
        $this->end_controls_section();

        $this->start_controls_section(
            'item_btns_style_section',
            [
                'label' => __( 'Buttons', 'ahura' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'btn_style_options',
            [
                'label' => esc_html__( 'Button', 'ahura' ),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'item_btn_show' => 'yes'
                ]
            ]
        );

        $this->start_controls_tabs('item_btn_style_tabs');
        $this->start_controls_tab('item_btn_style_normal_tab', ['label' => esc_html__( 'Normal', 'ahura' )]);

        $this->add_control(
			'btn_align',
			[
				'label' => esc_html__( 'Alignment', 'ahura' ),
				'type' => \Elementor\Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => esc_html__( 'Left', 'ahura' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'ahura' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'ahura' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'toggle' => true,
				'selectors' => [
					'{{WRAPPER}} .post-content-bottom' => 'text-align: {{VALUE}};',
				],
			]
		);

        $this->add_control(
            'item_btn_color',
            [
                'label' => esc_html__('Button Color', 'ahura'),
                'type' => Controls_Manager::COLOR,
                'default' => '#fff',
                'selectors' => [
                    '{{WRAPPER}} .item-main-btn' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'item_btn_show' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'btn_icon_color',
            [
                'label' => esc_html__('Icon Color', 'ahura'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .item-main-btn i' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .item-main-btn :is(svg, path)' => 'fill: {{VALUE}}',
                ],
                'condition' => [
                    'item_btn_show' => 'yes'
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'item_btn_typo',
                'selector' => '{{WRAPPER}} .item-main-btn',
                'fields_options' => [
                    'typography' => ['default' => 'yes'],
                    'font_weight' => ['default' => 400],
                    'font_size' => [
                        'default' => [
                            'unit' => 'px',
                            'size' => '13',
                        ]
                    ]
                ],
                'condition' => [
                    'item_btn_show' => 'yes'
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'item_btn_bg',
                'selector' => '{{WRAPPER}} .item-main-btn',
                'exclude' => ['image'],
                'fields_options' => [
                    'background' => ['default' => 'classic'],
                    'color' => ['default' => '#303030']
                ],
                'condition' => [
                    'item_btn_show' => 'yes'
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'item_btn_border',
                'selector' => '{{WRAPPER}} .item-main-btn',
                'condition' => [
                    'item_btn_show' => 'yes'
                ],
                'fields_options' => [
                    'border' => ['default' => 'solid'],
                    'width' => ['default' =>
                        [
                            'unit' => 'px',
                            'top' => 1,
                            'right' => 1,
                            'bottom' => 1,
                            'left' => 1,
                        ]
                    ],
                    'color' => ['default' => '#FFD342']
                ]
            ]
        );

        $this->add_control(
            'item_btn_border_radius',
            [
                'label' => esc_html__('Border Radius', 'ahura'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .item-main-btn' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'isLinked' => true,
                    'top' => 5,
                    'right' => 5,
                    'bottom' => 5,
                    'left' => 5,
                ],
                'condition' => [
                    'item_btn_show' => 'yes'
                ]
            ]
        );

        $this->add_responsive_control(
            'item_btn_padding',
            [
                'label' => esc_html__('Padding', 'ahura'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .item-main-btn' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'isLinked' => false,
                    'top' => 8,
                    'right' => 15,
                    'bottom' => 8,
                    'left' => 15,
                ],
                'condition' => [
                    'item_btn_show' => 'yes'
                ]
            ]
        );

        $this->add_responsive_control(
            'item_btn_margin',
            [
                'label' => esc_html__('Margin', 'ahura'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .item-main-btn' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'item_btn_shadow',
                'label' => esc_html__('Box Shadow', 'ahura'),
                'selector' => '{{WRAPPER}} .item-main-btn',
                'condition' => [
                    'item_btn_show' => 'yes'
                ]
            ]
        );

        $this->end_controls_tab();
        $this->start_controls_tab('item_btn_style_hover_tab', ['label' => esc_html__( 'Hover', 'ahura' )]);

        $this->add_control(
            'item_btn_color_hover',
            [
                'label' => esc_html__('Button Color', 'ahura'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .item-main-btn:hover' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'item_btn_show' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'btn_icon_color_hover',
            [
                'label' => esc_html__('Icon Color', 'ahura'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .item-main-btn:hover i' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .item-main-btn:hover :is(svg, path)' => 'fill: {{VALUE}}',
                ],
                'condition' => [
                    'item_btn_show' => 'yes'
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'item_btn_bg_hover',
                'selector' => '{{WRAPPER}} .item-main-btn:hover',
                'condition' => [
                    'item_btn_show' => 'yes'
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'item_btn_border_hover',
                'selector' => '{{WRAPPER}} .item-main-btn:hover',
                'condition' => [
                    'item_btn_show' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'item_btn_border_radius_hover',
            [
                'label' => esc_html__('Border Radius', 'ahura'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .item-main-btn:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'item_btn_show' => 'yes'
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'item_btn_shadow_hover',
                'label' => esc_html__('Box Shadow', 'ahura'),
                'selector' => '{{WRAPPER}} .item-main-btn:hover',
                'condition' => [
                    'item_btn_show' => 'yes'
                ]
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();

        $this->add_control(
            'btn_sec_style_options',
            [
                'label' => esc_html__( 'Secondary Button', 'ahura' ),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'item_sec_btn_show' => 'yes'
                ]
            ]
        );

        $this->start_controls_tabs('item_sec_btn_style_tabs');
        $this->start_controls_tab('item_sec_btn_style_normal_tab', ['label' => esc_html__( 'Normal', 'ahura' )]);

        $this->add_control(
            'item_sec_btn_color',
            [
                'label' => esc_html__('Button Color', 'ahura'),
                'type' => Controls_Manager::COLOR,
                'default' => '#000',
                'selectors' => [
                    '{{WRAPPER}} .item-sec-btn' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'item_sec_btn_show' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'sec_btn_icon_color',
            [
                'label' => esc_html__('Icon Color', 'ahura'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .item-sec-btn i' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .item-sec-btn :is(svg, path)' => 'fill: {{VALUE}}',
                ],
                'condition' => [
                    'item_sec_btn_show' => 'yes'
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'item_sec_btn_typo',
                'selector' => '{{WRAPPER}} .item-sec-btn',
                'fields_options' => [
                    'typography' => ['default' => 'yes'],
                    'font_weight' => ['default' => 400],
                    'font_size' => [
                        'default' => [
                            'unit' => 'px',
                            'size' => '13',
                        ]
                    ]
                ],
                'condition' => [
                    'item_sec_btn_show' => 'yes'
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'item_sec_btn_bg',
                'selector' => '{{WRAPPER}} .item-sec-btn',
                'exclude' => ['image'],
                'fields_options' => [
                    'background' => ['default' => 'classic'],
                    'color' => ['default' => '#FFD342']
                ],
                'condition' => [
                    'item_sec_btn_show' => 'yes'
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'item_sec_btn_border',
                'selector' => '{{WRAPPER}} .item-sec-btn',
                'condition' => [
                    'item_sec_btn_show' => 'yes'
                ],
                'fields_options' => [
                    'border' => ['default' => 'solid'],
                    'width' => ['default' =>
                        [
                            'unit' => 'px',
                            'top' => 1,
                            'right' => 1,
                            'bottom' => 1,
                            'left' => 1,
                        ]
                    ],
                    'color' => ['default' => '#FFD342']
                ]
            ]
        );

        $this->add_control(
            'item_sec_border_radius',
            [
                'label' => esc_html__('Border Radius', 'ahura'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .item-sec-btn' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'isLinked' => true,
                    'top' => 5,
                    'right' => 5,
                    'bottom' => 5,
                    'left' => 5,
                ],
                'condition' => [
                    'item_sec_btn_show' => 'yes'
                ]
            ]
        );

        $this->add_responsive_control(
            'item_sec_btn_padding',
            [
                'label' => esc_html__('Padding', 'ahura'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .item-sec-btn' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'isLinked' => false,
                    'top' => 8,
                    'right' => 15,
                    'bottom' => 8,
                    'left' => 15,
                ],
                'condition' => [
                    'item_sec_btn_show' => 'yes'
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'item_sec_btn_shadow',
                'label' => esc_html__('Box Shadow', 'ahura'),
                'selector' => '{{WRAPPER}} .item-sec-btn',
                'condition' => [
                    'item_sec_btn_show' => 'yes'
                ]
            ]
        );

        $this->end_controls_tab();
        $this->start_controls_tab('item_sec_btn_style_hover_tab', ['label' => esc_html__( 'Hover', 'ahura' )]);

        $this->add_control(
            'item_sec_btn_color_hover',
            [
                'label' => esc_html__('Button Color', 'ahura'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .item-sec-btn:hover' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'item_sec_btn_show' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'sec_btn_icon_color_hover',
            [
                'label' => esc_html__('Icon Color', 'ahura'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .item-sec-btn:hover i' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .item-sec-btn:hover :is(svg, path)' => 'fill: {{VALUE}}',
                ],
                'condition' => [
                    'item_sec_btn_show' => 'yes'
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'item_sec_btn_bg_hover',
                'selector' => '{{WRAPPER}} .item-sec-btn:hover',
                'condition' => [
                    'item_sec_btn_show' => 'yes'
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'item_sec_btn_border_hover',
                'selector' => '{{WRAPPER}} .item-sec-btn:hover',
                'condition' => [
                    'item_sec_btn_show' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'item_sec_border_radius_hover',
            [
                'label' => esc_html__('Border Radius', 'ahura'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .item-sec-btn:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'item_sec_btn_show' => 'yes'
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'item_sec_btn_shadow_hover',
                'label' => esc_html__('Box Shadow', 'ahura'),
                'selector' => '{{WRAPPER}} .item-sec-btn:hover',
                'condition' => [
                    'item_sec_btn_show' => 'yes'
                ]
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();
        $this->end_controls_section();

        $this->start_controls_section(
            'paginate_style_section',
            [
                'label' => __( 'Pagination', 'ahura' ),
                'tab'   => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_pagination' => 'yes'
                ]
            ]
        );

        $this->add_responsive_control(
            'paginate_width',
            [
                'label' => esc_html__('Width', 'ahura'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['%'],
                'range' => [
                    '%' => [
                        'min' => 0,
                        'max' => 100
                    ],
                ],
                'default' => [
                    'unit' => '%',
                    'size' => 50
                ],
                'selectors' => [
                    '{{WRAPPER}} .swiper-pagination' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'paginate_bg',
                'selector' => '{{WRAPPER}} .swiper-pagination',
                'exclude' => ['image'],
                'fields_options' => [
                    'background' => ['default' => 'classic'],
                    'color' => ['default' => '#f0f0f0']
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'active_paginate_bg',
                'label' => esc_html__('Active Background', 'ahura'),
                'selector' => '{{WRAPPER}} .swiper-pagination .swiper-pagination-progressbar-fill',
                'exclude' => ['image'],
                'fields_options' => [
                    'background' => ['default' => 'classic'],
                    'color' => ['default' => '#ffd342']
                ],
            ]
        );

        $this->end_controls_section();

        $this->register_additional_controls();
    }

    /**
     *
     * Get posts result
     *
     * @param array $params
     * @return \WP_Query
     */
    public function get_posts($params = [])
    {
        $settings = $this->get_settings_for_display();
        $args = [
            'post_type'      => $settings['post_type'],
            'posts_per_page' => $settings['per_page'],
            'post_status'    => 'publish',
        ];

        if ($settings['filter_by_tax'] === 'cat') {
            if ($settings['post_type'] === 'product') {
                $taxonomy = 'product_cat';
                $term_id = $settings['product_cat_id'] ?? [];
            } else {
                $taxonomy = 'category';
                $term_id = $settings['post_cat_id'] ?? [];
            }
        } else {
            if ($settings['post_type'] === 'product') {
                $taxonomy = 'product_tag';
                $term_id = $settings['product_tag'] ?? [];
            } else {
                $taxonomy = 'post_tag';
                $term_id = $settings['post_tag'] ?? [];
            }
        }

        if (!empty($term_id)) {
            $args['tax_query'] = [
                [
                    'taxonomy' => $taxonomy,
                    'field'    => 'term_id',
                    'terms'    => $term_id,
                    'operator' => 'IN',
                ],
            ];
        }

        return new \WP_Query(array_merge($args, $params));
    }


    /**
     *
     * Render content for display
     *
     */
    public function render()
    {
        $settings = $this->get_settings_for_display();
        $wid = $this->get_id();
        $chars_num = isset($settings['excerpt_chars_count']) && intval($settings['excerpt_chars_count']) ? $settings['excerpt_chars_count'] : false;
        $has_paginate = ($settings['show_pagination'] == 'yes');
        $show_like_btn = $settings['show_like_btn'] == 'yes';
		
		if ($settings['item_sec_btn_show'] == 'yes' && !empty($settings['item_sec_btn_link']['url'])) {
			$this->add_link_attributes('item_sec_btn_link', $settings['item_sec_btn_link']);
		}

        $args = [];

        $posts = $this->get_posts($args);

        $slider_data = [
            'widgetID' => $wid,
            'autoPlay' => $settings['autoplay'] == 'yes',
            'transitionDuration' => (intval($settings['transition_duration'])) ? $settings['transition_duration'] : 4500,
            'loop' => $settings['infinite_loop'] == 'yes',
            'slidesPerView' => (isset($settings['slides_per_view']) && intval($settings['slides_per_view'])) ? $settings['slides_per_view'] : 3,
            'tabletPerView' => (isset($settings['slides_per_view_tablet']) && intval($settings['slides_per_view_tablet'])) ? $settings['slides_per_view_tablet'] : 2,
            'mobilePerView' => (isset($settings['slides_per_view_mobile']) && intval($settings['slides_per_view_mobile'])) ? $settings['slides_per_view_mobile'] : 1,
            'pagination' => $has_paginate,
            'observer' => $settings['use_observer_slider'] == 'yes'
        ];
        
        if ($posts):
            ?>
            <div class="post-carousel11">
                <div class="swiper posts-carousel" data-slider='<?php echo json_encode($slider_data); ?>'>
                    <div class="swiper-wrapper">
                        <?php
                        while ($posts->have_posts()): $posts->the_post();
                            $liked = ahura_is_liked_post(get_the_ID());
                            ?>
                            <div class="swiper-slide">
                                <div class="post-item-content ah-has-bg <?php echo !has_post_thumbnail() ? 'without-thumb' : ''; ?>">
                                    <div class="post-content-top">
                                        <div class="post-cover">
                                            <a href="<?php echo esc_url(get_the_permalink()); ?>">
                                                <?php echo get_the_post_thumbnail(get_the_ID(), $settings['item_cover_size']); ?>
                                            </a>
                                        </div>
                                        <div class="post-details ah-has-text">
                                            <div class="post-content-header">
                                                <?php
                                                $post_categories = wp_get_post_categories(get_the_ID(), ['fields' => 'all']);
                                                if ($post_categories && $settings['show_cats'] === 'yes'):
                                                    $first_cat = $post_categories[array_key_first($post_categories)];
                                                    ?>
                                                    <div class="post-cats">
                                                        <?php \Elementor\Icons_Manager::render_icon($settings['item_category_icon'], ['aria-hidden' => 'true']); ?>
                                                        <a href="<?php echo esc_url(get_category_link($first_cat->term_id)); ?>" class="item-cat"><?php echo esc_html($first_cat->name); ?></a>
                                                    </div>
                                                <?php endif; ?>
                                                <div class="post-title-wrap <?php echo $show_like_btn ? 'has-like' : ''; ?>">
                                                    <a href="<?php echo esc_url(get_the_permalink()); ?>">
                                                        <h2 class="post-title"><?php the_title(); ?></h2>
                                                    </a>
                                                    <?php if ($show_like_btn): ?>
                                                        <span class="ah-post-like-btn btn-post-like <?php echo $liked ? 'liked' : ''; ?>" data-like="true" data-post-id="<?php echo esc_attr(get_the_ID()); ?>">
                                                            <svg xmlns="http://www.w3.org/2000/svg" class="icon icon-tabler icon-tabler-heart" width="44" height="44" viewBox="0 0 24 24" stroke-width="1.5" stroke="#333" fill="#fefefe" stroke-linecap="round" stroke-linejoin="round">
                                                                <path stroke="none" d="M0 0h24v24H0z" fill="none"/>
                                                                <path d="M19.5 12.572l-7.5 7.428l-7.5 -7.428a5 5 0 1 1 7.5 -6.566a5 5 0 1 1 7.5 6.572"/>
                                                            </svg>
                                                        </span>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                            <?php if ($settings['item_excerpt_show'] === 'yes'): ?>
                                                <div class="post-excerpt">
                                                    <?php
                                                    echo $chars_num ? '<p>' . wp_trim_words(get_the_excerpt(), $chars_num, '...') . '</p>' : get_the_excerpt();
                                                    ?>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <div class="post-content-bottom">
                                        <?php if ($settings['item_btn_show'] === 'yes'): ?>
                                            <a class="item-btn item-main-btn" href="<?php echo esc_url(get_the_permalink()); ?>">
                                                <?php \Elementor\Icons_Manager::render_icon($settings['post_btn_icon'], ['aria-hidden' => 'true']); ?>
                                                <span><?php echo esc_html($settings['post_box_button_text']); ?></span>
                                            </a>
                                        <?php endif; ?>

                                        <?php if ($settings['item_sec_btn_show'] === 'yes'): ?>
                                            <a <?php $this->print_render_attribute_string('item_sec_btn_link'); ?> class="item-btn item-sec-btn">
                                                <?php \Elementor\Icons_Manager::render_icon($settings['item_sec_btn_icon'], ['aria-hidden' => 'true']); ?>
                                                <span><?php echo esc_html($settings['item_sec_btn_text']); ?></span>
                                            </a>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        <?php endwhile; ?>
                    </div>
                    <?php if ($has_paginate): ?>
                        <div class="swiper-pagination"></div>
                    <?php endif; ?>
                </div>
            </div>
            <?php
        else:
            $this->renderElementorNotice(esc_html__('Sorry, no posts were found for display.', 'ahura'));
        endif;
        wp_reset_postdata();
    }
}