import os
from mysgi_api.settings.base import BASE_DIR
import json, environ
from rest_framework import status
from rest_framework.views import APIView
from rest_framework.response import Response
from rest_framework.permissions import IsAuthenticated
from .models import Users, Roles, Permissions
from .serializers import *
from permissions import CheckUserAccessLevel
from drf_spectacular.utils import extend_schema, OpenApiResponse, OpenApiExample, OpenApiParameter, OpenApiTypes
import logging
from pagination import custom_pagination
from drf_yasg import openapi
from django.db.models import F, Value, CharField, Q, ExpressionWrapper
from django.views.decorators.csrf import csrf_exempt
from django.utils.decorators import method_decorator



env = environ.Env()
environ.Env.read_env(os.path.join(BASE_DIR, '.env'))


class LoginView(APIView):
    
    http_method_names = ['post']
    serializer_class = UserLogin
    
    @extend_schema(
        request = UserLogin,
        responses = {
            200: OpenApiResponse(response=AuthenticationResponseSerializer, description='User Login Response'),
            406: OpenApiResponse(response=FailedMessageResponseAccessSerializer, description='User Login Response'),
        }
    )
    def post(self, request):
        serializer = UserLogin(data=request.data)
        if serializer.is_valid():
            result = serializer.authenticate_user(request.data)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class LogoutView(APIView):
    
    permission_classes = [IsAuthenticated]
    serializer_class = LogoutSerializer
     
    def post(self, request):
        serializer = LogoutSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.clear_tokens(request.data, request.META)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class GuestRegisterView(APIView):
    
    serializer_class = GuestUserSerializer
    
    def post(self, request):
        serializer = GuestUserSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.register_guest(request.data)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)



class GetUsersView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    
    @extend_schema(
        parameters=[
            OpenApiParameter(name='count', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Count Paginate Parameter', required=False),
            OpenApiParameter(name='page_number', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Page Number Paginate Parameter', required=False),
            OpenApiParameter(name='filter', location=OpenApiParameter.QUERY, type=OpenApiTypes.STR, description='Search Filter string', required=False)
        ],
        responses={
            200: OpenApiResponse(response=GetUsersSerializer, description='Get Users'),
        }
    )
    def get(self, request):
        self.check_object_permissions(request, 'users')
        if request.GET.get('filter') == None:
            users = Users.objects.all().order_by("created_at").values('id', 'user_name', 'first_name', 'last_name', 'personnel_code', 'mobile', 'email', 'job_title', 'department', 'office_name', 'national_code', 'is_active', 'is_guest', 'group_permission', 'last_login', 'birthday', 'role')
        else:
            users = Users.objects.filter(
                    Q(user_name__icontains=request.GET.get('filter')) |
                    Q(first_name__icontains=request.GET.get('filter')) |
                    Q(last_name__icontains=request.GET.get('filter')) |
                    Q(personnel_code__icontains=request.GET.get('filter')) |
                    Q(mobile__icontains=request.GET.get('filter')) |
                    Q(email__icontains=request.GET.get('filter')) |
                    Q(job_title__icontains=request.GET.get('filter')) |
                    Q(department__icontains=request.GET.get('filter')) |
                    Q(office_name__icontains=request.GET.get('filter')) |
                    Q(national_code__icontains=request.GET.get('filter'))
                ).order_by("created_at").values('id', 'user_name', 'first_name', 'last_name', 'personnel_code', 'mobile', 'email', 'job_title', 'department', 'office_name', 'national_code', 'is_active', 'is_guest', 'group_permission', 'last_login', 'birthday', 'role')
        datas = custom_pagination(users, request.GET.get('count', 100000), request.GET.get('page_number', 1))
        return Response(datas, status=status.HTTP_200_OK)


class GetRoleUsersView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetRoleUsersSerializers
    
    @extend_schema(
        parameters=[
            OpenApiParameter(name='count', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Count Paginate Parameter', required=False),
            OpenApiParameter(name='page_number', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Page Number Paginate Parameter', required=False),
            OpenApiParameter(name='filter', location=OpenApiParameter.QUERY, type=OpenApiTypes.STR, description='Search Filter string', required=False)
        ],
        request = GetRoleUsersSerializers,
        responses={
            200: OpenApiResponse(response=GetRoleUsersResponseSerializers, description='Get User Roles')
        },
    )
    def get(self, request, role):
        
        serializer = GetRoleUsersSerializers(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'roles-users')
            result = serializer.get_role_users(request.data, role, request)
            return Response(result[1], status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)  


class GetRolePermissionsView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetRolePermissionsSerializers
    
    @extend_schema(
        request = GetRolePermissionsSerializers,
        responses={
            200: OpenApiResponse(response=GetRoleUsersResponseSerializers, description='Get Role Permissions')
        },
    )
    def get(self, request, role):
        
        serializer = GetRolePermissionsSerializers(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'role-manager')
            result = serializer.get_role_permission(request.data, role)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)  
    

class GetRolesView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetRolesSeializer
    
    @extend_schema(
        responses={
            200: GetRolesSeializer(many=True)
        }
    )
    def get(self, request):
        roles = Roles.objects.all()
        self.check_object_permissions(request, 'role-manager')
        serialized_data = GetRolesSeializer(instance=roles, many=True)
        return Response(data=serialized_data.data, status=status.HTTP_200_OK)


class AddRolesView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = AddRoleSeializer
     
    @extend_schema(
        request = AddRoleSeializer,
        responses = {
            200: OpenApiResponse(response=ResponseSerialized, description='Add Role And Assign permission'),
        }
    )
    def post(self, request):
        
        serializer = AddRoleSeializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'roles-create')
            result = serializer.add_role(request.data, request.META)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class UpdateRoleView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = UpdateRoleSeializer
    
    def put(self, request, pk):
        
        serializer = UpdateRoleSeializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'roles-update')
            result = serializer.update(request.data, pk)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
        

class DeleteRoleView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = DeleteRoleSerializer
    
    def delete(self, request, pk):
        
        serializer = DeleteRoleSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'roles-delete')
            result = serializer.delete(request.data, pk)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class RoleAssignmentView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = RoleAssignSerializer
    
    @extend_schema(
        request = RoleAssignSerializer,
        responses = {
            200: OpenApiResponse(response=ResponseSerialized, description='Set Roles For Users'),
        }
    )
    def post(self, request):
        
        serializer = RoleAssignSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'roles-assign')
            result = serializer.create(request.data)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class GetPermissionsView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetPermissionSerializer
    
    @extend_schema(
        responses={
            200: GetPermissionSerializer(many=True)
        }
    )
    def get(self, request):
        permissions = Permissions.objects.all()
        self.check_object_permissions(request, 'users-permissions-get')
        serialized_data = GetPermissionSerializer(instance=permissions, many=True)
        return Response(data=serialized_data.data, status=status.HTTP_200_OK)


class AddPermissionsView(APIView):
    
    """
    Attention: if role has no parent_id , set 0 value on Create requests
    """
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = AddPermissionSerializer
    
    def post(self, request):
        
        serializer = AddPermissionSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'users-permissions')
            result = serializer.save()
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class UpdatePermissionView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = UpdatePermissionSeializer
    
    def put(self, request, pk):
        
        serializer = UpdatePermissionSeializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'roles-update')
            result = serializer.update(request.data, pk)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class DeletePermissionView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = DeletePermissionSerializer
    
    def delete(self, request, pk):
        serializer = DeletePermissionSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'roles-delete')
            result = serializer.delete(request.data, pk)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class PermissionAssignUserView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = PermissionAssignUserSerializer
    
    @extend_schema(
        request = PermissionAssignUserSerializer,
        responses = {
            200: OpenApiResponse(response=ResponseSerialized, description='Assign Permission To Users'),
        }
    )
    def post(self, request):
        serializer = PermissionAssignUserSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'roles-assign')
            result = serializer.assign_permission(request.data)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class PermissionAssignRoleView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = PermissionAssignRoleSerializer
    
    @extend_schema(
        request = PermissionAssignRoleSerializer,
        responses = {
            200: OpenApiResponse(response=ResponseSerialized, description='Assign Permission To Roles'),
        }
    )
    def post(self, request):
        serializer = PermissionAssignRoleSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'roles-assign')
            result = serializer.assign_permission_roles(request.data)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class GetUsersAccesslevel(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetUsersAccesslevelSerializer
    
    @extend_schema(
        request = GetUsersAccesslevelSerializer,
        responses = {
            200: OpenApiResponse(response=ResponseGetUsersAccesslevelSerializer, description='Get User Access Level'),
        }
    )
    def get(self, request, uid):
        serializer = GetUsersAccesslevelSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'users-permissions-get')
            result = serializer.get_user_access(request.data, uid)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class GetSidebarItemsView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetSidebarItemsSerializer

    @extend_schema(
        request = GetSidebarItemsSerializer,
        responses = {
            200: OpenApiResponse(response=GetSidebarItemsResponseSerializer, description='Sidebar Items'),
        }
    )
    def get(self, request):
        serializer = GetSidebarItemsSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.get_menu_items(request.data, request.META)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class GetInnerPagesView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetInnerPageSerializer
    
    @extend_schema(
        request = GetInnerPageSerializer,
        responses = {
            200: OpenApiResponse(response=GetSidebarItemsResponseSerializer, description='Inner Page access'),
        }
    )
    def get(self, request, pagename):
        serializer = GetInnerPageSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, pagename)
            result = serializer.get_menu_items(request.data, request.META, pagename)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class GetSidebarBadgeView(APIView):
    
    permission_classes = [IsAuthenticated]
    serializer_class = GetSidebarBadgeSerializer
    @extend_schema(
        request = GetSidebarBadgeSerializer,
        responses = {
            200: OpenApiResponse(response=GetSidebarItemsResponseSerializer, description='Get Badge Counts'),
        }
    )
    def get(self, request):
        serializer = GetSidebarBadgeSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.get_items(request.data, request.META)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
    


class GetNoteView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    
    @extend_schema(
        request = GetNoteSerializer,
        responses = {
            200: OpenApiResponse(response=GetNoteResponseSerializer, description='Get Daily Notes for Users'),
        }
    )
    def get(self, request, start_date, end_date):
        serializer = GetNoteSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.get_item(request.data, request.META, start_date, end_date)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class AddNoteView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    
    @extend_schema(
        request = AddNoteSerializer,
        responses = {
            200: OpenApiResponse(response=FailedMessageResponseAccessSerializer, description='Add Daily Note for Users'),
        }
    )
    def post(self, request):
        serializer = AddNoteSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.add_item(request.data, request.META)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class UpdateNoteView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    
    @extend_schema(
        request = UpdateNoteSerializer,
        responses = {
            200: OpenApiResponse(response=FailedMessageResponseAccessSerializer, description='Add Daily Note for Users'),
        }
    )
    def put(self, request, id):
        serializer = UpdateNoteSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.update_item(request.data, request.META, id)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class DeleteNoteView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    
    @extend_schema(
        request = DeleteNoteSerializer,
        responses = {
            200: OpenApiResponse(response=FailedMessageResponseAccessSerializer, description='Add Daily Note for Users'),
        }
    )
    def delete(self, request, id):
        serializer = DeleteNoteSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.delete_item(request.data, request.META, id)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)




class ListPhoneView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = ListPhoneSerializer
    
    @extend_schema(
        parameters=[
            OpenApiParameter(name='count', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Count Paginate Parameter', required=False),
            OpenApiParameter(name='page_number', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Page Number Paginate Parameter', required=False),
            OpenApiParameter(name='filter', location=OpenApiParameter.QUERY, type=OpenApiTypes.STR, description='Search Filter string', required=False)
        ],
        request=ListPhoneSerializer,
        responses={
            200: OpenApiResponse(response=ListPhoneSerializer, description='Phone number List')
        },
    )
    def get(self, request):
        serializer = ListPhoneSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'user_phone-manage')
            result = serializer.get_list(request.data, request)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
        

class AddPhoneView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = AddPhoneSerializer
    
    def post(self, request):
        serializer = AddPhoneSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'user_phone-create')
            result = serializer.add_phone(request.data, request.META)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class EditPhoneView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    
    @extend_schema(
        request = GetPhoneSerializer,
        responses={200: OpenApiResponse(response=MessageDoneSerializer, description='Edit Phone Number')},
    )
    def get(self, request, id):
        serializer = GetPhoneSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'user_phone-edit')
            result = serializer.get_phone(id)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
    
    @extend_schema(
        request = EditPhoneSerializer,
        responses={200: OpenApiResponse(response=MessageDoneSerializer, description='Edit Phone Number')},
    )
    def post(self, request, id):
        serializer = EditPhoneSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'user_phone-edit')
            result = serializer.edit_phone(request.data, request.META, id)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class DeletePhoneView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = DeletePhoneSerializer
    
    def post(self, request, id):
        serializer = DeletePhoneSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'user_phone-delete')
            result = serializer.delete_phone(request.data, id)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)





# @method_decorator(csrf_exempt, name='dispatch')
class AuthorizationCheckerView(APIView):
    
    serializer_class = AuthorizationCheckerSerializer
    
    @extend_schema(
        request = AuthorizationCheckerSerializer,
        responses = {
            200: OpenApiResponse(response=MessageDoneSerializer, description='Check Authorization'),
            406: OpenApiResponse(response=FailedMessageSerializer, description='Check Authorization'),
        }
    )
    def post(self, request):
        serializer = AuthorizationCheckerSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.check_authorization(request.data, request.META)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


