from rest_framework import serializers
from usermanagement.models import Users
from .models import *
import os
import json
import random
from django.utils import timezone
import environ
from rest_framework_simplejwt.tokens import AccessToken, RefreshToken
from mysgi_api.settings.base import BASE_DIR
import datetime
from datetime import timedelta
from jalali_date import datetime2jalali, date2jalali
from django.db import transaction
from django.db.models import F, Value, CharField, Q, ExpressionWrapper, Count
from django.db.models.functions import Concat
from permissions import CheckUserAccessLevel, check_admin_permission
from ticket.signals import ticket_created, ticket_status_changed
from pagination import custom_pagination
from persiantools.jdatetime import JalaliDate
from django.db.models import Sum



env = environ.Env()
environ.Env.read_env(os.path.join(BASE_DIR, '.env'))


class MessageDoneSerializer(serializers.Serializer):
    
    result = serializers.CharField()
    message = serializers.CharField()


class FailedMessageDueToAccessSerializer(serializers.Serializer):
    
    result = serializers.BooleanField()
    message = serializers.CharField()



class ListTicketStatusSerializer(serializers.Serializer):
    
    def get_items(self):
        
        status_obj = TicketStatus.objects.all().values('id', 'status_name')
        return [True, status_obj]


class AddTicketDepartmentSerializer(serializers.Serializer):
    
    department_name = serializers.CharField(max_length=255)
    description = serializers.CharField(max_length=255)
    
    def add_department(self, validated_data, request_meta) -> list:
        try:
            TicketDepartment.objects.create(
                department_name=validated_data['department_name'],
                description=validated_data['description']
            )
            return [True, "دپارتمان تیکت اضافه شد"]
        except:
            return [False, "مشکلی رخ داده است"]
    



class DepartmentChoiceField(serializers.ChoiceField):
    def __init__(self, **kwargs):
        kwargs['choices'] = []
        super().__init__(**kwargs)

    def to_representation(self, obj):
        return obj
    
class FieldTypeChoiceField(serializers.ChoiceField):
    def __init__(self, **kwargs):
        kwargs['choices'] = []
        super().__init__(**kwargs)

    def to_representation(self, obj):
        return obj

class AddCustomFieldSerializer(serializers.Serializer):
    
    field_name = serializers.CharField(max_length=255, required=True)
    field_type = FieldTypeChoiceField()
    description = serializers.CharField(max_length=10000, required=False)
    select_options = serializers.CharField(max_length=10000, required=False)
    display_order = serializers.IntegerField(required=True)
    ticket_department = DepartmentChoiceField()
    required = serializers.BooleanField(default=False, required=False)
    related_fields = serializers.CharField(max_length=255, required=False)
    
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields['ticket_department'].choices = self.get_ticket_department_choices()
        self.fields['field_type'].choices = self.get_field_type_choices()

    def get_ticket_department_choices(self):
        departments = TicketDepartment.objects.all()
        return [(str(department.id), department.department_name) for department in departments]
    
    def get_field_type_choices(self):
        fieldtypes = FieldType.objects.all()
        return [(str(field.id), field.field_type_name) for field in fieldtypes]
    
    def add_customfield(self, validated_data, request_meta) -> list:
        if validated_data['related_fields'] == '' or validated_data['related_fields'] is None:
            rel_field = None
        else:
            rel_field = CustomFields.objects.filter(field_name=validated_data['related_fields'], ticket_department_id=validated_data['ticket_department']).values_list('id', flat=True)
            print(rel_field)
            if rel_field.count() == 0:
                return [False, 'فیلد مرتبط وارد شده معتبر نمیباشد']
            else:
                rel_field = rel_field[0]
        
        '''
        Check CustomField string and ID and Ticket Department
        '''
        field_type_obj = FieldType.objects.filter(id=validated_data['field_type'])
        if field_type_obj.count() == 0:
            return [False, 'نوع فیلد مربوطه یافت نشد']
        
        ticket_department_obj = TicketDepartment.objects.filter(id=validated_data['ticket_department'])
        if ticket_department_obj.count() == 0:
            return [False, 'تیکت دپارتمان مربوطه یافت نشد']
        
        if validated_data['required'] == 'true' or validated_data['required'] == 'True':
            data_req = True
        else:
            data_req = False
        
        CustomFields.objects.create(
            field_name=validated_data['field_name'], field_type_id=validated_data['field_type'],
            description=validated_data['description'], select_options=validated_data['select_options'],
            display_order=validated_data['display_order'], ticket_department_id=validated_data['ticket_department'],
            related_field_id=rel_field, required=data_req
        )
        return [True, "فیلد اضافه شد"]


class UpdateCustomFieldSerializer(serializers.Serializer):
    
    field_name = serializers.CharField(max_length=255, required=True)
    field_type = FieldTypeChoiceField()
    description = serializers.CharField(max_length=10000, required=False)
    select_options = serializers.CharField(max_length=10000, required=False)
    display_order = serializers.IntegerField(required=True)
    ticket_department = DepartmentChoiceField()
    required = serializers.BooleanField(default=False, required=False)
    related_fields = serializers.CharField(max_length=255, required=False)
    
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields['ticket_department'].choices = self.get_ticket_department_choices()
        self.fields['field_type'].choices = self.get_field_type_choices()

    def get_ticket_department_choices(self):
        departments = TicketDepartment.objects.all()
        return [(str(department.id), department.department_name) for department in departments]
    
    def get_field_type_choices(self):
        fieldtypes = FieldType.objects.all()
        return [(str(field.id), field.field_type_name) for field in fieldtypes]
    
    def update_customfield(self, validated_data, request_meta, cfid) -> list:
        
        custom_field_obj = CustomFields.objects.filter(id=cfid)
        if custom_field_obj.count() == 0:
            return [False, 'فیلد یافت نشد']
        
        if validated_data['related_fields'] == '' or validated_data['related_fields'] is None:
            rel_field = None
        else:
            rel_field = CustomFields.objects.filter(field_name=validated_data['related_fields'], ticket_department_id=validated_data['ticket_department']).values_list('id', flat=True)
            print(rel_field)
            if rel_field.count() == 0:
                return [False, 'فیلد مرتبط وارد شده معتبر نمیباشد']
            else:
                rel_field = rel_field[0]
        
        custom_field_obj.update(
            field_name=validated_data['field_name'], field_type=validated_data['field_type'],
            description=validated_data['description'], select_options=validated_data['select_options'],
            display_order=validated_data['display_order'], ticket_department_id=validated_data['ticket_department'],
            related_field_id=rel_field
        )
        return [True, 'فیلد مربوطه به روزرسانی شد']


class DeleteCustomFieldSerializer(serializers.Serializer):
    
    def delete_customfield(self, validated_data, request_meta, cfid) -> list:
        
        custom_field_obj = CustomFields.objects.filter(id=cfid)
        if custom_field_obj.count() == 0:
            return [False, 'فیلد یافت نشد']
        
        custom_field_obj.delete()
        return [True, 'فیلد مربوطه حذف شد']


class GetCustomFieldSerializer(serializers.Serializer):
    
    def get_customfield(self, validated_data, request_meta, dname) -> list:
        
        department_obj = TicketDepartment.objects.filter(department_name=dname)
        if department_obj.count() == 0:
            return [False, 'دپارتمان مربوطه یافت نشد']
        
        custom_field_obj = CustomFields.objects.filter(ticket_department_id=department_obj.values_list('id', flat=True)[0]).values()
        return [True, custom_field_obj]
        



class TicketTemplateResponseSerializer(serializers.Serializer):
    result = serializers.BooleanField()
    message = serializers.JSONField()

class GetTicketTemplateSerializer(serializers.Serializer):
    
    def get_template(self, validated_data, request_meta, dname) -> list:
        
        department_obj = TicketDepartment.objects.filter(department_name=dname)
        if department_obj.count() == 0:
            return [False, 'دپارتمان مربوطه یافت نشد']
        
        custom_field_obj = CustomFields.objects.filter(ticket_department_id=department_obj.values_list('id', flat=True)[0]).values()
        last = list()
        for obj in custom_field_obj:
            obj['field_type'] = FieldType.objects.filter(id=obj['field_type_id']).values_list('field_type_name', flat=True)[0]
            obj.pop('field_type_id')
            obj['ticket_department'] = TicketDepartment.objects.filter(id=obj['ticket_department_id']).values_list('department_name', flat=True)[0]
            obj.pop('ticket_department_id')
            last.append(obj)
        return [True, last]




class ListTicketDataResponseSerializer(serializers.Serializer):
    id = serializers.IntegerField()
    user_name = serializers.CharField()
    status_name = serializers.CharField()
    title = serializers.CharField()
    department_name = serializers.CharField()
    created_at = serializers.DateTimeField()
    updated_at = serializers.DateTimeField()
    closed_date = serializers.DateTimeField()
    priority = serializers.CharField()
    service = serializers.CharField()
    title = serializers.CharField()
    result = serializers.CharField()
class ListTicketResponseSerializer(serializers.Serializer):
    datas = ListTicketDataResponseSerializer(many=True)
    has_next = serializers.BooleanField()
    has_previous = serializers.BooleanField()
    page_number = serializers.IntegerField()
    next_url = serializers.CharField()
    previous_url = serializers.CharField()
    total_page = serializers.IntegerField()
    total_items = serializers.IntegerField()
class ListTicketSerializer(serializers.Serializer):
    
    def get_ticket_list(self, validated_data, request_meta, dname, request) -> list:
        
        if 'user_name' not in validated_data or validated_data['user_name'] == '' or validated_data['user_name'] is None:
            jwt_token = request_meta.get('HTTP_AUTHORIZATION', '').split(' ')[1]
            access_token = AccessToken(jwt_token)
            user_id = access_token['user_id']
            user_object = Users.objects.filter(id=user_id)
        else:
            user_object = Users.objects.filter(user_name=validated_data['user_name'])
            if user_object.count() < 1:
                return [False, 'کاربر با این نام کاربری وجود ندارد']
            else:
                user_id = user_object.values_list('id', flat=True)[0]
            
        '''
        check user has manager permission or not
        '''
        is_admin = check_admin_permission(user_id, user_object, 'it_requests-manage')
        
        department_obj = TicketDepartment.objects.filter(department_name=dname)
        if department_obj.count() == 0:
            return [False, 'دپارتمان مربوطه یافت نشد']
        
        
        
        if is_admin:
            if request.GET.get('filter') == None:
                ticket_objs = Ticket.objects.filter(department_id=department_obj.values_list('id', flat=True)[0],status__in=[1,2]).annotate(user_name=Concat(F('user__first_name'), Value(' '), F('user__last_name'), output_field=CharField()), status_name=F('status__status_name'), department_name=F('user__department')).order_by("-created_at").values('id', 'user_name', 'title', 'status_name', 'department_name', 'closed_date', 'created_at', 'updated_at')
            else:
                ticket_objs = Ticket.objects.filter(department_id=department_obj.values_list('id', flat=True)[0]).filter(
                        Q(id__icontains=request.GET.get('filter')) |
                        Q(title__icontains=request.GET.get('filter')) |
                        Q(user__last_name__icontains=request.GET.get('filter')) |
                        Q(user__first_name__icontains=request.GET.get('filter')) |
                        Q(status__status_name__icontains=request.GET.get('filter')) |
                        Q(user__department__icontains=request.GET.get('filter'))
                    ).annotate(user_name=Concat(F('user__first_name'), Value(' '), F('user__last_name'), output_field=CharField()), status_name=F('status__status_name'), department_name=F('user__department')).order_by("-created_at").values('id', 'user_name', 'title', 'status_name', 'department_name', 'closed_date', 'created_at', 'updated_at')
        else:
            if request.GET.get('filter') == None:
                ticket_objs = Ticket.objects.filter(user_id=user_id, department_id=department_obj.values_list('id', flat=True)[0]).annotate(user_name=Concat(F('user__first_name'), Value(' '), F('user__last_name'), output_field=CharField()), status_name=F('status__status_name'), department_name=F('user__department')).order_by("-created_at").values('id', 'user_name', 'title', 'status_name', 'department_name', 'closed_date', 'created_at', 'updated_at')
            else:
                ticket_objs = Ticket.objects.filter(user_id=user_id, department_id=department_obj.values_list('id', flat=True)[0]).filter(
                        Q(id__icontains=request.GET.get('filter')) |
                        Q(title__icontains=request.GET.get('filter')) |
                        Q(user__last_name__icontains=request.GET.get('filter')) |
                        Q(user__first_name__icontains=request.GET.get('filter')) |
                        Q(status__status_name__icontains=request.GET.get('filter')) |
                        Q(user__department__icontains=request.GET.get('filter'))
                    ).annotate(user_name=Concat(F('user__first_name'), Value(' '), F('user__last_name'), output_field=CharField()), status_name=F('status__status_name'), department_name=F('user__department')).order_by("-created_at").values('id', 'user_name', 'title', 'status_name', 'department_name', 'closed_date', 'created_at', 'updated_at')

        
        last = list()
        # print(ticket_objs)
        if len(ticket_objs) == 0:
            return [True, {}]
        for one_ticket in ticket_objs:
            print(one_ticket['id'])
            one_item = {}
            # title_custom_field_id = CustomFields.objects.filter(field_name='عنوان').values_list('id', flat=True)[0]
            title_value = CustomFieldValues.objects.filter(custom_field_id=1, ticket_id=one_ticket['id']).values_list('value', flat=True)[0]
            # service_custom_field_id = CustomFields.objects.filter(field_name='نوع خدمات').values_list('id', flat=True)[0]
            service_value = CustomFieldValues.objects.filter(custom_field_id=5, ticket_id=one_ticket['id']).values_list('value', flat=True)[0]
            # priority_custom_field_id = CustomFields.objects.filter(field_name='فوریت درخواست').values_list('id', flat=True)[0]
            priority_value = CustomFieldValues.objects.filter(custom_field_id=3, ticket_id=one_ticket['id']).values_list('value', flat=True)[0]
            try:
                last_result = TicketReply.objects.filter(ticket_id=one_ticket['id'], is_staff=1).order_by('-id').values_list('message', flat=True)[0]
            except:
                last_result = ''
            one_item['id'] = one_ticket['id']
            one_item['user_name'] = one_ticket['user_name']
            one_item['title'] = title_value
            one_item['service'] = service_value
            one_item['status_name'] = one_ticket['status_name']
            one_item['department_name'] = one_ticket['department_name']
            one_item['closed_date'] = one_ticket['closed_date']
            one_item['created_at'] = one_ticket['created_at']
            one_item['updated_at'] = one_ticket['updated_at']
            one_item['priority'] = priority_value
            one_item['result'] = last_result
            last.append(one_item)
        
        datas = custom_pagination(last, request.GET.get('count', 10), request.GET.get('page_number', 1))
        return [True, datas]


class OpenTicketMessageSerializer(serializers.Serializer):

    ticket_id = serializers.CharField()

class OpenTicketResponseSerializer(serializers.Serializer):
    
    result = serializers.BooleanField()
    message = OpenTicketMessageSerializer()

class OpenTicketSerializer(serializers.Serializer):
    
    department_name = serializers.CharField(max_length=255, required=True)
    ticket_datas = serializers.JSONField()
    attachments = serializers.FileField(write_only=True, required=False)
    ticket_reply = serializers.CharField(max_length=100000, required=False)
    user_name = serializers.CharField(max_length=255, required=False)

    def open_ticket(self, validated_data, request_meta, request_files) -> list:
        
        department_obj = TicketDepartment.objects.filter(department_name=validated_data['department_name'])
        if department_obj.count() == 0:
            return [False, 'دپارتمان مربوطه یافت نشد']
        custom_fields_obj = CustomFields.objects.filter(ticket_department_id=department_obj.values_list('id', flat=True)[0])
        ticket_datas_obj = json.loads(validated_data['ticket_datas'])

        
        """
        Input Validation
        """
        if isinstance(ticket_datas_obj, list):
            counter = 0
            for ticket_obj in ticket_datas_obj:
                for obj in custom_fields_obj:
                    if obj.field_name in ticket_datas_obj[counter].keys():
                        pass
                    else:
                        return [False, 'send '+obj.field_name+' field']
                counter += 1
        else:
            for obj in custom_fields_obj:
                if obj.field_name in ticket_datas_obj.keys():
                    pass
                else:
                    return [False, 'send '+obj.field_name+' field']
            
        """
        Get User Object
        """
        if 'user_name' not in validated_data or validated_data['user_name'] == '' or validated_data['user_name'] is None:
            jwt_token = request_meta.get('HTTP_AUTHORIZATION', '').split(' ')[1]
            access_token = AccessToken(jwt_token)
            user_id = access_token['user_id']
        else:
            user_object = Users.objects.filter(user_name=validated_data['user_name'])
            if user_object.count() < 1:
                return [False, 'کاربر با این نام کاربری وجود ندارد']
            else:
                user_id = user_object.values_list('id', flat=True)[0]
        try:
            with transaction.atomic():
                """
                Add Ticket, Ticket Reply and attachments 
                Attention : Some tickets dont have Title.. we must generate random title for them
                """
                if isinstance(ticket_datas_obj, list):
                    ticket_title = 'درخواست شماره '+str(random.randint(10000, 500000))
                else:
                    ticket_title = ticket_datas_obj['عنوان']['value']


                ticket_obj = Ticket.objects.create(
                    title=ticket_title, user_id=user_id,
                    status_id=TicketStatus.objects.filter(status_name='درخواست جدید').values_list('id', flat=True)[0], department_id=department_obj.values_list('id', flat=True)[0]
                )
                # ticket_created.send(sender=Ticket, ticket=ticket_obj, department_name='خدمات فاوا')
                if validated_data['ticket_reply'] is not None or validated_data['ticket_reply'] != '':
                    ticket_reply_obj = TicketReply.objects.create(
                        ticket_id=ticket_obj.id, user_id=user_id,
                        message=validated_data['ticket_reply']
                    )

                    if 'attachments' in validated_data and ( request_files.getlist('attachments') != '' ):
                        for file1 in request_files.getlist('attachments'):
                            attach1 = Attachment.objects.create(reply_id=ticket_reply_obj.id, file=file1)
                            attach1.save()
                
                """
                Add Custom Filed Values
                """
                if isinstance(ticket_datas_obj, list):
                    counter = 0
                    for one_object in ticket_datas_obj:
                        for val in one_object.keys():
                            if 'جدول' in val:
                                pass
                            else:
                                CustomFieldValues.objects.create(
                                    custom_field_id=one_object[val]['id'], ticket_id=ticket_obj.id,
                                    value=one_object[val]['value']
                                )
                        counter += 1
                else:
                    for val in ticket_datas_obj.keys():
                        CustomFieldValues.objects.create(
                            custom_field_id=ticket_datas_obj[val]['id'], ticket_id=ticket_obj.id,
                            value=ticket_datas_obj[val]['value']
                        )
                return [True, {'ticket_id': str(ticket_obj.id)}]
                
        except Exception as e:
            transaction.rollback()
            return [False, 'ارور رخ داده است']


class AddReplyResponseSerializer(serializers.Serializer):
    
    result = serializers.BooleanField()
    message = serializers.CharField(max_length=255)

class AddReplySerializer(serializers.Serializer):
    
    message = serializers.CharField(max_length=225, required=True)
    user_name = serializers.CharField(max_length=255, required=False)
    attachments = serializers.FileField(write_only=True, required=False)
    is_staff = serializers.BooleanField(required=False)
    staff_name = serializers.CharField(max_length=60, required=False)
    
    def add_reply(self, validated_data, request_meta, request_files, tid) -> list:
        
        if 'user_name' in validated_data and ( validated_data['user_name'] != '' ):
            user_object = Users.objects.filter(user_name__exact=validated_data['user_name'])
            if user_object.count() < 1:
                return [False, 'کاربر با این نام کاربری وجود ندارد']
            else:
                user_id = user_object.values_list('id', flat=True)[0]
        else:
            jwt_token = request_meta.get('HTTP_AUTHORIZATION', '').split(' ')[1]
            access_token = AccessToken(jwt_token)
            user_id = access_token['user_id']
        
        
        
        ticket_update_obj = Ticket.objects.filter(id=tid)
        if ticket_update_obj.count() == 0:
            return [False, 'تیکت یافت نشد']
            
        
        if 'is_staff' in validated_data and ( validated_data['is_staff'] != '' and ( validated_data['is_staff'] == 'true' or validated_data['is_staff'] == True)):
            isstaff = True
            staff_user = validated_data['staff_name']
            ticket_obj = Ticket.objects.get(id=tid)
            ticket_old_status = ticket_update_obj.annotate(status_name=F('status__status_name')).values_list('status_name', flat=True)[0]
            Ticket.objects.filter(id=tid).update(
                status_id=TicketStatus.objects.filter(status_name="پاسخ داده شده").values_list('id', flat=True)[0],
                updated_at=timezone.now()
            )
            # ticket_status_changed.send(sender=Ticket, ticket=ticket_obj, old_status=ticket_old_status, new_status="پاسخ داده شده", department_name='خدمات فاوا')
        else:
            isstaff = False
            staff_user = None
            ticket_obj = Ticket.objects.get(id=tid)
            ticket_old_status = ticket_update_obj.annotate(status_name=F('status__status_name')).values_list('status_name', flat=True)[0]
            Ticket.objects.filter(id=tid).update(
                status_id=TicketStatus.objects.filter(status_name="پاسخ کاربر").values_list('id', flat=True)[0],
                updated_at=timezone.now()
            )
            # ticket_status_changed.send(sender=Ticket, ticket=ticket_obj, old_status=ticket_old_status, new_status="پاسخ کاربر", department_name='خدمات فاوا')
            
        ticket_reply = TicketReply.objects.create(
            ticket_id=tid, user_id=user_id,
            message=validated_data['message'], is_staff=isstaff,
            staff_name=staff_user
        )
        ticket_reply.save()
        
        if 'attachments' in validated_data and ( request_files.getlist('attachments') != '' ):
            for file1 in request_files.getlist('attachments'):
                attach1 = Attachment.objects.create(reply_id=ticket_reply.id, file=file1)
                attach1.save()
        return [True, 'پاسخ جدید ایجاد شد']
    
    class Meta:
        model = Ticket
        fields = ['message', 'user', 'attachments']


class GetTicketDetailSerializer(serializers.Serializer):
    
    def ticket_detail(self, validated_data, request_meta, tid) -> list:
        
        if 'user' in validated_data and ( validated_data['user'] != '' ):
            user_object = Users.objects.filter(email__exact=validated_data['user'])
            if user_object.count() < 1:
                return [False, 'کاربر با این نام کاربری وجود ندارد']
            else:
                user_id = user_object.values_list('id', flat=True)[0]
        else:
            jwt_token = request_meta.get('HTTP_AUTHORIZATION', '').split(' ')[1]
            access_token = AccessToken(jwt_token)
            user_id = access_token['user_id']
            user_object = Users.objects.filter(id=user_id)
        
        output = dict()
        ticket_detail = Ticket.objects.get(id=tid)
        output['title'] = ticket_detail.title
        print(ticket_detail.status.id)
        output['status'] = ticket_detail.status.status_name
        output['department'] = ticket_detail.department.department_name
        if ticket_detail.closed_date is not None:
            output['closed_date'] = ticket_detail.closed_date
        else:
            output['closed_date'] = None
        
        """
        GET Custom field and values
        """
        custom_field_obj = CustomFields.objects.filter(ticket_department_id=ticket_detail.department).values()
        last = list()
        
        for obj in custom_field_obj:
            if obj['field_type_id'] == 9:
                table_field = True
            else:
                table_field = False
        print(custom_field_obj)
        counter = 1
        total_obj = dict()
        for obj in custom_field_obj:
            print(obj)
            if table_field:
                if 'جدول' in obj['field_name']:
                    pass
                else:
                    obj['value'] = CustomFieldValues.objects.filter(custom_field_id=obj['id'], ticket_id=tid).values_list('value',flat=True)[0]
                    obj['field_type'] = FieldType.objects.filter(id=obj['field_type_id']).values_list('field_type_name', flat=True)[0]
                    obj.pop('field_type_id')
                    obj['ticket_department'] = TicketDepartment.objects.filter(id=obj['ticket_department_id']).values_list('department_name', flat=True)[0]
                    obj.pop('ticket_department_id')
                    total_obj[counter] = obj
                    last.append({counter:obj})
                    counter += 1
            else:
                obj['value'] = CustomFieldValues.objects.filter(custom_field_id=obj['id'], ticket_id=tid).values_list('value',flat=True)[0]
                obj['field_type'] = FieldType.objects.filter(id=obj['field_type_id']).values_list('field_type_name', flat=True)[0]
                obj.pop('field_type_id')
                obj['ticket_department'] = TicketDepartment.objects.filter(id=obj['ticket_department_id']).values_list('department_name', flat=True)[0]
                obj.pop('ticket_department_id')
                last.append(obj)
            
        output['detail_service'] = last
        
        output['user_name'] = ticket_detail.user.first_name +' '+ticket_detail.user.last_name
        
        if ticket_detail.updated_at is not None:
            output['updated_at'] = ticket_detail.updated_at
        else:
            output['updated_at'] = None
        if ticket_detail.created_at is not None:
            output['created_at'] = ticket_detail.created_at
        else:
            output['created_at'] = None
        output['replies_detail'] = list()
        replies_obj = TicketReply.objects.filter(ticket_id=tid).values_list('id', 'message', 'user', 'is_staff', 'staff_name', 'reply_at')
        output['admin_message'] = ""
        for reply in replies_obj:
            attchs_obj = Attachment.objects.filter(reply_id=reply[0]).values_list('file')
            attachs = list()
            for att in attchs_obj:
                attachs.append(att[0])
            reply_data = {'reply_id': reply[0], 'message': reply[1], 'user_id': reply[2], 'is_staff': reply[3], 'staff_name': reply[4], 'reply_at': reply[5], 'attachments': attachs}
            output['replies_detail'].append(reply_data)
            if reply[3] == True:
                output['admin_message'] = reply[1]
        return [True, output]


class ChangeTicketStatusResponseSerializer(serializers.Serializer):
    result = serializers.CharField()
    messsage = serializers.CharField()

class ChangeTicketStatusSerializer(serializers.Serializer):
    
    status = serializers.CharField(max_length=255, required=True)
    ticket_id = serializers.CharField(max_length=255, required=True)
    
    def change_status(self, validated_data, request_meta):
            
        '''
        check status and ticket exist
        '''
        status_obj = TicketStatus.objects.filter(status_name=validated_data['status'])
        if status_obj.count == 0:
            return [False, 'وضعیت مربوطه یافت نشد']
        ticket_obj = Ticket.objects.filter(id=validated_data['ticket_id'])
        if ticket_obj.count() == 0:
            return [False, 'تیکت مربوطه یافت نشد']
        
        ticket_obj.update(status_id=status_obj.values_list('id', flat=True)[0])
        ticket_old_status = ticket_obj.annotate(status_name=F('status__status_name')).values_list('status_name', flat=True)[0]
        # ticket_status_changed.send(sender=Ticket, ticket=Ticket.objects.get(id=validated_data['ticket_id']), old_status=ticket_old_status, new_status=status_obj.values_list('status_name', flat=True)[0], department_name='خدمات فاوا')
        return [True, 'وضعیت تیکت به روزرسانی شد']


class DeleteTicketResponseSerializer(serializers.Serializer):
    result = serializers.CharField()
    messsage = serializers.CharField()

class DeleteTicketSerializer(serializers.Serializer):
    
    def delete_ticket(self, validated_data, request_meta, tid):
        
        '''
        check ticket exist
        '''
        ticket_obj = Ticket.objects.filter(id=tid)
        if ticket_obj.count() == 0:
            return [False, 'تیکت مربوطه یافت نشد']
        if ticket_obj.values_list('status', flat=True)[0] == 3:
            return [False, 'تیکت بسته شده است']
        
        ticket_obj.delete()
        return [True, 'تیکت حذف شد']


class UpdateTicketmethodUpdateSerializer(serializers.Serializer):
    
    ticket_datas = serializers.JSONField()
    attachments = serializers.FileField(write_only=True, required=False)
    
    def update_ticket_and_reply(self, validated_data, request_meta, tid, request_files):
        
        if 'user_name' not in validated_data or validated_data['user_name'] == '' or validated_data['user_name'] is None:
            jwt_token = request_meta.get('HTTP_AUTHORIZATION', '').split(' ')[1]
            access_token = AccessToken(jwt_token)
            user_id = access_token['user_id']
            user_object = Users.objects.filter(id=user_id)
        else:
            user_object = Users.objects.filter(user_name=validated_data['user_name'])
            if user_object.count() < 1:
                return [False, 'کاربر با این نام کاربری وجود ندارد']
            else:
                user_id = user_object.values_list('id', flat=True)[0]
        
        is_admin = check_admin_permission(user_id, user_object, 'it_requests-manage')
        
        if is_admin:
            ticket_obj = Ticket.objects.filter(id=tid)
            if ticket_obj.count() == 0:
                return [False, 'تیکت مربوطه یافت نشد']
            ticket_datas = json.loads(validated_data['ticket_datas'])
            for cvalue in ticket_datas:
                if cvalue == 'عنوان':
                    ticket_obj.update(title=ticket_datas[cvalue]['value'])
                elif cvalue != 'متن تیکت':
                    print(ticket_datas[cvalue])
                    CustomFieldValues.objects.filter(id=ticket_datas[cvalue]['id']).update(value=ticket_datas[cvalue]['value'])
                else:
                    TicketReply.objects.filter(id=ticket_datas[cvalue]['id']).update(message=ticket_datas[cvalue]['value'])
                    if 'attachments' in validated_data and ( request_files.getlist('attachments') != '' ):
                        for file1 in request_files.getlist('attachments'):
                            attach_obj = Attachment.objects.filter(reply_id=ticket_datas[cvalue]['id'])
                            if attach_obj.count() == 0:
                                attach1 = Attachment.objects.create(reply_id=ticket_datas[cvalue]['id'], file=file1)
                                attach1.save()
                            else:
                                attach_obj.update(file=file1)
            return [True, 'آپدیت انجام شد']
        else:
            ticket_obj = Ticket.objects.filter(id=tid)
            if ticket_obj.count() == 0:
                return [False, 'تیکت مربوطه یافت نشد']
            tstatus_name = ticket_obj.annotate(status_name=F('status__status_name')).values_list('status_name', flat=True)[0]
            if tstatus_name != "درخواست جدید":
                return [False, 'امکان به روزرسانی نمیباشد']
            else:
                ticket_datas = json.loads(validated_data['ticket_datas'])
                for cvalue in ticket_datas:
                    if cvalue == 'عنوان':
                        ticket_obj.update(title=ticket_datas[cvalue]['value'])
                    elif cvalue != 'متن تیکت':
                        print(ticket_datas[cvalue])
                        CustomFieldValues.objects.filter(id=ticket_datas[cvalue]['id']).update(value=ticket_datas[cvalue]['value'])
                    else:
                        TicketReply.objects.filter(id=ticket_datas[cvalue]['id']).update(message=ticket_datas[cvalue]['value'])
                        if 'attachments' in validated_data and ( request_files.getlist('attachments') != '' ):
                            for file1 in request_files.getlist('attachments'):
                                attach_obj = Attachment.objects.filter(reply_id=ticket_datas[cvalue]['id'])
                                if attach_obj.count() == 0:
                                    attach1 = Attachment.objects.create(reply_id=ticket_datas[cvalue]['id'], file=file1)
                                    attach1.save()
                                else:
                                    attach_obj.update(file=file1)
                return [True, 'آپدیت انجام شد']


class UpdateTicketmethodGetSerializer(serializers.Serializer):
    
    
    def get_ticket(self, validated_data, request_meta, tid):
        
        if 'user' in validated_data and ( validated_data['user'] != '' ):
            user_object = Users.objects.filter(email__exact=validated_data['user'])
            if user_object.count() < 1:
                return [False, 'کاربر با این نام کاربری وجود ندارد']
            else:
                user_id = user_object.values_list('id', flat=True)[0]
        else:
            jwt_token = request_meta.get('HTTP_AUTHORIZATION', '').split(' ')[1]
            access_token = AccessToken(jwt_token)
            user_id = access_token['user_id']
            user_object = Users.objects.filter(id=user_id)
        
        output = dict()
        ticket_detail = Ticket.objects.get(id=tid)
        output['title'] = ticket_detail.title
        print(ticket_detail.status.id)
        output['status'] = ticket_detail.status.status_name
        output['department'] = ticket_detail.department.department_name
        
        """
        GET Custom field and values
        """
        custom_field_obj = CustomFields.objects.filter(ticket_department_id=ticket_detail.department).values()
        print(custom_field_obj)
        last = list()
        for obj in custom_field_obj:
            custom_field_value_object = CustomFieldValues.objects.filter(custom_field_id=obj['id'], ticket_id=tid)
            obj['value'] = custom_field_value_object.values_list('value',flat=True)[0]
            obj['field_type'] = FieldType.objects.filter(id=obj['field_type_id']).values_list('field_type_name', flat=True)[0]
            obj.pop('field_type_id')
            obj['ticket_department'] = TicketDepartment.objects.filter(id=obj['ticket_department_id']).values_list('department_name', flat=True)[0]
            obj.pop('ticket_department_id')
            obj['id'] = custom_field_value_object.values_list('id',flat=True)[0]
            last.append(obj)
        output['detail_service'] = last
        
        output['user_name'] = user_object.values_list('first_name', flat=True)[0]+' '+user_object.values_list('last_name', flat=True)[0]

        replies_obj = TicketReply.objects.filter(ticket_id=tid).values_list('id', 'message', 'user', 'is_staff', 'staff_name', 'reply_at')[0]
        attach_obj = Attachment.objects.filter(reply_id=replies_obj[0])
        if attach_obj.count() > 0:
            attachment = attach_obj.values_list('file',flat=True)[0]
        else:
            attachment = ''
        last.append({'field_name': 'متن تیکت', 'reply_id': replies_obj[0], 'value': replies_obj[1], 'attachment': attachment})
        return [True, output]
        

class SaveResponseITRequestsSerializer(serializers.Serializer):
    
    description = serializers.CharField(max_length=255, required=False)
    status = serializers.CharField(max_length=255)
    
    def save_response(self, validated_data, request_meta, tid):
        
        if 'user' in validated_data and ( validated_data['user'] != '' ):
            user_object = Users.objects.filter(email__exact=validated_data['user'])
            if user_object.count() < 1:
                return [False, 'کاربر با این نام کاربری وجود ندارد']
            else:
                user_id = user_object.values_list('id', flat=True)[0]
        else:
            jwt_token = request_meta.get('HTTP_AUTHORIZATION', '').split(' ')[1]
            access_token = AccessToken(jwt_token)
            user_id = access_token['user_id']
            user_object = Users.objects.filter(id=user_id)

        status_id = TicketStatus.objects.filter(status_name=validated_data['status'])
        if status_id.count() == 0:
            return [False, 'وضعیت یافت نشد']
        ticket_obj = Ticket.objects.filter(id=tid)
        old_status = ticket_obj.annotate(status_name=F('status__status_name')).values_list('status_name', flat=True)[0]
        ticket_obj.update(status_id=status_id.values_list('id', flat=True)[0])

        if validated_data['description'] is not None or validated_data['description'] != '':
            ticket_owner = ticket_obj.values_list('user_id', flat=True)[0]
            TicketReply.objects.create(
                ticket_id=tid, message=validated_data['description'], user_id=ticket_owner,
                is_staff=True, staff_name=user_object.values_list('first_name', flat=True)[0] + ' '+user_object.values_list('last_name', flat=True)[0]
            )
        # ticket_status_changed.send(sender=Ticket, ticket=Ticket.objects.get(id=tid), old_status=old_status, new_status=validated_data['status'], department_name='خدمات فاوا')
        return [True, 'انجام شد']
            
            



class ITEquipmentsListunitsSerializer(serializers.ModelSerializer):
    
    class Meta:
        model = ITEquipmentUnits
        fields = '__all__'


class ITEquipmentsListDataResponseSerializer(serializers.Serializer):
    id = serializers.IntegerField()
    user_id = serializers.IntegerField()
    description = serializers.CharField(max_length=255)
    close = serializers.BooleanField()
    created_at = serializers.DateField()
    updated_at = serializers.DateField()
    status_name = serializers.CharField()
    department_name = serializers.CharField()
    result = serializers.CharField()
    detail = serializers.JSONField()
class ITEquipmentsListResponseSerializer(serializers.Serializer):
    datas = ITEquipmentsListDataResponseSerializer(many=True)
    has_next = serializers.BooleanField()
    has_previous = serializers.BooleanField()
    page_number = serializers.IntegerField()
    next_url = serializers.CharField()
    previous_url = serializers.CharField()
    total_page = serializers.IntegerField()
    total_items = serializers.IntegerField()
class ITEquipmentsListSerializer(serializers.Serializer):
    
    def get_items(self, validated_data, request_meta, request):
        
        if 'user_name' not in validated_data or validated_data['user_name'] == '' or validated_data['user_name'] is None:
            jwt_token = request_meta.get('HTTP_AUTHORIZATION', '').split(' ')[1]
            access_token = AccessToken(jwt_token)
            user_id = access_token['user_id']
            user_object = Users.objects.filter(id=user_id)
        else:
            user_object = Users.objects.filter(user_name=validated_data['user_name'])
            if user_object.count() < 1:
                return [False, 'کاربر با این نام کاربری وجود ندارد']
            else:
                user_id = user_object.values_list('id', flat=True)[0]
                
        '''
        check user has manager permission or not
        '''
        is_admin = check_admin_permission(user_id, user_object, 'it_equipment_requests-manage')
        
        if is_admin:
            if request.GET.get('filter') == None:
                it_request_obj = ITEquipmentRequests.objects.all().annotate(status_name=F('status__status_name'), user_name=Concat(F('user__first_name'), Value(' '), F('user__last_name'), output_field=CharField()), department_name=F('user__department')).values().order_by("-id")
            else:
                it_request_obj = ITEquipmentRequests.objects.all().filter(
                        Q(id__icontains=request.GET.get('filter')) |
                        Q(user__last_name__icontains=request.GET.get('filter')) |
                        Q(user__first_name__icontains=request.GET.get('filter')) |
                        Q(status__status_name__icontains=request.GET.get('filter')) |
                        Q(user__department__icontains=request.GET.get('filter'))
                    ).annotate(status_name=F('status__status_name'), user_name=Concat(F('user__first_name'), Value(' '), F('user__last_name'), output_field=CharField()), department_name=F('user__department')).values().order_by("-id")
        else:
            if request.GET.get('filter') == None:
                it_request_obj = ITEquipmentRequests.objects.filter(user_id=user_id).annotate(status_name=F('status__status_name'), user_name=Concat(F('user__first_name'), Value(' '), F('user__last_name'), output_field=CharField()), department_name=F('user__department')).values().order_by("-id")
            else:
                it_request_obj = ITEquipmentRequests.objects.filter(user_id=user_id).filter(
                        Q(id__icontains=request.GET.get('filter')) |
                        Q(user__last_name__icontains=request.GET.get('filter')) |
                        Q(user__first_name__icontains=request.GET.get('filter')) |
                        Q(status__status_name__icontains=request.GET.get('filter')) |
                        Q(user__department__icontains=request.GET.get('filter'))
                    ).annotate(status_name=F('status__status_name'), user_name=Concat(F('user__first_name'), Value(' '), F('user__last_name'), output_field=CharField()), department_name=F('user__department')).values().order_by("-id")
        datas = custom_pagination(it_request_obj, request.GET.get('count', 10), request.GET.get('page_number', 1))
        return [True, datas]


class ITEquipmentsAddSerializer(serializers.Serializer):
    
    equipment_list = serializers.JSONField()
    description = serializers.CharField(max_length=100000, required=False)
    user_name = serializers.CharField(required=False)

    def add_item(self, validated_data, request_meta):
        
        if 'user_name' in validated_data and ( validated_data['user_name'] != '' ):
            user_object = Users.objects.filter(user_name__exact=validated_data['user_name'])
            if user_object.count() < 1:
                return [False, 'کاربر با این نام کاربری وجود ندارد']
            else:
                user_id = user_object.values_list('id', flat=True)[0]
        else:
            jwt_token = request_meta.get('HTTP_AUTHORIZATION', '').split(' ')[1]
            access_token = AccessToken(jwt_token)
            user_id = access_token['user_id']
            user_object = Users.objects.filter(id=user_id)
        
        equipment_items = json.loads(validated_data['equipment_list'])
        
        if validated_data['description'] is None or validated_data['description'] == '':
            descr = ''
        else:
            descr = validated_data['description']
        
        equipment_obj = ITEquipmentRequests.objects.create(user_id=user_id, description=descr, status_id=TicketStatus.objects.filter(status_name='درخواست جدید').values_list('id'))
        for item in equipment_items:
            '''
            validate fields
            '''
            unit_obj=ITEquipmentUnits.objects.filter(id=item['unit'])
            if unit_obj.count() == 0:
                return [False, 'واحد یافت نشد']
            
            if item['description'] is None or item['description'] == '':
                item_descr = ''
            else:
                item_descr = item['description']
        
            ITEquipmentRequestDetail.objects.create(
                title=item['title'], unit_id=unit_obj.values_list('id', flat=True)[0],
                count=item['count'], description=item_descr,
                it_equipment_request_id=equipment_obj.id
            )
        # ticket_created.send(sender=ITEquipmentRequests, ticket=equipment_obj, department_name='تامین تجهیزات IT')
        return [True, 'انجام شد']
        

class ITEquipmentsDeleteSerializer(serializers.Serializer):
    
    def delete_item(self, validated_data, request_meta, tid):
        
        equipment_obj = ITEquipmentRequests.objects.filter(id=tid)
        if equipment_obj.count() == 0:
            return [False, 'درخواست مدنظرتان یافت نشد']
        
        equipment_obj.delete()
        return [True, 'حذف شد']


class ITEquipmentsUpdateGetMethodMessageDetailResponseSerializer(serializers.Serializer):
    id = serializers.IntegerField()
    it_equipment_request_id = serializers.IntegerField()
    title = serializers.CharField()
    unit_id = serializers.IntegerField()
    count = serializers.IntegerField()
    description = serializers.CharField()
    created_at = serializers.DateField()
    updated_at = serializers.DateField()

class ITEquipmentsUpdateGetMethodMessageResponseSerializer(serializers.Serializer):
    description = serializers.CharField()
    user_name = serializers.CharField()
    result = serializers.CharField()
    detail = ITEquipmentsUpdateGetMethodMessageDetailResponseSerializer(many=True)
    
class ITEquipmentsUpdateGetMethodResponseSerializer(serializers.Serializer):
    result = serializers.CharField()
    message = ITEquipmentsUpdateGetMethodMessageResponseSerializer()
    
class ITEquipmentsUpdateGetMethodSerializer(serializers.Serializer):
    
    def get_items(self, validated_data, request_meta, tid):
        
        if 'user_name' in validated_data and ( validated_data['user_name'] != '' ):
            user_object = Users.objects.filter(user_name__exact=validated_data['user_name'])
            if user_object.count() < 1:
                return [False, 'کاربر با این نام کاربری وجود ندارد']
            else:
                user_id = user_object.values_list('id', flat=True)[0]
        else:
            jwt_token = request_meta.get('HTTP_AUTHORIZATION', '').split(' ')[1]
            access_token = AccessToken(jwt_token)
            user_id = access_token['user_id']
            user_object = Users.objects.filter(id=user_id)
        
        '''
        check user has manager permission or not
        '''
        is_admin = check_admin_permission(user_id, user_object, 'it_equipment_requests-manage')
        
        if is_admin:
            request_object = ITEquipmentRequests.objects.filter(id=tid)
        else:
            request_object = ITEquipmentRequests.objects.filter(user_id=user_id, id=tid)
        
        
        if request_object.count() == 0:
            return [False, 'چنین درخواستی وجود ندارد']
        
        it_equipments_details = ITEquipmentRequestDetail.objects.filter(it_equipment_request_id=request_object.values_list('id',flat=True)[0]).values()
        last = dict()
        last['description'] = request_object.values_list('description', flat=True)[0]
        last['result'] = request_object.values_list('result', flat=True)[0]
        last['user_name'] = request_object.annotate(user_name=Concat(F('user__first_name'), Value(' '), F('user__last_name'), output_field=CharField())).values_list('user_name', flat=True)[0]
        last['department'] = 'تامین تجهیزات IT'
        last['detail'] = it_equipments_details
            
        return [True, last]


class ITEquipmentsUpdateUpdateMethodSerializer(serializers.Serializer):
    
    equipment_list = serializers.JSONField()
    description = serializers.CharField(max_length=100000, required=False)
    
    def update_items(self, validated_data, request_meta, tid):
        
        if 'user_name' not in validated_data or validated_data['user_name'] == '' or validated_data['user_name'] is None:
            jwt_token = request_meta.get('HTTP_AUTHORIZATION', '').split(' ')[1]
            access_token = AccessToken(jwt_token)
            user_id = access_token['user_id']
            user_object = Users.objects.filter(id=user_id)
        else:
            user_object = Users.objects.filter(user_name=validated_data['user_name'])
            if user_object.count() < 1:
                return [False, 'کاربر با این نام کاربری وجود ندارد']
            else:
                user_id = user_object.values_list('id', flat=True)[0]
        
        is_admin = check_admin_permission(user_id, user_object, 'it_equipment_requests-manage')
        
        if is_admin:
            pass
        else:
            ticket_obj = ITEquipmentRequests.objects.filter(id=tid)
            if ticket_obj.count() == 0:
                return [False, 'تیکت مربوطه یافت نشد']
            tstatus_name = ticket_obj.annotate(status_name=F('status__status_name')).values_list('status_name', flat=True)[0]
            if tstatus_name != "درخواست جدید":
                return [False, 'امکان به روزرسانی نمیباشد']
        try:
            with transaction.atomic():
                request_details_object = ITEquipmentRequestDetail.objects.filter(it_equipment_request_id=tid)
                if request_details_object.count() == 0:
                    return [False, 'چنین درخواستی وجود ندارد']
                
                request_details_object.delete()
                ITEquipmentRequests.objects.filter(id=tid).update(description=validated_data['description'])
                
                equipment_items = json.loads(validated_data['equipment_list'])
                
                for item in equipment_items:
                    '''
                    validate fields
                    '''
                    unit_obj=ITEquipmentUnits.objects.filter(id=item['unit'])
                    if unit_obj.count() == 0:
                        return [False, 'واحد یافت نشد']
                
                    ITEquipmentRequestDetail.objects.create(
                        title=item['title'], unit_id=unit_obj.values_list('id', flat=True)[0],
                        count=item['count'], description=item['description'],
                        it_equipment_request_id=tid
                    )
                return [True, 'انجام شد']
        except Exception as e:
            transaction.rollback()
            return [False, 'ارور رخ داده است']


class ITEquipmentsChangeStatusSerializer(serializers.Serializer):
    
    status = serializers.CharField(max_length=255, required=True)
    request_id = serializers.CharField(max_length=255, required=True)

    def change_status(self, validated_data, request_meta):
        
        '''
        check status and ticket exist
        '''
        status_obj = TicketStatus.objects.filter(status_name=validated_data['status'])
        if status_obj.count == 0:
            return [False, 'وضعیت مربوطه یافت نشد']
        request_obj = ITEquipmentRequests.objects.filter(id=validated_data['request_id'])
        if request_obj.count() == 0:
            return [False, 'درخواست مربوطه یافت نشد']
        old_status = request_obj.annotate(status_name=F('status__status_name')).values_list('status_name', flat=True)[0]
        request_obj.update(status_id=status_obj.values_list('id', flat=True)[0])
        # ticket_status_changed.send(sender=ITEquipmentRequests, ticket=ITEquipmentRequests.objects.get(id=validated_data['request_id']), old_status=old_status, new_status=validated_data['status'], department_name='تامین تجهیزات IT')
        return [True, 'وضعیت درخواست به روزرسانی شد']


class ITEquipmentsAdminResponseSerializer(serializers.Serializer):
    
    status = serializers.CharField(max_length=255, required=True)
    result = serializers.CharField(max_length=255, required=False)

    def save_response(self, validated_data, request_meta, tid):
        
        '''
        check status and ticket exist
        '''
        status_obj = TicketStatus.objects.filter(status_name=validated_data['status'])
        if status_obj.count == 0:
            return [False, 'وضعیت مربوطه یافت نشد']
        request_obj = ITEquipmentRequests.objects.filter(id=tid)
        if request_obj.count() == 0:
            return [False, 'درخواست مربوطه یافت نشد']
            
        if validated_data['result'] == '' or validated_data['result'] is None:
            tresult = ""
        else:
            tresult = validated_data['result']
        
        old_status = request_obj.annotate(status_name=F('status__status_name')).values_list('status_name', flat=True)[0]
        request_obj.update(status_id=status_obj.values_list('id', flat=True)[0], result=tresult)
        # ticket_status_changed.send(sender=ITEquipmentRequests, ticket=ITEquipmentRequests.objects.get(id=tid), old_status=old_status, new_status=validated_data['status'], department_name='تامین تجهیزات IT')
        return [True, 'وضعیت درخواست به روزرسانی شد']





class MeetingRequestContentTypeSerializer(serializers.Serializer):
    
    def get_items(self):
        
        queryset = MeetingRequestContentType.objects.values_list('name', flat=True)
        return [True, list(queryset)]


class MeetingRequestLocationSerializer(serializers.Serializer):
    
    def get_items(self):
        
        queryset = MeetingRequestLocation.objects.values('id', 'building', 'room')
        last = []
        for q in queryset:
            last.append({'id': q['id'], 'building': q['building'], 'room': q['room']})
        return [True, last]


class MeetingRequestServeSerializer(serializers.Serializer):
    
    def get_items(self):
        
        queryset = MeetingRequestServe.objects.values_list('name', flat=True)
        return [True, list(queryset)]


class MeetingRequestTypeSerializer(serializers.Serializer):
    
    def get_items(self):
        
        queryset = MeetingRequestType.objects.values_list('name', flat=True)
        return [True, list(queryset)]


class MeetingRequestGetSerializer(serializers.Serializer):
    
    start_date = serializers.DateField()
    end_date = serializers.DateField()
    
    def get_meetings(self, validated_data, request_meta, lid) -> list:
        
        if validated_data['start_date'] >= validated_data['end_date']:
            return [False, 'تاریخ شروع و پایان نمیتواند یکسان باشد']
        
        meet_time_ranges = ["07:00","07:30","08:00","08:30","09:00","09:30","10:00","10:30","11:00","11:30","12:00","12:30","13:00","13:30","14:00","14:30","15:00","15:30","16:00","16:30","17:00","17:30","18:00","18:30"]
        
        requests = MeetingRequest.objects.filter(status__in=[1,7,8,9,11]).filter(start_date_time__gte=validated_data['start_date'], end_date_time__lte=validated_data['end_date'], meeting_location_id=lid).annotate(status_name=F('status__status_name')).values('start_date_time', 'end_date_time', 'status_name', 'id')
        
        print(requests)
        
        start_date = datetime.datetime.strptime(validated_data['start_date'], '%Y-%m-%d')
        end_date = datetime.datetime.strptime(validated_data['end_date'], '%Y-%m-%d')
        
        current_date = start_date
        dates_between = []
        while current_date <= end_date:
            dates_between.append(current_date)
            current_date += timedelta(days=1)
        
        last = []
        for one_date in dates_between:
            days_json = {}
            for mrtime in meet_time_ranges:
                if len(requests) > 0:
                    for req in list(requests):
                        if datetime.datetime.strptime((str(one_date).replace(' 00:00:00', ' '+mrtime)), '%Y-%m-%d %H:%M') >= req['start_date_time'] and datetime.datetime.strptime((str(one_date).replace(' 00:00:00', ' '+mrtime)), '%Y-%m-%d %H:%M') < req['end_date_time'] :
                            
                            if str(one_date) in days_json:
                                days_json[str(one_date)][mrtime] = [req['status_name'], req['id']]
                                break
                            else:
                                days_json[str(one_date)] = {mrtime: [req['status_name'], req['id']]}
                                break
                        else:
                            print(one_date, mrtime, req['status_name'], 1111)
                            if str(one_date) in days_json:
                                days_json[str(one_date)][mrtime] = ["empty", None]
                            else:
                                days_json[str(one_date)] = {mrtime: ["empty", None]}
                else:
                    if str(one_date) in days_json:
                        days_json[str(one_date)][mrtime] = ["empty", None]
                    else:
                        days_json[str(one_date)] = {mrtime: ["empty", None]}
            last.append(days_json)
        
        
        
        return [True, last]
    

class MeetingRequestListSerializer(serializers.Serializer):
    
    def list_requests(self, validated_data, request_meta, request):
        
        if 'user_name' not in validated_data or validated_data['user_name'] == '' or validated_data['user_name'] is None:
            jwt_token = request_meta.get('HTTP_AUTHORIZATION', '').split(' ')[1]
            access_token = AccessToken(jwt_token)
            user_id = access_token['user_id']
            user_object = Users.objects.filter(id=user_id)
        else:
            user_object = Users.objects.filter(user_name=validated_data['user_name'])
            if user_object.count() < 1:
                return [False, 'کاربر با این نام کاربری وجود ندارد']
            else:
                user_id = user_object.values_list('id', flat=True)[0]
                
        '''
        check user has manager permission or not
        '''
        is_admin = check_admin_permission(user_id, user_object, 'meeting_requests-manage')
        
        if is_admin:
            if request.GET.get('filter') == None:
                all_meetings = MeetingRequest.objects.all().annotate(
                        user_name=Concat(F('user__first_name'), Value(' '), F('user__last_name'), output_field=CharField()), 
                        department_name=F('user__department'), 
                        location_name=Concat(F('meeting_location__building'), Value(' ,'), F('meeting_location__room'), output_field=CharField()),
                        status_name=F('status__status_name')
                    ).values('id', 'user_name', 'department_name', 'location_name', 'status_name', 'created_at', 'start_date_time', 'title').order_by('-created_at')
            else:
                all_meetings = MeetingRequest.objects.all().filter(
                        Q(id__icontains=request.GET.get('filter')) |
                        Q(title__icontains=request.GET.get('filter')) |
                        Q(user__last_name__icontains=request.GET.get('filter')) |
                        Q(user__first_name__icontains=request.GET.get('filter')) |
                        Q(status__status_name__icontains=request.GET.get('filter')) |
                        Q(user__department__icontains=request.GET.get('filter')) |
                        Q(meeting_location__building__icontains=request.GET.get('filter')) |
                        Q(meeting_location__room__icontains=request.GET.get('filter'))
                    ).annotate(
                        user_name=Concat(F('user__first_name'), Value(' '), F('user__last_name'), output_field=CharField()), 
                        department_name=F('user__department'), 
                        location_name=Concat(F('meeting_location__building'), Value(' ,'), F('meeting_location__room'), output_field=CharField()),
                        status_name=F('status__status_name')
                    ).values('id', 'user_name', 'department_name', 'location_name', 'status_name', 'created_at', 'start_date_time', 'title').order_by('-created_at')
        else:
            if request.GET.get('filter') == None:
                all_meetings = MeetingRequest.objects.filter(user_id=user_id).annotate(
                        user_name=Concat(F('user__first_name'), Value(' '), F('user__last_name'), output_field=CharField()), 
                        department_name=F('user__department'), 
                        location_name=Concat(F('meeting_location__building'), Value(' ,'), F('meeting_location__room'), output_field=CharField()),
                        status_name=F('status__status_name')
                    ).values('id', 'user_name', 'department_name', 'location_name', 'status_name', 'created_at', 'start_date_time', 'title').order_by('-created_at')
            else:
                all_meetings = MeetingRequest.objects.filter(user_id=user_id).filter(
                        Q(id__icontains=request.GET.get('filter')) |
                        Q(title__icontains=request.GET.get('filter')) |
                        Q(user__last_name__icontains=request.GET.get('filter')) |
                        Q(user__first_name__icontains=request.GET.get('filter')) |
                        Q(status__status_name__icontains=request.GET.get('filter')) |
                        Q(user__department__icontains=request.GET.get('filter')) |
                        Q(meeting_location__building__icontains=request.GET.get('filter')) |
                        Q(meeting_location__room__icontains=request.GET.get('filter'))
                    ).annotate(
                        user_name=Concat(F('user__first_name'), Value(' '), F('user__last_name'), output_field=CharField()), 
                        department_name=F('user__department'), 
                        location_name=Concat(F('meeting_location__building'), Value(' ,'), F('meeting_location__room'), output_field=CharField()),
                        status_name=F('status__status_name')
                    ).values('id', 'user_name', 'department_name', 'location_name', 'status_name', 'created_at', 'start_date_time', 'title').order_by('-created_at')
        
        datas = custom_pagination(all_meetings, request.GET.get('count', 10), request.GET.get('page_number', 1))
        return [True, datas]


class MeetingRequestAddSerializer(serializers.Serializer):
    
    internal_phone = serializers.CharField(max_length=255, required=True)
    title = serializers.CharField(max_length=255, required=True)
    people_count = serializers.IntegerField(required=True)
    meet_type = serializers.CharField(required=True)
    meeting_content_type = serializers.CharField(required=True)
    meeting_location = serializers.IntegerField(required=True)
    meeting_serve = serializers.CharField(required=True)
    start_date_time = serializers.DateTimeField(required=True)
    end_date_time = serializers.DateTimeField(required=True)
    description = serializers.CharField(required=False)
    user_name = serializers.CharField(required=False)
    
    def add_meet(self, validated_data, request_meta):
        
        if 'user_name' not in validated_data or validated_data['user_name'] == '' or validated_data['user_name'] is None:
            jwt_token = request_meta.get('HTTP_AUTHORIZATION', '').split(' ')[1]
            access_token = AccessToken(jwt_token)
            user_id = access_token['user_id']
            user_object = Users.objects.filter(id=user_id)
        else:
            user_object = Users.objects.filter(user_name=validated_data['user_name'])
            if user_object.count() < 1:
                return [False, 'کاربر با این نام کاربری وجود ندارد']
            else:
                user_id = user_object.values_list('id', flat=True)[0] 
        
        print(validated_data['meet_type'])
        meet_type = MeetingRequestType.objects.filter(name=validated_data['meet_type'])
        if meet_type.count() == 0:
            return [False, 'نوع ملاقات یافت نشد']
        
        meet_request_contet = MeetingRequestContentType.objects.filter(name=validated_data['meeting_content_type'])
        if meet_request_contet.count() == 0:
            return [False, 'نوع محتوای ملاقات یافت نشد']
        
        meet_serve = MeetingRequestServe.objects.filter(name=validated_data['meeting_serve'])
        if meet_serve.count() == 0:
            return [False, 'نوع پذیرایی یافت نشد']
        
        meet_location = MeetingRequestLocation.objects.filter(id=validated_data['meeting_location'])
        if meet_location.count() == 0:
            return [False, 'ساختمان یافت نشد']
        
        if validated_data['description'] is None or validated_data['description'] == '':
            descr = ''
        else:
            descr = validated_data['description']
        
        '''
        check meet location is free or not
        '''
        overlapping_reservations = MeetingRequest.objects.filter(status__in=[1,7,8,9,11,6,10], meeting_location_id=validated_data['meeting_location']).filter(
            Q(start_date_time__lte=validated_data['start_date_time'], end_date_time__gt=validated_data['start_date_time']) |
            Q(start_date_time__lt=validated_data['end_date_time'], end_date_time__gte=validated_data['end_date_time']) |
            Q(start_date_time__gte=validated_data['start_date_time'], end_date_time__lte=validated_data['end_date_time'])
        )
        if overlapping_reservations.exists():
            return [False, 'زمان از  قبل رزرو شده است']
        
        meeting_object = MeetingRequest.objects.create(
            user_id=user_id, internal_phone=validated_data['internal_phone'],
            people_count=validated_data['people_count'], meeting_type_id=meet_type.values_list('id', flat=True)[0],
            meeting_content_type_id=meet_request_contet.values_list('id', flat=True)[0],
            meeting_location_id=validated_data['meeting_location'], meeting_serve_id=meet_serve.values_list('id', flat=True)[0],
            start_date_time=validated_data['start_date_time'], end_date_time=validated_data['end_date_time'],
            description=descr, status_id=1, title=validated_data['title']
        )
        # ticket_created.send(sender=MeetingRequest, ticket=meeting_object, department_name='رزرو اتاق جلسه')
        return [True, 'درخواست شما اضافه شد']
    

class MeetingRequestEditSerializer(serializers.Serializer):

    internal_phone = serializers.CharField(max_length=255, required=True)
    title = serializers.CharField(max_length=255, required=True)
    people_count = serializers.IntegerField(required=True)
    meet_type = serializers.CharField(required=True)
    meeting_content_type = serializers.CharField(required=True)
    meeting_serve = serializers.CharField(required=True)
    description = serializers.CharField(required=False)
    
    def edit_meet(self, validated_data, request_meta,id):
        
        if 'user_name' not in validated_data or validated_data['user_name'] == '' or validated_data['user_name'] is None:
            jwt_token = request_meta.get('HTTP_AUTHORIZATION', '').split(' ')[1]
            access_token = AccessToken(jwt_token)
            user_id = access_token['user_id']
            user_object = Users.objects.filter(id=user_id)
        else:
            user_object = Users.objects.filter(user_name=validated_data['user_name'])
            if user_object.count() < 1:
                return [False, 'کاربر با این نام کاربری وجود ندارد']
            else:
                user_id = user_object.values_list('id', flat=True)[0] 
        
        meet_obj = MeetingRequest.objects.filter(id=id)
        if meet_obj.count() == 0:
            return [False, 'ملاقات یافت نشد']
        
        is_admin = check_admin_permission(user_id, user_object, 'meeting_requests-manage')
        
        if is_admin:
            pass
        else:
            tstatus_name = meet_obj.annotate(status_name=F('status__status_name')).values_list('status_name', flat=True)[0]
            if tstatus_name != "درخواست جدید":
                return [False, 'امکان به روزرسانی نمیباشد']
        
        meet_type = MeetingRequestType.objects.filter(name=validated_data['meet_type'])
        if meet_type.count() == 0:
            return [False, 'نوع ملاقات یافت نشد']
        
        meet_request_contet = MeetingRequestContentType.objects.filter(name=validated_data['meeting_content_type'])
        if meet_request_contet.count() == 0:
            return [False, 'نوع محتوای ملاقات یافت نشد']
        
        meet_serve = MeetingRequestServe.objects.filter(name=validated_data['meeting_serve'])
        if meet_serve.count() == 0:
            return [False, 'نوع پذیرایی یافت نشد']
        
        if validated_data['description'] is None or validated_data['description'] == '':
            descr = None
        else:
            descr = validated_data['description']
        
        
        
        meet_obj.update(
            internal_phone=validated_data['internal_phone'],
            people_count=validated_data['people_count'], meeting_type_id=meet_type.values_list('id', flat=True)[0],
            meeting_content_type_id=meet_request_contet.values_list('id', flat=True)[0],
            meeting_serve_id=meet_serve.values_list('id', flat=True)[0],
            description=descr, title=validated_data['title']
        )
        
        return [True, 'درخواست شما به روزرسانی شد']


class MeetingRequestDetailSerializer(serializers.Serializer):
    
    def get_item(self, validated_data, meta_request, id) -> list:
        
        meeting_obj = MeetingRequest.objects.filter(id=id)
        if meeting_obj.count() == 0:
            return [False, 'درخواست یافت نشد']
        
        requests = meeting_obj.annotate(
            member_name=Concat(F('user__first_name'), Value(' '), F('user__last_name'), output_field=CharField()),
            type_name=F('meeting_type__name'), content_type_name=F('meeting_content_type__name'),
            location_name=Concat(F('meeting_location__building'), Value(' '), F('meeting_location__room'), output_field=CharField()),
            serve_name=F('meeting_serve__name'), status_name=F('status__status_name')
            
        ).values('internal_phone', 'member_name', 'people_count', 'type_name', 'content_type_name', 'meeting_location', 'location_name', 'serve_name', 'status_name', 'close', 'start_date_time', 'end_date_time', 'title', 'description', 'result')

        for req in requests:
            meeting = {}
            meeting['internal_phone'] = req['internal_phone']
            meeting['member_name'] = req['member_name']
            meeting['people_count'] = req['people_count']
            meeting['type_name'] = req['type_name']
            meeting['content_type_name'] = req['content_type_name']
            meeting['location_name'] = req['location_name']
            meeting['location_id'] = req['meeting_location']
            meeting['serve_name'] = req['serve_name']
            meeting['status_name'] = req['status_name']
            meeting['close'] = req['close']
            meeting['title'] = req['title']
            meeting['description'] = req['description']
            meeting['start_date_time'] = req['start_date_time']
            meeting['end_date_time'] = req['end_date_time']
            meeting['result'] = req['result']
        return [True, meeting]


class MeetingRequestChangeStatusSerializer(serializers.Serializer):
    
    status = serializers.CharField(max_length=255, required=True)
    request_id = serializers.CharField(max_length=255, required=True)

    def change_status(self, validated_data, request_meta):
        
        '''
        check status and ticket exist
        '''
        status_obj = TicketStatus.objects.filter(status_name=validated_data['status'])
        if status_obj.count == 0:
            return [False, 'وضعیت مربوطه یافت نشد']
        request_obj = MeetingRequest.objects.filter(id=validated_data['request_id'])
        if request_obj.count() == 0:
            return [False, 'درخواست مربوطه یافت نشد']
        old_status = request_obj.annotate(status_name=F('status__status_name')).values_list('status_name', flat=True)[0]
        if 'result' not in validated_data:
            res = ''
        elif validated_data['result'] is None or validated_data['result'] == '':
            res = ''
        else:
            res = validated_data['result']
        request_obj.update(status_id=status_obj.values_list('id', flat=True)[0], result=res)
        # ticket_status_changed.send(sender=MeetingRequest, ticket=MeetingRequest.objects.get(id=validated_data['request_id']), old_status=old_status, new_status=validated_data['status'], department_name='رزرو اتاق جلسه')
        return [True, 'وضعیت درخواست به روزرسانی شد']
    

class MeetingRequestDeleteSerializer(serializers.Serializer):
    
    def delete_meet(self, validated_data, request_meta, id):
        
        if 'user_name' not in validated_data or validated_data['user_name'] == '' or validated_data['user_name'] is None:
            jwt_token = request_meta.get('HTTP_AUTHORIZATION', '').split(' ')[1]
            access_token = AccessToken(jwt_token)
            user_id = access_token['user_id']
            user_object = Users.objects.filter(id=user_id)
        else:
            user_object = Users.objects.filter(user_name=validated_data['user_name'])
            if user_object.count() < 1:
                return [False, 'کاربر با این نام کاربری وجود ندارد']
            else:
                user_id = user_object.values_list('id', flat=True)[0] 
        
        meeting_obj = MeetingRequest.objects.filter(id=id)
        if meeting_obj.count() == 0:
            return [False, 'درخواست یافت نشد']
        
        if meeting_obj.annotate(status_name=F('status__status_name')).values_list('status_name', flat=True)[0] == "بسته شده":
            return [False, 'تیکت بسته شده است و امکان حذف ندارد']

        meeting_obj.delete()
        return [True, 'تیکت حذف شد']





class FurnitureListunitsSerializer(serializers.ModelSerializer):
    
    class Meta:
        model = FurnitureUnits
        fields = '__all__'


class FurnitureListDataResponseSerializer(serializers.Serializer):
    id = serializers.IntegerField()
    user_id = serializers.IntegerField()
    description = serializers.CharField(max_length=255)
    close = serializers.BooleanField()
    created_at = serializers.DateField()
    updated_at = serializers.DateField()
    status_name = serializers.CharField()
    department_name = serializers.CharField()
    result = serializers.CharField()
    detail = serializers.JSONField()
class FurnitureListResponseSerializer(serializers.Serializer):
    datas = FurnitureListDataResponseSerializer(many=True)
    has_next = serializers.BooleanField()
    has_previous = serializers.BooleanField()
    page_number = serializers.IntegerField()
    next_url = serializers.CharField()
    previous_url = serializers.CharField()
    total_page = serializers.IntegerField()
    total_items = serializers.IntegerField()
class FurnitureListSerializer(serializers.Serializer):
    
    def get_items(self, validated_data, request_meta, request):
        
        if 'user_name' not in validated_data or validated_data['user_name'] == '' or validated_data['user_name'] is None:
            jwt_token = request_meta.get('HTTP_AUTHORIZATION', '').split(' ')[1]
            access_token = AccessToken(jwt_token)
            user_id = access_token['user_id']
            user_object = Users.objects.filter(id=user_id)
        else:
            user_object = Users.objects.filter(user_name=validated_data['user_name'])
            if user_object.count() < 1:
                return [False, 'کاربر با این نام کاربری وجود ندارد']
            else:
                user_id = user_object.values_list('id', flat=True)[0]
                
        '''
        check user has manager permission or not
        '''
        is_admin = check_admin_permission(user_id, user_object, 'furniture_requests-manage')
        
        if is_admin:
            if request.GET.get('filter') == None:
                furniture_request_obj = FurnitureRequests.objects.all().annotate(status_name=F('status__status_name'), user_name=Concat(F('user__first_name'), Value(' '), F('user__last_name'), output_field=CharField()), department_name=F('user__department')).values().order_by("-id")
            else:
                furniture_request_obj = FurnitureRequests.objects.all().filter(
                        Q(id__icontains=request.GET.get('filter')) |
                        Q(user__last_name__icontains=request.GET.get('filter')) |
                        Q(user__first_name__icontains=request.GET.get('filter')) |
                        Q(status__status_name__icontains=request.GET.get('filter')) |
                        Q(user__department__icontains=request.GET.get('filter'))
                    ).annotate(status_name=F('status__status_name'), user_name=Concat(F('user__first_name'), Value(' '), F('user__last_name'), output_field=CharField()), department_name=F('user__department')).values().order_by("-id")
        else:
            if request.GET.get('filter') == None:
                furniture_request_obj = FurnitureRequests.objects.filter(user_id=user_id).annotate(status_name=F('status__status_name'), user_name=Concat(F('user__first_name'), Value(' '), F('user__last_name'), output_field=CharField()), department_name=F('user__department')).values().order_by("-id")
            else:
                furniture_request_obj = FurnitureRequests.objects.filter(user_id=user_id).filter(
                        Q(id__icontains=request.GET.get('filter')) |
                        Q(user__last_name__icontains=request.GET.get('filter')) |
                        Q(user__first_name__icontains=request.GET.get('filter')) |
                        Q(status__status_name__icontains=request.GET.get('filter')) |
                        Q(user__department__icontains=request.GET.get('filter'))
                    ).annotate(status_name=F('status__status_name'), user_name=Concat(F('user__first_name'), Value(' '), F('user__last_name'), output_field=CharField()), department_name=F('user__department')).values().order_by("-id")
        datas = custom_pagination(furniture_request_obj, request.GET.get('count', 10), request.GET.get('page_number', 1))
        return [True, datas]


class FurnitureAddSerializer(serializers.Serializer):
    
    furniture_list = serializers.JSONField()
    description = serializers.CharField(max_length=100000, required=False)
    user_name = serializers.CharField(required=False)

    def add_item(self, validated_data, request_meta):
        
        if 'user_name' in validated_data and ( validated_data['user_name'] != '' ):
            user_object = Users.objects.filter(user_name__exact=validated_data['user_name'])
            if user_object.count() < 1:
                return [False, 'کاربر با این نام کاربری وجود ندارد']
            else:
                user_id = user_object.values_list('id', flat=True)[0]
        else:
            jwt_token = request_meta.get('HTTP_AUTHORIZATION', '').split(' ')[1]
            access_token = AccessToken(jwt_token)
            user_id = access_token['user_id']
            user_object = Users.objects.filter(id=user_id)
        
        furniture_items = json.loads(validated_data['furniture_list'])
        
        if validated_data['description'] is None or validated_data['description'] == '':
            descr = ''
        else:
            descr = validated_data['description']
        
        furniture_obj = FurnitureRequests.objects.create(user_id=user_id, description=descr, status_id=TicketStatus.objects.filter(status_name='درخواست جدید').values_list('id'))
        for item in furniture_items:
            '''
            validate fields
            '''
            unit_obj=FurnitureUnits.objects.filter(id=item['unit'])
            if unit_obj.count() == 0:
                return [False, 'واحد یافت نشد']
            
            if item['description'] is None or item['description'] == '':
                item_descr = ''
            else:
                item_descr = item['description']
        
            FurnitureRequestDetail.objects.create(
                title=item['title'], unit_id=unit_obj.values_list('id', flat=True)[0],
                count=item['count'], description=item_descr,
                furniture_request_id=furniture_obj.id
            )
        # ticket_created.send(sender=FurnitureRequests, ticket=furniture_obj, department_name='تامین تجهیزات پشتیبانی')
        return [True, 'انجام شد']
        

class FurnitureDeleteSerializer(serializers.Serializer):
    
    def delete_item(self, validated_data, request_meta, tid):
        
        furniture_obj = FurnitureRequests.objects.filter(id=tid)
        if furniture_obj.count() == 0:
            return [False, 'درخواست مدنظرتان یافت نشد']
        
        furniture_obj.delete()
        return [True, 'حذف شد']


class FurnitureUpdateGetMethodMessageDetailResponseSerializer(serializers.Serializer):
    id = serializers.IntegerField()
    furniture_request_id = serializers.IntegerField()
    title = serializers.CharField()
    unit_id = serializers.IntegerField()
    count = serializers.IntegerField()
    description = serializers.CharField()
    created_at = serializers.DateField()
    updated_at = serializers.DateField()

class FurnitureUpdateGetMethodMessageResponseSerializer(serializers.Serializer):
    description = serializers.CharField()
    user_name = serializers.CharField()
    result = serializers.CharField()
    detail = FurnitureUpdateGetMethodMessageDetailResponseSerializer(many=True)
    
class FurnitureUpdateGetMethodResponseSerializer(serializers.Serializer):
    result = serializers.CharField()
    message = FurnitureUpdateGetMethodMessageResponseSerializer()
    
class FurnitureUpdateGetMethodSerializer(serializers.Serializer):
    
    def get_items(self, validated_data, request_meta, tid):
        
        if 'user_name' in validated_data and ( validated_data['user_name'] != '' ):
            user_object = Users.objects.filter(user_name__exact=validated_data['user_name'])
            if user_object.count() < 1:
                return [False, 'کاربر با این نام کاربری وجود ندارد']
            else:
                user_id = user_object.values_list('id', flat=True)[0]
        else:
            jwt_token = request_meta.get('HTTP_AUTHORIZATION', '').split(' ')[1]
            access_token = AccessToken(jwt_token)
            user_id = access_token['user_id']
            user_object = Users.objects.filter(id=user_id)
        
        '''
        check user has manager permission or not
        '''
        is_admin = check_admin_permission(user_id, user_object, 'furniture_requests-manage')
        
        if is_admin:
            request_object = FurnitureRequests.objects.filter(id=tid)
        else:
            request_object = FurnitureRequests.objects.filter(user_id=user_id, id=tid)
        
        
        if request_object.count() == 0:
            return [False, 'چنین درخواستی وجود ندارد']
        
        furniture_details = FurnitureRequestDetail.objects.filter(furniture_request_id=request_object.values_list('id',flat=True)[0]).values()
        last = dict()
        last['description'] = request_object.values_list('description', flat=True)[0]
        last['result'] = request_object.values_list('result', flat=True)[0]
        last['user_name'] = request_object.annotate(user_name=Concat(F('user__first_name'), Value(' '), F('user__last_name'), output_field=CharField())).values_list('user_name', flat=True)[0]
        last['department'] = 'تامین تجهیزات پشتیبانی'
        last['detail'] = furniture_details
            
        return [True, last]


class FurnitureUpdateUpdateMethodSerializer(serializers.Serializer):
    
    furniture_list = serializers.JSONField()
    description = serializers.CharField(max_length=100000, required=False)
    
    def update_items(self, validated_data, request_meta, tid):
        
        if 'user_name' not in validated_data or validated_data['user_name'] == '' or validated_data['user_name'] is None:
            jwt_token = request_meta.get('HTTP_AUTHORIZATION', '').split(' ')[1]
            access_token = AccessToken(jwt_token)
            user_id = access_token['user_id']
            user_object = Users.objects.filter(id=user_id)
        else:
            user_object = Users.objects.filter(user_name=validated_data['user_name'])
            if user_object.count() < 1:
                return [False, 'کاربر با این نام کاربری وجود ندارد']
            else:
                user_id = user_object.values_list('id', flat=True)[0]
        
        is_admin = check_admin_permission(user_id, user_object, 'furniture_requests-manage')
        
        if is_admin:
            pass
        else:
            ticket_obj = FurnitureRequests.objects.filter(id=tid)
            if ticket_obj.count() == 0:
                return [False, 'تیکت مربوطه یافت نشد']
            tstatus_name = ticket_obj.annotate(status_name=F('status__status_name')).values_list('status_name', flat=True)[0]
            if tstatus_name != "درخواست جدید":
                return [False, 'امکان به روزرسانی نمیباشد']
        
        try:
            with transaction.atomic():
                request_details_object = FurnitureRequestDetail.objects.filter(furniture_request_id=tid)
                if request_details_object.count() == 0:
                    return [False, 'چنین درخواستی وجود ندارد']
                
                request_details_object.delete()
                FurnitureRequests.objects.filter(id=tid).update(description=validated_data['description'])
                
                furniture_items = json.loads(validated_data['furniture_list'])
                
                for item in furniture_items:
                    '''
                    validate fields
                    '''
                    unit_obj=FurnitureUnits.objects.filter(id=item['unit'])
                    if unit_obj.count() == 0:
                        return [False, 'واحد یافت نشد']
                
                    FurnitureRequestDetail.objects.create(
                        title=item['title'], unit_id=unit_obj.values_list('id', flat=True)[0],
                        count=item['count'], description=item['description'],
                        furniture_request_id=tid
                    )
                return [True, 'انجام شد']
        except Exception as e:
            transaction.rollback()
            return [False, 'ارور رخ داده است']


class FurnitureChangeStatusSerializer(serializers.Serializer):
    
    status = serializers.CharField(max_length=255, required=True)
    request_id = serializers.CharField(max_length=255, required=True)

    def change_status(self, validated_data, request_meta):
        
        '''
        check status and ticket exist
        '''
        status_obj = TicketStatus.objects.filter(status_name=validated_data['status'])
        if status_obj.count == 0:
            return [False, 'وضعیت مربوطه یافت نشد']
        request_obj = FurnitureRequests.objects.filter(id=validated_data['request_id'])
        if request_obj.count() == 0:
            return [False, 'درخواست مربوطه یافت نشد']
        old_status = request_obj.annotate(status_name=F('status__status_name')).values_list('status_name', flat=True)[0]
        request_obj.update(status_id=status_obj.values_list('id', flat=True)[0])
        # ticket_status_changed.send(sender=FurnitureRequests, ticket=FurnitureRequests.objects.get(id=validated_data['request_id']), old_status=old_status, new_status=validated_data['status'], department_name='تامین تجهیزات پشتیبانی')
        return [True, 'وضعیت درخواست به روزرسانی شد']


class FurnitureAdminResponseSerializer(serializers.Serializer):
    
    status = serializers.CharField(max_length=255, required=False)
    result = serializers.CharField(max_length=255, required=False)

    def save_response(self, validated_data, request_meta, tid):
        
        '''
        check status and ticket exist
        '''
        status_obj = TicketStatus.objects.filter(status_name=validated_data['status'])
        if status_obj.count == 0:
            return [False, 'وضعیت مربوطه یافت نشد']
        request_obj = FurnitureRequests.objects.filter(id=tid)
        if request_obj.count() == 0:
            return [False, 'درخواست مربوطه یافت نشد']
        old_status = request_obj.annotate(status_name=F('status__status_name')).values_list('status_name', flat=True)[0]
        if validated_data['result'] is not None or validated_data['result'] != '':
            request_obj.update(status_id=status_obj.values_list('id', flat=True)[0], result=validated_data['result'])
        else:
            request_obj.update(status_id=status_obj.values_list('id', flat=True)[0])
        # ticket_status_changed.send(sender=FurnitureRequests, ticket=FurnitureRequests.objects.get(id=tid), old_status=old_status, new_status=validated_data['status'], department_name='تامین تجهیزات پشتیبانی')
        return [True, 'وضعیت درخواست به روزرسانی شد']





class GamingRoomListTypesSerializer(serializers.Serializer):
    
    def get_types(self, validated_data, request_meta):
        
        queryset = GamingRequestType.objects.values('id', 'name')
        last = []
        for q in queryset:
            last.append({'id': q['id'], 'name': q['name']})
        return [True, last]


class GamingRoomListRequestSerializer(serializers.Serializer):
    
    def get_requests(self, validated_data, request_meta, request):
        
        if 'user_name' not in validated_data or validated_data['user_name'] == '' or validated_data['user_name'] is None:
            jwt_token = request_meta.get('HTTP_AUTHORIZATION', '').split(' ')[1]
            access_token = AccessToken(jwt_token)
            user_id = access_token['user_id']
            user_object = Users.objects.filter(id=user_id)
        else:
            user_object = Users.objects.filter(user_name=validated_data['user_name'])
            if user_object.count() < 1:
                return [False, 'کاربر با این نام کاربری وجود ندارد']
            else:
                user_id = user_object.values_list('id', flat=True)[0]
                
        '''
        check user has manager permission or not
        '''
        is_admin = check_admin_permission(user_id, user_object, 'gaming_room_requests-manage')
        
        if is_admin:
            if request.GET.get('filter') == None:
                all_rooms = GamingRoomRequest.objects.all().annotate(
                        user_name=Concat(F('applicant__first_name'), Value(' '), F('applicant__last_name'), output_field=CharField()), 
                        department_name=F('applicant__department'),
                        status_name=F('status__status_name'),
                        num_users=Count('users')
                    ).values('id', 'user_name', 'department_name', 'status_name', 'created_at', 'start_date_time', 'internal_phone', 'num_users').order_by('-created_at')
            else:
                all_rooms = GamingRoomRequest.objects.all().filter(
                        Q(id__icontains=request.GET.get('filter')) |
                        Q(internal_phone__icontains=request.GET.get('filter')) |
                        Q(applicant__last_name__icontains=request.GET.get('filter')) |
                        Q(applicant__first_name__icontains=request.GET.get('filter')) |
                        Q(status__status_name__icontains=request.GET.get('filter')) |
                        Q(applicant__department__icontains=request.GET.get('filter'))
                    ).annotate(
                        user_name=Concat(F('applicant__first_name'), Value(' '), F('applicant__last_name'), output_field=CharField()), 
                        department_name=F('applicant__department'), 
                        status_name=F('status__status_name'),
                        num_users=Count('users')
                    ).values('id', 'user_name', 'department_name', 'status_name', 'created_at', 'start_date_time', 'internal_phone', 'num_users').order_by('-created_at')
        else:
            if request.GET.get('filter') == None:
                all_rooms = GamingRoomRequest.objects.filter(applicant_id=user_id).annotate(
                        user_name=Concat(F('applicant__first_name'), Value(' '), F('applicant__last_name'), output_field=CharField()), 
                        department_name=F('applicant__department'),
                        status_name=F('status__status_name'),
                        num_users=Count('users')
                    ).values('id', 'user_name', 'department_name', 'status_name', 'created_at', 'start_date_time', 'internal_phone', 'num_users').order_by('-created_at')
            else:
                all_rooms = GamingRoomRequest.objects.filter(applicant_id=user_id).filter(
                        Q(id__icontains=request.GET.get('filter')) |
                        Q(internal_phone__icontains=request.GET.get('filter')) |
                        Q(applicant__last_name__icontains=request.GET.get('filter')) |
                        Q(applicant__first_name__icontains=request.GET.get('filter')) |
                        Q(status__status_name__icontains=request.GET.get('filter')) |
                        Q(applicant__department__icontains=request.GET.get('filter'))
                    ).annotate(
                        user_name=Concat(F('applicant__first_name'), Value(' '), F('applicant__last_name'), output_field=CharField()), 
                        department_name=F('applicant__department'),
                        status_name=F('status__status_name'),
                        num_users=Count('users')
                    ).values('id', 'user_name', 'department_name', 'status_name', 'created_at', 'start_date_time', 'internal_phone', 'num_users').order_by('-created_at')
        
        datas = custom_pagination(all_rooms, request.GET.get('count', 10), request.GET.get('page_number', 1))
        return [True, datas]


class GamingRoomAddRequestSerializer(serializers.Serializer):

    users = serializers.ListField()
    # start_datetime = serializers.DateTimeField()
    # end_datetime = serializers.DateTimeField()
    time_ranges = serializers.CharField()
    gaming_type = serializers.IntegerField(required=False)
    applicant = serializers.CharField()
    internal_phone = serializers.CharField()
    description = serializers.CharField(required=False)

    def add_requests(self, validated_data, request_meta):
        
        if 'user_name' not in validated_data or validated_data['user_name'] == '' or validated_data['user_name'] is None:
            jwt_token = request_meta.get('HTTP_AUTHORIZATION', '').split(' ')[1]
            access_token = AccessToken(jwt_token)
            user_id = access_token['user_id']
            user_object = Users.objects.filter(id=user_id)
        else:
            user_object = Users.objects.filter(user_name=validated_data['user_name'])
            if user_object.count() < 1:
                return [False, 'کاربر با این نام کاربری وجود ندارد']
            else:
                user_id = user_object.values_list('id', flat=True)[0]   
        
        lusers = validated_data['users'].split(',')
        if len(lusers) < 2:
            return [False, 'حداقل 2 نفر برای رزرو اتاق بازی نیاز است']
        applicantid = Users.objects.filter(user_name=validated_data['applicant']).values_list('id', flat=True)[0]
        
        all_times = validated_data['time_ranges'].split(',')
        if len(all_times) > 2:
            return [False, 'بازه انتخابی صحیح نمیباشد']
        
        for index, t in enumerate(all_times):
            if len(all_times) == 2:
                if index == 0:
                    starttime = t.split('_')[0]
                elif index == 1:
                    endtime = t.split('_')[1]
            elif len(all_times) == 1:
                starttime = t.split('_')[0]
                endtime = t.split('_')[1]
                
        ## check start and end time that user entered
        if endtime < starttime:
            return [False, 'بازه انتخابی صحیح نمیباشد']
        time_difference = datetime.datetime.strptime(endtime, '%Y-%m-%d %H:%M') - datetime.datetime.strptime(starttime, '%Y-%m-%d %H:%M')
        user_hours_requested = time_difference.total_seconds() / 3600
        print(user_hours_requested)
        if user_hours_requested > 1.0:
            return [False, 'امکان رزرو بیش از یک ساعت نمیباشد']
        
        ## check users maximum entertainment in a week
        today = JalaliDate.today()
        days_to_shambe = (today.weekday() - 0) % 7
        shambe_date = today - timedelta(days=days_to_shambe)
        days_to_jome = (6 - today.weekday()) % 7
        jome_date = today + timedelta(days=days_to_jome)
        current_week_start =  datetime.datetime.combine(shambe_date.to_gregorian(), datetime.datetime.min.time())
        current_week_end = datetime.datetime.combine(jome_date.to_gregorian(), datetime.datetime.max.time())
        while datetime.datetime.strptime(starttime, '%Y-%m-%d %H:%M') > current_week_end:
            current_week_start = current_week_start + timedelta(days=7)
            current_week_end = current_week_end + timedelta(days=7)
            
        for uid in lusers:
            total_hours_used_this_week = GamingRoomRequest.objects.filter(
                users__user_name=uid,
                start_date_time__gte=current_week_start,
                end_date_time__lte=current_week_end,
                status__in=[1,2,3,4,5,6,7,8,9,10,11]
            ).aggregate(total_hours=Sum('hours_requested'))['total_hours']
            if total_hours_used_this_week is not None and total_hours_used_this_week + user_hours_requested > 1.00:
                full_name = Users.objects.filter(user_name=uid).annotate(fulltime_username=Concat(('first_name'), Value(' '), F('last_name'), output_field=CharField())).values_list('fulltime_username', flat=True)[0]
                return [False, 'کاربر '+str(full_name)+' 1 ساعت هفتگی خود را استفاده کرده است']

        ## check time overlapping
        overlapping_reservations = GamingRoomRequest.objects.filter(status__in=[1,7,8,9,11,6,10]).filter(
            Q(start_date_time__lte=starttime, end_date_time__gt=starttime) |
            Q(start_date_time__lt=endtime, end_date_time__gte=endtime) |
            Q(start_date_time__gte=starttime, end_date_time__lte=endtime)
        )
        if overlapping_reservations.exists():
            return [False, 'زمان از  قبل رزرو شده است']
        
        if validated_data['description'] is None or validated_data['description'] == '':
            descr = ''
        else:
            descr = validated_data['description']
        
        gaming_room_request = GamingRoomRequest.objects.create(hours_requested=user_hours_requested, applicant_id=applicantid, start_date_time=starttime, end_date_time=endtime, status_id=1, internal_phone=validated_data['internal_phone'], description=descr)
        for usr in lusers:
            user1 = get_user_model().objects.filter(user_name=usr).values_list('id', flat=True)[0]
            gaming_room_request.users.add(user1)
        

        return [True, 'درخواست شما اضافه شد']


class GamingRoomGetTimeRangesSerializer(serializers.Serializer):
     
    start_date = serializers.DateField()
    end_date = serializers.DateField()
    
    def get_times(self, validated_data, request_meta) -> list:
        
        if validated_data['start_date'] >= validated_data['end_date']:
            return [False, 'تاریخ شروع و پایان نمیتواند یکسان باشد']
        
        meet_time_ranges = ["10:00_10:29","10:30_10:59","11:00_11:29","11:30_11:59", "12:00_12:29","12:30_12:59", "13:00_13:29", "13:30_13:59", "14:00_14:29", "14:30_14:59"]
        
        requests = GamingRoomRequest.objects.filter(status__in=[1,7,8,9,11]).filter(start_date_time__gte=validated_data['start_date'], end_date_time__lte=validated_data['end_date']).annotate(status_name=F('status__status_name')).values('start_date_time', 'end_date_time', 'status_name', 'id')
        
        start_date = datetime.datetime.strptime(validated_data['start_date'], '%Y-%m-%d')
        end_date = datetime.datetime.strptime(validated_data['end_date'], '%Y-%m-%d')
        
        current_date = start_date
        dates_between = []
        while current_date <= end_date:
            dates_between.append(current_date)
            current_date += timedelta(days=1)
        
        last = []
        for one_date in dates_between:
            days_json = {}
            for mrtime in meet_time_ranges:
                
                if len(requests) > 0:
                    for req in list(requests):
                        if datetime.datetime.strptime((str(one_date).replace(' 00:00:00', ' '+mrtime.split('_')[1])), '%Y-%m-%d %H:%M') >= req['start_date_time'] and datetime.datetime.strptime((str(one_date).replace(' 00:00:00', ' '+mrtime.split('_')[0])), '%Y-%m-%d %H:%M') < req['end_date_time'] :
                            
                            if str(one_date) in days_json:
                                days_json[str(one_date)][mrtime] = [req['status_name'], req['id']]
                                break
                            else:
                                days_json[str(one_date)] = {mrtime: [req['status_name'], req['id']]}
                                break
                        else:
                            if str(one_date) in days_json:
                                days_json[str(one_date)][mrtime] = ["empty", None]
                            else:
                                days_json[str(one_date)] = {mrtime: ["empty", None]}
                else:
                    if str(one_date) in days_json:
                        days_json[str(one_date)][mrtime] = ["empty", None]
                    else:
                        days_json[str(one_date)] = {mrtime: ["empty", None]}
            last.append(days_json)
        
        
        
        return [True, last]
 

class GamingRoomDeleteRequestSerializer(serializers.Serializer):
    
    def delete_room_request(self, validated_data, request_meta, id):
        
        if 'user_name' not in validated_data or validated_data['user_name'] == '' or validated_data['user_name'] is None:
            jwt_token = request_meta.get('HTTP_AUTHORIZATION', '').split(' ')[1]
            access_token = AccessToken(jwt_token)
            user_id = access_token['user_id']
            user_object = Users.objects.filter(id=user_id)
        else:
            user_object = Users.objects.filter(user_name=validated_data['user_name'])
            if user_object.count() < 1:
                return [False, 'کاربر با این نام کاربری وجود ندارد']
            else:
                user_id = user_object.values_list('id', flat=True)[0] 
        
        gameroom_obj = GamingRoomRequest.objects.filter(id=id)
        if gameroom_obj.count() == 0:
            return [False, 'درخواست یافت نشد']
        
        if gameroom_obj.annotate(status_name=F('status__status_name')).values_list('status_name', flat=True)[0] == "بسته شده":
            return [False, 'تیکت بسته شده است و امکان حذف ندارد']

        gameroom_obj.delete()
        return [True, 'تیکت حذف شد']


class GamingRoomRequestDetailSerializer(serializers.Serializer):
    
    def get_item(self, validated_data, meta_request, id) -> list:
        
        gamingroom_obj = GamingRoomRequest.objects.filter(id=id)
        if gamingroom_obj.count() == 0:
            return [False, 'درخواست یافت نشد']
        
        requests = gamingroom_obj.annotate(
            member_name=Concat(F('applicant__first_name'), Value(' '), F('applicant__last_name'), output_field=CharField()),
            type_name=F('game_type__name'), status_name=F('status__status_name'),
            user_name=Concat(F('users__first_name'), Value(' '), F('users__last_name'), output_field=CharField())
        ).values('internal_phone', 'member_name', 'type_name', 'status_name', 'start_date_time', 'end_date_time', 'description', 'user_name', 'result')

        user_names = []
        meetings = {}
        for req in requests:
            user_names.append(req['user_name'])
            meetings['internal_phone'] = req['internal_phone']
            meetings['member_name'] = req['member_name']
            meetings['type_name'] = req['type_name']
            meetings['status_name'] = req['status_name']
            meetings['start_date_time'] = req['start_date_time']
            meetings['end_date_time'] = req['end_date_time']
            meetings['description'] = req['description']
            meetings['result'] = req['result']
        
        meetings['users'] = user_names
        return [True, meetings]


class GamingRoomRequestEditSerializer(serializers.Serializer):

    internal_phone = serializers.CharField(max_length=255, required=True)
    gaming_type = serializers.IntegerField()
    description = serializers.CharField(required=False)
    
    def edit_room(self, validated_data, request_meta,id):
        
        if 'user_name' not in validated_data or validated_data['user_name'] == '' or validated_data['user_name'] is None:
            jwt_token = request_meta.get('HTTP_AUTHORIZATION', '').split(' ')[1]
            access_token = AccessToken(jwt_token)
            user_id = access_token['user_id']
            user_object = Users.objects.filter(id=user_id)
        else:
            user_object = Users.objects.filter(user_name=validated_data['user_name'])
            if user_object.count() < 1:
                return [False, 'کاربر با این نام کاربری وجود ندارد']
            else:
                user_id = user_object.values_list('id', flat=True)[0] 
        
        meet_obj = GamingRoomRequest.objects.filter(id=id)
        if meet_obj.count() == 0:
            return [False, 'ملاقات یافت نشد']
        
        is_admin = check_admin_permission(user_id, user_object, 'gaming_room_requests-manage')
        
        if is_admin:
            pass
        else:
            tstatus_name = meet_obj.annotate(status_name=F('status__status_name')).values_list('status_name', flat=True)[0]
            if tstatus_name != "درخواست جدید":
                return [False, 'امکان به روزرسانی نمیباشد']
        
        gaming_type = MeetingRequestType.objects.filter(id=validated_data['gaming_type'])
        if gaming_type.count() == 0:
            return [False, 'نوع بازی یافت نشد']
        
        if validated_data['description'] is None or validated_data['description'] == '':
            descr = ''
        else:
            descr = validated_data['description']
        
        meet_obj.update(
            game_type_id=gaming_type.values_list('id', flat=True)[0],
            description=descr, internal_phone=validated_data['internal_phone']
        )
        
        return [True, 'درخواست شما به روزرسانی شد']


class GamingRoomChangeStatusSerializer(serializers.Serializer):
    
    status = serializers.CharField(max_length=255, required=True)
    result = serializers.CharField(max_length=255, required=False)
    request_id = serializers.CharField(max_length=255, required=True)

    def change_status(self, validated_data, request_meta):
        
        '''
        check status and ticket exist
        '''
        status_obj = TicketStatus.objects.filter(status_name=validated_data['status'])
        if status_obj.count == 0:
            return [False, 'وضعیت مربوطه یافت نشد']
        request_obj = GamingRoomRequest.objects.filter(id=validated_data['request_id'])
        if request_obj.count() == 0:
            return [False, 'درخواست مربوطه یافت نشد']
        old_status = request_obj.annotate(status_name=F('status__status_name')).values_list('status_name', flat=True)[0]
        
        if 'result' not in validated_data:
            res = ''
        elif validated_data['result'] is None or validated_data['result'] == '':
            res = ''
        else:
            res = validated_data['result']
        request_obj.update(status_id=status_obj.values_list('id', flat=True)[0], result=res)
        return [True, 'وضعیت درخواست به روزرسانی شد']




class GetVersioningSerializer(serializers.Serializer):
    
    def get_versions(self, validated_data, request_meta) -> list:
        
        version_obj = VersioningSGI.objects.latest('id')
        last = {}
        last["version_number"] = version_obj.version_number
        last["version_date"] = version_obj.version_date
        last["features"] = json.loads(version_obj.features)
        return [True, last]
    
    
class AddVersioningSerializer(serializers.Serializer):
    
    version_number = serializers.CharField()
    version_date = serializers.DateTimeField()
    features = serializers.JSONField()
    
    def add_versions(self, validated_data, request_meta):
        version_obj = VersioningSGI.objects.filter(version_number=validated_data['version_number'])
        if version_obj.count() > 0:
            version_obj.update(
                version_number=validated_data['version_number'],
                version_date=validated_data['version_date'],
                features=validated_data['features']
            )
            return [True, 'ورژن ویرایش شد']
        else:
            VersioningSGI.objects.create(
                version_number=validated_data['version_number'],
                version_date=validated_data['version_date'],
                features=validated_data['features']
            )
        return [True, 'ورژن جدید اضافه شد']
        

