import os
from mysgi_api.settings.base import BASE_DIR
from rest_framework import status
from rest_framework.views import APIView
from rest_framework.response import Response
from rest_framework.permissions import IsAuthenticated
from .models import Notification, Announcement, AnnouncementCategory
from .serializers import *
from permissions import CheckUserAccessLevel
from drf_spectacular.utils import extend_schema, OpenApiResponse, OpenApiExample, OpenApiParameter, OpenApiTypes


class GetCategoryAnnouncementView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetCategoryAnnouncementSerializer
    
    @extend_schema(
        request = GetCategoryAnnouncementSerializer,
        responses = {
            200: OpenApiResponse(response=GetCategoryAnnouncementResponseSerializer, description='Get Category Announcement'),
            406: OpenApiResponse(response=FailedMessageDueToAccessSerializer, description='Get Category Announcement'),
        }
    )
    def get(self, request):
        serializer = GetCategoryAnnouncementSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'announcements')
            result = serializer.get_items()
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class AddAnnouncementView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = AddAnnouncementSerializer
    
    @extend_schema(
        request = AddAnnouncementSerializer,
        responses = {
            200: OpenApiResponse(response=MessageDoneSerializer, description='Add Announcement'),
            406: OpenApiResponse(response=FailedMessageDueToAccessSerializer, description='Add Announcement'),
        }
    )
    def post(self, request):
        serializer = AddAnnouncementSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'announcements')
            result = serializer.add_item(request.data, request.META, request.FILES)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class ListAnnouncementView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = ListAnnouncementSerializer
    
    @extend_schema(
        parameters=[
            OpenApiParameter(name='count', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Count Paginate Parameter', required=False),
            OpenApiParameter(name='page_number', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Page Number Paginate Parameter', required=False),
            OpenApiParameter(name='filter', location=OpenApiParameter.QUERY, type=OpenApiTypes.STR, description='Search Filter string', required=False)
        ],
        request = ListAnnouncementSerializer,
        responses = {
            200: OpenApiResponse(response=ListAnnouncementResponseSerializer, description='List All Announcement'),
            406: OpenApiResponse(response=FailedMessageDueToAccessSerializer, description='List All Announcement'),
        }
    )
    def get(self, request):
        serializer = ListAnnouncementSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'announcements')
            result = serializer.get_items(request)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1], "num_items": result[2]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class GetOneAnnouncementView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetOneAnnouncementSerializer
    
    @extend_schema(
        request = GetOneAnnouncementSerializer,
        responses = {
            200: OpenApiResponse(response=GetAnnouncementResponseSerializer, description='Get One Announcement'),
            406: OpenApiResponse(response=FailedMessageDueToAccessSerializer, description='Get One Announcement'),
        }
    )
    def get(self, request, aid):
        serializer = GetOneAnnouncementSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'announcements')
            result = serializer.get_item(aid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class UpdateAnnouncementView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    
    @extend_schema(
        request = GetAnnouncementSerializer,
        responses = {
            200: OpenApiResponse(response=GetAnnouncementResponseSerializer, description='Get One Announcement'),
            406: OpenApiResponse(response=FailedMessageDueToAccessSerializer, description='Get One Announcement'),
        }
    )
    def get(self, request, aid):
        serializer = GetAnnouncementSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'announcements-edit')
            result = serializer.get_item(aid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
    
    
    @extend_schema(
        request = UpdateAnnouncementSerializer,
        responses = {
            200: OpenApiResponse(response=MessageDoneSerializer, description='Update One Announcement'),
            406: OpenApiResponse(response=FailedMessageDueToAccessSerializer, description='Update One Announcement'),
        }
    )
    def put(self, request, aid):
        serializer = UpdateAnnouncementSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'announcements-update')
            result = serializer.update_item(request.data, request.META, request.FILES, aid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class DeleteAnnouncementView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = DeleteAnnouncementSerializer
    
    @extend_schema(
        request = DeleteAnnouncementSerializer,
        responses = {
            200: OpenApiResponse(response=MessageDoneSerializer, description='Delete Announcement'),
            406: OpenApiResponse(response=FailedMessageDueToAccessSerializer, description='Delete Announcement'),
        }
    )
    def delete(self, request, aid):
        serializer = DeleteAnnouncementSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'announcements-destroy')
            result = serializer.delete_item(aid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class GetCountAnnouncementView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetCountAnnouncementSerializer
    
    @extend_schema(
        request = GetCountAnnouncementSerializer,
        responses = {
            200: OpenApiResponse(response=ListAnnouncementResponseSerializer, description='Get Count Records Announcement'),
            406: OpenApiResponse(response=FailedMessageDueToAccessSerializer, description='Get Count Records Announcement'),
        }
    )
    def get(self, request, count):
        serializer = GetCountAnnouncementSerializer(data=request.data)
        if serializer.is_valid():
            # self.check_object_permissions(request, 'announcements')
            result = serializer.get_items(count)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)





class GetInternalphonesView(APIView):
    
    permission_classes = [IsAuthenticated]
    serializer_class = GetInternalPhonesSerializer
    
    @extend_schema(
        request = GetInternalPhonesSerializer,
        responses = {
            200: OpenApiResponse(response=MessageDoneSerializer, description='Get Internal Phones'),
            406: OpenApiResponse(response=FailedMessageDueToAccessSerializer, description='Get Internal Phones'),
        }
    )
    def get(self, request):
        serializer = GetInternalPhonesSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.get_items()
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class SearchInternalphonesView(APIView):
    
    permission_classes = [IsAuthenticated]
    serializer_class = SearchInternalphonesSerializer
    
    @extend_schema(
        request = SearchInternalphonesSerializer,
        responses = {
            200: OpenApiResponse(response=MessageDoneSerializer, description='Search Internal Phones'),
            406: OpenApiResponse(response=FailedMessageDueToAccessSerializer, description='Search Internal Phones'),
        }
    )
    def get(self, request, search_string):
        serializer = SearchInternalphonesSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.search_item(search_string)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class GetInternetUsageView(APIView):
    
    permission_classes = [IsAuthenticated]
    serializer_class = GetInternetUsageSerializer
    
    @extend_schema(
        request = GetInternetUsageSerializer,
        responses = {
            200: OpenApiResponse(response=MessageDoneSerializer, description='Get Internet Usage'),
            406: OpenApiResponse(response=FailedMessageDueToAccessSerializer, description='Get Internet Usage'),
        }
    )
    def get(self, request, email):
        serializer = GetInternetUsageSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.get_usage(email)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)




class GetNotificationView(APIView):
    
    permission_classes = [IsAuthenticated]
    serializer_class = GetNotificationSerializer
    
    @extend_schema(
        request = GetNotificationSerializer,
        responses = {
            200: OpenApiResponse(response=GetNotificationResponseSerializer, description='Get Notifications View'),
            406: OpenApiResponse(response=FailedMessageDueToAccessSerializer, description='Get Notifications View'),
        }
    )
    def get(self, request):
        serializer = GetNotificationSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.get_items(request.data, request.META)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class UpdateNotificationView(APIView):

    permission_classes = [IsAuthenticated]
    serializer_class = UpdateNotificationSerializer
    
    @extend_schema(
        request = UpdateNotificationSerializer,
        responses = {
            200: OpenApiResponse(response=GetNotificationResponseSerializer, description='Update Notifications View'),
            406: OpenApiResponse(response=FailedMessageDueToAccessSerializer, description='Update Notifications View'),
        }
    )
    def put(self, request):
        serializer = UpdateNotificationSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.update_items(request.data, request.META)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
