from rest_framework import serializers
from usermanagement.models import Users
import environ
from rest_framework_simplejwt.tokens import AccessToken, RefreshToken
from mysgi_api.settings.base import BASE_DIR
import os
from .models import Notification, Announcement, AnnouncementCategory, InternalPhones
from insurance.models import Flags
from django.core.files.storage import FileSystemStorage
from django.db import transaction
from django.db.models import F, Value, CharField, Q, ExpressionWrapper
from django.db.models.functions import Concat
from random import sample
from .utils import Kerio
from pagination import custom_pagination
from .signals import announcement_created, announcement_deleted



env = environ.Env()
environ.Env.read_env(os.path.join(BASE_DIR, '.env'))

class PaginateParamSerializer(serializers.Serializer):
    count = serializers.IntegerField()
    page_number = serializers.IntegerField()


class MessageDoneSerializer(serializers.Serializer):
    
    result = serializers.BooleanField()
    message = serializers.CharField()


class FailedMessageDueToAccessSerializer(serializers.Serializer):
    
    result = serializers.BooleanField()
    message = serializers.CharField()


class GetCategoryAnnouncementResponseMessageSerializer(serializers.Serializer):
    name = serializers.CharField()
    logo_path = serializers.CharField()
    id = serializers.IntegerField()
class GetCategoryAnnouncementResponseSerializer(serializers.Serializer):
    result = serializers.BooleanField()
    message = GetCategoryAnnouncementResponseMessageSerializer(many=True)

class GetCategoryAnnouncementSerializer(serializers.Serializer):
    
    def get_items(self):
        
        categories = AnnouncementCategory.objects.all().values('name', 'logo_path', 'id')
        return [True, categories]
     

class AddAnnouncementSerializer(serializers.Serializer):
    
    title = serializers.CharField(max_length=255)
    category = serializers.CharField()
    short_description = serializers.CharField(max_length=100000)
    attachments = serializers.FileField(write_only=True, required=False)
    is_active = serializers.CharField()
    content = serializers.CharField(max_length=10000000, required=False)
    user = serializers.CharField(max_length=255)
    flag = serializers.CharField()
    
    def add_item(self, validated_data, request_meta, request_files) -> list:
        
        user_obj = Users.objects.filter(user_name=validated_data['user'])
        if user_obj.count() == 0:
            return [False, 'نام کاربری یافت نشد']
        user_id = user_obj.values_list('id', flat=True)[0]
        
        category_obj = AnnouncementCategory.objects.filter(id=validated_data['category'])
        if category_obj.count() == 0:
            return [False, 'دسته بندی یافت نشد']
        
        flag_obj = Flags.objects.filter(id=validated_data['flag'])
        if flag_obj.count() == 0:
            return [False, ' وضعیت یافت نشد']
        
        if validated_data['content'] == '' or validated_data['content'] is None:
            content = ""
        else:
            content = validated_data['content']
        
        if 'attachments' in validated_data and validated_data['attachments'] is not None and validated_data['attachments'] != '' and ( request_files.getlist('attachments') != '' ):
            uploaded_file = request_files['attachments']
            storage = FileSystemStorage(location='media/notification/')
            filename = storage.save(uploaded_file.name, uploaded_file)
            file_url = storage.url(filename)
        else:
            file_url = ''
        
        if validated_data['is_active'] is None or validated_data['is_active'] == '':
            if validated_data['is_active'] == 'true':
                isactive = True
            elif validated_data['is_active'] == 'false':
                isactive = False
        else:
            isactive = True
        
        announce_obj = Announcement.objects.create(
            category_id=validated_data['category'], user_id=user_id,
            flag_id=validated_data['flag'], title=validated_data['title'],
            short_description=validated_data['short_description'], content=content,
            attachment_path=file_url, is_active=isactive
        )
        announcement_created.send(sender=Announcement, title=validated_data['title'], tid=announce_obj.id, department_name='خدمات فاوا')
        return [True, 'اضافه شد']


class ListAnnouncementDataResponseSerializer(serializers.Serializer):
    id = serializers.IntegerField()
    category_name = serializers.CharField()
    category_id = serializers.IntegerField()
    title = serializers.CharField()
    short_description = serializers.CharField()
    content = serializers.CharField()
    attachment_path = serializers.CharField()
    created_at = serializers.DateTimeField()
    user_name = serializers.CharField()
    flag_name = serializers.CharField()
    flag_id = serializers.IntegerField()
    expired_at = serializers.DateTimeField()
class ListAnnouncementResponseSerializer(serializers.Serializer):
    datas = ListAnnouncementDataResponseSerializer(many=True)
    has_next = serializers.BooleanField()
    has_previous = serializers.BooleanField()
    page_number = serializers.IntegerField()
    next_url = serializers.CharField()
    previous_url = serializers.CharField()
    total_page = serializers.IntegerField()

class ListAnnouncementSerializer(serializers.Serializer):
    
    def get_items(self, request):
        
        if request.GET.get('filter') == None:
            announcements = Announcement.objects.filter(is_active=True).annotate(
                category_name=F('category__name'),
                user_name=Concat(F('user__first_name'), Value(' '), F('user__last_name'), output_field=CharField()),
                flag_name=F('flag__name'),
            ).order_by("-created_at").values('id', 'category_name', 'category', 'user_name', 'flag_name', 'flag', 'title', 'short_description', 'content', 'attachment_path', 'created_at', 'expired_at')
            num_items = len(announcements)
        else:
            announcements = Announcement.objects.filter(is_active=True).filter(
                    Q(category__name__icontains=request.GET.get('filter')) |
                    Q(user__user_name__icontains=request.GET.get('filter')) |
                    Q(user__first_name__icontains=request.GET.get('filter')) |
                    Q(user__last_name__icontains=request.GET.get('filter')) |
                    Q(flag__name__icontains=request.GET.get('filter')) |
                    Q(title__icontains=request.GET.get('filter')) 
                ).annotate(
                category_name=F('category__name'),
                user_name=Concat(F('user__first_name'), Value(' '), F('user__last_name'), output_field=CharField()),
                flag_name=F('flag__name'),
            ).order_by("-created_at").values('id', 'category_name', 'category', 'user_name', 'flag_name', 'flag', 'title', 'short_description', 'content', 'attachment_path', 'created_at', 'expired_at')
            num_items = len(announcements)
        datas = custom_pagination(announcements, request.GET.get('count', 10), request.GET.get('page_number', 1))
        return [True, datas, num_items]


class GetAnnouncementResponseMessageSerializer(serializers.Serializer):
    id = serializers.IntegerField()
    title = serializers.CharField()
    short_description = serializers.CharField()
    content = serializers.CharField()
    attachment_path = serializers.CharField()
    created_at = serializers.DateTimeField()
    category_name = serializers.CharField()
    category_id = serializers.IntegerField()
    user_name = serializers.CharField()
    flag_name = serializers.CharField()
    flag_id = serializers.IntegerField()
    expired_at = serializers.DateTimeField()
class GetAnnouncementResponseSerializer(serializers.Serializer):
    result = serializers.BooleanField()
    message = GetAnnouncementResponseMessageSerializer()

class GetAnnouncementSerializer(serializers.Serializer):
    
    def get_item(self, aid):
        announce_obj = Announcement.objects.filter(id=aid)
        if announce_obj.count() == 0:
            return [False, 'اعلان مربوطه یافت نشد']
        
        one_obj = announce_obj.annotate(
            category_name=F('category__name'),
            user_name=Concat(F('user__first_name'), Value(' '), F('user__last_name'), output_field=CharField()),
            flag_name=F('flag__name'),
        ).values('id', 'category_name', 'category', 'user_name', 'flag_name', 'flag', 'title', 'short_description', 'content', 'attachment_path', 'created_at', 'expired_at')[0]
        last = {
            'id': one_obj['id'],
            'category_name': one_obj['category_name'],
            'category_id': one_obj['category'],
            'user_name': one_obj['user_name'],
            'flag_name': one_obj['flag_name'],
            'flag_id': one_obj['flag'],
            'title': one_obj['title'],
            'short_description': one_obj['short_description'],
            'content': one_obj['content'],
            'attachment_path': one_obj['attachment_path'],
            'created_at': one_obj['created_at'],
            'expired_at': one_obj['expired_at'],
        }
        return [True, last]


class UpdateAnnouncementSerializer(serializers.Serializer):
    
    title = serializers.CharField(max_length=255)
    category = serializers.IntegerField()
    short_description = serializers.CharField(max_length=100000)
    attachments = serializers.FileField(write_only=True, required=False)
    expire_date = serializers.DateTimeField(required=False)
    is_active = serializers.BooleanField(default=True)
    content = serializers.CharField(max_length=10000000)
    user = serializers.CharField(max_length=255)
    flag = serializers.IntegerField()
    
    def update_item(self, validated_data, request_meta, request_files, aid) -> list:
        
        announce_obj = Announcement.objects.filter(id=aid)
        if announce_obj.count() == 0:
            return [False, 'اعلان مربوطه یافت نشد']
        
        user_obj = Users.objects.filter(user_name=validated_data['user'])
        if user_obj.count() == 0:
            return [False, 'کاربر با این یوزرنیم یافت نشد']
        user_id = user_obj.values_list('id', flat=True)[0]
        
        category_obj = AnnouncementCategory.objects.filter(id=validated_data['category'])
        if category_obj.count() == 0:
            return [False, 'دسته بندی یافت نشد']
        
        flag_obj = Flags.objects.filter(id=validated_data['flag'])
        if flag_obj.count() == 0:
            return [False, ' وضعیت یافت نشد']
        
        if 'attachments' in validated_data and validated_data['attachments'] is not None and validated_data['attachments'] != '' and ( request_files.getlist('attachments') != '' ):
            uploaded_file = request_files['attachments']
            storage = FileSystemStorage(location='media/notification/')
            filename = storage.save(uploaded_file.name, uploaded_file)
            file_url = storage.url(filename)
        else:
            file_url = ''
        
        if validated_data['is_active'] is None or validated_data['is_active'] == '':
            if validated_data['is_active'] == 'true':
                isactive = True
            elif validated_data['is_active'] == 'false':
                isactive = False
        else:
            isactive = True
        
        announce_obj.update(
            category_id=validated_data['category'], user_id=user_id,
            flag_id=validated_data['flag'], title=validated_data['title'],
            short_description=validated_data['short_description'], content=validated_data['content'],
            attachment_path=file_url, is_active=isactive
        )
        
        return [True, 'به روزرسانی شد']


class DeleteAnnouncementSerializer(serializers.Serializer):
    
    def delete_item(self, aid):
        announce_obj = Announcement.objects.filter(id=aid)
        if announce_obj.count() == 0:
            return [False, 'اعلان مربوطه یافت نشد']
        
        announce_obj.delete()
        announcement_deleted.send(sender=Announcement, tid=aid)
        return [True, 'اعلان مربوطه حذف شد']


class GetOneAnnouncementSerializer(serializers.Serializer):
    
    def get_item(self, aid):
        
        announcement = Announcement.objects.filter(id=aid).annotate(
            category_name=F('category__name'),
            user_name=Concat(F('user__first_name'), Value(' '), F('user__last_name'), output_field=CharField()),
            flag_name=F('flag__name'),
        ).values('id', 'category_name', 'category', 'user_name', 'flag_name', 'flag', 'title', 'short_description', 'content', 'attachment_path', 'created_at', 'expired_at')[0]
        last = {
            'id': announcement['id'],
            'category_name': announcement['category_name'],
            'category_id': announcement['category'],
            'user_name': announcement['user_name'],
            'flag_name': announcement['flag_name'],
            'flag_id': announcement['flag'],
            'title': announcement['title'],
            'short_description': announcement['short_description'],
            'content': announcement['content'],
            'attachment_path': announcement['attachment_path'],
            'created_at': announcement['created_at'],
            'expired_at': announcement['expired_at'],
        }
        return [True, last]
    

class GetCountAnnouncementSerializer(serializers.Serializer):
    
    def get_items(self, count):
        
        announcements = Announcement.objects.filter(is_active=True).annotate(
            category_name=F('category__name'),
            user_name=Concat(F('user__first_name'), Value(' '), F('user__last_name'), output_field=CharField()),
            flag_name=F('flag__name'),
        ).order_by('-created_at').values('id', 'category_name', 'category', 'user_name', 'flag_name', 'flag', 'title', 'short_description', 'content', 'attachment_path', 'created_at', 'expired_at')[:int(count)]
        last = list()
        for announce in announcements:
            last.append(
                {
                    'id': announce['id'],
                    'category_name': announce['category_name'],
                    'category_id': announce['category'],
                    'user_name': announce['user_name'],
                    'flag_name': announce['flag_name'],
                    'flag_id': announce['flag'],
                    'title': announce['title'],
                    'short_description': announce['short_description'],
                    'content': announce['content'],
                    'attachment_path': announce['attachment_path'],
                    'created_at': announce['created_at'],
                    'expired_at': announce['expired_at'],
                }
            )
        return [True, last]


class GetInternalPhonesSerializer(serializers.Serializer):
    
    def get_items(self):
        
        phones = InternalPhones.objects.filter(name__isnull=False).values('name', 'number')
        random_rows = sample(list(phones), 5)
        return [True, random_rows]
        

class SearchInternalphonesSerializer(serializers.Serializer):
    
    def search_item(self, search_string):
        
        phones = InternalPhones.objects.filter(
            Q(name__icontains=search_string) |
            Q(number__icontains=search_string)
        ).values('name', 'number')
        if len(phones) > 0:
            # random_rows = sample(list(phones), min(5, len(phones)))
            random_rows = phones
        else:
            random_rows = []
        return [True, random_rows]


class GetInternetUsageSerializer(serializers.Serializer):
    
    def get_usage(self, email):
        kerio = Kerio()
        answer = kerio.get_kerio_summary(email)
        if answer[0]:
            return [True, answer[1]]
        else:
            return [False, 'ارور در دریافت اطلاعات']



class GetNotificationResponseMessageSerializer(serializers.Serializer):
    id = serializers.IntegerField()
    message = serializers.CharField()
    department = serializers.CharField()
    notif_id = serializers.CharField()

class GetNotificationResponseSerializer(serializers.Serializer):
    result = serializers.BooleanField()
    message = GetNotificationResponseMessageSerializer(many=True)

class GetNotificationSerializer(serializers.Serializer):
    
    def get_items(self, validated_data, request_meta):
        if 'user_name' in validated_data and ( validated_data['user_name'] != '' ):
            user_object = Users.objects.filter(user_name__exact=validated_data['user_name'])
            if user_object.count() < 1:
                return [False, 'کاربر با این نام کاربری وجود ندارد']
            else:
                user_id = user_object.values_list('id', flat=True)[0]
        else:
            jwt_token = request_meta.get('HTTP_AUTHORIZATION', '').split(' ')[1]
            access_token = AccessToken(jwt_token)
            user_id = access_token['user_id']

        notifs = Notification.objects.filter(
            user_id=user_id, is_read=False
        ).values('id', 'message', 'department', 'notif_id').order_by("-created_at")
        return [True, notifs]


class UpdateNotificationSerializer(serializers.Serializer):
    
    notif_ids = serializers.CharField()
    
    def update_items(self, validated_data, request_meta):
        
        ids = validated_data['notif_ids'].split(',')
        for notif in ids:
            print(notif)
            try:
                notif_obj = Notification.objects.get(id=notif)
            except Notification.DoesNotExist:
                return [False, 'اعلان مربوطه یافت نشد']
            
            notif_obj.is_read = True
            notif_obj.save()
            
        return [True, 'انجام شد']

