import os
from mysgi_api.settings.base import BASE_DIR
from rest_framework import status
from rest_framework.views import APIView
from rest_framework.response import Response
from rest_framework.permissions import IsAuthenticated
from .models import InsuranceMembers, InsuranceContracts, InsuranceDocuments, InsurancePeriods, InsuranceContractMembers, Relationship, Flags, InsuranceServices, InsuranceRequests
from .serializers import *
from permissions import CheckUserAccessLevel
from drf_spectacular.utils import extend_schema, OpenApiResponse, OpenApiExample, OpenApiParameter, OpenApiTypes
from openpyxl import Workbook, load_workbook
from django.http import HttpResponse
from django.http import FileResponse
import pdfkit 
from usermanagement.jalali import Persian, Gregorian
import jdatetime
from pathlib import Path
import datetime




env = environ.Env()
environ.Env.read_env(os.path.join(BASE_DIR, '.env'))

class MembersGetView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetInsuranceMemberSeializer
    
    def get(self, request):
        self.check_object_permissions(request, 'insured_members')
        members = InsuranceMembers.objects.all()
        serialized_data = GetInsuranceMemberSeializer(instance=members, many=True)
        return Response(data=serialized_data.data)


class MembersAddView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = AddInsuranceMemberSerializer
    
    def post(self, request):
        self.check_object_permissions(request, 'insured_members-create')
        serializer = AddInsuranceMemberSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.save()
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class MembersUpdateView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = UpdateInsuranceMemberSerializer
    
    def put(self, request, pk):
        self.check_object_permissions(request, 'insured_members-edit')
        serializer = UpdateInsuranceMemberSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.update(request.data, pk)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class MembersDeleteView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = DeleteInsuranceMemberSerializer
    
    def delete(self, request, pk):
        self.check_object_permissions(request, 'insured_members-destroy')
        serializer = DeleteInsuranceMemberSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.delete(request.data, pk)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)




class ContractsListView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    
    @extend_schema(
        parameters=[
            OpenApiParameter(name='count', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Count Paginate Parameter', required=False),
            OpenApiParameter(name='page_number', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Page Number Paginate Parameter', required=False),
            OpenApiParameter(name='filter', location=OpenApiParameter.QUERY, type=OpenApiTypes.STR, description='Search Filter string', required=False)
        ],
        responses={
            200: OpenApiResponse(response=ListContractsSerializer, description='List Contracts'),
        }
    )
    def get(self, request):
        self.check_object_permissions(request, 'insurance_documents')
        if request.GET.get('filter') == None:
            contracts = InsuranceContracts.objects.all().values('id', 'company', 'title', 'number', 'start_date', 'end_date', 'is_active', 'description', 'close', 'created_at', 'updated_at').order_by("-created_at")
        else:
            contracts = InsuranceContracts.objects.filter(
                    Q(company__icontains=request.GET.get('filter')) |
                    Q(title__icontains=request.GET.get('filter')) | 
                    Q(number__icontains=request.GET.get('filter')) | 
                    Q(description__icontains=request.GET.get('filter'))
                ).values('id', 'company', 'title', 'number', 'start_date', 'end_date', 'is_active', 'description', 'close', 'created_at', 'updated_at').order_by("-created_at")
        datas = custom_pagination(contracts, request.GET.get('count', 5), request.GET.get('page_number', 1))
        return Response(datas, status=status.HTTP_200_OK)


class ContractsGetView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetInsuranceContractsSeializer
    
    def get(self, request, cid):
        self.check_object_permissions(request, 'insurance_documents')
        serializer = GetInsuranceContractsSeializer(data=request.data)
        if serializer.is_valid():
            result = serializer.get_contract_detail(request.data, cid)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class ContractsAddView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = AddInsuranceContractsSerializer
    
    @extend_schema(
        request = AddInsuranceContractsSerializer,
        responses = {
            200: OpenApiResponse(response=GetDocumentMemberResponseSerializer, description='Add Contract Members'),
            406: OpenApiResponse(response=FailedMessageSerializer, description='Add Contract Members'),
        }
    )
    def post(self, request):
        self.check_object_permissions(request, 'insurance_contracts-create')
        serializer = AddInsuranceContractsSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.add_contract(request.data)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class ContractsUpdateView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = UpdateInsuranceContractsSerializer
    
    def put(self, request, pk):
        self.check_object_permissions(request, 'insurance_contracts-update')
        serializer = UpdateInsuranceContractsSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.update_contract(request.data, pk)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class ContractsDeleteView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = DeleteInsuranceContractsSerializer
    
    def delete(self, request, pk):
        self.check_object_permissions(request, 'insurance_contracts-destroy')
        serializer = DeleteInsuranceContractsSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.delete(request.data, pk)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)




class ContractMemberGetView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetInsuranceContractMemeberSeializer
    
    @extend_schema(
        parameters=[
            OpenApiParameter(name='count', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Count Paginate Parameter', required=False),
            OpenApiParameter(name='page_number', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Page Number Paginate Parameter', required=False),
            OpenApiParameter(name='filter', location=OpenApiParameter.QUERY, type=OpenApiTypes.STR, description='Search Filter string', required=False)
        ],
        request = GetInsuranceContractMemeberSeializer,
        responses = {
            200: OpenApiResponse(response=GetDocumentMemberResponseSerializer, description='Get Contract Members'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Get Contract Members'),
        }
    )
    def get(self, request, cid):
        serializer = GetInsuranceContractMemeberSeializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_contracts')
            result = serializer.get_items(request.data, request.META, cid, request)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1], "is_active": result[2]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class ContractMemberAddView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = AddInsuranceContractMemeberSerializer
    
    @extend_schema(
        request = AddInsuranceContractMemeberSerializer,
        responses = {
            200: OpenApiResponse(response=DeleteInsuranceDocumentsResponseSerializer, description='Add Contract Members'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Add Contract Members'),
        }
    )
    def post(self, request, cid):
        serializer = AddInsuranceContractMemeberSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_contracts-create')
            result = serializer.add_member(request.data, request.META, cid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class ContractMemberUpdateView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    
    @extend_schema(
        request = UpdateInsuranceContractMemeberGetSerializer,
        responses = {
            200: OpenApiResponse(response=DeleteInsuranceDocumentsResponseSerializer, description='Add Contract Members'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Add Contract Members'),
        }
    )
    def get(self, request, cid, mid):
        serializer = UpdateInsuranceContractMemeberGetSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_contracts-edit')
            result = serializer.get_member(request.data, request.META, cid, mid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
    
    @extend_schema(
        request = UpdateInsuranceContractMemeberUpdateSerializer,
        responses = {
            200: OpenApiResponse(response=DeleteInsuranceDocumentsResponseSerializer, description='Update Contract Members'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Update Contract Members'),
        }
    )
    def put(self, request, cid, mid):
        serializer = UpdateInsuranceContractMemeberUpdateSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_contracts-update')
            result = serializer.update_item(request.data, request.META, cid, mid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class ContractMemberDeleteView(APIView):
    
    permission_classes = [IsAuthenticated]
    serializer_class = DeleteInsuranceContractMemeberSerializer
    
    def delete(self, request, cid, mid):
        serializer = DeleteInsuranceContractMemeberSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.delete(request.data, cid, mid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class ContractParentMemberView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = ContractParentMemberSerializer
    
    @extend_schema(
        request = ContractParentMemberSerializer,
        responses = {
            200: OpenApiResponse(response=ContractParentMemberSerializer, description='Get Parent Members of Contract'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Get Parent Members of Contract'),
        }
    )
    def get(self, request, cid):
        serializer = ContractParentMemberSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_contracts')
            result = serializer.get_parent(request.data, request.META, cid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
    



class GetContractPeriodsView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetContractPeriodSerializer
    
    @extend_schema(
        parameters=[
            OpenApiParameter(name='count', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Count Paginate Parameter', required=False),
            OpenApiParameter(name='page_number', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Page Number Paginate Parameter', required=False),
            OpenApiParameter(name='filter', location=OpenApiParameter.QUERY, type=OpenApiTypes.STR, description='Search Filter string', required=False)
        ],
        request = GetContractPeriodSerializer,
        responses = {
            200: OpenApiResponse(response=GetContractPeriodResponseSerializer, description='Get Contract Periods'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Get Contract Periods'),
        }
    )
    def get(self, request):
        serializer = GetContractPeriodSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_periods')
            result = serializer.get_period(request.data, request.META, request)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class GetOneContractPeriodsView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetOneContractPeriodSerializer
    
    @extend_schema(
        parameters=[
            OpenApiParameter(name='count', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Count Paginate Parameter', required=False),
            OpenApiParameter(name='page_number', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Page Number Paginate Parameter', required=False),
            OpenApiParameter(name='filter', location=OpenApiParameter.QUERY, type=OpenApiTypes.STR, description='Search Filter string', required=False)
        ],
        request = GetOneContractPeriodSerializer,
        responses = {
            200: OpenApiResponse(response=GetContractPeriodResponseSerializer, description='Get Contract Periods'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Get Contract Periods'),
        }
    )
    def get(self, request, cid):
        serializer = GetOneContractPeriodSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_periods')
            result = serializer.get_period(request.data, request.META, cid, request)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class GetRequestsContractPeriodsView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetRequestsContractPeriodSerializer
    
    @extend_schema(
        parameters=[
            OpenApiParameter(name='count', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Count Paginate Parameter', required=False),
            OpenApiParameter(name='page_number', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Page Number Paginate Parameter', required=False),
            OpenApiParameter(name='filter', location=OpenApiParameter.QUERY, type=OpenApiTypes.STR, description='Search Filter string', required=False)
        ],
        request = GetRequestsContractPeriodSerializer,
        responses = {
            200: OpenApiResponse(response=GetRequestsContractPeriodResponseSerializer, description='Get Contract Periods'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Get Contract Periods'),
        }
    )
    def get(self, request, cid, pid):
        serializer = GetRequestsContractPeriodSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_periods')
            result = serializer.get_requests(request.data, request.META, cid, pid, request)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class AddContractPeriodsView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    
    @extend_schema(
        parameters=[
            OpenApiParameter(name='count', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Count Paginate Parameter', required=False),
            OpenApiParameter(name='page_number', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Page Number Paginate Parameter', required=False),
            OpenApiParameter(name='filter', location=OpenApiParameter.QUERY, type=OpenApiTypes.STR, description='Search Filter string', required=False)
        ],
        request = AddContractPeriodGetSerializer,
        responses = {
            200: OpenApiResponse(response=AddContractPeriodGetResponseSerializer, description='Add Contract Periods Get Method'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Add Contract Periods Get Method'),
        }
    )
    def get(self, request, cid):
        serializer = AddContractPeriodGetSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_periods-create')
            result = serializer.get_requests(request.data, request.META, cid, request)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1], "last_period_number": result[2], "all_ids": result[3]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
    
    @extend_schema(
        request = AddContractPeriodSerializer,
        responses = {
            200: OpenApiResponse(response=GetContractPeriodResponseSerializer, description='Add Contract Periods'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Add Contract Periods'),
        }
    )
    def post(self, request, cid):
        serializer = AddContractPeriodSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_periods-create')
            result = serializer.add_item(request.data, request.META, cid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class DeleteContractPeriodsView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = DeleteContractPeriodSerializer
    
    @extend_schema(
        request = DeleteContractPeriodSerializer,
        responses = {
            200: OpenApiResponse(response=MessageDoneSerializer, description='Delete Contract Periods'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Delete Contract Periods'),
        }
    )
    def delete(self, request, pid):
        serializer = DeleteContractPeriodSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_periods-destroy')
            result = serializer.delete_period(request.data, request.META, pid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class UpdateContractPeriodsView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    
    @extend_schema(
        parameters=[
            OpenApiParameter(name='count', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Count Paginate Parameter', required=False),
            OpenApiParameter(name='page_number', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Page Number Paginate Parameter', required=False),
            OpenApiParameter(name='filter', location=OpenApiParameter.QUERY, type=OpenApiTypes.STR, description='Search Filter string', required=False)
        ],
        request = UpdateContractPeriodGetSerializer,
        responses = {
            200: OpenApiResponse(response=UpdateContractPeriodGetResponseSerializer, description='Update Contract Periods Get Method'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Update Contract Periods Get Method'),
        }
    )
    def get(self, request, cid, pid):
        serializer = UpdateContractPeriodGetSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_periods-create')
            result = serializer.get_requests(request.data, request.META, cid, pid, request)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1], "last_period_number": result[2], "current_period_date": result[3], "current_period_number": result[4]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
    
    @extend_schema(
        request = UpdateContractPeriodSerializer,
        responses = {
            200: OpenApiResponse(response=MessageDoneSerializer, description='Update Contract Periods'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Update Contract Periods'),
        }
    )
    def put(self, request, cid, pid):
        serializer = UpdateContractPeriodSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_periods-update')
            result = serializer.update_item(request.data, request.META, cid, pid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class DeleteRequestsPeriodView(APIView):

    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = DeleteRequestsPeriodSerializer
    
    @extend_schema(
        request = DeleteRequestsPeriodSerializer,
        responses = {
            200: OpenApiResponse(response=MessageDoneSerializer, description='Delete Requests from Periods'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Delete Requests from Periods'),
        }
    )
    def post(self, request, pid):
        serializer = DeleteRequestsPeriodSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_periods-update')
            result = serializer.delete_items(request.data, request.META, pid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)





class GetContractServicesView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetContractServicesSerializer
    
    @extend_schema(
        parameters=[
            OpenApiParameter(name='count', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Count Paginate Parameter', required=False),
            OpenApiParameter(name='page_number', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Page Number Paginate Parameter', required=False),
            OpenApiParameter(name='filter', location=OpenApiParameter.QUERY, type=OpenApiTypes.STR, description='Search Filter string', required=False)
        ],
        request = GetContractServicesSerializer,
        responses = {
            200: OpenApiResponse(response=DeleteInsuranceDocumentsResponseSerializer, description='Get Contract Services'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Get Contract Services'),
        }
    )
    def get(self, request, cid):
        serializer = GetContractServicesSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_contracts')
            result = serializer.get_item(request.data, request.META, cid, request)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1], "contract_title": result[2], "is_active": result[3]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class AddContractServicesView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = AddContractServicesSerializer
    
    @extend_schema(
        request = AddContractServicesSerializer,
        responses = {
            200: OpenApiResponse(response=DeleteInsuranceDocumentsResponseSerializer, description='Add Contract Services'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Add Contract Services'),
        }
    )
    def post(self, request, cid):
        serializer = AddContractServicesSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_services-create')
            result = serializer.add_item(request.data, request.META, cid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class UpdateContractServicesView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    
    @extend_schema(
        request = GetOneContractServicesSerializer,
        responses = {
            200: OpenApiResponse(response=DeleteInsuranceDocumentsResponseSerializer, description='Edit Contract Services'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Edit Contract Services'),
        }
    )
    def get(self, request, csid):
        serializer = GetOneContractServicesSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_services-edit')
            result = serializer.get_item(request.data, request.META, csid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
    
    @extend_schema(
        request = UpdateContractServicesSerializer,
        responses = {
            200: OpenApiResponse(response=DeleteInsuranceDocumentsResponseSerializer, description='Update Contract Services'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Update Contract Services'),
        }
    )
    def put(self, request, csid):
        serializer = UpdateContractServicesSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_services-update')
            result = serializer.update_item(request.data, request.META, csid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class DeleteContractServicesView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = DeleteContractServicesSerializer
    
    @extend_schema(
        request = DeleteContractServicesSerializer,
        responses = {
            200: OpenApiResponse(response=MessageDoneSerializer, description='Delete Contract Services'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Delete Contract Services'),
        }
    )
    def delete(self, request, csid):
        serializer = DeleteContractServicesSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_services-destroy')
            result = serializer.delete_item(request.data, request.META, csid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class DocumentsGetView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetInsuranceDocumentsSeializer
    
    @extend_schema(
        parameters=[
            OpenApiParameter(name='count', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Count Paginate Parameter', required=False),
            OpenApiParameter(name='page_number', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Page Number Paginate Parameter', required=False),
            OpenApiParameter(name='filter', location=OpenApiParameter.QUERY, type=OpenApiTypes.STR, description='Search Filter string', required=False)
        ],
        request = GetInsuranceDocumentsSeializer,
        responses = {
            200: OpenApiResponse(response=GetInsuranceDocumentsContractResponseErrorSeializer, description='Get Insurance Documents'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Get Insurance Documents'),
        }
    )
    def get(self, request):
        serializer = GetInsuranceDocumentsSeializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_documents')
            result = serializer.get_items(request.data, request.META, request)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1], "contract": result[2]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class DocumentsGetContractView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetInsuranceDocumentsContractSeializer
    
    @extend_schema(
        parameters=[
            OpenApiParameter(name='count', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Count Paginate Parameter', required=False),
            OpenApiParameter(name='page_number', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Page Number Paginate Parameter', required=False),
            OpenApiParameter(name='filter', location=OpenApiParameter.QUERY, type=OpenApiTypes.STR, description='Search Filter string', required=False)
        ],
        request = GetInsuranceDocumentsContractSeializer,
        responses = {
            200: OpenApiResponse(response=GetInsuranceDocumentsResponseSeializer, description='Get Insurance Documents'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Get Insurance Documents'),
        }
    )
    def get(self, request, cid):
        pass
        serializer = GetInsuranceDocumentsContractSeializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_documents')
            result = serializer.get_items(request.data, request.META, cid, request)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class DocumentsAddView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = AddInsuranceDocumentsSerializer
    
    def post(self, request):
        serializer = AddInsuranceDocumentsSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_documents-create')
            result = serializer.add_document(request.data)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class GetDocumentRequestsView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetDocumentRequestsSerializer
    
    @extend_schema(
        parameters=[
            OpenApiParameter(name='count', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Count Paginate Parameter', required=False),
            OpenApiParameter(name='page_number', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Page Number Paginate Parameter', required=False),
            OpenApiParameter(name='filter', location=OpenApiParameter.QUERY, type=OpenApiTypes.STR, description='Search Filter string', required=False)
        ],
        request = GetDocumentRequestsSerializer,
        responses = {
            200: OpenApiResponse(response=GetDocumentRequestsResponseSerializer, description='Get Insurance Document Requests'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Get Insurance Document Requests')
        }
    )
    def get(self, request, mid, did):
        serializer = GetDocumentRequestsSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_documents-show')
            result = serializer.get_items(request.data, request.META, mid, did, request)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class DocumentsUpdateView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    
    @extend_schema(
        request = GetDocumentRequestsSerializer,
        responses = {
            200: OpenApiResponse(response=GetDocumentRequestsResponseSerializer, description='Get-Update Insurance Document'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Get-Update Insurance Document'),
        }
    )
    def get(self, request, mid, did):
        serializer = GetDocumentRequestsSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_documents-update')
            result = serializer.get_items(request.data, request.META, mid, did, request)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
    
    @extend_schema(
        request = UpdateInsuranceDocumentsSerializer,
        responses = {
            200: OpenApiResponse(response=GetDocumentRequestsResponseSerializer, description='Update Insurance Document'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Update Insurance Document'),
        }
    )
    def put(self, request, mid, did):
        serializer = UpdateInsuranceDocumentsSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_documents-update')
            result = serializer.update_item(request.data, request.META, mid, did)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class DocumentsDeleteView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = DeleteInsuranceDocumentsSerializer
    
    @extend_schema(
        request = DeleteInsuranceDocumentsSerializer,
        responses = {
            200: OpenApiResponse(response=DeleteInsuranceDocumentsResponseSerializer, description='Delete insurance Documents'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Delete Insurance Document'),
        }
    )
    def delete(self, request, mid, did):
        serializer = DeleteInsuranceDocumentsSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_documents-destroy')
            result = serializer.delete(request.data, request.META, mid, did)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class ExportDocumentView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = ExportDocumentSerializer
    
    @extend_schema(
        request = ExportDocumentSerializer,
        responses = {
            200: OpenApiResponse(response=GetDocumentRequestsResponseSerializer, description='Download Export Insurance Document Requests'),
        }
    )
    def get(self, request, mid, did):
        serializer = ExportDocumentSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_documents-export_pdf')
            result = serializer.get_items(request.data, request.META, mid, did)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class DocumentGetMembersView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetDocumentMemberSerializer
    
    @extend_schema(
        request = GetDocumentMemberSerializer,
        responses = {
            200: OpenApiResponse(response=GetDocumentMemberResponseSerializer, description='Get Members of Documents'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Get Members of Documents'),
        }
    )
    def get(self, request, cid):
        serializer = GetDocumentMemberSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_documents-create')
            result = serializer.get_members(request.data, request.META, cid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class DocumentGetServicesView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetDocumentServiceSerializer
    
    @extend_schema(
        request = GetDocumentServiceSerializer,
        responses = {
            200: OpenApiResponse(response=GetDocumentServiceResponseSerializer, description='Get Services of Contracts'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Get Services of Contracts'),
        }
    )
    def get(self, request, cid):
        serializer = GetDocumentServiceSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_documents-create')
            result = serializer.get_services(request.data, request.META, cid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class DocumentsFilterView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = DocumentsFilterSerializer
    
    @extend_schema(
        parameters=[
            OpenApiParameter(name='count', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Count Paginate Parameter', required=False),
            OpenApiParameter(name='page_number', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Page Number Paginate Parameter', required=False)
        ],
        request = DocumentsFilterSerializer,
        responses = {
            200: OpenApiResponse(response=GetInsuranceDocumentsResponseSeializer, description='Filter In Documents and Requests'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Filter In Documents and Requests'),
        }
    )
    def post(self, request):
        serializer = DocumentsFilterSerializer(data=request.data)
        if serializer.is_valid():
            # self.check_object_permissions(request, 'insurance_contracts')
            result = serializer.search_in_tables(request.data, request.META, request)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1], "full_ids": result[2], "services_list": result[3]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)




class RelationshipGetView(APIView):
    
    permission_classes = [IsAuthenticated]
    serializer_class = GetRelationshipSerializer
    
    def get(self, request):
        relationship = Relationship.objects.all()
        self.check_object_permissions(request, relationship)
        serialized_data = GetRelationshipSerializer(instance=relationship, many=True)
        return Response(data=serialized_data.data)


class RelationshipAddView(APIView):
    
    permission_classes = [IsAuthenticated]
    serializer_class = AddRelationshipSerializer
    
    def post(self, request):
        serializer = AddRelationshipSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.save()
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class RelationshipUpdateView(APIView):
    
    permission_classes = [IsAuthenticated]
    serializer_class = UpdateRelationshipSerializer
    
    def put(self, request, pk):
        serializer = UpdateRelationshipSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.update(request.data, pk)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class RelationshipDeleteView(APIView):
    
    permission_classes = [IsAuthenticated]
    serializer_class = DeleteRelationshipSerializer
    
    def delete(self, request, pk):
        serializer = DeleteRelationshipSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.delete(request.data, pk)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)




class FlagsGetView(APIView):
    
    permission_classes = [IsAuthenticated]
    serializer_class = GetFlagsSerializer
    
    def get(self, request):
        flags = Flags.objects.all()
        self.check_object_permissions(request, flags)
        serialized_data = GetFlagsSerializer(instance=flags, many=True)
        return Response(data=serialized_data.data)


class FlagsAddView(APIView):
    
    permission_classes = [IsAuthenticated]
    serializer_class = AddFlagsSerializer
    
    def post(self, request):
        serializer = AddFlagsSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.save()
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class FlagsUpdateView(APIView):
    
    permission_classes = [IsAuthenticated]
    serializer_class = UpdateFlagsSerializer
    
    def put(self, request, pk):
        serializer = UpdateFlagsSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.update(request.data, pk)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class FlagsDeleteView(APIView):
    
    permission_classes = [IsAuthenticated]
    serializer_class = DeleteFlagsSerializer
    
    def delete(self, request):
        serializer = DeleteFlagsSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.save()
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)




class RequestsGetView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetRequestsSerializer
    
    @extend_schema(
        request = GetRequestsSerializer,
        responses = {
            200: OpenApiResponse(response=GetRequestsResponseSerializer, description='Get Requests Detail'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Get Requests Detail'),
        }
    )
    def get(self, request, rid, mid):
        serializer = GetRequestsSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_requests-show')
            result = serializer.get_detail(request.data, request.META, rid, mid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class RequestsAddView(APIView):
    
    permission_classes = [IsAuthenticated]
    serializer_class = AddRequestsSerializer
    
    def post(self, request):
        serializer = AddRequestsSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.save()
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class RequestsUpdateView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    
    @extend_schema(
        request = GetRequestsSerializer,
        responses = {
            200: OpenApiResponse(response=GetRequestsResponseSerializer, description='Get Edit Requests'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Get Edit Requests'),
        }
    )
    def get(self, request, rid, mid):
        serializer = GetRequestsSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_requests-edit')
            result = serializer.get_detail(request.data, request.META, rid, mid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


    @extend_schema(
        request = UpdateRequestsSerializer,
        responses = {
            200: OpenApiResponse(response=MessageDoneSerializer, description='Update Edit Requests'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Update Edit Requests'),
        }
    )
    def put(self, request, rid, mid):
        serializer = UpdateRequestsSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_requests-update')
            result = serializer.update_item(request.data, request.META, rid, mid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class RequestsDeleteView(APIView):
    
    permission_classes = [IsAuthenticated]
    serializer_class = DeleteFlagsSerializer
    
    def delete(self, request):
        serializer = DeleteFlagsSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.save()
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class RequestsPeriodsUpdateView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = UpdateRequestsPeriodsSerializer


    @extend_schema(
        request = UpdateRequestsPeriodsSerializer,
        responses = {
            200: OpenApiResponse(response=MessageDoneSerializer, description='Update Edit Requests'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Update Edit Requests'),
        }
    )
    def put(self, request):
        serializer = UpdateRequestsPeriodsSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_requests-update')
            result = serializer.update_item(request.data, request.META)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class GetPeriodsView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetPeriodsSerializer
    
    @extend_schema(
        request = GetPeriodsSerializer,
        responses = {
            200: OpenApiResponse(response=GetContractPeriodResponseSerializer, description='Get Contract Periods'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Get Contract Periods'),
        }
    )
    def get(self, request, cid, pid):
        serializer = GetPeriodsSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_periods')
            result = serializer.get_item(cid, pid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class GetExcelRequestsPeriodsView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetExcelRequestsPeriodsSerializer


    @extend_schema(
        request = GetExcelRequestsPeriodsSerializer,
        responses = {
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Get Excel'),
        }
    )
    def post(self, request, cid, pid):
        serializer = GetExcelRequestsPeriodsSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_documents-show')
            result = serializer.get_requests(request.data, request.META, cid, pid)
            if result[0] == True:
                response = HttpResponse(content_type='application/ms-excel')
                response['Content-Disposition'] = 'attachment; filename="درخواست-بیمه-دوره‌ای"'
                
                wb = Workbook()
                ws = wb.active
                ws.title = "درخواست های دوره "+str(pid)
                # ws.row_dimensions.height = 70
                ws.column_dimensions['A'].width = 15
                ws.column_dimensions['B'].width = 15
                ws.column_dimensions['C'].width = 25
                ws.column_dimensions['D'].width = 20
                ws.column_dimensions['E'].width = 15
                ws.column_dimensions['F'].width = 25
                ws.column_dimensions['G'].width = 20
                ws.column_dimensions['H'].width = 25
                ws.column_dimensions['I'].width = 25
                ws.column_dimensions['J'].width = 20
                ws.column_dimensions['K'].width = 25
                ws.column_dimensions['L'].width = 25
                ws.column_dimensions['M'].width = 25
                ws.column_dimensions['N'].width = 25
                
                headers = ["شماره درخواست", "شماره سند", "سرپرست", "کدملی سرپرست","نسبت", "مشخصات بیمه شده", "کدملی بیمه شده", "تعهد اصلی", "تاریخ فاکتور", "مبلغ فاکتور", "تاریخ تحویل به بیمه", "مبلغ پرداخت بیمه", "توضیحات", "دوره"]
                ws.append(headers)
                
                requests = result[1]
                for request in requests:
                    ws.append([request['request_id'], request['document_number'], request['parent_name'], request['parent_national_code'], request['relationship'], request['member_name'], request['national_code'], request['service_name'], request['claimed_date'], request['claimed_fee'], request['period_date'], request['benefit'], request['description'], request['period_number']])

                ws.sheet_view.rightToLeft = True
                wb.save(response)
                return response
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class ExportInsuranceMembersView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]


    @extend_schema(
        request = ExportInsuranceMembersSerializer,
        responses = {
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Get Excel of members'),
        }
    )
    def post(self, request, cid):
        serializer = ExportInsuranceMembersSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.get_members(request.data, request.META, cid)
            if result[0] == True:
                response = HttpResponse(content_type='application/ms-excel')
                response['Content-Disposition'] = 'attachment; filename="کاربران دوره بیمه"'
                
                wb = Workbook()
                ws = wb.active
                ws.title = "کاربران دوره بیمه"
                # ws.row_dimensions.height = 70
                ws.column_dimensions['A'].width = 25
                ws.column_dimensions['B'].width = 15
                ws.column_dimensions['C'].width = 25
                ws.column_dimensions['D'].width = 20
                ws.column_dimensions['E'].width = 15
                ws.column_dimensions['F'].width = 25
                ws.column_dimensions['G'].width = 20
                ws.column_dimensions['H'].width = 25
                
                
                headers = ["نام و نام خانوادگی", "نسبت", "نام پدر", "جنسیت", "شماره شناسنامه", "کد ملی", "تاریخ تولد", "کد بیمه"]
                ws.append(headers)
                
                # print(result[1])
                requests = result[1]
                for request in requests:
                    ws.append([request['user_name'], request['ralation_name'], request['father_name'], request['gender'], request['identity_number'], request['national_code'], request['birth_date'], request['insurance_code']])

                ws.sheet_view.rightToLeft = True
                wb.save(response)
                return response
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
    

class ExportExcelView(APIView):
    
    permission_classes = [IsAuthenticated]
    serializer_class = ExportInsuranceSerializer
    
    def post(self, request):
        serializer = ExportInsuranceSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.save()
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class ImportExcelView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = ImportInsuranceMembers
    
    @extend_schema(
        request = ImportInsuranceMembers,
        responses = {
            200: OpenApiResponse(response=ImportInsuranceMembers, description='Import Insurance Member Excel'),
        }
    )
    
    def post(self, request):
        serializer = ImportInsuranceMembers(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_contracts')
            print(request.FILES)
            if "excel_file" in request.FILES:
                
                excel_file = request.FILES["excel_file"]
                workbook = load_workbook(excel_file)
                sheet = workbook.active
                
                header = [cell.value for cell in sheet[1]]

                data = []
                for row in sheet.iter_rows(min_row=2, values_only=True):
                    row_dict = dict(zip(header, row))
                    data.append(row_dict)
                
                
                members_parent = {}
                parents_db_id = {}
                for one_person in data:
                    member_obj = InsuranceMembers.objects.create(
                        first_name=one_person['first'], last_name=one_person['last'],
                        father_name=one_person['father'], relationship_id=one_person['relationship'],
                        gender=one_person['gender'], identity_number=str(one_person['identity']),
                        national_code=str(one_person['national']), birth_date=datetime.datetime.strptime(str(one_person['birth']), '%Y-%m-%d %H:%M:%S'), 
                        insurance_code=str(one_person['insure']), parent_id=None,
                    )
                    
                    if one_person['relationship'] == 10000:
                        parents_db_id[str(one_person['id'])] = member_obj.id
                        members_parent[member_obj.id] = [str(one_person['id']), str(one_person['national']), True]
                    else:
                        members_parent[member_obj.id] = [str(one_person['id']), str(one_person['national']), False]
                    
                    InsuranceContractMembers.objects.create(
                        insurance_contract_id=request.POST.get('contract_id'), insurance_member_id=member_obj.id
                    )
                    
                    
                for member_id in members_parent:
                    mem_obj = members_parent[member_id]
                    if mem_obj[2]:
                        pass
                    else:
                        InsuranceMembers.objects.filter(id=member_id).update(parent_id=parents_db_id[members_parent[member_id][0]])  

                workbook.close()
                
                return Response({'message': 'فایل مربوطه در حال پردازش میباشد تا 10 دقیقه آتی وضعیت را بررسی نمایید'}, status=status.HTTP_200_OK)
            else:
                return Response({'message': 'لطفا فایل با نام و پسوند user-bime.xlsx آپلود نمایید'}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    
class GeneratePDFForDocumentView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GeneratePDFForDocumentSerializer
    
    @extend_schema(
        request = GeneratePDFForDocumentSerializer,
        responses = {
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Get PDF'),
        }
    )
    def post(self, request, did):
        
        serializer = GeneratePDFForDocumentSerializer(data=request.data)
        
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_documents-export_pdf')
            result = serializer.get_requests(request.data, did)
            
            if result[0]:
                
                # path_wkthmltopdf = b'C:\Program Files\wkhtmltopdf\\bin\wkhtmltopdf.exe'
                # config = pdfkit.configuration(wkhtmltopdf=path_wkthmltopdf)
                options = {
                    'enable-local-file-access': True,
                    "encoding": "UTF-8",
                    'page-size': 'A4',
                    'disable-javascript': False,
                    'javascript-delay': 2000,
                }
                
                print(result[2])
                date = result[2]['created_at']
                one_year_later = date + datetime.timedelta(days=365)
                start_date_shamsi = jdatetime.date.fromgregorian(day=date.day,month=date.month,year=date.year)
                end_date_shamsi = jdatetime.date.fromgregorian(day=one_year_later.day,month=one_year_later.month,year=one_year_later.year)
                
                print(start_date_shamsi, end_date_shamsi)
                member_datas = '''
                <tr>
                    <th width="16%">بیمه شده اصلی:</th>
                    <th width="16%">'''+result[2]['member_name']+'''</th>
                    <th width="16%">کد بیمه شده اصلی:</th>
                    <th width="16%" class="numbers">'''+result[2]['national_code']+'''</th>
                    <th width="16%">تاریخ انقضاء:</th>
                    <th width="16%" class="numbers">'''+str(end_date_shamsi)+'''</th>
                </tr>
                
                '''
                

                user_datas = ''
                count = 1
                for req in result[1]:
                    claime_date = req["claimed_date"]
                    claime_date = jdatetime.date.fromgregorian(day=claime_date.day, month=claime_date.month, year=claime_date.year)
                    user_datas += '''
                        <tr>
                            <th>'''+str(count)+'''</th>
                            <th>
                                <table style="width:100%">
                                    <tr>
                                        <th>
                                            '''+str(req["member_name"])+'''
                                        </th>
                                    </tr>
                                    <tr>
                                        <th class="numbers">
                                            '''+str(req["national_code"])+'''
                                        </th>
                                    </tr>
                                </table>
                            </th>
                            <th>'''+str(req["service_name"])+'''</th>
                            <th class="numbers">'''+str(claime_date)+'''</th>
                            <th class="numbers"></th>
                            <th></th>
                            <th></th>
                            <th></th>
                        </tr>
                    '''
                    count += 1

                html_codes = '''
                <!DOCTYPE html>
                <html>
                <head>
                    <meta charset="UTF-8">
                </head>
                <style>
                    @font-face {
                        font-family: "vazir";
                        src: url("file:////app/insurance/pdf-assets/Vazir-Light.ttf") format("truetype") ;
                    }
                    @font-face {
                        font-family: "iransans";
                        src: url("file:////app/insurance/pdf-assets/IRANSans(FaNum)_Light.ttf") format("truetype") ;
                    }
                    body {
                        direction: rtl;
                        font-family: "vazir";
                        font-size:14px;
                    }
                    table, th, td {
                        border:1px solid black;
                    }
                    th, td {
                        padding-top: 2px;
                        padding-bottom: 2px;
                        padding-left: 2px;
                        padding-right: 2px;
                    }
                    .numbers {
                        font-family: "iransans";
                    }
                </style>
                <body>

                    <table style="width:100%">
                        <tr>
                            <th><img style="width:70px" src="file:////app/insurance/pdf-assets/dana.jpg" ></th>
                            <th style="text-align:center">فرم ارسال اسناد سرپائی (به استثناء دندانپزشکی)</th>
                            <th><img style="width:70px" src="file:////app/insurance/pdf-assets/sgi.png" ></th>
                        </tr>
                    </table>
                    <table style="width:100%">
                        <tr>
                            <th width="16%">بیمه گذار:</th>
                            <th width="16%">شرکت علوم سبز</th>
                            <th width="16%">شماره قرارداد:</th>
                            <th width="16%" class="numbers">'''+result[2]['contract_number']+'''</th>
                            <th width="16%">تاریخ شروع:</th>
                            <th width="16%" class="numbers">'''+str(start_date_shamsi)+'''</th>
                        </tr>
                        '''+member_datas+'''
                    </table>
                    <br><br>

                    <table style="width:100%">
                        <tr>
                            <th style="width:50%">* مشخصات نسخ سرپایی *</th>
                            <th style="width:50%">* لطفا در این قسمت چیزی ننویسید *</th>
                        </tr>
                    </table>
                    <table style="width:100%">

                        <tr>
                            <th>ردیف</th>
                            <th>
                                <table style="width:100%">
                                    <tr>
                                        <th>
                                            نام و نام خانوادگی بیمار
                                        </th>
                                    </tr>
                                    <tr>
                                        <th>
                                            کد ملی بیمار
                                        </th>
                                    </tr>
                                </table>
                            </th>
                            <th>موضوع هزینه</th>
                            <th>تاریخ هزینه</th>
                            <th>مبلغ هزینه ( ریال )</th>
                            <th>تفاوت تعرفه</th>
                            <th>فرانشیز</th>
                            <th> مبلغ قابل پرداخت (ریال)</th>
                        </tr>
                        '''+user_datas+'''
                    </table>
                    <br><br>

                    <table style="width:100%">
                        <tr>
                            <th style="height:150px;vertical-align: top;">کارشناس پذیرش اسناد:</th>
                            <th style="height:150px;vertical-align: top;">کارشناس محاسب:</th>
                        </tr>
                        <tr>
                            <th style="height:150px;vertical-align: top;">توضیحات:</th>
                            <th style="height:150px;vertical-align: top;">کارشناس دارویی:</th>
                        </tr>
                        <tr>
                            <th style="height:150px">
                                <table style="width:100%;height:100%">
                                    <tr>
                                        <th style="width:25%">
                                            کد ردیف:
                                        </th>
                                        <th style="width:25%">
                                        </th>
                                        <th style="width:25%">
                                        </th>
                                        <th style="width:25%">
                                        </th>
                                    </tr>
                                    <tr>
                                        <th style="width:25%">
                                            کد جراحی:
                                        </th>
                                        <th style="width:25%">
                                        </th>
                                        <th style="width:25%">
                                        </th>
                                        <th style="width:25%">
                                        </th>
                                    </tr>
                                </table>
                            </th>
                            <th style="height:150px;vertical-align: top;">کارشناس محاسب:</th>
                        </tr>
                    </table>

                <script>
                    function onDOMContentLoaded() {
                        console.log('DOM is ready');
                    }
                    function ConvertNumberToPersion() {
                        persian = { 0: '۰', 1: '۱', 2: '۲', 3: '۳', 4: '۴', 5: '۵', 6: '۶', 7: '۷', 8: '۸', 9: '۹' };
                        function traverse(el) {
                            if (el.nodeType == 3) {
                                var list = el.data.match(/[0-9]/g);
                                if (list != null && list.length != 0) {
                                    for (var i = 0; i < list.length; i++)
                                        el.data = el.data.replace(list[i], persian[list[i]]);
                                }
                            }
                            for (var i = 0; i < el.childNodes.length; i++) {
                                traverse(el.childNodes[i]);
                            }
                        }
                        traverse(document.body);
                    }
                    if (document.readyState === 'loading') {
                        document.addEventListener('DOMContentLoaded', ConvertNumberToPersion);
                    } else {
                        // If the DOM is already loaded, execute the function immediately
                        console.log('qal is ready');
                        ConvertNumberToPersion();
                    }
                </script>

                </body>
                </html>

                '''

                # pdfkit.from_string(html_codes, output_path=str(did)+'.pdf', configuration=config, options=options)
                pdfkit.from_string(html_codes, output_path=str(did)+'.pdf', options=options)
                
                response = FileResponse(open(str(did)+'.pdf', 'rb'))
                response['Content-Type'] = 'application/pdf'
                response['Content-Disposition'] = 'inline; filename="{'+date.strftime("%d")+'_'+date.strftime("%M")+'_'+date.strftime("%Y")+'__'+str(did)+'}"'
                
                return response
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class GetSampleExcelView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    
    def post(self, request):
        
        app_directory = os.path.dirname(os.path.abspath(__file__))
        file_path = os.path.join(app_directory, 'pdf-assets', 'sample.xlsx')

        response = FileResponse(open(file_path, 'rb'), as_attachment=True, content_type='application/vnd.openxmlformats-officedocument.spreadsheetml.sheet')
        response['Content-Disposition'] = 'attachment; filename="sample.xlsx"'
        return response


class GenerateExcelRequestsView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GenerateExcelRequestsSerializer


    @extend_schema(
        request = GenerateExcelRequestsSerializer,
        responses = {
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Get Excel'),
        }
    )
    def post(self, request):
        serializer = GenerateExcelRequestsSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_documents-show')
            result = serializer.get_requests(request.data, request.META)
            if result[0] == True:
                response = HttpResponse(content_type='application/ms-excel')
                response['Content-Disposition'] = 'attachment; filename="درخواست-بیمه-دوره‌ای"'
                
                wb = Workbook()
                ws = wb.active
                ws.title = "درخواست های دوره"
                # ws.row_dimensions.height = 70
                ws.column_dimensions['A'].width = 15
                ws.column_dimensions['B'].width = 15
                ws.column_dimensions['C'].width = 25
                ws.column_dimensions['D'].width = 20
                ws.column_dimensions['E'].width = 15
                ws.column_dimensions['F'].width = 25
                ws.column_dimensions['G'].width = 20
                ws.column_dimensions['H'].width = 25
                ws.column_dimensions['I'].width = 25
                ws.column_dimensions['J'].width = 20
                ws.column_dimensions['K'].width = 25
                ws.column_dimensions['L'].width = 40
                ws.column_dimensions['M'].width = 40
                ws.column_dimensions['N'].width = 25
                ws.column_dimensions['O'].width = 15
                ws.column_dimensions['P'].width = 15
                
                headers = ["شماره درخواست", "شماره سند", "سرپرست", "کدملی سرپرست","نسبت", "مشخصات بیمه شده", "کدملی بیمه شده", "تعهد اصلی", "تاریخ فاکتور", "مبلغ فاکتور", "تاریخ تحویل به بیمه", "مبلغ پرداخت بیمه", "توضیحات", "نتیجه درخواست", "وضعیت", "دوره"]
                ws.append(headers)
                
                requests = result[1]
                for request in requests:
                    ws.append([request['request_id'], request['document_number'], request['parent_name'], request['parent_national_code'], request['relationship'], request['member_name'], request['national_code'], request['service_name'], request['claimed_date'], request['claimed_fee'],request['period_date'],  request['benefit'], request['description'], request['result'], request['status'], request['period_number']])

                ws.sheet_view.rightToLeft = True
                wb.save(response)
                return response
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class GeneratePDFRequestsView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GeneratePDFRequestsSerializer
    
    @extend_schema(
        request = GeneratePDFRequestsSerializer,
        responses = {
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Get PDF'),
        }
    )
    def post(self, request):
        
        serializer = GeneratePDFRequestsSerializer(data=request.data)
        
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_documents-export_pdf')
            result = serializer.get_requests(request.data)
            
            if result[0]:
                
                # path_wkthmltopdf = b'C:\Program Files\wkhtmltopdf\\bin\wkhtmltopdf.exe'
                # config = pdfkit.configuration(wkhtmltopdf=path_wkthmltopdf)
                options = {
                    'enable-local-file-access': True,
                    "encoding": "UTF-8",
                    'page-size': 'A4',
                    'disable-javascript': False,
                    'javascript-delay': 2000,
                }

                user_datas = ''
                count = 1
                for req in result[1]:
                    claime_date = req["claimed_date"]
                    claime_date = jdatetime.date.fromgregorian(day=claime_date.day, month=claime_date.month, year=claime_date.year)
                    user_datas += '''
                        <tr>
                            <th>'''+str(count)+'''</th>
                            <th>
                                <table style="width:100%">
                                    <tr>
                                        <th>
                                            '''+str(req["member_name"])+'''
                                        </th>
                                    </tr>
                                    <tr>
                                        <th class="numbers">
                                            '''+str(req["national_code"])+'''
                                        </th>
                                    </tr>
                                </table>
                            </th>
                            <th>'''+str(req["service_name"])+'''</th>
                            <th class="numbers">'''+str(claime_date)+'''</th>
                            <th class="numbers"></th>
                            <th></th>
                            <th></th>
                            <th></th>
                        </tr>
                    '''
                    count += 1

                html_codes = '''
                <!DOCTYPE html>
                <html>
                <head>
                    <meta charset="UTF-8">
                </head>
                <style>
                    @font-face {
                        font-family: "vazir";
                        src: url("file:////app/insurance/pdf-assets/Vazir-Light.ttf") format("truetype") ;
                    }
                    @font-face {
                        font-family: "iransans";
                        src: url("file:////app/insurance/pdf-assets/IRANSans(FaNum)_Light.ttf") format("truetype") ;
                    }
                    body {
                        direction: rtl;
                        font-family: "vazir";
                        font-size:14px;
                    }
                    table, th, td {
                        border:1px solid black;
                    }
                    th, td {
                        padding-top: 2px;
                        padding-bottom: 2px;
                        padding-left: 2px;
                        padding-right: 2px;
                    }
                    .numbers {
                        font-family: "iransans";
                    }
                </style>
                <body>

                    <table style="width:100%">
                        <tr>
                            <th><img style="width:70px" src="file:////app/insurance/pdf-assets/dana.jpg" ></th>
                            <th style="text-align:center">فرم ارسال اسناد سرپائی (به استثناء دندانپزشکی)</th>
                            <th><img style="width:70px" src="file:////app/insurance/pdf-assets/sgi.png" ></th>
                        </tr>
                    </table>
                    <table style="width:100%">
                        <tr>
                            <th width="16%">شماره درخواست</th>
                            <th width="16%">شماره سند</th>
                            <th width="16%">سرپرست</th>
                            <th width="16%" class="numbers">کد ملی سرپرست</th>
                            <th width="16%">تاریخ شروع:</th>
                            <th width="16%" class="numbers">'''+str(start_date_shamsi)+'''</th>
                        </tr>
                        '''+member_datas+'''
                    </table>
                    <br><br>
                    
                    <br><br>
                <script>
                    function onDOMContentLoaded() {
                        console.log('DOM is ready');
                    }
                    function ConvertNumberToPersion() {
                        persian = { 0: '۰', 1: '۱', 2: '۲', 3: '۳', 4: '۴', 5: '۵', 6: '۶', 7: '۷', 8: '۸', 9: '۹' };
                        function traverse(el) {
                            if (el.nodeType == 3) {
                                var list = el.data.match(/[0-9]/g);
                                if (list != null && list.length != 0) {
                                    for (var i = 0; i < list.length; i++)
                                        el.data = el.data.replace(list[i], persian[list[i]]);
                                }
                            }
                            for (var i = 0; i < el.childNodes.length; i++) {
                                traverse(el.childNodes[i]);
                            }
                        }
                        traverse(document.body);
                    }
                    if (document.readyState === 'loading') {
                        document.addEventListener('DOMContentLoaded', ConvertNumberToPersion);
                    } else {
                        // If the DOM is already loaded, execute the function immediately
                        console.log('qal is ready');
                        ConvertNumberToPersion();
                    }
                </script>

                </body>
                </html>

                '''

                # pdfkit.from_string(html_codes, output_path=str(did)+'.pdf', configuration=config, options=options)
                pdfkit.from_string(html_codes, output_path='filterapi.pdf', options=options)
                
                response = FileResponse('filterapi.pdf', 'rb')
                response['Content-Type'] = 'application/pdf'
                response['Content-Disposition'] = 'inline; filename="{'+date.strftime("%d")+'_'+date.strftime("%M")+'_'+date.strftime("%Y")+'__filterapi}"'
                
                return response
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class GetAllServicesView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetAllServicesSerializer
    
    @extend_schema(
        request = GetAllServicesSerializer,
        responses = {
            200: OpenApiResponse(response=GetDocumentServiceResponseSerializer, description='Get All Services'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Get All Services'),
        }
    )
    def get(self, request):
        serializer = GetAllServicesSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_documents-create')
            result = serializer.get_services(request.data, request.META)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
    

class GetAllUsersView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetAllUsersSerializer
    
    @extend_schema(
        request = GetAllUsersSerializer,
        responses = {
            200: OpenApiResponse(response=GetDocumentServiceResponseSerializer, description='Get All Users'),
            406: OpenApiResponse(response=GetInsuranceDocumentsResponseErrorSeializer, description='Get All Users'),
        }
    )
    def get(self, request):
        serializer = GetAllUsersSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'insurance_documents-create')
            result = serializer.get_users(request.data, request.META, request)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


