#!/usr/bin/env php
<?php

chdir(__DIR__);

require_once '../../include/vars.php';
require_once '../../include/console.php';

$opts = [];

if ( empty($is_cli) )
{
	header("Content-Type: text/plain", true, 500);
	die("The requested operation is only supported in the cli.\n");
}

foreach ( ['../../include/mizban.logo', '../../include/mizbancloud.figlet'] as $l ) if ( $fp = realpath($l) )
	require $fp;

if ( !is_file($fp = 'data.json') )
	touch($fp) or die("Unable to create file!");

if ( empty($data = file_get_contents($fp)) && !is_string($data) )
	die("Unable to read file!");

if ( empty($data = json_decode($data ?: '[]', true)) && json_last_error() != JSON_ERROR_NONE )
	die("Invalid file content!");

if ( in_array('-h', $argv) || in_array('--help', $argv) )
	die( is_file($h = 'help.data') ? require $h : console_text(sprintf('No help file "%s" was found!', $h), [CONSOLE['l_red']]) );

for ($i = 1; $i < count($argv); $i++) switch($argv[$i])
{
	case '--start':
	case '--end':
		if ( !preg_match('/^\d{4}[\/-]\d{2}[\/-]\d{2} \d{2}:\d{2}:\d{2}$/', $parm = $argv[$i + 1] ?? '') )
			die( console_text(sprintf(empty($parm) ? 'No value specified for %s' : 'Invalid parameter specified for %s: %s', $argv[$i], $parm), [CONSOLE['l_red']]) );
		$opts[trim($argv[$i], '-')] = str_replace('-', '/', $parm);
		$i++;
		break;

	case '--all':
	case '-A':
		$opts['all'] = true;
		break;

	case '--force':
	case '-f':
		unset($data['last_gap']);
		break;

	case '--list':
	case '-l':
		$opts['list'] = true;
		break;

	default:
		die( console_text("Unrecognised parameter: " . $argv[$i], [CONSOLE['l_red']]) );
		break;
}

if ( !empty($opts['all']) )
{
	echo console_text(">> Processing all records...", [CONSOLE['l_black']]) . "\r";

	if ( empty($opts['start']) || empty($opts['end']) )
	{
		die( console_text("Please provide both --start and --end parameters when using the --all option.", [CONSOLE['l_red']]) );
	}

	if ( !empty($opts['list']) )
	{
		die( console_text("The --list option is not compatible with the --all option.", [CONSOLE['l_red']]) );
	}

	$range = FINANCIAL_INVOICEDETAILSBYHOUR_ADMIN::GetRangeOfInvoiceDates($opts['start'], $opts['end']);

	$range = array_map(function($str) { return str_replace('-', '/', $str); }, $range);

	$is_adjusted = ( (!empty($range['min_start']) && $range['min_start'] != $opts['start']) || (!empty($range['max_end']) && $range['max_end'] != $opts['end']) );

	list($opts['start'], $opts['end']) = [$range['min_start'] ?? $opts['start'], $range['max_end'] ?? $opts['end']];

	$gaps = [];

	$gaps [] =
	[
		'missing_start' => $opts['start'],
		'missing_end' => $opts['end']
	];

	if ($is_adjusted) echo console_text("* NOTE: Adjusting the time range to between {$opts['start']} and {$opts['end']}", [CONSOLE['l_blue']]) . "\n";
}

else
{

	echo console_text(">> Finding invoice gaps...", [CONSOLE['l_black']]);

	$gaps = FINANCIAL_INVOICEDETAILSBYHOUR_ADMIN::FindAllGaps();

	if ( empty($gaps) )
		die("\nNo gaps found!");

	echo "\r\e[2K>> Found " . console_text($c = count($gaps), [CONSOLE['l_yellow'], CONSOLE['bold']]) . " gap" . ($c == 1 ? '' : 's') . "\n\n";

	$ln = 36;

	echo console_text(str_pad(" No. ", 8, ' ', STR_PAD_RIGHT), [CONSOLE['l_yellow']]);
	echo console_text(str_pad("Start Time", $ln, ' ', STR_PAD_RIGHT), [CONSOLE['l_yellow']]);
	echo console_text(str_pad("End Time", $ln, ' ', STR_PAD_RIGHT), [CONSOLE['l_yellow']]);
	echo "\n";
	echo console_text(str_repeat("=", $ln * 2), [CONSOLE['l_black']]) . "\n";

	$i = 0;

	foreach ( $gaps as $gap )
	{
		echo console_text(str_pad(' '.++$i, 8, ' ', STR_PAD_RIGHT), [CONSOLE['bold'], CONSOLE['l_yellow']]);
		echo str_pad($gap['missing_start'], $ln, ' ', STR_PAD_RIGHT);
		echo str_pad($gap['missing_end'], $ln, ' ', STR_PAD_RIGHT);
		echo "\n";
	}

	echo console_text(str_repeat("=", $ln * 2), [CONSOLE['l_black']]) . "\n";

}

if ( !empty($opts['list']) )
	die("\nNo further operation will be executed.\n");

$s_start = console_text('Start:', [CONSOLE['bold'], CONSOLE['l_red']]);
$s_end = console_text('End:', [CONSOLE['bold'], CONSOLE['l_red']]);

foreach ( $gaps as $gap )
{
	$gap = array_map(function($str) { return str_replace('-', '/', $str); }, $gap);

	echo sprintf("\n* Fixing gap, $s_start %s, $s_end %s\n", $gap['missing_start'], $gap['missing_end']);

	if ( !empty($data['last_gap']) && $gap['missing_start'] < $data['last_gap'] && empty($opts['all']) )
	{
		echo console_text('-- Skipping gap due to already being processed before --', [CONSOLE['l_green']]) . "\n";
		continue;
	}

	if ( !empty($opts['start']) && $gap['missing_start'] < $opts['start'] )
	{
		echo console_text('-- Skipping gap due to start date being older than specified start time --', [CONSOLE['l_yellow']]) . "\n";
		continue;
	}

	if ( !empty($opts['end']) && $gap['missing_end'] > $opts['end'] )
	{
		echo console_text('-- Skipping gap due to end date being more recent than specified end time --', [CONSOLE['l_yellow']]) . "\n";
		continue;
	}

	echo "\r\e[0J" . console_text('-- Start filling CDN gaps --', [CONSOLE['l_green']]) . "\n\e[s";
    FINANCIAL_INVOICEBYHOURFIX_ADMIN::RunJobForCalculateAndGenerateInvoiceForCDN( $gap['missing_start'], $gap['missing_end'] ); echo "\e[u";
	echo "\r\e[0J" . console_text('-- End filling CDN gaps --', [CONSOLE['l_green']]) . "\n\e[s";

	echo "\r\e[K\n\e[K";

	echo "\r\e[0J" . console_text('-- Start filling Cloud gaps --', [CONSOLE['l_green']]) . "\n\e[s";
	FINANCIAL_INVOICEBYHOURFIX_ADMIN::RunJobForCalculateAndGenerateInvoiceForCloudServer( $gap['missing_start'], $gap['missing_end'] ); echo "\e[u";
	echo "\r\e[0J" . console_text('-- End filling Cloud gaps --', [CONSOLE['l_green']]) . "\n\e[s";

	$data['last_gap'] = $gap['missing_end'];

	file_put_contents($fp, json_encode($data, JSON_PRETTY_PRINT));
}

echo "\n" . console_text("Done!", [CONSOLE['l_green'], CONSOLE['bold']]) . "\n";
