<?php
/**
 * PRTG Traffic Usage Cron Job - Improved Version
 * 
 * This script fetches bandwidth usage from PRTG sensors and stores it in the database.
 * Set up a cron job to run this script daily or hourly.
 * 
 * Example cron entry (daily at 1 AM):
 * 0 1 * * * php -q /path/to/whmcs/modules/addons/prtgmonitoring/cron_improved.php
 */

// Initialize WHMCS
if (!defined("WHMCS")) {
    define("WHMCS", true);
}

// Include WHMCS configuration
$whmcsPath = dirname(dirname(dirname(dirname(__FILE__))));
require_once $whmcsPath . '/configuration.php';
require_once $whmcsPath . '/includes/functions.php';
require_once $whmcsPath . '/includes/dbfunctions.php';
require_once $whmcsPath . '/init.php';

use WHMCS\Database\Capsule;

// Connect to database
if (!Capsule::connection()->getPdo()) {
    die("Could not connect to database");
}

// Log function
function logActivity2($message)
{
    // Write to log file
    $logFile = dirname(__FILE__) . '/traffic_cron.log';
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($logFile, "[$timestamp] $message" . PHP_EOL, FILE_APPEND);
}

logActivity2("Starting PRTG traffic usage update");

// Get all monitored groups
$monitoredGroups = Capsule::table('mod_prtg_groups')->get();

// Build group IDs for query
$groupIds = [];
foreach ($monitoredGroups as $group) {
    $groupIds[] = $group->group_id;
}

// Build the query for monitored services
$whereClause = "";
if (!empty($groupIds)) {
    $groupIdsStr = implode(',', $groupIds);
    $whereClause = "and `gid` IN ($groupIdsStr)";
} else {
    // If no groups are defined, use a default (for backward compatibility)
    $whereClause = "and `gid` = 26";
}

// Get all active services in monitored groups
$query = "SELECT tblhosting.id as hostingid, tblproducts.id as productid, dedicatedip 
          FROM `tblproducts`, tblhosting, tblclients 
          WHERE userid=tblclients.id $whereClause and packageid=tblproducts.id and domainstatus='Active'";

$result = full_query($query);
$updatedCount = 0;
$errorCount = 0;

while ($service = mysql_fetch_array($result)) {
    try {
        // Get monitoring ID for this service
        $sql = full_query("SELECT * from tblcustomfields, tblcustomfieldsvalues
                          WHERE tblcustomfields.relid=" . $service["productid"] . " 
                          AND tblcustomfieldsvalues.relid = " . $service["hostingid"] . " 
                          AND tblcustomfieldsvalues.fieldid = tblcustomfields.id 
                          AND tblcustomfields.fieldname='monitoringID'");
        $monitoringData = mysql_fetch_array($sql);

        if (!$monitoringData || empty($monitoringData["value"])) {
            logActivity2("No monitoring ID found for service ID: " . $service["hostingid"]);
            continue;
        }

        $sensorId = $monitoringData["value"];
        logActivity2("Processing sensor ID: " . $sensorId . " for service ID: " . $service["hostingid"]);

        // Get server for this product
        $serverMapping = Capsule::table('mod_prtg_product_servers')
            ->where('product_id', $service["productid"])
            ->first();

        $serverUsername = 'admin';
        $serverPasshash = '1314677760';
        $serverUrl = '';

        if ($serverMapping) {
            $server = Capsule::table('mod_prtg_servers')
                ->where('id', $serverMapping->server_id)
                ->first();

            if ($server) {
                $serverUsername = $server->username;
                $serverPasshash = $server->passhash;
                $serverUrl = $server->url;
                logActivity2("Using server: " . $server->name . " (" . $serverUrl . ")");
            }
        }

        if (empty($serverUrl)) {
            logActivity2("No server URL found for service ID: " . $service["hostingid"]);
            continue;
        }

        // Calculate date range (last month)
        $endDate = date("Y-m-d-H-i-s");
        $startDate = date("Y-m-d-00-00-00", strtotime("-1 month"));
        logActivity2("Date range: " . $startDate . " to " . $endDate);

        // Try to get data using the PRTG API
        $totalTraffic = getPRTGTrafficData($serverUrl, $serverUsername, $serverPasshash, $sensorId, $startDate, $endDate, $service["hostingid"]);
        
        if ($totalTraffic === false) {
            $errorCount++;
            continue;
        }

        // Update or insert into mod_trafficusage table
        $existingRecord = Capsule::table('mod_trafficusage')
            ->where('product_id', $service["hostingid"])
            ->first();

        if ($existingRecord) {
            // Update existing record
            Capsule::table('mod_trafficusage')
                ->where('product_id', $service["hostingid"])
                ->update([
                    'usage' => $totalTraffic,
                    'lastupdate' => Capsule::raw('NOW()')
                ]);
        } else {
            // Insert new record
            Capsule::table('mod_trafficusage')->insert([
                'product_id' => $service["hostingid"],
                'usage' => $totalTraffic,
                'lastupdate' => Capsule::raw('NOW()')
            ]);
        }

        logActivity2("Updated traffic usage for service ID: " . $service["hostingid"] . " - Usage: " . $totalTraffic . " GB");
        $updatedCount++;

    } catch (Exception $e) {
        logActivity2("Error processing service ID " . $service["hostingid"] . ": " . $e->getMessage());
        $errorCount++;
    }
}

logActivity2("Completed PRTG traffic usage update. Updated: $updatedCount, Errors: $errorCount");

/**
 * Get traffic data from PRTG API
 * 
 * @param string $serverUrl PRTG server URL
 * @param string $username PRTG username
 * @param string $passhash PRTG passhash
 * @param int $sensorId Sensor ID
 * @param string $startDate Start date (Y-m-d-H-i-s)
 * @param string $endDate End date (Y-m-d-H-i-s)
 * @param int $serviceId Service ID for logging
 * @return float|bool Total traffic in GB or false on error
 */
function getPRTGTrafficData($serverUrl, $username, $passhash, $sensorId, $startDate, $endDate, $serviceId)
{
    // First try the historicdata.json API
    $apiUrl = rtrim($serverUrl, '/') . "/api/historicdata.json";
    $apiUrl .= "?id=" . $sensorId;
    $apiUrl .= "&avg=86400"; // Daily average
    $apiUrl .= "&sdate=" . $startDate;
    $apiUrl .= "&edate=" . $endDate;
    $apiUrl .= "&username=" . $username;
    $apiUrl .= "&passhash=" . $passhash;
    
    logActivity2("Requesting data from: " . $apiUrl);

    // Fetch data from PRTG API
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 60);
    curl_setopt($ch, CURLOPT_USERAGENT, 'WHMCS PRTG Module');
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    $response = curl_exec($ch);
    
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    if (curl_errno($ch)) {
        logActivity2("cURL Error for service ID " . $serviceId . ": " . curl_error($ch));
        curl_close($ch);
        return false;
    }
    
    if ($httpCode != 200) {
        logActivity2("HTTP Error " . $httpCode . " for service ID " . $serviceId);
        logActivity2("Response: " . substr($response, 0, 500));
        curl_close($ch);
        
        // Try alternative method - table.json API
        return getPRTGTrafficDataAlternative($serverUrl, $username, $passhash, $sensorId, $startDate, $endDate, $serviceId);
    }

    curl_close($ch);

    // Parse JSON response
    $data = json_decode($response, true);

    if (!$data || !isset($data['histdata']) || empty($data['histdata'])) {
        logActivity2("Invalid or empty response for service ID: " . $serviceId);
        logActivity2("Response: " . substr($response, 0, 500));
        
        // Try alternative method
        return getPRTGTrafficDataAlternative($serverUrl, $username, $passhash, $sensorId, $startDate, $endDate, $serviceId);
    }

    // Calculate total traffic (in GB)
    $totalTraffic = 0;
    foreach ($data['histdata'] as $record) {
        // Assuming the traffic value is in bytes, convert to GB
        if (isset($record['Traffic Total (volume)'])) {
            $totalTraffic += $record['Traffic Total (volume)'] / (1024 * 1024 * 1024);
        } elseif (isset($record['Traffic Total'])) {
            $totalTraffic += $record['Traffic Total'] / (1024 * 1024 * 1024);
        }
    }

    // Round to 2 decimal places
    return round($totalTraffic, 2);
}

/**
 * Alternative method to get traffic data from PRTG API
 */
function getPRTGTrafficDataAlternative($serverUrl, $username, $passhash, $sensorId, $startDate, $endDate, $serviceId)
{
    logActivity2("Trying alternative API method for service ID: " . $serviceId);
    
    // Try the table.json API
    $apiUrl = rtrim($serverUrl, '/') . "/api/table.json";
    $apiUrl .= "?content=sensors";
    $apiUrl .= "&columns=sensor,lastvalue";
    $apiUrl .= "&id=" . $sensorId;
    $apiUrl .= "&username=" . $username;
    $apiUrl .= "&passhash=" . $passhash;
    
    logActivity2("Requesting data from alternative API: " . $apiUrl);

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 60);
    curl_setopt($ch, CURLOPT_USERAGENT, 'WHMCS PRTG Module');
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    $response = curl_exec($ch);
    
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    if (curl_errno($ch) || $httpCode != 200) {
        logActivity2("Alternative API also failed for service ID " . $serviceId);
        curl_close($ch);
        return false;
    }

    curl_close($ch);
    
    // Parse JSON response
    $data = json_decode($response, true);
    
    if (!$data || !isset($data['sensors']) || empty($data['sensors'])) {
        logActivity2("Invalid or empty response from alternative API for service ID: " . $serviceId);
        return false;
    }
    
    // Try to extract traffic value from the sensor data
    foreach ($data['sensors'] as $sensor) {
        if (isset($sensor['lastvalue_raw'])) {
            // Convert to GB if needed (assuming bytes)
            $trafficValue = $sensor['lastvalue_raw'] / (1024 * 1024 * 1024);
            return round($trafficValue, 2);
        }
    }
    
    logActivity2("Could not extract traffic value from alternative API for service ID: " . $serviceId);
    return false;
}