<?php
/**
 * PRTG Traffic Usage Cron Job
 * 
 * This script fetches bandwidth usage from PRTG sensors and stores it in the database.
 * Set up a cron job to run this script daily or hourly.
 * 
 * Example cron entry (daily at 1 AM):
 * 0 1 * * * php -q /path/to/whmcs/modules/addons/prtgmonitoring/cron.php
 */

// Initialize WHMCS
if (!defined("WHMCS")) {
    define("WHMCS", true);
}

// Include WHMCS configuration
$whmcsPath = dirname(dirname(dirname(dirname(__FILE__))));
require_once $whmcsPath . '/configuration.php';
require_once $whmcsPath . '/includes/functions.php';
require_once $whmcsPath . '/includes/dbfunctions.php';
require_once $whmcsPath . '/init.php';

use WHMCS\Database\Capsule;

// Connect to database
if (!Capsule::connection()->getPdo()) {
    die("Could not connect to database");
}

// Log function
function logActivity1($message)
{
    // Write to log file
    $logFile = dirname(__FILE__) . '/traffic_cron.log';
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($logFile, "[$timestamp] $message" . PHP_EOL, FILE_APPEND);
}

logActivity1("Starting PRTG traffic usage update");

// Get all monitored groups
$monitoredGroups = Capsule::table('mod_prtg_groups')->get();

// Build group IDs for query
$groupIds = [];
foreach ($monitoredGroups as $group) {
    $groupIds[] = $group->group_id;
}

// Build the query for monitored services
$whereClause = "";
if (!empty($groupIds)) {
    $groupIdsStr = implode(',', $groupIds);
    $whereClause = "and `gid` IN ($groupIdsStr)";
} else {
    // If no groups are defined, use a default (for backward compatibility)
    $whereClause = "and `gid` = 26";
}

// Get all active services in monitored groups
$query = "SELECT tblhosting.id as hostingid, tblproducts.id as productid, dedicatedip 
          FROM `tblproducts`, tblhosting, tblclients 
          WHERE userid=tblclients.id $whereClause and packageid=tblproducts.id and domainstatus='Active'";

$result = full_query($query);
$updatedCount = 0;
$errorCount = 0;

while ($service = mysql_fetch_array($result)) {
    try {
        // Get monitoring ID for this service
        $sql = full_query("SELECT * from tblcustomfields, tblcustomfieldsvalues
                          WHERE tblcustomfields.relid=" . $service["productid"] . " 
                          AND tblcustomfieldsvalues.relid = " . $service["hostingid"] . " 
                          AND tblcustomfieldsvalues.fieldid = tblcustomfields.id 
                          AND tblcustomfields.fieldname='monitoringID'");
        $monitoringData = mysql_fetch_array($sql);
        if (!$monitoringData || empty($monitoringData["value"])) {
            logActivity1("No monitoring ID found for service ID: " . $service["hostingid"]);
            continue;
        }
        $sensorId = $monitoringData["value"];
        // Get server for this product
        $serverMapping = Capsule::table('mod_prtg_product_servers')
            ->where('product_id', $service["productid"])
            ->first();
        $serverUsername = 'admin';
        $serverPasshash = '1314677760';
        $serverUrl = '';
        if ($serverMapping) {
            $server = Capsule::table('mod_prtg_servers')
                ->where('id', $serverMapping->server_id)
                ->first();

            if ($server) {
                $serverUsername = $server->username;
                $serverPasshash = $server->passhash;
                $serverUrl = $server->url;
            }
        }
        if (empty($serverUrl)) {
            logActivity1(
                "No server URL found for service ID: " . $service["hostingid"]
            );
            continue;
        }
        // Get next due date for this service
        $nextDueDateQuery = full_query("SELECT nextduedate FROM tblhosting WHERE id=" . $service["hostingid"]);
        $nextDueDateRow = mysql_fetch_array($nextDueDateQuery);
        $nextDueDate = isset($nextDueDateRow["nextduedate"]) ? $nextDueDateRow["nextduedate"] : null;
        // Calculate date range (from next due date minus 1 month to current date)
        $endDate = date("Y-m-d-H-i-s");
        if ($nextDueDate) {
            $date = date_create($nextDueDate);
            date_modify($date, '-1 month');
            $startDate = date_format($date, 'Y-m-d-00-00-00');
            logActivity1("Using date range from next due date: " . $startDate . " to " . $endDate . " for service ID: " . $service["hostingid"]);
        } else {
            // Fallback to default (last month from today)
            $startDate = date("Y-m-d-00-00-00", strtotime("-1 month"));
            logActivity1("No next due date found, using default date range: " . $startDate . " to " . $endDate . " for service ID: " . $service["hostingid"]);
        }
        // Check if we have an API key
        $apiKey = '';
        if ($serverMapping && $server && isset($server->api_key) && !empty($server->api_key)) {
            $apiKey = $server->api_key;

            // Include API client
            require_once(dirname(__FILE__) . '/api_client.php');

            // Use API key method
            logActivity1("Using API key for service ID: " . $service["hostingid"]);

            // Get sensor data using API key
            $sensorData = prtg_get_sensor_data($serverUrl, $apiKey, $sensorId, $startDate, $endDate);

            if ($sensorData) {
                // Skip the regular API call
                $response = json_encode($sensorData);
                $data = $sensorData;
            } else {
                // Fall back to username/password method
                logActivity1("API key method failed, falling back to username/password for service ID: " . $service["hostingid"]);
                $apiKey = ''; // Reset to use the fallback method
            }
        }
        // Only use the traditional method if API key method wasn't used or failed
        if (empty($apiKey) || !isset($data)) {
            // Build API URL to get sensor data
            $apiUrl = rtrim($serverUrl, '/') . "/api/historicdata.json";
            $apiUrl .= "?id=" . $sensorId;
            $apiUrl .= "&usecaption=1"; // Daily average
            $apiUrl .= "&avg=3600"; // Daily average
            $apiUrl .= "&sdate=" . $startDate;
            $apiUrl .= "&edate=" . $endDate;
            $apiUrl .= "&username=" . $serverUsername;
            $apiUrl .= "&passhash=" . $serverPasshash;
            // Fetch data from PRTG API
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $apiUrl);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            $response = curl_exec($ch);
            if (curl_errno($ch)) {
                logActivity1("cURL Error for service ID " . $service["hostingid"] . ": " . curl_error($ch));
                $errorCount++;
                curl_close($ch);
                continue;
            }
            curl_close($ch);
            // Parse JSON response
            $data = json_decode($response, true);
            if (!$data || !isset($data['histdata']) || empty($data['histdata'])) {
                logActivity1("Invalid or empty response for service ID: " . $service["hostingid"]);
                logActivity1("Json: " . $response);
                logActivity1("Url : " . $apiUrl);
                $errorCount++;
                continue;
            }
            // Get traffic type preference for this server
            $trafficType = 'total'; // Default
            if ($serverMapping && $server && isset($server->traffic_type)) {
                $trafficType = $server->traffic_type;
            }
            logActivity1("Using traffic type: " . $trafficType . " for service ID: " . $service["hostingid"]);
            // Calculate traffic based on selected type (in GB)
            $totalTraffic = 0;
            foreach ($data['histdata'] as $record) {
                // Process based on traffic type preference
                if ($trafficType == 'download') {
                    // Download traffic only
                    if (isset($record['Traffic In (volume)'])) {
                        $totalTraffic += $record['Traffic In (volume)'] / (1024 * 1024 * 1024);
                    } elseif (isset($record['Traffic In'])) {
                        $totalTraffic += $record['Traffic In'] / (1024 * 1024 * 1024);
                    }
                } elseif ($trafficType == 'upload') {
                    // Upload traffic only
                    if (isset($record['Traffic Out (volume)'])) {
                        $totalTraffic += $record['Traffic Out (volume)'] / (1024 * 1024 * 1024);
                    } elseif (isset($record['Traffic Out'])) {
                        $totalTraffic += $record['Traffic Out'] / (1024 * 1024 * 1024);
                    }
                } else {
                    // Total traffic (default)
                    if (isset($record['Traffic Total (volume)'])) {
                        $totalTraffic += $record['Traffic Total (volume)'] / (1024 * 1024 * 1024);
                    } elseif (isset($record['Traffic Total'])) {
                        $totalTraffic += $record['Traffic Total'] / (1024 * 1024 * 1024);
                    }
                }
            }
            // Round to 2 decimal places
            $totalTraffic = round($totalTraffic, 2);
            // Update or insert into mod_trafficusage table
            $existingRecord = Capsule::table('mod_trafficusage')
                ->where('product_id', $service["hostingid"])
                ->first();
            if ($existingRecord) {
                // Update existing record
                Capsule::table('mod_trafficusage')
                    ->where('product_id', $service["hostingid"])
                    ->update([
                        'usage' => $totalTraffic,
                        'lastupdate' => Capsule::raw('NOW()')
                    ]);
            } else {
                // Insert new record
                Capsule::table('mod_trafficusage')->insert([
                    'product_id' => $service["hostingid"],
                    'usage' => $totalTraffic,
                    'lastupdate' => Capsule::raw('NOW()')
                ]);
            }
            logActivity1("Updated traffic usage for service ID: " . $service["hostingid"] . " - Usage: " . $totalTraffic . " GB");
            logActivity1("Api Url: " . $apiUrl);
            $updatedCount++;

        } else {
            logActivity1("No Apikey Found " . $service["hostingid"]);
        }
    } catch (Exception $e) {
        logActivity1("Error processing service ID " . $service["hostingid"] . ": " . $e->getMessage());
        $errorCount++;
    }
}

logActivity1("Completed PRTG traffic usage update. Updated: $updatedCount, Errors: $errorCount");