<?php
/**
 * PRTG API Client
 * 
 * This file provides functions to interact with the PRTG API using API keys
 */

if (!defined("WHMCS")) {
    die("This file cannot be accessed directly");
}

/**
 * Get data from PRTG API using API key
 * 
 * @param string $serverUrl PRTG server URL
 * @param string $apiKey PRTG API key
 * @param string $endpoint API endpoint to call
 * @param array $params Additional parameters
 * @return array|false Response data or false on error
 */
function prtg_api_request($serverUrl, $apiKey, $endpoint, $params = [])
{
    // Build API URL
    $url = rtrim($serverUrl, '/') . '/api/' . $endpoint;
    
    // Add API key to parameters
    $params['apikey'] = $apiKey;
    $params['username'] = 'dummy'; // Required by PRTG API but not used with API key
    
    // Build query string
    $queryString = http_build_query($params);
    $url .= '?' . $queryString;
    
    // Initialize cURL
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    // Execute request
    $response = curl_exec($ch);
    
    // Check for errors
    if (curl_errno($ch)) {
        logActivity("PRTG API Error: " . curl_error($ch));
        curl_close($ch);
        return false;
    }
    
    curl_close($ch);
    
    // Parse response
    $data = json_decode($response, true);
    if (!$data) {
        logActivity("PRTG API Error: Invalid response format");
        return false;
    }
    
    return $data;
}

/**
 * Get sensor data using API key
 * 
 * @param string $serverUrl PRTG server URL
 * @param string $apiKey PRTG API key
 * @param int $sensorId Sensor ID
 * @param string $startDate Start date (Y-m-d-H-i-s)
 * @param string $endDate End date (Y-m-d-H-i-s)
 * @return array|false Sensor data or false on error
 */
function prtg_get_sensor_data($serverUrl, $apiKey, $sensorId, $startDate, $endDate)
{
    $params = [
        'id' => $sensorId,
        'avg' => 3600,
        'sdate' => $startDate,
        'edate' => $endDate
    ];
    
    return prtg_api_request($serverUrl, $apiKey, 'historicdata.json', $params);
}

/**
 * Get traffic usage from sensor data
 * 
 * @param array $sensorData Sensor data from API
 * @param string $trafficType Type of traffic to calculate (total, download, upload)
 * @return float Traffic usage in GB
 */
function prtg_calculate_traffic($sensorData, $trafficType = 'total')
{
    if (!isset($sensorData['histdata']) || empty($sensorData['histdata'])) {
        return 0;
    }
    
    $totalTraffic = 0;
    foreach ($sensorData['histdata'] as $record) {
        // Process based on traffic type preference
        if ($trafficType == 'download') {
            // Download traffic only
            if (isset($record['Traffic In (volume)'])) {
                $totalTraffic += $record['Traffic In (volume)'] / (1024 * 1024 * 1024);
            } elseif (isset($record['Traffic In'])) {
                $totalTraffic += $record['Traffic In'] / (1024 * 1024 * 1024);
            }
        } elseif ($trafficType == 'upload') {
            // Upload traffic only
            if (isset($record['Traffic Out (volume)'])) {
                $totalTraffic += $record['Traffic Out (volume)'] / (1024 * 1024 * 1024);
            } elseif (isset($record['Traffic Out'])) {
                $totalTraffic += $record['Traffic Out'] / (1024 * 1024 * 1024);
            }
        } else {
            // Total traffic (default)
            if (isset($record['Traffic Total (volume)'])) {
                $totalTraffic += $record['Traffic Total (volume)'] / (1024 * 1024 * 1024);
            } elseif (isset($record['Traffic Total'])) {
                $totalTraffic += $record['Traffic Total'] / (1024 * 1024 * 1024);
            }
        }
    }
    
    // Round to 2 decimal places
    return round($totalTraffic, 2);
}