<?php

namespace WHMCS\Module\Addon\HourlyAddonSeller\Admin;

use WHMCS\Database\Capsule;

/**
 * Admin Area Controller
 */
class Controller
{

    /**
     * Generate navigation menu
     *
     * @param string $currentAction Current active menu item
     * @param string $modulelink Base module link
     * @return string HTML for navigation menu
     */


    /**
     * Close the content container
     *
     * @return string HTML to close the content container
     */
    private function closeContentContainer()
    {
        return '</div></div></div>';
    }

    /**
     * Index action.
     *
     * @param array $vars Module configuration parameters
     *
     * @return string
     */
    public function index($vars)
    {
        // Get common module parameters
        $modulelink = $vars['modulelink']; // eg. addonmodules.php?module=addonmodule
        $version = $vars['version']; // eg. 1.0
        $LANG = $vars['_lang']; // an array of the currently loaded language variables

        $output = $this->generateMenu('index', $modulelink);

        $output .= <<<EOF
        <div class="row">
            <div class="col-md-6">
                <div class="panel panel-default">
                    <div class="panel-heading">
                        <h3 class="panel-title">API Key Management</h3>
                    </div>
                    <div class="panel-body">
                        <p>Manage your API key for external integrations.</p>
                        <p><a href="$modulelink&action=apikey" class="btn btn-primary">Manage API Key</a></p>
                    </div>
                </div>
            </div>
            
            <div class="col-md-6">
                <div class="panel panel-default">
                    <div class="panel-heading">
                        <h3 class="panel-title">Product Management</h3>
                    </div>
                    <div class="panel-body">
                        <p>Manage hourly billing products.</p>
                        <p><a href="$modulelink&action=products" class="btn btn-primary">Manage Products</a></p>
                    </div>
                </div>
            </div>
        </div>

        <div class="row">
            <div class="col-md-6">
                <div class="panel panel-default">
                    <div class="panel-heading">
                        <h3 class="panel-title">Reports</h3>
                    </div>
                    <div class="panel-body">
                        <p>View usage and billing reports.</p>
                        <p><a href="$modulelink&action=reports" class="btn btn-primary">View Reports</a></p>
                    </div>
                </div>
            </div>
            
            <div class="col-md-6">
                <div class="panel panel-default">
                    <div class="panel-heading">
                        <h3 class="panel-title">Settings</h3>
                    </div>
                    <div class="panel-body">
                        <p>Configure module settings.</p>
                        <p><a href="$modulelink&action=settings" class="btn btn-primary">Configure Settings</a></p>
                    </div>
                </div>
            </div>
        </div>
        EOF;

        $output .= $this->closeContentContainer();

        return $output;
    }

    /**
     * API Key management page.
     *
     * @param array $vars Module configuration parameters
     *
     * @return string
     */
    public function apikey($vars)
    {
        $modulelink = $vars['modulelink'];
        $success = '';
        $error = '';
        $currentApiKey = '';

        // Process form submission
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['apikey'])) {
            $apikey = trim($_POST['apikey']);

            if (empty($apikey)) {
                $error = 'API Key cannot be empty';
            } else {
                try {
                    // Check if record exists
                    $exists = Capsule::table('mod_HourlyAddonSeller')
                        ->where('id', '=', 1)
                        ->exists();

                    if ($exists) {
                        // Update existing record
                        Capsule::table('mod_HourlyAddonSeller')
                            ->where('id', '=', 1)
                            ->update([
                                'apikey' => $apikey,
                                'updated_at' => date('Y-m-d H:i:s')
                            ]);
                        $success = 'API Key updated successfully';
                    } else {
                        // Insert new record
                        Capsule::table('mod_HourlyAddonSeller')
                            ->insert([
                                'apikey' => $apikey,
                                'status' => 'Active',
                                'created_at' => date('Y-m-d H:i:s'),
                                'updated_at' => date('Y-m-d H:i:s')
                            ]);
                        $success = 'API Key added successfully';
                    }
                } catch (\Exception $e) {
                    $error = 'Database error: ' . $e->getMessage();
                }
            }
        }

        // Get current API key
        try {
            $record = Capsule::table('mod_HourlyAddonSeller')
                ->where('id', '=', 1)
                ->first();

            if ($record) {
                $currentApiKey = $record->apikey;
            }
        } catch (\Exception $e) {
            $error = 'Error retrieving API key: ' . $e->getMessage();
        }

        $buttonText = $currentApiKey ? 'Update' : 'Add';
        $cardTitle = $currentApiKey ? 'Update API Key' : 'Add API Key';

        $output = $this->generateMenu('apikey', $modulelink);

        if ($success) {
            $output .= '<div class="alert alert-success">' . $success . '</div>';
        }

        if ($error) {
            $output .= '<div class="alert alert-danger">' . $error . '</div>';
        }

        $output .= <<<EOF
        <div class="panel panel-default">
            <div class="panel-heading">
                <h3 class="panel-title">$cardTitle</h3>
            </div>
            <div class="panel-body">
                <form method="post" action="$modulelink">
                    <input type="hidden" name="originalAction" value="apikey">
                    <div class="form-group">
                        <label for="apikey">API Key</label>
                        <input type="text" class="form-control" id="apikey" name="apikey" value="$currentApiKey" required>
                    </div>
                    <button type="submit" class="btn btn-primary">
                        $buttonText API Key
                    </button>
                </form>
            </div>
        </div>
        EOF;

        $output .= $this->closeContentContainer();

        return $output;
    }

    /**
     * Products management page.
     *
     * @param array $vars Module configuration parameters
     *
     * @return string
     */
    public function products($vars)
    {
        $modulelink = $vars['modulelink'];
        $success = '';
        $error = '';

        // Handle form submissions
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            if (isset($_POST['action'])) {
                switch ($_POST['action']) {
                    case 'add':
                        $success = $this->addProduct($_POST);
                        break;
                    case 'edit':
                        $success = $this->updateProduct($_POST);
                        break;
                    case 'delete':
                        $success = $this->deleteProduct($_POST['id']);
                        break;
                }
            }
        }

        // Get all WHMCS products for dropdown
        $whmcsProducts = [];
        try {
            $products = Capsule::table('tblproducts')
                ->select('id', 'name')
                // ->where('servertype', '!=', '')
                ->orderBy('name')
                ->get();

            foreach ($products as $product) {
                $whmcsProducts[$product->id] = $product->name;
            }
        } catch (\Exception $e) {
            $error = 'Error retrieving products: ' . $e->getMessage();
        }

        // Get all hourly products
        $hourlyProducts = [];
        try {
            $products = Capsule::table('mod_HourlyAddonSeller_products')
                ->select('*')
                ->orderBy('id', 'desc')
                ->get();

            foreach ($products as $product) {
                $hourlyProducts[] = (array) $product;
            }
        } catch (\Exception $e) {
            $error = 'Error retrieving hourly products: ' . $e->getMessage();
        }

        // Start building the output
        $output = $this->generateMenu('products', $modulelink);

        if ($success) {
            $output .= '<div class="alert alert-success">' . $success . '</div>';
        }

        if ($error) {
            $output .= '<div class="alert alert-danger">' . $error . '</div>';
        }

        // Add product form
        $output .= <<<EOF
            <div class="panel panel-default">
                <div class="panel-heading">
                    <h3 class="panel-title">Add New Hourly Product</h3>
                </div>
                <div class="panel-body">
                    <form method="post" action="$modulelink">
                        <input type="hidden" name="action" value="add">
                        <input type="hidden" name="originalAction" value="products">
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="product_id">WHMCS Product</label>
                                    <select class="form-control" id="product_id" name="product_id" required>
                                        <option value="">-- Select Product --</option>
            EOF;

        foreach ($whmcsProducts as $id => $name) {
            $output .= '<option value="' . $id . '">' . htmlspecialchars($name) . '</option>';
        }

        $output .= <<<EOF
                                    </select>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="minimum_balance">Minimum Balance</label>
                                    <input type="number" class="form-control" id="minimum_balance" name="minimum_balance" required>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="service_amount">Hourly Service Amount</label>
                                    <input type="number" class="form-control" id="service_amount" name="service_amount" required>
                                </div>
                            </div>
                            
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="traffic_amount">Traffic Amount (per GB)</label>
                                    <input type="number" class="form-control" id="traffic_amount" name="traffic_amount" required>
                                </div>
                            </div>
                            
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="free_traffic">Free Traffic (GB)</label>
                                    <input type="number" class="form-control" id="free_traffic" name="free_traffic" value="0" required>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="traffic_type">Traffic Type</label>
                                    <select class="form-control" id="traffic_type" name="traffic_type" required>
                                        <option value="download">Download Only</option>
                                        <option value="upload">Upload Only</option>
                                        <option value="both">Both (Download + Upload)</option>
                                    </select>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="prtype">Product Type</label>
                                    <input type="text" class="form-control" id="prtype" name="prtype" value="virtualizor">
                                </div>
                            </div>
                        </div>
                        
                        <button type="submit" class="btn btn-success">
                            Add Product
                        </button>
                    </form>
                </div>
            </div>
            EOF;

        // Product list
        $output .= <<<EOF
            <div class="panel panel-default">
                <div class="panel-heading">
                    <h3 class="panel-title">Hourly Products</h3>
                </div>
                <div class="panel-body">
                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Product ID</th>
                                    <th>Min Balance</th>
                                    <th>Hourly Rate</th>
                                    <th>Traffic Rate</th>
                                    <th>Free Traffic</th>
                                    <th>Traffic Type</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
            EOF;

        if (empty($hourlyProducts)) {
            $output .= '<tr><td colspan="8" class="text-center">No products found</td></tr>';
        } else {
            foreach ($hourlyProducts as $product) {
                $productName = isset($whmcsProducts[$product['product_id']]) ? $whmcsProducts[$product['product_id']] : 'Unknown';

                $output .= '<tr>';
                $output .= '<td>' . $product['id'] . '</td>';
                $output .= '<td>' . $product['product_id'] . ' - ' . htmlspecialchars($productName) . '</td>';
                $output .= '<td>' . number_format($product['minimum_balance']) . '</td>';
                $output .= '<td>' . number_format($product['service_amount']) . '</td>';
                $output .= '<td>' . number_format($product['traffic_amount']) . '</td>';
                $output .= '<td>' . number_format($product['free_traffic']) . '</td>';
                $output .= '<td>' . $product['traffic_type'] . '</td>';
                $output .= '<td>';
                $output .= '<a href="' . $modulelink . '&action=editproduct&id=' . $product['id'] . '" class="btn btn-sm btn-primary">Edit</a> ';
                $output .= '<form method="post" action="' . $modulelink . '" style="display:inline;">';
                //$output .= '<input type="hidden" name="action" value="delete">';
                $output .= '<input type="hidden" name="originalAction" value="products">';
                $output .= '<input type="hidden" name="id" value="' . $product['id'] . '">';
                //$output .= '<button type="submit" class="btn btn-sm btn-danger" onclick="return confirm(\'Are you sure you want to delete this product?\')">Delete</button>';
                $output .= '</form>';
                $output .= '</td>';
                $output .= '</tr>';
            }
        }

        $output .= <<<EOF
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
            EOF;

        $output .= $this->closeContentContainer();

        return $output;
    }

    /**
     * Edit product page.
     *
     * @param array $vars Module configuration parameters
     *
     * @return string
     */
    public function editproduct($vars)
    {
        $modulelink = $vars['modulelink'];
        $id = isset($_GET['id']) ? (int) $_GET['id'] : 0;
        $error = '';

        if (!$id) {
            return '<div class="alert alert-danger">Invalid product ID</div>';
        }

        // Get product details
        try {
            $product = Capsule::table('mod_HourlyAddonSeller_products')
                ->where('id', '=', $id)
                ->first();

            if (!$product) {
                return '<div class="alert alert-danger">Product not found</div>';
            }
        } catch (\Exception $e) {
            return '<div class="alert alert-danger">Error retrieving product: ' . $e->getMessage() . '</div>';
        }

        // Get all WHMCS products for dropdown
        $whmcsProducts = [];
        try {
            $products = Capsule::table('tblproducts')
                ->select('id', 'name')
                ->where('servertype', '!=', '')
                ->orderBy('name')
                ->get();

            foreach ($products as $p) {
                $whmcsProducts[$p->id] = $p->name;
            }
        } catch (\Exception $e) {
            $error = 'Error retrieving products: ' . $e->getMessage();
        }

        $output = $this->generateMenu('products', $modulelink);

        if ($error) {
            $output .= '<div class="alert alert-danger">' . $error . '</div>';
        }

        // Edit form
        $output .= <<<EOF
            <div class="panel panel-default">
                <div class="panel-heading">
                    <h3 class="panel-title">Edit Hourly Product</h3>
                </div>
                <div class="panel-body">
                    <form method="post" action="$modulelink">
                        <input type="hidden" name="action" value="edit">
                        <input type="hidden" name="originalAction" value="products">
                        <input type="hidden" name="id" value="{$product->id}">
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="product_id">WHMCS Product</label>
                                    <select class="form-control" id="product_id" name="product_id" required>
                                        <option value="">-- Select Product --</option>
            EOF;

        foreach ($whmcsProducts as $pid => $name) {
            $selected = ($pid == $product->product_id) ? 'selected' : '';
            $output .= '<option value="' . $pid . '" ' . $selected . '>' . htmlspecialchars($name) . '</option>';
        }

        $output .= <<<EOF
                                    </select>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="minimum_balance">Minimum Balance</label>
                                    <input type="number" class="form-control" id="minimum_balance" name="minimum_balance" value="{$product->minimum_balance}" required>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="service_amount">Hourly Service Amount</label>
                                    <input type="number" class="form-control" id="service_amount" name="service_amount" value="{$product->service_amount}" required>
                                </div>
                            </div>
                            
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="traffic_amount">Traffic Amount (per GB)</label>
                                    <input type="number" class="form-control" id="traffic_amount" name="traffic_amount" value="{$product->traffic_amount}" required>
                                </div>
                            </div>
                            
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="free_traffic">Free Traffic (GB)</label>
                                    <input type="number" class="form-control" id="free_traffic" name="free_traffic" value="{$product->free_traffic}" required>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="traffic_type">Traffic Type</label>
                                    <select class="form-control" id="traffic_type" name="traffic_type" required>
            EOF;

        $types = ['download', 'upload', 'both'];
        foreach ($types as $type) {
            $selected = ($type == $product->traffic_type) ? 'selected' : '';
            $output .= '<option value="' . $type . '" ' . $selected . '>' . ucfirst($type) . '</option>';
        }

        $output .= <<<EOF
                                    </select>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="prtype">Product Type</label>
                                    <input type="text" class="form-control" id="prtype" name="prtype" value="{$product->prtype}">
                                </div>
                            </div>
                        </div>
                        
                        <div>
                            <button type="submit" class="btn btn-primary">
                                Update Product
                            </button>
                            <a href="$modulelink&action=products" class="btn btn-default">
                                Cancel
                            </a>
                        </div>
                    </form>
                </div>
            </div>
            EOF;

        $output .= $this->closeContentContainer();

        return $output;
    }

    /**
     * Reports page.
     *
     * @param array $vars Module configuration parameters
     *
     * @return string
     */
    public function reports($vars)
    {
        $modulelink = $vars['modulelink'];

        $output = $this->generateMenu('reports', $modulelink);

        $output .= <<<EOF
        <div class="panel panel-default">
            <div class="panel-heading">
                <h3 class="panel-title">Usage Reports</h3>
            </div>
            <div class="panel-body">
                <p>This section will contain usage reports for hourly billing services.</p>
                <p>Coming soon...</p>
            </div>
        </div>
        EOF;

        $output .= $this->closeContentContainer();

        return $output;
    }

    /**
     * Settings page.
     *
     * @param array $vars Module configuration parameters
     *
     * @return string
     */
    public function settings($vars)
    {
        $modulelink = $vars['modulelink'];

        $output = $this->generateMenu('settings', $modulelink);

        $output .= <<<EOF
        <div class="panel panel-default">
            <div class="panel-heading">
                <h3 class="panel-title">Module Settings</h3>
            </div>
            <div class="panel-body">
                <p>This section will contain module configuration settings.</p>
                <p>Coming soon...</p>
            </div>
        </div>
        EOF;

        $output .= $this->closeContentContainer();

        return $output;
    }

    /**
     * Add a new product
     * 
     * @param array $data Form data
     * @return string Success message
     */
    private function addProduct($data)
    {
        try {
            // Check if product with same WHMCS product ID already exists
            $exists = Capsule::table('mod_HourlyAddonSeller_products')
                ->where('product_id', '=', (int) $data['product_id'])
                ->exists();

            if ($exists) {
                return 'Error: A product with this WHMCS product ID already exists';
            }

            // Insert into database
            $productId = Capsule::table('mod_HourlyAddonSeller_products')
                ->insertGetId([
                    'product_id' => (int) $data['product_id'],
                    'minimum_balance' => (int) $data['minimum_balance'],
                    'service_amount' => (int) $data['service_amount'],
                    'traffic_amount' => (int) $data['traffic_amount'],
                    'free_traffic' => (int) $data['free_traffic'],
                    'traffic_type' => $data['traffic_type'],
                    'prtype' => $data['prtype'],
                    'created_at' => date('Y-m-d H:i:s'),
                    'updated_at' => date('Y-m-d H:i:s')
                ]);

            // Include hooks file to access API queue functions
            require_once(__DIR__ . '/../../hooks.php');

            // Add to API queue
            $apiParams = [
                'action' => 'product_add',
                'product_id' => $productId,
                'whmcs_product_id' => (int) $data['product_id'],
                'minimum_balance' => (int) $data['minimum_balance'],
                'service_amount' => (int) $data['service_amount'],
                'traffic_amount' => (int) $data['traffic_amount'],
                'free_traffic' => (int) $data['free_traffic'],
                'traffic_type' => $data['traffic_type'],
                'prtype' => $data['prtype']
            ];

            addToApiQueue($apiParams);

            return 'Product added successfully';
        } catch (\Exception $e) {
            return 'Error adding product: ' . $e->getMessage();
        }
    }

    /**
     * Update an existing product
     * 
     * @param array $data Form data
     * @return string Success message
     */
    private function updateProduct($data)
    {
        try {
            // Check if another product with the same WHMCS product ID exists
            $exists = Capsule::table('mod_HourlyAddonSeller_products')
                ->where('product_id', '=', (int) $data['product_id'])
                ->where('id', '!=', (int) $data['id'])
                ->exists();

            if ($exists) {
                return 'Error: Another product with this WHMCS product ID already exists';
            }

            // Update database
            Capsule::table('mod_HourlyAddonSeller_products')
                ->where('id', '=', (int) $data['id'])
                ->update([
                    'product_id' => (int) $data['product_id'],
                    'minimum_balance' => (int) $data['minimum_balance'],
                    'service_amount' => (int) $data['service_amount'],
                    'traffic_amount' => (int) $data['traffic_amount'],
                    'free_traffic' => (int) $data['free_traffic'],
                    'traffic_type' => $data['traffic_type'],
                    'prtype' => $data['prtype'],
                    'updated_at' => date('Y-m-d H:i:s')
                ]);

            // Include hooks file to access API queue functions
            require_once(__DIR__ . '/../../hooks.php');

            // Add to API queue
            $apiParams = [
                'action' => 'product_update',
                'product_id' => (int) $data['id'],
                'whmcs_product_id' => (int) $data['product_id'],
                'minimum_balance' => (int) $data['minimum_balance'],
                'service_amount' => (int) $data['service_amount'],
                'traffic_amount' => (int) $data['traffic_amount'],
                'free_traffic' => (int) $data['free_traffic'],
                'traffic_type' => $data['traffic_type'],
                'prtype' => $data['prtype']
            ];

            addToApiQueue($apiParams);

            return 'Product updated successfully';
        } catch (\Exception $e) {
            return 'Error updating product: ' . $e->getMessage();
        }
    }

    /**
     * Delete a product
     * 
     * @param int $id Product ID
     * @return string Success message
     */
    private function deleteProduct($id)
    {
        try {
            // Get product details before deletion
            $product = Capsule::table('mod_HourlyAddonSeller_products')
                ->where('id', '=', (int) $id)
                ->first();

            if (!$product) {
                return 'Product not found';
            }

            // Delete from database
            Capsule::table('mod_HourlyAddonSeller_products')
                ->where('id', '=', (int) $id)
                ->delete();

            // Include hooks file to access API queue functions
            require_once(__DIR__ . '/../../hooks.php');

            // Add to API queue
            $apiParams = [
                'action' => 'product_delete',
                'product_id' => (int) $id,
                'whmcs_product_id' => $product->product_id
            ];

            addToApiQueue($apiParams);

            return 'Product deleted successfully';
        } catch (\Exception $e) {
            return 'Error deleting product: ' . $e->getMessage();
        }
    }

    /**
     * Need to configure page.
     *
     * @param array $vars Module configuration parameters
     *
     * @return string
     */
    public function NeedToConfigure($vars)
    {
        $modulelink = $vars['modulelink'];

        return <<<EOF
        <div class="alert alert-warning">
            <h3>Configuration Required</h3>
            <p>Please configure your license key before using this module.</p>
            <p><a href="../modules/addons/AddonSeller/license.php" class="btn btn-warning">Configure License</a></p>
        </div>
        EOF;
    }

    /**
     * Need to upgrade page.
     *
     * @param array $vars Module configuration parameters
     *
     * @return string
     */
    public function NeedToUpgrade($vars)
    {
        $modulelink = $vars['modulelink'];

        return <<<EOF
        <div class="alert alert-warning">
            <h3>Upgrade Required</h3>
            <p>A new version of this module is available. Please upgrade to continue using all features.</p>
        </div>
        EOF;
    }
    /**
     * Generate navigation menu
     *
     * @param string $currentAction Current active menu item
     * @param string $modulelink Base module link
     * @return string HTML for navigation menu
     */
    private function generateMenu($currentAction, $modulelink)
    {
        $menuItems = [
            'index' => 'Dashboard',
            'apikey' => 'API Key',
            'products' => 'Products',
            'services' => 'Services',
            'balances' => 'Balances',
            'invoices' => 'Invoices',
            'logs' => 'Logs',
            'settings' => 'Settings'
        ];

        $output = '<div class="tabs-nav"><ul class="nav nav-tabs">';

        foreach ($menuItems as $action => $label) {
            $active = ($currentAction == $action) ? ' class="active"' : '';
            $output .= '<li' . $active . '><a href="' . $modulelink . '&action=' . $action . '">' . $label . '</a></li>';
        }

        $output .= '</ul></div><div class="tab-content"><div class="tab-pane active"><div class="content-padded">';

        return $output;
    }

    /**
     * Services management page.
     *
     * @param array $vars Module configuration parameters
     *
     * @return string
     */
    public function services($vars)
    {
        $modulelink = $vars['modulelink'];
        $success = '';
        $error = '';

        // Handle form submissions
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            if (isset($_POST['action'])) {
                switch ($_POST['action']) {
                    case 'add_service':
                        $success = $this->addService($_POST);
                        break;
                    case 'edit_service':
                        $success = $this->updateService($_POST);
                        break;
                    case 'delete_service':
                        $success = $this->deleteService($_POST['id']);
                        break;
                }
            }
        }

        // Get all services
        $services = [];
        try {
            $services = Capsule::table('mod_HourlyAddonSeller_services')
                ->select('*')
                ->orderBy('id', 'desc')
                ->get();
        } catch (\Exception $e) {
            $error = 'Error retrieving services: ' . $e->getMessage();
        }

        // Start building the output
        $output = $this->generateMenu('services', $modulelink);

        if ($success) {
            $output .= '<div class="alert alert-success">' . $success . '</div>';
        }

        if ($error) {
            $output .= '<div class="alert alert-danger">' . $error . '</div>';
        }

        // Add service form
        $output .= <<<EOF
            <div class="panel panel-default">
                <div class="panel-heading">
                    <h3 class="panel-title">Add New Service</h3>
                </div>
                <div class="panel-body">
                    <form method="post" action="$modulelink">
                        <input type="hidden" name="action" value="add_service">
                        <input type="hidden" name="originalAction" value="services">
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="service_id">Service ID</label>
                                    <input type="number" class="form-control" id="service_id" name="service_id" required>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="status">Status</label>
                                    <select class="form-control" id="status" name="status" required>
                                        <option value="Active">Active</option>
                                        <option value="Suspended">Suspended</option>
                                        <option value="Terminated">Terminated</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="last_traffic">Last Traffic</label>
                                    <input type="number" class="form-control" id="last_traffic" name="last_traffic" value="0">
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="start_time">Start Time</label>
                                    <input type="text" class="form-control" id="start_time" name="start_time" value="'date('Y-m-d H:i:s');'">
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="last_calculated">Last Calculated</label>
                                    <input type="text" class="form-control" id="last_calculated" name="last_calculated" value="<?php echo date('Y-m-d H:i:s'); ?>">
                                </div>
                            </div>
                        </div>
                        
                        <button type="submit" class="btn btn-success">
                            Add Service
                        </button>
                    </form>
                </div>
            </div>
            EOF;

        // Services list
        $output .= <<<EOF
            <div class="panel panel-default">
                <div class="panel-heading">
                    <h3 class="panel-title">Services</h3>
                </div>
                <div class="panel-body">
                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Service ID</th>
                                    <th>Status</th>
                                    <th>Start Time</th>
                                    <th>Last Calculated</th>
                                    <th>Last Traffic</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
            EOF;

        if (count($services) == 0) {
            $output .= '<tr><td colspan="7" class="text-center">No services found</td></tr>';
        } else {
            foreach ($services as $service) {
                $output .= '<tr>';
                $output .= '<td>' . $service->id . '</td>';
                $output .= '<td>' . $service->service_id . '</td>';
                $output .= '<td>' . $service->status . '</td>';
                $output .= '<td>' . $service->start_time . '</td>';
                $output .= '<td>' . $service->last_calculated . '</td>';
                $output .= '<td>' . $service->last_traffic . '</td>';
                $output .= '<td>';
                $output .= '<a href="' . $modulelink . '&action=editservice&id=' . $service->id . '" class="btn btn-sm btn-primary">Edit</a> ';
                $output .= '<form method="post" action="' . $modulelink . '" style="display:inline;">';
                //$output .= '<input type="hidden" name="action" value="delete_service">';
                $output .= '<input type="hidden" name="originalAction" value="services">';
                $output .= '<input type="hidden" name="id" value="' . $service->id . '">';
                //$output .= '<button type="submit" class="btn btn-sm btn-danger" onclick="return confirm(\'Are you sure you want to delete this service?\')">Delete</button>';
                $output .= '</form>';
                $output .= '</td>';
                $output .= '</tr>';
            }
        }

        $output .= <<<EOF
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
            EOF;

        $output .= $this->closeContentContainer();

        return $output;
    }

    /**
     * Logs management page.
     *
     * @param array $vars Module configuration parameters
     *
     * @return string
     */
    public function logs($vars)
    {
        $modulelink = $vars['modulelink'];
        $success = '';
        $error = '';

        // Handle form submissions
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            if (isset($_POST['action'])) {
                switch ($_POST['action']) {
                    case 'retry_queue':
                        if (isset($_POST['queue_id'])) {
                            $success = $this->retryQueueItem($_POST['queue_id']);
                        }
                        break;
                }
            }
        }

        // Get all queue items
        $queueItems = [];
        try {
            $queueItems = Capsule::table('mod_HourlyAddonSeller_api_queue')
                ->select('*')
                ->orderBy('id', 'desc')
                ->limit(100)
                ->get();
        } catch (\Exception $e) {
            $error = 'Error retrieving queue items: ' . $e->getMessage();
        }

        // Start building the output
        $output = $this->generateMenu('logs', $modulelink);

        if ($success) {
            $output .= '<div class="alert alert-success">' . $success . '</div>';
        }

        if ($error) {
            $output .= '<div class="alert alert-danger">' . $error . '</div>';
        }

        // Queue items list
        $output .= <<<EOF
            <div class="panel panel-default">
                <div class="panel-heading">
                    <h3 class="panel-title">API Queue Items</h3>
                </div>
                <div class="panel-body">
                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Request Data</th>
                                    <th>Status</th>
                                    <th>Attempts</th>
                                    <th>Last Attempt</th>
                                    <th>Error</th>
                                    <th>Created</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
            EOF;

        if (count($queueItems) == 0) {
            $output .= '<tr><td colspan="8" class="text-center">No queue items found</td></tr>';
        } else {
            foreach ($queueItems as $item) {
                // Parse request data to show action
                $requestData = json_decode($item->request_data, true);
                $action = isset($requestData['action']) ? $requestData['action'] : 'unknown';

                $output .= '<tr>';
                $output .= '<td>' . $item->id . '</td>';
                $output .= '<td>' . htmlspecialchars($action) . '</td>';
                $output .= '<td>' . ucfirst($item->status) . '</td>';
                $output .= '<td>' . $item->attempts . '</td>';
                $output .= '<td>' . ($item->last_attempt ?: 'Never') . '</td>';
                $output .= '<td>' . htmlspecialchars(substr($item->error, 0, 50)) . (strlen($item->error) > 50 ? '...' : '') . '</td>';
                $output .= '<td>' . $item->created_at . '</td>';
                $output .= '<td>';

                // Only show retry button for failed items
                if ($item->status == 'failed') {
                    $output .= '<form method="post" action="' . $modulelink . '" style="display:inline;">';
                    $output .= '<input type="hidden" name="action" value="retry_queue">';
                    $output .= '<input type="hidden" name="originalAction" value="logs">';
                    $output .= '<input type="hidden" name="queue_id" value="' . $item->id . '">';
                    $output .= '<button type="submit" class="btn btn-sm btn-warning">Retry</button>';
                    $output .= '</form>';
                }

                $output .= '</td>';
                $output .= '</tr>';
            }
        }

        $output .= <<<EOF
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
            EOF;

        $output .= $this->closeContentContainer();

        return $output;
    }

    /**
     * Retry a failed queue item
     * 
     * @param int $queueId Queue item ID
     * @return string Success message
     */
    private function retryQueueItem($queueId)
    {
        try {
            // Get the queue item
            $queueItem = Capsule::table('mod_HourlyAddonSeller_api_queue')
                ->where('id', '=', (int) $queueId)
                ->first();

            if (!$queueItem) {
                return 'Queue item not found';
            }

            // Reset the status and attempts
            Capsule::table('mod_HourlyAddonSeller_api_queue')
                ->where('id', '=', (int) $queueId)
                ->update([
                    'status' => 'pending',
                    'error' => null,
                    'updated_at' => date('Y-m-d H:i:s')
                ]);

            // Include hooks.php to access the processApiQueueItem function
            require_once(__DIR__ . '/../../hooks.php');

            // Process the queue item immediately
            $result = processApiQueueItem((array) $queueItem);

            if ($result) {
                return 'Queue item processed successfully';
            } else {
                return 'Queue item requeued for processing';
            }
        } catch (\Exception $e) {
            return 'Error retrying queue item: ' . $e->getMessage();
        }
    }

    public function balances($vars)
    {
        $modulelink = $vars['modulelink'];
        $success = '';
        $error = '';

        // Handle form submissions
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            if (isset($_POST['action'])) {
                switch ($_POST['action']) {
                    case 'add_balance':
                        $success = $this->addBalance($_POST);
                        break;
                    case 'edit_balance':
                        $success = $this->updateBalance($_POST);
                        break;
                    case 'delete_balance':
                        $success = $this->deleteBalance($_POST['userid']);
                        break;
                }
            }
        }

        // Get all balances
        $balances = [];
        try {
            $balances = Capsule::table('mod_HourlyAddonSeller_balance')
                ->select('mod_HourlyAddonSeller_balance.*', 'tblclients.firstname', 'tblclients.lastname', 'tblclients.email')
                ->leftJoin('tblclients', 'mod_HourlyAddonSeller_balance.userid', '=', 'tblclients.id')
                ->orderBy('userid', 'asc')
                ->get();
        } catch (\Exception $e) {
            $error = 'Error retrieving balances: ' . $e->getMessage();
        }

        // Get all clients for dropdown
        $clients = [];
        try {
            $clientsData = Capsule::table('tblclients')
                ->select('id', 'firstname', 'lastname', 'email', 'companyname')
                ->orderBy('id', 'asc')
                ->get();

            foreach ($clientsData as $client) {
                $displayName = $client->id . ' - ' . $client->firstname . ' ' . $client->lastname;
                if (!empty($client->companyname)) {
                    $displayName .= ' (' . $client->companyname . ')';
                }
                $displayName .= ' - ' . $client->email;
                $clients[$client->id] = $displayName;
            }
        } catch (\Exception $e) {
            $error = 'Error retrieving clients: ' . $e->getMessage();
        }

        // Start building the output
        $output = $this->generateMenu('balances', $modulelink);

        if ($success) {
            $output .= '<div class="alert alert-success">' . $success . '</div>';
        }

        if ($error) {
            $output .= '<div class="alert alert-danger">' . $error . '</div>';
        }

        // Add balance form
        $output .= <<<EOF
            <div class="panel panel-default">
            <div class="panel-heading">
                <h3 class="panel-title">Add New Balance</h3>
            </div>
            <div class="panel-body">
                <form method="post" action="$modulelink">
                    <input type="hidden" name="action" value="add_balance">
                    <input type="hidden" name="originalAction" value="balances">
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="userid">User</label>
                                <select class="form-control chosen-select" id="userid" name="userid" required data-placeholder="Select a user...">
                                    <option value=""></option>
            EOF;

        foreach ($clients as $id => $name) {
            $output .= '<option value="' . $id . '">' . htmlspecialchars($name) . '</option>';
        }

        $output .= <<<EOF
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="balance">Balance</label>
                                <input type="number" class="form-control" id="balance" name="balance" value="0" required>
                            </div>
                        </div>
                    </div>
                    
                    <button type="submit" class="btn btn-success">
                        Add Balance
                    </button>
                </form>
            </div>
            </div>

            <script>
            jQuery(document).ready(function() {
            jQuery(".chosen-select").chosen({
                width: "100%",
                search_contains: true
            });
            });
            </script>
            EOF;

        // Balances list
        $output .= <<<EOF
            <div class="panel panel-default">
            <div class="panel-heading">
                <h3 class="panel-title">User Balances</h3>
            </div>
            <div class="panel-body">
                <div class="table-responsive">
                    <table class="table table-striped">
                        <thead>
                            <tr>
                                <th>User ID</th>
                                <th>Name</th>
                                <th>Email</th>
                                <th>Balance</th>
                                <th>Created</th>
                                <th>Updated</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
            EOF;

        if (count($balances) == 0) {
            $output .= '<tr><td colspan="7" class="text-center">No balances found</td></tr>';
        } else {
            foreach ($balances as $balance) {
                $output .= '<tr>';
                $output .= '<td>' . $balance->userid . '</td>';
                $output .= '<td>' . htmlspecialchars($balance->firstname . ' ' . $balance->lastname) . '</td>';
                $output .= '<td>' . htmlspecialchars($balance->email) . '</td>';
                $output .= '<td>' . number_format($balance->balance) . '</td>';
                $output .= '<td>' . $balance->created_at . '</td>';
                $output .= '<td>' . $balance->updated_at . '</td>';
                $output .= '<td>';
                $output .= '<a href="' . $modulelink . '&action=editbalance&userid=' . $balance->userid . '" class="btn btn-sm btn-primary">Edit</a> ';
                $output .= '<form method="post" action="' . $modulelink . '" style="display:inline;">';
                //$output .= '<input type="hidden" name="action" value="delete_balance">';
                $output .= '<input type="hidden" name="originalAction" value="balances">';
                $output .= '<input type="hidden" name="userid" value="' . $balance->userid . '">';
                //$output .= '<button type="submit" class="btn btn-sm btn-danger" onclick="return confirm(\'Are you sure you want to delete this balance?\')">Delete</button>';
                $output .= '</form>';
                $output .= '</td>';
                $output .= '</tr>';
            }
        }

        $output .= <<<EOF
                        </tbody>
                    </table>
                </div>
            </div>
            </div>
            EOF;

        $output .= $this->closeContentContainer();

        return $output;
    }

    /**
     * Add a new balance
     * 
     * @param array $data Form data
     * @return string Success message
     */
    private function addBalance($data)
    {
        try {
            // Check if user already has a balance record
            $exists = Capsule::table('mod_HourlyAddonSeller_balance')
                ->where('userid', '=', (int) $data['userid'])
                ->exists();

            $action = $exists ? 'balance_update' : 'balance_add';
            $message = $exists ? 'Balance updated successfully' : 'Balance added successfully';

            if ($exists) {
                // Update existing record
                Capsule::table('mod_HourlyAddonSeller_balance')
                    ->where('userid', '=', (int) $data['userid'])
                    ->update([
                        'balance' => (int) $data['balance'],
                        'updated_at' => date('Y-m-d H:i:s')
                    ]);
            } else {
                // Insert new record
                Capsule::table('mod_HourlyAddonSeller_balance')
                    ->insert([
                        'userid' => (int) $data['userid'],
                        'balance' => (int) $data['balance'],
                        'created_at' => date('Y-m-d H:i:s'),
                        'updated_at' => date('Y-m-d H:i:s')
                    ]);
            }

            // Include hooks file to access API queue functions
            require_once(__DIR__ . '/../../hooks.php');

            // Add to API queue
            $apiParams = [
                'action' => $action,
                'userid' => (int) $data['userid'],
                'balance' => (int) $data['balance']
            ];

            addToApiQueue($apiParams);

            return $message;
        } catch (\Exception $e) {
            return 'Error adding balance: ' . $e->getMessage();
        }
    }

    /**
     * Update an existing balance
     * 
     * @param array $data Form data
     * @return string Success message
     */
    private function updateBalance($data)
    {
        try {
            // Update database
            Capsule::table('mod_HourlyAddonSeller_balance')
                ->where('userid', '=', (int) $data['userid'])
                ->update([
                    'balance' => (int) $data['balance'],
                    'updated_at' => date('Y-m-d H:i:s')
                ]);

            // Include hooks file to access API queue functions
            require_once(__DIR__ . '/../../hooks.php');

            // Add to API queue
            $apiParams = [
                'action' => 'balance_update',
                'userid' => (int) $data['userid'],
                'balance' => (int) $data['balance']
            ];

            addToApiQueue($apiParams);

            return 'Balance updated successfully';
        } catch (\Exception $e) {
            return 'Error updating balance: ' . $e->getMessage();
        }
    }
    public function editbalance($vars)
    {
        $modulelink = $vars['modulelink'];
        $userid = isset($_GET['userid']) ? (int) $_GET['userid'] : 0;
        $error = '';

        if (!$userid) {
            return '<div class="alert alert-danger">Invalid user ID</div>';
        }

        // Get balance details
        try {
            $balance = Capsule::table('mod_HourlyAddonSeller_balance')
                ->where('userid', '=', $userid)
                ->first();

            if (!$balance) {
                return '<div class="alert alert-danger">Balance not found</div>';
            }

            // Get user details
            $user = Capsule::table('tblclients')
                ->where('id', '=', $userid)
                ->first();

        } catch (\Exception $e) {
            return '<div class="alert alert-danger">Error retrieving balance: ' . $e->getMessage() . '</div>';
        }

        $output = $this->generateMenu('balances', $modulelink);

        if ($error) {
            $output .= '<div class="alert alert-danger">' . $error . '</div>';
        }

        // Edit form
        $output .= <<<EOF
        <div class="panel panel-default">
            <div class="panel-heading">
                <h3 class="panel-title">Edit Balance</h3>
            </div>
            <div class="panel-body">
                <form method="post" action="$modulelink">
                    <input type="hidden" name="action" value="edit_balance">
                    <input type="hidden" name="originalAction" value="balances">
                    <input type="hidden" name="userid" value="{$balance->userid}">
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>User</label>
                                <p class="form-control-static">{$user->id} - {$user->firstname} {$user->lastname} - {$user->email}</p>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="balance">Balance</label>
                                <input type="number" class="form-control" id="balance" name="balance" value="{$balance->balance}" required>
                            </div>
                        </div>
                    </div>
                    
                    <div>
                        <button type="submit" class="btn btn-primary">
                            Update Balance
                        </button>
                        <a href="$modulelink&action=balances" class="btn btn-default">
                            Cancel
                        </a>
                    </div>
                </form>
            </div>
        </div>
        EOF;

        $output .= $this->closeContentContainer();

        return $output;
    }
    /**
     * Delete a balance
     * 
     * @param int $userid User ID
     * @return string Success message
     */
    private function deleteBalance($userid)
    {
        try {
            // Get balance before deletion
            $balance = Capsule::table('mod_HourlyAddonSeller_balance')
                ->where('userid', '=', (int) $userid)
                ->first();

            if (!$balance) {
                return 'Balance not found';
            }

            // Delete from database
            Capsule::table('mod_HourlyAddonSeller_balance')
                ->where('userid', '=', (int) $userid)
                ->delete();

            // Include hooks file to access API queue functions
            require_once(__DIR__ . '/../../hooks.php');

            // Add to API queue
            $apiParams = [
                'action' => 'balance_delete',
                'userid' => (int) $userid,
                'previous_balance' => $balance->balance
            ];

            addToApiQueue($apiParams);

            return 'Balance deleted successfully';
        } catch (\Exception $e) {
            return 'Error deleting balance: ' . $e->getMessage();
        }
    }

    /**
     * Invoices management page.
     *
     * @param array $vars Module configuration parameters
     *
     * @return string
     */
    public function invoices($vars)
    {
        $modulelink = $vars['modulelink'];
        $success = '';
        $error = '';

        // Handle form submissions
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            if (isset($_POST['action'])) {
                switch ($_POST['action']) {
                    case 'add_invoice':
                        $success = $this->addInvoice($_POST);
                        break;
                    case 'edit_invoice':
                        $success = $this->updateInvoice($_POST);
                        break;
                    case 'delete_invoice':
                        $success = $this->deleteInvoice($_POST['invoice_id']);
                        break;
                }
            }
        }

        // Get all invoices
        $invoices = [];
        try {
            $invoices = Capsule::table('mod_HourlyAddonSeller_invoice')
                ->select('*')
                ->orderBy('invoice_id', 'desc')
                ->get();
        } catch (\Exception $e) {
            $error = 'Error retrieving invoices: ' . $e->getMessage();
        }

        // Start building the output
        $output = $this->generateMenu('invoices', $modulelink);

        if ($success) {
            $output .= '<div class="alert alert-success">' . $success . '</div>';
        }

        if ($error) {
            $output .= '<div class="alert alert-danger">' . $error . '</div>';
        }

        // Add invoice form
        $output .= <<<EOF
        <div class="panel panel-default">
            <div class="panel-heading">
                <h3 class="panel-title">Add New Invoice</h3>
            </div>
            <div class="panel-body">
                <form method="post" action="$modulelink">
                    <input type="hidden" name="action" value="add_invoice">
                    <input type="hidden" name="originalAction" value="invoices">
                    
                    <div class="row">
                        <div class="col-md-4">
                            <div class="form-group">
                                <label for="invoice_id">Invoice ID</label>
                                <input type="number" class="form-control" id="invoice_id" name="invoice_id" required>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-group">
                                <label for="amount">Amount</label>
                                <input type="number" class="form-control" id="amount" name="amount" required>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-group">
                                <label for="status">Status</label>
                                <select class="form-control" id="status" name="status" required>
                                    <option value="unpaid">Unpaid</option>
                                    <option value="paid">Paid</option>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <button type="submit" class="btn btn-success">
                        Add Invoice
                    </button>
                </form>
            </div>
        </div>
        EOF;

        // Invoices list
        $output .= <<<EOF
        <div class="panel panel-default">
            <div class="panel-heading">
                <h3 class="panel-title">Invoices</h3>
            </div>
            <div class="panel-body">
                <div class="table-responsive">
                    <table class="table table-striped">
                        <thead>
                            <tr>
                                <th>Invoice ID</th>
                                <th>Amount</th>
                                <th>Status</th>
                                <th>Created</th>
                                <th>Updated</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
        EOF;

        if (count($invoices) == 0) {
            $output .= '<tr><td colspan="6" class="text-center">No invoices found</td></tr>';
        } else {
            foreach ($invoices as $invoice) {
                $output .= '<tr>';
                $output .= '<td>' . $invoice->invoice_id . '</td>';
                $output .= '<td>' . number_format($invoice->amount) . '</td>';
                $output .= '<td>' . ucfirst($invoice->status) . '</td>';
                $output .= '<td>' . $invoice->created_at . '</td>';
                $output .= '<td>' . $invoice->updated_at . '</td>';
                $output .= '<td>';
                $output .= '<a href="' . $modulelink . '&action=editinvoice&invoice_id=' . $invoice->invoice_id . '" class="btn btn-sm btn-primary">Edit</a> ';
                $output .= '<form method="post" action="' . $modulelink . '" style="display:inline;">';
                //$output .= '<input type="hidden" name="action" value="delete_invoice">';
                $output .= '<input type="hidden" name="originalAction" value="invoices">';
                $output .= '<input type="hidden" name="invoice_id" value="' . $invoice->invoice_id . '">';
                //$output .= '<button type="submit" class="btn btn-sm btn-danger" onclick="return confirm(\'Are you sure you want to delete this invoice?\')">Delete</button>';
                $output .= '</form>';
                $output .= '</td>';
                $output .= '</tr>';
            }
        }

        $output .= <<<EOF
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        EOF;

        $output .= $this->closeContentContainer();

        return $output;
    }

    public function editinvoice($vars)
    {
        $modulelink = $vars['modulelink'];
        $invoice_id = isset($_GET['invoice_id']) ? (int) $_GET['invoice_id'] : 0;
        $error = '';

        if (!$invoice_id) {
            return '<div class="alert alert-danger">Invalid invoice ID</div>';
        }

        // Get invoice details
        try {
            $invoice = Capsule::table('mod_HourlyAddonSeller_invoice')
                ->where('invoice_id', '=', $invoice_id)
                ->first();

            if (!$invoice) {
                return '<div class="alert alert-danger">Invoice not found</div>';
            }
        } catch (\Exception $e) {
            return '<div class="alert alert-danger">Error retrieving invoice: ' . $e->getMessage() . '</div>';
        }

        $output = $this->generateMenu('invoices', $modulelink);

        if ($error) {
            $output .= '<div class="alert alert-danger">' . $error . '</div>';
        }

        // Edit form
        $output .= <<<EOF
        <div class="panel panel-default">
            <div class="panel-heading">
                <h3 class="panel-title">Edit Invoice</h3>
            </div>
            <div class="panel-body">
                <form method="post" action="$modulelink">
                    <input type="hidden" name="action" value="edit_invoice">
                    <input type="hidden" name="originalAction" value="invoices">
                    <input type="hidden" name="invoice_id" value="{$invoice->invoice_id}">
                    
                    <div class="row">
                        <div class="col-md-4">
                            <div class="form-group">
                                <label for="invoice_id_display">Invoice ID</label>
                                <input type="text" class="form-control" id="invoice_id_display" value="{$invoice->invoice_id}" disabled>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-group">
                                <label for="amount">Amount</label>
                                <input type="number" class="form-control" id="amount" name="amount" value="{$invoice->amount}" required>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-group">
                                <label for="status">Status</label>
                                <select class="form-control" id="status" name="status" required>
        EOF;

        $statuses = ['unpaid', 'paid'];
        foreach ($statuses as $status) {
            $selected = ($status == $invoice->status) ? 'selected' : '';
            $output .= '<option value="' . $status . '" ' . $selected . '>' . ucfirst($status) . '</option>';
        }

        $output .= <<<EOF
                        </select>
                    </div>
                </div>
            </div>
            
            <div>
                <button type="submit" class="btn btn-primary">
                    Update Invoice
                </button>
                <a href="$modulelink&action=invoices" class="btn btn-default">
                    Cancel
                </a>
            </div>
        </form>
    </div>
</div>
EOF;

        $output .= $this->closeContentContainer();

        return $output;
    }


}