/**
 * Handle suspend service request
 * 
 * @param array $data Request data
 * @return array Result data
 */
function handleSuspendService($data)
{
    // Validate required fields
    if (!isset($data['service_id'])) {
        throw new Exception("Missing required field: service_id");
    }
    
    // Check if service exists
    $service = Capsule::table('mod_HourlyAddonSeller_services')
        ->where('service_id', '=', (int) $data['service_id'])
        ->first();
        
    if (!$service) {
        throw new Exception("Service not found");
    }
    
    // Get WHMCS service details
    $whmcsService = Capsule::table('tblhosting')
        ->where('id', '=', (int) $data['service_id'])
        ->first();
    
    // Update service status to Suspended
    Capsule::table('mod_HourlyAddonSeller_services')
        ->where('service_id', '=', (int) $data['service_id'])
        ->update([
            'status' => 'Suspended',
            'updated_at' => date('Y-m-d H:i:s')
        ]);
    
    // Add to API queue if needed
    if (isset($data['queue_api']) && $data['queue_api']) {
        $apiParams = [
            'action' => 'suspend',
            'service_id' => (int) $data['service_id'],
            'product_id' => $whmcsService->packageid,
            'client_id' => $whmcsService->userid,
            'domain' => $whmcsService->domain,
            'server_ip' => $whmcsService->dedicatedip,
            'status' => 'Suspended',
            'suspend_reason' => isset($data['suspend_reason']) ? $data['suspend_reason'] : 'API Request'
        ];
        
        $queueId = addToApiQueue($apiParams);
        return ['service_id' => (int) $data['service_id'], 'status' => 'Suspended', 'queue_id' => $queueId];
    }
    
    return ['service_id' => (int) $data['service_id'], 'status' => 'Suspended'];
}

/**
 * Handle unsuspend service request
 * 
 * @param array $data Request data
 * @return array Result data
 */
function handleUnsuspendService($data)
{
    // Validate required fields
    if (!isset($data['service_id'])) {
        throw new Exception("Missing required field: service_id");
    }
    
    // Check if service exists
    $service = Capsule::table('mod_HourlyAddonSeller_services')
        ->where('service_id', '=', (int) $data['service_id'])
        ->first();
        
    if (!$service) {
        throw new Exception("Service not found");
    }
    
    // Get WHMCS service details
    $whmcsService = Capsule::table('tblhosting')
        ->where('id', '=', (int) $data['service_id'])
        ->first();
    
    // Update service status to Active
    Capsule::table('mod_HourlyAddonSeller_services')
        ->where('service_id', '=', (int) $data['service_id'])
        ->update([
            'status' => 'Active',
            'updated_at' => date('Y-m-d H:i:s')
        ]);
    
    // Add to API queue if needed
    if (isset($data['queue_api']) && $data['queue_api']) {
        $apiParams = [
            'action' => 'unsuspend',
            'service_id' => (int) $data['service_id'],
            'product_id' => $whmcsService->packageid,
            'client_id' => $whmcsService->userid,
            'domain' => $whmcsService->domain,
            'server_ip' => $whmcsService->dedicatedip,
            'status' => 'Active'
        ];
        
        $queueId = addToApiQueue($apiParams);
        return ['service_id' => (int) $data['service_id'], 'status' => 'Active', 'queue_id' => $queueId];
    }
    
    return ['service_id' => (int) $data['service_id'], 'status' => 'Active'];
}

/**
 * Handle suspend all services for a user
 * 
 * @param array $data Request data
 * @return array Result data
 */
function handleSuspendUserServices($data)
{
    // Validate required fields
    if (!isset($data['userid'])) {
        throw new Exception("Missing required field: userid");
    }
    
    // Check if user exists
    $userExists = Capsule::table('tblclients')
        ->where('id', '=', (int) $data['userid'])
        ->exists();
        
    if (!$userExists) {
        throw new Exception("User not found");
    }
    
    // Get all active services for this user
    $services = Capsule::table('tblhosting')
        ->where('userid', '=', (int) $data['userid'])
        ->whereIn('domainstatus', ['Active', 'Suspended'])
        ->get();
    
    if (count($services) == 0) {
        return ['userid' => (int) $data['userid'], 'services_suspended' => 0];
    }
    
    $suspendedCount = 0;
    $queueIds = [];
    
    // Process each service
    foreach ($services as $whmcsService) {
        // Check if service exists in our system
        $service = Capsule::table('mod_HourlyAddonSeller_services')
            ->where('service_id', '=', $whmcsService->id)
            ->first();
            
        if (!$service) {
            continue; // Skip if not in our system
        }
        
        // Update service status to Suspended
        Capsule::table('mod_HourlyAddonSeller_services')
            ->where('service_id', '=', $whmcsService->id)
            ->update([
                'status' => 'Suspended',
                'updated_at' => date('Y-m-d H:i:s')
            ]);
        
        $suspendedCount++;
        
        // Add to API queue if needed
        if (isset($data['queue_api']) && $data['queue_api']) {
            $apiParams = [
                'action' => 'suspend',
                'service_id' => $whmcsService->id,
                'product_id' => $whmcsService->packageid,
                'client_id' => $whmcsService->userid,
                'domain' => $whmcsService->domain,
                'server_ip' => $whmcsService->dedicatedip,
                'status' => 'Suspended',
                'suspend_reason' => isset($data['suspend_reason']) ? $data['suspend_reason'] : 'User Suspension'
            ];
            
            $queueIds[] = addToApiQueue($apiParams);
        }
    }
    
    $result = [
        'userid' => (int) $data['userid'],
        'services_suspended' => $suspendedCount
    ];
    
    if (isset($data['queue_api']) && $data['queue_api'] && !empty($queueIds)) {
        $result['queue_ids'] = $queueIds;
    }
    
    return $result;
}

/**
 * Handle unsuspend all services for a user
 * 
 * @param array $data Request data
 * @return array Result data
 */
function handleUnsuspendUserServices($data)
{
    // Validate required fields
    if (!isset($data['userid'])) {
        throw new Exception("Missing required field: userid");
    }
    
    // Check if user exists
    $userExists = Capsule::table('tblclients')
        ->where('id', '=', (int) $data['userid'])
        ->exists();
        
    if (!$userExists) {
        throw new Exception("User not found");
    }
    
    // Get all suspended services for this user
    $services = Capsule::table('tblhosting')
        ->where('userid', '=', (int) $data['userid'])
        ->where('domainstatus', '=', 'Suspended')
        ->get();
    
    if (count($services) == 0) {
        return ['userid' => (int) $data['userid'], 'services_unsuspended' => 0];
    }
    
    $unsuspendedCount = 0;
    $queueIds = [];
    
    // Process each service
    foreach ($services as $whmcsService) {
        // Check if service exists in our system
        $service = Capsule::table('mod_HourlyAddonSeller_services')
            ->where('service_id', '=', $whmcsService->id)
            ->first();
            
        if (!$service) {
            continue; // Skip if not in our system
        }
        
        // Update service status to Active
        Capsule::table('mod_HourlyAddonSeller_services')
            ->where('service_id', '=', $whmcsService->id)
            ->update([
                'status' => 'Active',
                'updated_at' => date('Y-m-d H:i:s')
            ]);
        
        $unsuspendedCount++;
        
        // Add to API queue if needed
        if (isset($data['queue_api']) && $data['queue_api']) {
            $apiParams = [
                'action' => 'unsuspend',
                'service_id' => $whmcsService->id,
                'product_id' => $whmcsService->packageid,
                'client_id' => $whmcsService->userid,
                'domain' => $whmcsService->domain,
                'server_ip' => $whmcsService->dedicatedip,
                'status' => 'Active'
            ];
            
            $queueIds[] = addToApiQueue($apiParams);
        }
    }
    
    $result = [
        'userid' => (int) $data['userid'],
        'services_unsuspended' => $unsuspendedCount
    ];
    
    if (isset($data['queue_api']) && $data['queue_api'] && !empty($queueIds)) {
        $result['queue_ids'] = $queueIds;
    }
    
    return $result;
}