<?php
/**
 * API Request Samples
 * 
 * This file contains sample JSON payloads for all API requests sent by the HourlyAddonSeller module.
 * Use these samples for documentation and testing purposes.
 */

// Load configuration from external file
$config = require_once(__DIR__ . '/api_config.php');

// Get parameters from config
$sampleApiKey = $config['api_key'];
$apiEndpoint = $config['api_endpoint'];
$sampleServiceId = $config['service_id'];
$sampleProductId = $config['product_id'];
$sampleClientId = $config['client_id'];
$sampleDomain = $config['domain'];
$sampleServerIp = $config['server_ip'];
$sampleInvoiceId = $config['invoice_id'];
$sampleAmount = $config['amount'];
$sampleBalance = $config['balance'];
$sampleHours = $config['hours'];
$sampleTraffic = $config['traffic'];
$sampleUsageAmount = $config['usage_amount'];
$sampleCreditAmount = $config['credit_amount'];
$sampleAdminId = $config['admin_id'];
$sampleMinimumBalance = $config['minimum_balance'] ?? 1000;
$sampleServiceAmount = $config['service_amount'] ?? 100;
$sampleTrafficAmount = $config['traffic_amount'] ?? 50;
$sampleFreeTraffic = $config['free_traffic'] ?? 10;
$sampleTrafficType = $config['traffic_type'] ?? 'both';
$samplePrType = $config['prtype'] ?? 'virtualizor';

/**
 * 1. Service Creation
 * Sent when a new service is created
 */
$createServiceRequest = [
    'action' => 'create',
    'service_id' => $sampleServiceId,
    'product_id' => $sampleProductId,
    'client_id' => $sampleClientId,
    'domain' => $sampleDomain,
    'server_ip' => $sampleServerIp,
    'status' => 'Active'
];

/**
 * 2. Service Suspension
 * Sent when a service is suspended
 */
$suspendServiceRequest = [
    'action' => 'suspend',
    'service_id' => $sampleServiceId,
    'product_id' => $sampleProductId,
    'client_id' => $sampleClientId,
    'domain' => $sampleDomain,
    'server_ip' => $sampleServerIp,
    'status' => 'Suspended',
    'suspend_reason' => 'Overdue payment'
];

/**
 * 3. Service Termination
 * Sent when a service is terminated
 */
$terminateServiceRequest = [
    'action' => 'terminate',
    'service_id' => $sampleServiceId,
    'product_id' => $sampleProductId,
    'client_id' => $sampleClientId,
    'domain' => $sampleDomain,
    'server_ip' => $sampleServerIp,
    'status' => 'Terminated'
];

/**
 * 4. Service Unsuspension
 * Sent when a service is unsuspended
 */
$unsuspendServiceRequest = [
    'action' => 'unsuspend',
    'service_id' => $sampleServiceId,
    'product_id' => $sampleProductId,
    'client_id' => $sampleClientId,
    'domain' => $sampleDomain,
    'server_ip' => $sampleServerIp,
    'status' => 'Active'
];

/**
 * 5. Package Change
 * Sent when a service's package is changed
 */
$packageChangeRequest = [
    'action' => 'package_change',
    'service_id' => $sampleServiceId,
    'product_id' => $sampleProductId, // New product ID
    'client_id' => $sampleClientId,
    'status' => 'Active'
];

/**
 * 6. Invoice Payment
 * Sent when an invoice is paid
 */
$invoicePaymentRequest = [
    'action' => 'invoice_paid',
    'invoice_id' => $sampleInvoiceId,
    'client_id' => $sampleClientId,
    'amount' => $sampleAmount,
    'balance' => $sampleBalance, // User's balance after payment
    'payment_date' => date('Y-m-d H:i:s')
];

/**
 * 7. Usage Update
 * Sent when updating service usage (typically from cron)
 */
$usageUpdateRequest = [
    'action' => 'update_usage',
    'service_id' => $sampleServiceId,
    'client_id' => $sampleClientId,
    'hours' => $sampleHours,
    'traffic' => $sampleTraffic, // GB
    'amount' => $sampleUsageAmount, // Amount charged
    'timestamp' => date('Y-m-d H:i:s')
];

/**
 * 8. Balance Check
 * Request to check user's balance
 */
$balanceCheckRequest = [
    'action' => 'check_balance',
    'client_id' => $sampleClientId
];

/**
 * 9. Service Status Check
 * Request to check service status
 */
$serviceStatusRequest = [
    'action' => 'check_service',
    'service_id' => $sampleServiceId
];

/**
 * 10. Add Credit
 * Request to add credit to user's account
 */
$addCreditRequest = [
    'action' => 'add_credit',
    'client_id' => $sampleClientId,
    'amount' => $sampleCreditAmount,
    'description' => 'Manual credit addition',
    'admin_id' => $sampleAdminId
];

/**
 * 11. Product Add
 * Sent when a new product is added
 */
$productAddRequest = [
    'action' => 'product_add',
    'product_id' => $sampleProductId,
    'whmcs_product_id' => $sampleProductId,
    'minimum_balance' => $sampleMinimumBalance,
    'service_amount' => $sampleServiceAmount,
    'traffic_amount' => $sampleTrafficAmount,
    'free_traffic' => $sampleFreeTraffic,
    'traffic_type' => $sampleTrafficType,
    'prtype' => $samplePrType
];

/**
 * 12. Product Update
 * Sent when a product is updated
 */
$productUpdateRequest = [
    'action' => 'product_update',
    'product_id' => $sampleProductId,
    'whmcs_product_id' => $sampleProductId,
    'minimum_balance' => $sampleMinimumBalance,
    'service_amount' => $sampleServiceAmount,
    'traffic_amount' => $sampleTrafficAmount,
    'free_traffic' => $sampleFreeTraffic,
    'traffic_type' => $sampleTrafficType,
    'prtype' => $samplePrType
];

/**
 * 13. Product Delete
 * Sent when a product is deleted
 */
$productDeleteRequest = [
    'action' => 'product_delete',
    'product_id' => $sampleProductId,
    'whmcs_product_id' => $sampleProductId
];

/**
 * 14. Balance Add
 * Sent when a new balance is added
 */
$balanceAddRequest = [
    'action' => 'balance_add',
    'userid' => $sampleClientId,
    'balance' => $sampleBalance
];

/**
 * 15. Balance Update
 * Sent when a balance is updated
 */
$balanceUpdateRequest = [
    'action' => 'balance_update',
    'userid' => $sampleClientId,
    'balance' => $sampleBalance
];

/**
 * 16. Balance Delete
 * Sent when a balance is deleted
 */
$balanceDeleteRequest = [
    'action' => 'balance_delete',
    'userid' => $sampleClientId,
    'previous_balance' => $sampleBalance
];

// Function to display the request as formatted JSON
function displayRequestAsJson($request) {
    return json_encode($request, JSON_PRETTY_PRINT);
}

// Example of how to use this file for testing
if (isset($_GET['test'])) {
    header('Content-Type: application/json');
    
    switch ($_GET['test']) {
        case 'create':
            echo displayRequestAsJson($createServiceRequest);
            break;
        case 'suspend':
            echo displayRequestAsJson($suspendServiceRequest);
            break;
        case 'terminate':
            echo displayRequestAsJson($terminateServiceRequest);
            break;
        case 'unsuspend':
            echo displayRequestAsJson($unsuspendServiceRequest);
            break;
        case 'package_change':
            echo displayRequestAsJson($packageChangeRequest);
            break;
        case 'invoice_paid':
            echo displayRequestAsJson($invoicePaymentRequest);
            break;
        case 'update_usage':
            echo displayRequestAsJson($usageUpdateRequest);
            break;
        case 'check_balance':
            echo displayRequestAsJson($balanceCheckRequest);
            break;
        case 'check_service':
            echo displayRequestAsJson($serviceStatusRequest);
            break;
        case 'add_credit':
            echo displayRequestAsJson($addCreditRequest);
            break;
        case 'product_add':
            echo displayRequestAsJson($productAddRequest);
            break;
        case 'product_update':
            echo displayRequestAsJson($productUpdateRequest);
            break;
        case 'product_delete':
            echo displayRequestAsJson($productDeleteRequest);
            break;
        case 'balance_add':
            echo displayRequestAsJson($balanceAddRequest);
            break;
        case 'balance_update':
            echo displayRequestAsJson($balanceUpdateRequest);
            break;
        case 'balance_delete':
            echo displayRequestAsJson($balanceDeleteRequest);
            break;
        default:
            // Display all requests
            $allRequests = [
                'create_service' => $createServiceRequest,
                'suspend_service' => $suspendServiceRequest,
                'terminate_service' => $terminateServiceRequest,
                'unsuspend_service' => $unsuspendServiceRequest,
                'package_change' => $packageChangeRequest,
                'invoice_paid' => $invoicePaymentRequest,
                'update_usage' => $usageUpdateRequest,
                'check_balance' => $balanceCheckRequest,
                'check_service' => $serviceStatusRequest,
                'add_credit' => $addCreditRequest,
                'product_add' => $productAddRequest,
                'product_update' => $productUpdateRequest,
                'product_delete' => $productDeleteRequest,
                'balance_add' => $balanceAddRequest,
                'balance_update' => $balanceUpdateRequest,
                'balance_delete' => $balanceDeleteRequest
            ];
            echo displayRequestAsJson($allRequests);
    }
    exit;
}

/**
 * Example cURL request to test the API
 */
function sendTestRequest($request, $apiKey) {
    global $apiEndpoint;
    $ch = curl_init($apiEndpoint);
    
    $headers = [
        'Authorization: Bearer ' . $apiKey,
        'Content-Type: application/json'
    ];
    
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($request));
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    
    curl_close($ch);
    
    return [
        'response' => $response,
        'http_code' => $httpCode,
        'error' => $error
    ];
}

// Uncomment to test a specific request
/*
$result = sendTestRequest($createServiceRequest, $sampleApiKey);
echo "Response: " . $result['response'] . "\n";
echo "HTTP Code: " . $result['http_code'] . "\n";
if ($result['error']) {
    echo "Error: " . $result['error'] . "\n";
}
*/