<?php
/**
 * API Request Samples
 * 
 * This file contains sample JSON payloads for all API requests that can be sent to the API endpoint.
 * Use these samples for documentation and testing purposes.
 */

// Load configuration from external file
$config = require_once(__DIR__ . '/api_config.php');

// Get parameters from config
$sampleApiKey = $config['api_key'];
$apiEndpoint = $config['api_endpoint'];
$sampleServiceId = $config['service_id'];
$sampleProductId = $config['product_id'];
$sampleClientId = $config['client_id'];
$sampleDomain = $config['domain'];
$sampleServerIp = $config['server_ip'];
$sampleInvoiceId = $config['invoice_id'];
$sampleAmount = $config['amount'];
$sampleBalance = $config['balance'];

// API endpoint URL
$apiUrl = "https://your-whmcs-domain.com/modules/addons/AddonSeller/api.php";

/**
 * 1. Add Product
 * Add a new product to the system
 */
$addProductRequest = [
    'security_key' => $sampleApiKey,
    'action' => 'add_product',
    'product_id' => $sampleProductId,
    'minimum_balance' => 1000,
    'service_amount' => 100,
    'traffic_amount' => 50,
    'free_traffic' => 10,
    'traffic_type' => 'both',
    'prtype' => 'virtualizor',
    'queue_api' => true
];

/**
 * 2. Update Product
 * Update an existing product
 */
$updateProductRequest = [
    'security_key' => $sampleApiKey,
    'action' => 'update_product',
    'id' => 5, // Product ID in the database
    'product_id' => $sampleProductId,
    'minimum_balance' => 2000,
    'service_amount' => 150,
    'traffic_amount' => 75,
    'free_traffic' => 15,
    'traffic_type' => 'both',
    'prtype' => 'virtualizor',
    'queue_api' => true
];

/**
 * 3. Delete Product
 * Delete an existing product
 */
$deleteProductRequest = [
    'security_key' => $sampleApiKey,
    'action' => 'delete_product',
    'id' => 5, // Product ID in the database
    'queue_api' => true
];

/**
 * 4. Add Balance
 * Add a new balance for a user
 */
$addBalanceRequest = [
    'security_key' => $sampleApiKey,
    'action' => 'add_balance',
    'userid' => $sampleClientId,
    'balance' => 50000,
    'queue_api' => true
];

/**
 * 5. Update Balance
 * Update an existing balance
 */
$updateBalanceRequest = [
    'security_key' => $sampleApiKey,
    'action' => 'update_balance',
    'userid' => $sampleClientId,
    'balance' => 75000,
    'queue_api' => true
];

/**
 * 6. Delete Balance
 * Delete a user's balance
 */
$deleteBalanceRequest = [
    'security_key' => $sampleApiKey,
    'action' => 'delete_balance',
    'userid' => $sampleClientId,
    'queue_api' => true
];

/**
 * 7. Add Service
 * Add a new service to the system
 */
$addServiceRequest = [
    'security_key' => $sampleApiKey,
    'action' => 'add_service',
    'service_id' => $sampleServiceId,
    'status' => 'Active',
    'last_traffic' => 0,
    'start_time' => date('Y-m-d H:i:s'),
    'last_calculated' => date('Y-m-d H:i:s'),
    'queue_api' => true
];

/**
 * 8. Update Service
 * Update an existing service
 */
$updateServiceRequest = [
    'security_key' => $sampleApiKey,
    'action' => 'update_service',
    'service_id' => $sampleServiceId,
    'status' => 'Active',
    'last_traffic' => 5.2,
    'last_calculated' => date('Y-m-d H:i:s'),
    'queue_api' => true
];

/**
 * 9. Delete Service
 * Delete an existing service
 */
$deleteServiceRequest = [
    'security_key' => $sampleApiKey,
    'action' => 'delete_service',
    'service_id' => $sampleServiceId,
    'queue_api' => true
];

/**
 * 10. Get Service
 * Get details about a service
 */
$getServiceRequest = [
    'security_key' => $sampleApiKey,
    'action' => 'get_service',
    'service_id' => $sampleServiceId
];

/**
 * 11. Suspend Service
 * Suspend a single service
 */
$suspendServiceRequest = [
    'security_key' => $sampleApiKey,
    'action' => 'suspend_service',
    'service_id' => $sampleServiceId,
    'suspend_reason' => 'Overdue payment',
    'queue_api' => true
];

/**
 * 12. Unsuspend Service
 * Unsuspend a single service
 */
$unsuspendServiceRequest = [
    'security_key' => $sampleApiKey,
    'action' => 'unsuspend_service',
    'service_id' => $sampleServiceId,
    'queue_api' => true
];

/**
 * 13. Suspend User Services
 * Suspend all services for a user
 */
$suspendUserServicesRequest = [
    'security_key' => $sampleApiKey,
    'action' => 'suspend_user_services',
    'userid' => $sampleClientId,
    'suspend_reason' => 'Account suspension',
    'queue_api' => true
];

/**
 * 14. Unsuspend User Services
 * Unsuspend all services for a user
 */
$unsuspendUserServicesRequest = [
    'security_key' => $sampleApiKey,
    'action' => 'unsuspend_user_services',
    'userid' => $sampleClientId,
    'queue_api' => true
];

/**
 * 15. Process Queue
 * Start the queue processor
 */
$processQueueRequest = [
    'security_key' => $sampleApiKey,
    'action' => 'process_queue'
];

// Function to display the request as formatted JSON
function displayRequestAsJson($request) {
    return json_encode($request, JSON_PRETTY_PRINT);
}

// Function to send a test request
function sendTestRequest($request, $apiUrl) {
    $ch = curl_init($apiUrl);
    
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($request));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json'
    ]);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    
    curl_close($ch);
    
    return [
        'response' => $response,
        'http_code' => $httpCode,
        'error' => $error
    ];
}

// Example of how to use this file for testing
if (isset($_GET['test'])) {
    header('Content-Type: application/json');
    
    switch ($_GET['test']) {
        case 'add_product':
            echo displayRequestAsJson($addProductRequest);
            break;
        case 'update_product':
            echo displayRequestAsJson($updateProductRequest);
            break;
        case 'delete_product':
            echo displayRequestAsJson($deleteProductRequest);
            break;
        case 'add_balance':
            echo displayRequestAsJson($addBalanceRequest);
            break;
        case 'update_balance':
            echo displayRequestAsJson($updateBalanceRequest);
            break;
        case 'delete_balance':
            echo displayRequestAsJson($deleteBalanceRequest);
            break;
        case 'add_service':
            echo displayRequestAsJson($addServiceRequest);
            break;
        case 'update_service':
            echo displayRequestAsJson($updateServiceRequest);
            break;
        case 'delete_service':
            echo displayRequestAsJson($deleteServiceRequest);
            break;
        case 'get_service':
            echo displayRequestAsJson($getServiceRequest);
            break;
        case 'suspend_service':
            echo displayRequestAsJson($suspendServiceRequest);
            break;
        case 'unsuspend_service':
            echo displayRequestAsJson($unsuspendServiceRequest);
            break;
        case 'suspend_user_services':
            echo displayRequestAsJson($suspendUserServicesRequest);
            break;
        case 'unsuspend_user_services':
            echo displayRequestAsJson($unsuspendUserServicesRequest);
            break;
        case 'process_queue':
            echo displayRequestAsJson($processQueueRequest);
            break;
        default:
            // Display all requests
            $allRequests = [
                'add_product' => $addProductRequest,
                'update_product' => $updateProductRequest,
                'delete_product' => $deleteProductRequest,
                'add_balance' => $addBalanceRequest,
                'update_balance' => $updateBalanceRequest,
                'delete_balance' => $deleteBalanceRequest,
                'add_service' => $addServiceRequest,
                'update_service' => $updateServiceRequest,
                'delete_service' => $deleteServiceRequest,
                'get_service' => $getServiceRequest,
                'suspend_service' => $suspendServiceRequest,
                'unsuspend_service' => $unsuspendServiceRequest,
                'suspend_user_services' => $suspendUserServicesRequest,
                'unsuspend_user_services' => $unsuspendUserServicesRequest,
                'process_queue' => $processQueueRequest
            ];
            echo displayRequestAsJson($allRequests);
    }
    exit;
}

// Example usage:
/*
// To test a specific request:
$result = sendTestRequest($suspendServiceRequest, $apiUrl);
echo "Response: " . $result['response'] . "\n";
echo "HTTP Code: " . $result['http_code'] . "\n";
if ($result['error']) {
    echo "Error: " . $result['error'] . "\n";
}
*/