<?php
/**
 * External API Endpoint
 * 
 * This file handles incoming API requests from external websites
 */

// Initialize WHMCS
require_once(__DIR__ . '/../../../init.php');
require_once(__DIR__ . '/hooks.php');

use WHMCS\Database\Capsule;

// Set headers for JSON response
header('Content-Type: application/json');

// Get API security key from database
$apiKey = Capsule::table('mod_HourlyAddonSeller_apikey')
    ->value('api_key');

if (!$apiKey) {
    sendResponse(500, 'error', 'API key not configured in database');
    exit;
}

define('API_SECURITY_KEY', $apiKey);

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

// Only allow POST requests
if ($method !== 'POST') {
    sendResponse(405, 'Method Not Allowed', 'Only POST requests are accepted');
    exit;
}

// Get JSON input
$input = file_get_contents('php://input');
$data = json_decode($input, true);

// Validate input
if (!$data || !isset($data['action'])) {
    sendResponse(400, 'Bad Request', 'Invalid JSON data or missing action');
    exit;
}

// Validate security key
if (!isset($data['security_key']) || $data['security_key'] !== API_SECURITY_KEY) {
    sendResponse(401, 'Unauthorized', 'Invalid security key');
    exit;
}

// Process the request based on action
try {
    switch ($data['action']) {
        case 'add_product':
            $result = handleAddProduct($data);
            break;
            
        case 'update_product':
            $result = handleUpdateProduct($data);
            break;
            
        case 'delete_product':
            $result = handleDeleteProduct($data);
            break;
            
        case 'add_balance':
            $result = handleAddBalance($data);
            break;
            
        case 'update_balance':
            $result = handleUpdateBalance($data);
            break;
            
        case 'delete_balance':
            $result = handleDeleteBalance($data);
            break;
            
        case 'add_service':
            $result = handleAddService($data);
            break;
            
        case 'update_service':
            $result = handleUpdateService($data);
            break;
            
        case 'delete_service':
            $result = handleDeleteService($data);
            break;
            
        case 'get_service':
            $result = handleGetService($data);
            break;
            
        case 'suspend_service':
            $result = handleSuspendService($data);
            break;
            
        case 'unsuspend_service':
            $result = handleUnsuspendService($data);
            break;
            
        case 'suspend_user_services':
            $result = handleSuspendUserServices($data);
            break;
            
        case 'unsuspend_user_services':
            $result = handleUnsuspendUserServices($data);
            break;
            
        case 'process_queue':
            $result = handleProcessQueue($data);
            break;
            
        default:
            sendResponse(400, 'Bad Request', 'Unknown action: ' . $data['action']);
            exit;
    }
    
    sendResponse(200, 'success', 'Action processed successfully', $result);
} catch (Exception $e) {
    sendResponse(500, 'error', $e->getMessage());
}

/**
 * Send JSON response
 * 
 * @param int $httpCode HTTP status code
 * @param string $status Status (success/error)
 * @param string $message Response message
 * @param array $data Additional data
 */
function sendResponse($httpCode, $status, $message, $data = [])
{
    http_response_code($httpCode);
    
    $response = [
        'status' => $status,
        'message' => $message
    ];
    
    if (!empty($data)) {
        $response['data'] = $data;
    }
    
    echo json_encode($response);
    exit;
}

/**
 * Handle add product request
 * 
 * @param array $data Request data
 * @return array Result data
 */
function handleAddProduct($data)
{
    // Validate required fields
    $requiredFields = ['product_id', 'minimum_balance', 'service_amount', 'traffic_amount', 'free_traffic', 'traffic_type', 'prtype'];
    foreach ($requiredFields as $field) {
        if (!isset($data[$field])) {
            throw new Exception("Missing required field: $field");
        }
    }
    
    // Check if product already exists
    $exists = Capsule::table('mod_HourlyAddonSeller_products')
        ->where('product_id', '=', (int) $data['product_id'])
        ->exists();
        
    if ($exists) {
        throw new Exception("Product with this WHMCS product ID already exists");
    }
    
    // Insert into database
    $productId = Capsule::table('mod_HourlyAddonSeller_products')
        ->insertGetId([
            'product_id' => (int) $data['product_id'],
            'minimum_balance' => (int) $data['minimum_balance'],
            'service_amount' => (int) $data['service_amount'],
            'traffic_amount' => (int) $data['traffic_amount'],
            'free_traffic' => (int) $data['free_traffic'],
            'traffic_type' => $data['traffic_type'],
            'prtype' => $data['prtype'],
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s')
        ]);
    
    // Add to API queue if needed
    if (isset($data['queue_api']) && $data['queue_api']) {
        $apiParams = [
            'action' => 'product_add',
            'product_id' => $productId,
            'whmcs_product_id' => (int) $data['product_id'],
            'minimum_balance' => (int) $data['minimum_balance'],
            'service_amount' => (int) $data['service_amount'],
            'traffic_amount' => (int) $data['traffic_amount'],
            'free_traffic' => (int) $data['free_traffic'],
            'traffic_type' => $data['traffic_type'],
            'prtype' => $data['prtype']
        ];
        
        $queueId = addToApiQueue($apiParams);
        return ['product_id' => $productId, 'queue_id' => $queueId];
    }
    
    return ['product_id' => $productId];
}

/**
 * Handle update product request
 * 
 * @param array $data Request data
 * @return array Result data
 */
function handleUpdateProduct($data)
{
    // Validate required fields
    if (!isset($data['id'])) {
        throw new Exception("Missing required field: id");
    }
    
    // Check if product exists
    $product = Capsule::table('mod_HourlyAddonSeller_products')
        ->where('id', '=', (int) $data['id'])
        ->first();
        
    if (!$product) {
        throw new Exception("Product not found");
    }
    
    // Prepare update data
    $updateData = [
        'updated_at' => date('Y-m-d H:i:s')
    ];
    
    // Add fields to update data if provided
    $fields = ['product_id', 'minimum_balance', 'service_amount', 'traffic_amount', 'free_traffic', 'traffic_type', 'prtype'];
    foreach ($fields as $field) {
        if (isset($data[$field])) {
            $updateData[$field] = $data[$field];
        }
    }
    
    // Update database
    Capsule::table('mod_HourlyAddonSeller_products')
        ->where('id', '=', (int) $data['id'])
        ->update($updateData);
    
    // Add to API queue if needed
    if (isset($data['queue_api']) && $data['queue_api']) {
        $apiParams = [
            'action' => 'product_update',
            'product_id' => (int) $data['id'],
            'whmcs_product_id' => isset($data['product_id']) ? (int) $data['product_id'] : $product->product_id,
            'minimum_balance' => isset($data['minimum_balance']) ? (int) $data['minimum_balance'] : $product->minimum_balance,
            'service_amount' => isset($data['service_amount']) ? (int) $data['service_amount'] : $product->service_amount,
            'traffic_amount' => isset($data['traffic_amount']) ? (int) $data['traffic_amount'] : $product->traffic_amount,
            'free_traffic' => isset($data['free_traffic']) ? (int) $data['free_traffic'] : $product->free_traffic,
            'traffic_type' => isset($data['traffic_type']) ? $data['traffic_type'] : $product->traffic_type,
            'prtype' => isset($data['prtype']) ? $data['prtype'] : $product->prtype
        ];
        
        $queueId = addToApiQueue($apiParams);
        return ['product_id' => (int) $data['id'], 'queue_id' => $queueId];
    }
    
    return ['product_id' => (int) $data['id']];
}

/**
 * Handle delete product request
 * 
 * @param array $data Request data
 * @return array Result data
 */
function handleDeleteProduct($data)
{
    // Validate required fields
    if (!isset($data['id'])) {
        throw new Exception("Missing required field: id");
    }
    
    // Check if product exists
    $product = Capsule::table('mod_HourlyAddonSeller_products')
        ->where('id', '=', (int) $data['id'])
        ->first();
        
    if (!$product) {
        throw new Exception("Product not found");
    }
    
    // Delete from database
    Capsule::table('mod_HourlyAddonSeller_products')
        ->where('id', '=', (int) $data['id'])
        ->delete();
    
    // Add to API queue if needed
    if (isset($data['queue_api']) && $data['queue_api']) {
        $apiParams = [
            'action' => 'product_delete',
            'product_id' => (int) $data['id'],
            'whmcs_product_id' => $product->product_id
        ];
        
        $queueId = addToApiQueue($apiParams);
        return ['product_id' => (int) $data['id'], 'queue_id' => $queueId];
    }
    
    return ['product_id' => (int) $data['id']];
}

/**
 * Handle add balance request
 * 
 * @param array $data Request data
 * @return array Result data
 */
function handleAddBalance($data)
{
    // Validate required fields
    if (!isset($data['userid']) || !isset($data['balance'])) {
        throw new Exception("Missing required fields: userid or balance");
    }
    
    // Check if user exists
    $userExists = Capsule::table('tblclients')
        ->where('id', '=', (int) $data['userid'])
        ->exists();
        
    if (!$userExists) {
        throw new Exception("User not found");
    }
    
    // Check if balance record already exists
    $exists = Capsule::table('mod_HourlyAddonSeller_balance')
        ->where('userid', '=', (int) $data['userid'])
        ->exists();
    
    if ($exists) {
        // Update existing record
        Capsule::table('mod_HourlyAddonSeller_balance')
            ->where('userid', '=', (int) $data['userid'])
            ->update([
                'balance' => (int) $data['balance'],
                'updated_at' => date('Y-m-d H:i:s')
            ]);
            
        $action = 'balance_update';
    } else {
        // Insert new record
        Capsule::table('mod_HourlyAddonSeller_balance')
            ->insert([
                'userid' => (int) $data['userid'],
                'balance' => (int) $data['balance'],
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            ]);
            
        $action = 'balance_add';
    }
    
    // Add to API queue if needed
    if (isset($data['queue_api']) && $data['queue_api']) {
        $apiParams = [
            'action' => $action,
            'userid' => (int) $data['userid'],
            'balance' => (int) $data['balance']
        ];
        
        $queueId = addToApiQueue($apiParams);
        return ['userid' => (int) $data['userid'], 'queue_id' => $queueId];
    }
    
    return ['userid' => (int) $data['userid']];
}

/**
 * Handle update balance request
 * 
 * @param array $data Request data
 * @return array Result data
 */
function handleUpdateBalance($data)
{
    // Validate required fields
    if (!isset($data['userid']) || !isset($data['balance'])) {
        throw new Exception("Missing required fields: userid or balance");
    }
    
    // Check if balance record exists
    $exists = Capsule::table('mod_HourlyAddonSeller_balance')
        ->where('userid', '=', (int) $data['userid'])
        ->exists();
        
    if (!$exists) {
        throw new Exception("Balance record not found");
    }
    
    // Update database
    Capsule::table('mod_HourlyAddonSeller_balance')
        ->where('userid', '=', (int) $data['userid'])
        ->update([
            'balance' => (int) $data['balance'],
            'updated_at' => date('Y-m-d H:i:s')
        ]);
    
    // Add to API queue if needed
    if (isset($data['queue_api']) && $data['queue_api']) {
        $apiParams = [
            'action' => 'balance_update',
            'userid' => (int) $data['userid'],
            'balance' => (int) $data['balance']
        ];
        
        $queueId = addToApiQueue($apiParams);
        return ['userid' => (int) $data['userid'], 'queue_id' => $queueId];
    }
    
    return ['userid' => (int) $data['userid']];
}

/**
 * Handle delete balance request
 * 
 * @param array $data Request data
 * @return array Result data
 */
function handleDeleteBalance($data)
{
    // Validate required fields
    if (!isset($data['userid'])) {
        throw new Exception("Missing required field: userid");
    }
    
    // Check if balance record exists
    $balance = Capsule::table('mod_HourlyAddonSeller_balance')
        ->where('userid', '=', (int) $data['userid'])
        ->first();
        
    if (!$balance) {
        throw new Exception("Balance record not found");
    }
    
    // Delete from database
    Capsule::table('mod_HourlyAddonSeller_balance')
        ->where('userid', '=', (int) $data['userid'])
        ->delete();
    
    // Add to API queue if needed
    if (isset($data['queue_api']) && $data['queue_api']) {
        $apiParams = [
            'action' => 'balance_delete',
            'userid' => (int) $data['userid'],
            'previous_balance' => $balance->balance
        ];
        
        $queueId = addToApiQueue($apiParams);
        return ['userid' => (int) $data['userid'], 'queue_id' => $queueId];
    }
    
    return ['userid' => (int) $data['userid']];
}

/**
 * Handle add service request
 * 
 * @param array $data Request data
 * @return array Result data
 */
function handleAddService($data)
{
    // Validate required fields
    $requiredFields = ['service_id', 'status'];
    foreach ($requiredFields as $field) {
        if (!isset($data[$field])) {
            throw new Exception("Missing required field: $field");
        }
    }
    
    // Check if service already exists
    $exists = Capsule::table('mod_HourlyAddonSeller_services')
        ->where('service_id', '=', (int) $data['service_id'])
        ->exists();
        
    if ($exists) {
        throw new Exception("Service with this ID already exists");
    }
    
    // Check if service exists in WHMCS
    $whmcsService = Capsule::table('tblhosting')
        ->where('id', '=', (int) $data['service_id'])
        ->first();
        
    if (!$whmcsService) {
        throw new Exception("Service not found in WHMCS");
    }
    
    // Prepare insert data
    $insertData = [
        'service_id' => (int) $data['service_id'],
        'status' => $data['status'],
        'created_at' => date('Y-m-d H:i:s'),
        'updated_at' => date('Y-m-d H:i:s')
    ];
    
    // Add optional fields if provided
    $optionalFields = ['last_traffic', 'start_time', 'last_calculated'];
    foreach ($optionalFields as $field) {
        if (isset($data[$field])) {
            $insertData[$field] = $data[$field];
        }
    }
    
    // Set default values for optional fields if not provided
    if (!isset($insertData['last_traffic'])) {
        $insertData['last_traffic'] = 0;
    }
    
    if (!isset($insertData['start_time'])) {
        $insertData['start_time'] = date('Y-m-d H:i:s');
    }
    
    if (!isset($insertData['last_calculated'])) {
        $insertData['last_calculated'] = date('Y-m-d H:i:s');
    }
    
    // Insert into database
    $id = Capsule::table('mod_HourlyAddonSeller_services')
        ->insertGetId($insertData);
    
    // Add to API queue if needed
    if (isset($data['queue_api']) && $data['queue_api']) {
        $apiParams = [
            'action' => 'create',
            'service_id' => (int) $data['service_id'],
            'product_id' => $whmcsService->packageid,
            'client_id' => $whmcsService->userid,
            'domain' => $whmcsService->domain,
            'server_ip' => $whmcsService->dedicatedip,
            'status' => $data['status']
        ];
        
        $queueId = addToApiQueue($apiParams);
        return ['id' => $id, 'service_id' => (int) $data['service_id'], 'queue_id' => $queueId];
    }
    
    return ['id' => $id, 'service_id' => (int) $data['service_id']];
}

/**
 * Handle update service request
 * 
 * @param array $data Request data
 * @return array Result data
 */
function handleUpdateService($data)
{
    // Validate required fields
    if (!isset($data['service_id'])) {
        throw new Exception("Missing required field: service_id");
    }
    
    // Check if service exists
    $service = Capsule::table('mod_HourlyAddonSeller_services')
        ->where('service_id', '=', (int) $data['service_id'])
        ->first();
        
    if (!$service) {
        throw new Exception("Service not found");
    }
    
    // Prepare update data
    $updateData = [
        'updated_at' => date('Y-m-d H:i:s')
    ];
    
    // Add fields to update data if provided
    $fields = ['status', 'last_traffic', 'last_calculated'];
    foreach ($fields as $field) {
        if (isset($data[$field])) {
            $updateData[$field] = $data[$field];
        }
    }
    
    // Update database
    Capsule::table('mod_HourlyAddonSeller_services')
        ->where('service_id', '=', (int) $data['service_id'])
        ->update($updateData);
    
    // Get WHMCS service details
    $whmcsService = Capsule::table('tblhosting')
        ->where('id', '=', (int) $data['service_id'])
        ->first();
    
    // Add to API queue if needed
    if (isset($data['queue_api']) && $data['queue_api']) {
        $apiAction = isset($data['status']) ? strtolower($data['status']) : 'update';
        
        $apiParams = [
            'action' => $apiAction,
            'service_id' => (int) $data['service_id'],
            'product_id' => $whmcsService->packageid,
            'client_id' => $whmcsService->userid,
            'domain' => $whmcsService->domain,
            'server_ip' => $whmcsService->dedicatedip,
            'status' => isset($data['status']) ? $data['status'] : $service->status
        ];
        
        $queueId = addToApiQueue($apiParams);
        return ['service_id' => (int) $data['service_id'], 'queue_id' => $queueId];
    }
    
    return ['service_id' => (int) $data['service_id']];
}

/**
 * Handle delete service request
 * 
 * @param array $data Request data
 * @return array Result data
 */
function handleDeleteService($data)
{
    // Validate required fields
    if (!isset($data['service_id'])) {
        throw new Exception("Missing required field: service_id");
    }
    
    // Check if service exists
    $service = Capsule::table('mod_HourlyAddonSeller_services')
        ->where('service_id', '=', (int) $data['service_id'])
        ->first();
        
    if (!$service) {
        throw new Exception("Service not found");
    }
    
    // Get WHMCS service details before deletion
    $whmcsService = Capsule::table('tblhosting')
        ->where('id', '=', (int) $data['service_id'])
        ->first();
    
    // Delete from database
    Capsule::table('mod_HourlyAddonSeller_services')
        ->where('service_id', '=', (int) $data['service_id'])
        ->delete();
    
    // Add to API queue if needed
    if (isset($data['queue_api']) && $data['queue_api']) {
        $apiParams = [
            'action' => 'terminate',
            'service_id' => (int) $data['service_id'],
            'product_id' => $whmcsService->packageid,
            'client_id' => $whmcsService->userid,
            'domain' => $whmcsService->domain,
            'server_ip' => $whmcsService->dedicatedip,
            'status' => 'Terminated'
        ];
        
        $queueId = addToApiQueue($apiParams);
        return ['service_id' => (int) $data['service_id'], 'queue_id' => $queueId];
    }
    
    return ['service_id' => (int) $data['service_id']];
}

/**
 * Handle get service request
 * 
 * @param array $data Request data
 * @return array Result data
 */
function handleGetService($data)
{
    // Validate required fields
    if (!isset($data['service_id'])) {
        throw new Exception("Missing required field: service_id");
    }
    
    // Get service details
    $service = Capsule::table('mod_HourlyAddonSeller_services')
        ->where('service_id', '=', (int) $data['service_id'])
        ->first();
        
    if (!$service) {
        throw new Exception("Service not found");
    }
    
    // Get WHMCS service details
    $whmcsService = Capsule::table('tblhosting')
        ->where('id', '=', (int) $data['service_id'])
        ->first();
    
    // Get product details
    $product = Capsule::table('mod_HourlyAddonSeller_products')
        ->where('product_id', '=', $whmcsService->packageid)
        ->first();
    
    // Get client balance
    $balance = Capsule::table('mod_HourlyAddonSeller_balance')
        ->where('userid', '=', $whmcsService->userid)
        ->value('balance');
    
    // Prepare response data
    $serviceData = [
        'id' => $service->id,
        'service_id' => $service->service_id,
        'status' => $service->status,
        'last_traffic' => $service->last_traffic,
        'start_time' => $service->start_time,
        'last_calculated' => $service->last_calculated,
        'created_at' => $service->created_at,
        'updated_at' => $service->updated_at,
        'whmcs_service' => [
            'id' => $whmcsService->id,
            'userid' => $whmcsService->userid,
            'packageid' => $whmcsService->packageid,
            'domain' => $whmcsService->domain,
            'dedicatedip' => $whmcsService->dedicatedip,
            'domainstatus' => $whmcsService->domainstatus
        ]
    ];
    
    if ($product) {
        $serviceData['product'] = [
            'id' => $product->id,
            'product_id' => $product->product_id,
            'minimum_balance' => $product->minimum_balance,
            'service_amount' => $product->service_amount,
            'traffic_amount' => $product->traffic_amount,
            'free_traffic' => $product->free_traffic,
            'traffic_type' => $product->traffic_type,
            'prtype' => $product->prtype
        ];
    }
    
    if ($balance !== null) {
        $serviceData['client_balance'] = $balance;
    }
    
    return $serviceData;
}

/**
 * Handle process queue request
 * 
 * @param array $data Request data
 * @return array Result data
 */
function handleProcessQueue($data)
{
    // Start the queue processor in the background
    $queueProcessorPath = __DIR__ . '/process_queue.php';
    $command = "php " . escapeshellarg($queueProcessorPath) . " > /dev/null 2>&1 &";
    exec($command);
    
    return ['status' => 'Queue processor started'];
}



/**
 * Handle suspend service request
 * 
 * @param array $data Request data
 * @return array Result data
 */
function handleSuspendService($data)
{
    // Validate required fields
    if (!isset($data['service_id'])) {
        throw new Exception("Missing required field: service_id");
    }

    // Check if service exists
    $service = Capsule::table('mod_HourlyAddonSeller_services')
        ->where('service_id', '=', (int) $data['service_id'])
        ->first();

    if (!$service) {
        throw new Exception("Service not found");
    }

    // Get WHMCS service details
    $whmcsService = Capsule::table('tblhosting')
        ->where('id', '=', (int) $data['service_id'])
        ->first();

    // Update service status to Suspended
    Capsule::table('mod_HourlyAddonSeller_services')
        ->where('service_id', '=', (int) $data['service_id'])
        ->update([
            'status' => 'Suspended',
            'updated_at' => date('Y-m-d H:i:s')
        ]);

    // Add to API queue if needed
    if (isset($data['queue_api']) && $data['queue_api']) {
        $apiParams = [
            'action' => 'suspend',
            'service_id' => (int) $data['service_id'],
            'product_id' => $whmcsService->packageid,
            'client_id' => $whmcsService->userid,
            'domain' => $whmcsService->domain,
            'server_ip' => $whmcsService->dedicatedip,
            'status' => 'Suspended',
            'suspend_reason' => isset($data['suspend_reason']) ? $data['suspend_reason'] : 'API Request'
        ];

        $queueId = addToApiQueue($apiParams);
        return ['service_id' => (int) $data['service_id'], 'status' => 'Suspended', 'queue_id' => $queueId];
    }

    return ['service_id' => (int) $data['service_id'], 'status' => 'Suspended'];
}

/**
 * Handle unsuspend service request
 * 
 * @param array $data Request data
 * @return array Result data
 */
function handleUnsuspendService($data)
{
    // Validate required fields
    if (!isset($data['service_id'])) {
        throw new Exception("Missing required field: service_id");
    }

    // Check if service exists
    $service = Capsule::table('mod_HourlyAddonSeller_services')
        ->where('service_id', '=', (int) $data['service_id'])
        ->first();

    if (!$service) {
        throw new Exception("Service not found");
    }

    // Get WHMCS service details
    $whmcsService = Capsule::table('tblhosting')
        ->where('id', '=', (int) $data['service_id'])
        ->first();

    // Update service status to Active
    Capsule::table('mod_HourlyAddonSeller_services')
        ->where('service_id', '=', (int) $data['service_id'])
        ->update([
            'status' => 'Active',
            'updated_at' => date('Y-m-d H:i:s')
        ]);

    // Add to API queue if needed
    if (isset($data['queue_api']) && $data['queue_api']) {
        $apiParams = [
            'action' => 'unsuspend',
            'service_id' => (int) $data['service_id'],
            'product_id' => $whmcsService->packageid,
            'client_id' => $whmcsService->userid,
            'domain' => $whmcsService->domain,
            'server_ip' => $whmcsService->dedicatedip,
            'status' => 'Active'
        ];

        $queueId = addToApiQueue($apiParams);
        return ['service_id' => (int) $data['service_id'], 'status' => 'Active', 'queue_id' => $queueId];
    }

    return ['service_id' => (int) $data['service_id'], 'status' => 'Active'];
}

/**
 * Handle suspend all services for a user
 * 
 * @param array $data Request data
 * @return array Result data
 */
function handleSuspendUserServices($data)
{
    // Validate required fields
    if (!isset($data['userid'])) {
        throw new Exception("Missing required field: userid");
    }

    // Check if user exists
    $userExists = Capsule::table('tblclients')
        ->where('id', '=', (int) $data['userid'])
        ->exists();

    if (!$userExists) {
        throw new Exception("User not found");
    }

    // Get all active services for this user
    $services = Capsule::table('tblhosting')
        ->where('userid', '=', (int) $data['userid'])
        ->whereIn('domainstatus', ['Active', 'Suspended'])
        ->get();

    if (count($services) == 0) {
        return ['userid' => (int) $data['userid'], 'services_suspended' => 0];
    }

    $suspendedCount = 0;
    $queueIds = [];

    // Process each service
    foreach ($services as $whmcsService) {
        // Check if service exists in our system
        $service = Capsule::table('mod_HourlyAddonSeller_services')
            ->where('service_id', '=', $whmcsService->id)
            ->first();

        if (!$service) {
            continue; // Skip if not in our system
        }

        // Update service status to Suspended
        Capsule::table('mod_HourlyAddonSeller_services')
            ->where('service_id', '=', $whmcsService->id)
            ->update([
                'status' => 'Suspended',
                'updated_at' => date('Y-m-d H:i:s')
            ]);

        $suspendedCount++;

        // Add to API queue if needed
        if (isset($data['queue_api']) && $data['queue_api']) {
            $apiParams = [
                'action' => 'suspend',
                'service_id' => $whmcsService->id,
                'product_id' => $whmcsService->packageid,
                'client_id' => $whmcsService->userid,
                'domain' => $whmcsService->domain,
                'server_ip' => $whmcsService->dedicatedip,
                'status' => 'Suspended',
                'suspend_reason' => isset($data['suspend_reason']) ? $data['suspend_reason'] : 'User Suspension'
            ];

            $queueIds[] = addToApiQueue($apiParams);
        }
    }

    $result = [
        'userid' => (int) $data['userid'],
        'services_suspended' => $suspendedCount
    ];

    if (isset($data['queue_api']) && $data['queue_api'] && !empty($queueIds)) {
        $result['queue_ids'] = $queueIds;
    }

    return $result;
}

/**
 * Handle unsuspend all services for a user
 * 
 * @param array $data Request data
 * @return array Result data
 */
function handleUnsuspendUserServices($data)
{
    // Validate required fields
    if (!isset($data['userid'])) {
        throw new Exception("Missing required field: userid");
    }

    // Check if user exists
    $userExists = Capsule::table('tblclients')
        ->where('id', '=', (int) $data['userid'])
        ->exists();

    if (!$userExists) {
        throw new Exception("User not found");
    }

    // Get all suspended services for this user
    $services = Capsule::table('tblhosting')
        ->where('userid', '=', (int) $data['userid'])
        ->where('domainstatus', '=', 'Suspended')
        ->get();

    if (count($services) == 0) {
        return ['userid' => (int) $data['userid'], 'services_unsuspended' => 0];
    }

    $unsuspendedCount = 0;
    $queueIds = [];

    // Process each service
    foreach ($services as $whmcsService) {
        // Check if service exists in our system
        $service = Capsule::table('mod_HourlyAddonSeller_services')
            ->where('service_id', '=', $whmcsService->id)
            ->first();

        if (!$service) {
            continue; // Skip if not in our system
        }

        // Update service status to Active
        Capsule::table('mod_HourlyAddonSeller_services')
            ->where('service_id', '=', $whmcsService->id)
            ->update([
                'status' => 'Active',
                'updated_at' => date('Y-m-d H:i:s')
            ]);

        $unsuspendedCount++;

        // Add to API queue if needed
        if (isset($data['queue_api']) && $data['queue_api']) {
            $apiParams = [
                'action' => 'unsuspend',
                'service_id' => $whmcsService->id,
                'product_id' => $whmcsService->packageid,
                'client_id' => $whmcsService->userid,
                'domain' => $whmcsService->domain,
                'server_ip' => $whmcsService->dedicatedip,
                'status' => 'Active'
            ];

            $queueIds[] = addToApiQueue($apiParams);
        }
    }

    $result = [
        'userid' => (int) $data['userid'],
        'services_unsuspended' => $unsuspendedCount
    ];

    if (isset($data['queue_api']) && $data['queue_api'] && !empty($queueIds)) {
        $result['queue_ids'] = $queueIds;
    }

    return $result;
}