<?php

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Simple EDD Software Licensing class for license validation only.
 * No auto-update functionality included.
 * Automatically detects whether to use custom parent or plugin parent based on parent_slug.
 *
 * @version 1.0
 */
class EDD_SL {

	private $api_url;
	private $plugin_slug;
	private $plugin_name;
	private $license_option;
	private $status_option;
	private $item_id;
	private $item_name;
	private $admin_page_slug;
	private $admin_page_title;
	private $parent_slug;
	private $is_custom_parent;

	/**
	 * Class constructor.
	 *
	 * @param array $args Configuration arguments
	 */
	public function __construct( $args = array() ) {
		$defaults = array(
			'api_url'          => 'https://elementorfa.ir/',
			'plugin_slug'      => '',
			'plugin_name'      => '',
			'item_id'          => '',
			'item_name'        => '',
			'admin_page_title' => 'مدیریت لایسنس',
			'parent_slug'      => '', // اگر خالی باشد، از plugin parent استفاده می‌شود
		);

		$args = wp_parse_args( $args, $defaults );

		$this->api_url           = trailingslashit( $args['api_url'] );
		$this->plugin_slug       = $args['plugin_slug'];
		$this->plugin_name       = !empty($args['plugin_name']) ? $args['plugin_name'] : $this->generate_plugin_name($args['plugin_slug']);
		$this->item_id           = $args['item_id'];
		$this->item_name         = !empty($args['item_name']) ? $args['item_name'] : $this->plugin_name;
		$this->admin_page_title  = $args['admin_page_title'];
		$this->parent_slug       = $args['parent_slug'];
		$this->license_option    = $this->plugin_slug . '_license_key';
		$this->status_option     = $this->plugin_slug . '_license_status';
		$this->admin_page_slug   = $this->plugin_slug . '_license';
		
		// تشخیص نوع parent بر اساس وجود parent_slug
		$this->is_custom_parent  = !empty($this->parent_slug);

		$this->init();
	}

	/**
	 * Generate plugin name from slug.
	 */
	private function generate_plugin_name($slug) {
		// تبدیل خط تیره به فاصله و بزرگ کردن حرف اول هر کلمه
		$name = str_replace('-', ' ', $slug);
		$name = ucwords($name);
		return $name;
	}

	/**
	 * Initialize hooks.
	 */
	private function init() {
		add_action( 'admin_menu', array( $this, 'add_license_page' ), 99 );
		add_action( 'admin_init', array( $this, 'register_license_setting' ) );
		add_action( 'admin_init', array( $this, 'activate_license' ) );
		add_action( 'admin_init', array( $this, 'deactivate_license' ) );
		add_action( 'admin_init', array( $this, 'manual_check_license' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_styles' ) );
		add_action( 'admin_notices', array( $this, 'license_notice' ) );
		
		// بررسی وضعیت لایسنس هنگام لود admin
		add_action( 'admin_init', array( $this, 'maybe_check_license_status' ) );
	}

	/**
	 * Add license page to admin menu.
	 */
	public function add_license_page() {
		if ( $this->is_custom_parent ) {
			// استفاده از custom parent
			add_submenu_page(
				$this->parent_slug,
				$this->admin_page_title,
				$this->admin_page_title,
				'manage_options',
				$this->admin_page_slug,
				array( $this, 'license_page' )
			);
		} else {
			// استفاده از plugin parent
			add_plugins_page(
				$this->admin_page_title,
				$this->admin_page_title,
				'manage_options',
				$this->admin_page_slug,
				array( $this, 'license_page' )
			);
		}
	}

	/**
	 * Get admin URL based on parent type.
	 */
	private function get_admin_url() {
		if ( $this->is_custom_parent ) {
			return admin_url( 'admin.php' );
		} else {
			return admin_url( 'plugins.php' );
		}
	}

	/**
	 * Enqueue admin styles.
	 */
	public function admin_styles( $hook ) {
		if ( strpos( $hook, $this->admin_page_slug ) !== false ) {
			wp_enqueue_style( 'edd-sl-admin', plugin_dir_url( __FILE__ ) . 'admin.css', array(), '1.2' );
		}
	}

	/**
	 * Display license page.
	 */
	public function license_page() {
		$license = get_option( $this->license_option );
		$status = get_option( $this->status_option );
		?>
		<div class="wrap udm-library-settings">
			<div class="udm-library-header">
				<div class="udm-library-header-main">
					<div class="udm-library-logo">
						<div class="udm-logo-icon">
							<span class="dashicons dashicons-admin-network"></span>
						</div>
					</div>
					<div class="udm-library-header-title">
						<h4><?php echo esc_html( $this->admin_page_title ); ?></h4>
						<p style="color: #6d7882; font-size: 14px; margin: 15px 0 0;">
							در این صفحه می‌توانید لایسنس <?php echo esc_html( $this->plugin_name ); ?> خود را مدیریت کنید.</p>
					</div>
				</div>
			</div>
			
			<div id="udm-library-messages">
				<?php
				// نمایش پیام‌های بررسی وضعیت لایسنس
				if (isset($_GET['message']) && isset($_GET['type'])) {
					$message = sanitize_text_field( rawurldecode($_GET['message']) );
					$type = sanitize_text_field( $_GET['type'] );
					
					if (in_array($type, ['success', 'error', 'info', 'warning'])) {
						$class = 'notice notice-' . $type;
						if ( $type === 'success' ) {
							$class .= ' is-dismissible';
						}
						echo '<div class="' . esc_attr($class) . '"><p>' . esc_html($message) . '</p></div>';
					}
				}
				?>
			</div>
			
			<div class="udm-library-main">
				<div class="udm-library-content">
					<div class="udm-library-card">
						<div class="udm-library-card-header">
							<h3>وضعیت سایت</h3>
						</div>
						<div class="udm-library-card-body">
							<div class="udm-library-settings-row">
								<div class="udm-library-settings-icon">
									<span class="dashicons dashicons-admin-site-alt3"></span>
								</div>
								<div class="udm-library-settings-content">
									<div class="udm-library-settings-title">دامنه فعلی</div>
									<div class="udm-library-settings-description"><?php echo esc_html(home_url()); ?></div>
								</div>
							</div>
							
							<div class="udm-library-settings-row">
								<div class="udm-library-settings-icon">
									<span class="dashicons dashicons-shield"></span>
								</div>
								<div class="udm-library-settings-content">
									<div class="udm-library-settings-title">وضعیت لایسنس</div>
									<div class="udm-library-settings-description">
										<?php
										if ( 'valid' === $status ) {
											echo '<span class="status-active">🟢 فعال</span>';
											$message = 'لایسنس شما فعال است';
										} elseif ( !empty($license) ) {
											echo '<span class="status-error">🟡 غیر فعال</span>';
											$message = 'لایسنس وارد شده اما فعال نیست';
										} else {
											echo '<span class="status-inactive">⚫ غیر فعال</span>';
											$message = 'لایسنس وارد نشده است';
										}
										?>
										<br><small><?php echo esc_html($message); ?></small>
									</div>
								</div>
							</div>
						</div>
					</div>

					<div class="udm-library-card">
						<div class="udm-library-card-header">
							<h3>فعال‌سازی لایسنس</h3>
						</div>
						<div class="udm-library-card-body">
							<form method="post" action="options.php" id="udm-library-license-form">
								<?php 
								settings_fields( $this->plugin_slug . '_license' );
								wp_nonce_field( $this->plugin_slug . '_nonce', $this->plugin_slug . '_nonce' ); 
								?>
								
								<div class="udm-library-settings-row">
									<div class="udm-library-settings-icon">
										<span class="dashicons dashicons-admin-network"></span>
									</div>
									<div class="udm-library-settings-content">
										<div class="udm-library-settings-title">کلید لایسنس</div>
										<div class="udm-library-settings-description">
											<?php if ( 'valid' === $status && !empty($license) ): ?>
												<input type="text" 
													   id="<?php echo esc_attr( $this->license_option ); ?>" 
													   value="<?php echo esc_attr( substr($license, 0, 4) . str_repeat('*', strlen($license) - 8) . substr($license, -4) ); ?>" 
													   class="regular-text" 
													   readonly 
													   style="background: #f1f1f1; color: #666;" />
												<p class="description">لایسنس شما با موفقیت فعال شده است.</p>
											<?php else: ?>
												<input type="text" 
													   id="<?php echo esc_attr( $this->license_option ); ?>" 
													   name="<?php echo esc_attr( $this->license_option ); ?>" 
													   value="<?php echo esc_attr( $license ); ?>" 
													   class="regular-text" 
													   placeholder="لایسنس خود را وارد کنید"
													   required />
												<p class="description">کلید لایسنس خود را در این قسمت وارد کنید</p>
											<?php endif; ?>
										</div>
									</div>
								</div>
							</div>
							
							<div class="udm-library-submit">
								<?php if ( 'valid' === $status && !empty($license) ): ?>
					</form>
					<form method="post" action="<?php echo esc_url( $this->get_admin_url() ); ?>" id="deactivate-form">
						<?php wp_nonce_field( $this->plugin_slug . '_nonce', $this->plugin_slug . '_nonce' ); ?>
						<button type="submit" name="<?php echo esc_attr( $this->plugin_slug ); ?>_license_deactivate" id="deactivate-license" class="button udm-library-button-danger" onclick="return confirm('آیا مطمئن هستید که می‌خواهید لایسنس را غیر فعال کنید؟')">غیر فعال کردن لایسنس</button>
					</form>
								<?php else: ?>
									<button type="submit" name="<?php echo esc_attr( $this->plugin_slug ); ?>_license_activate" class="button button-primary udm-library-button-primary">فعال‌سازی لایسنس</button>
									</form>
								<?php endif; ?>
							</div>
					</div>
				</div>
				
				<div class="udm-library-sidebar">
					<div class="udm-library-premium-ad">
						<h5>نیاز به پشتیبانی دارید؟</h5>
						<p>برای دریافت پشتیبانی و راهنمایی بیشتر، با تیم ما در ارتباط باشید.</p>
						<a href="#" target="_blank" class="button button-secondary">تماس با پشتیبانی</a>
					</div>
					
					<?php if ( 'valid' === $status && !empty($license) ): ?>
			<div class="udm-license-status-check">
				<h5>بررسی وضعیت لایسنس</h5>
				<p>آخرین بررسی: <?php 
					$last_check = $this->get_last_check_time();
					if ( $last_check ) {
						// نمایش زمان بر اساس ساعت محلی کاربر
						$local_time = $last_check + ( get_option( 'gmt_offset' ) * HOUR_IN_SECONDS );
						echo esc_html( date_i18n( 'Y/m/d H:i', $local_time ) );
					} else {
						echo 'هرگز';
					}
				?></p>
				<form method="post" action="<?php echo esc_url( $this->get_admin_url() ); ?>" style="margin: 0;">
					<?php wp_nonce_field( $this->plugin_slug . '_nonce', $this->plugin_slug . '_nonce' ); ?>
					<button type="submit" name="<?php echo esc_attr( $this->plugin_slug ); ?>_check_license" class="button button-secondary" style="width: 100%;">بررسی وضعیت از سرور</button>
				</form>
			</div>
			<?php endif; ?>
			
			<div class="udm-library-about">
				<h5>درباره سیستم لایسنس</h5>
				<p>این سیستم به منظور فعال‌سازی و مدیریت لایسنس افزونه‌های خریداری شده طراحی شده است.</p>
				<ul>
					<li>مدیریت لایسنس</li>
					<li>کنترل دسترسی دامنه</li>
					<li>بررسی خودکار وضعیت</li>
				</ul>
			</div>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Register license setting.
	 */
	public function register_license_setting() {
		register_setting( $this->plugin_slug . '_license', $this->license_option, array( $this, 'sanitize_license' ) );
	}

	/**
	 * Sanitize license key.
	 */
	public function sanitize_license( $new ) {
		$old = get_option( $this->license_option );
		if ( $old && $old !== $new ) {
			delete_option( $this->status_option );
		}
		return sanitize_text_field( $new );
	}

	/**
	 * Activate license.
	 */
	public function activate_license() {
		if ( ! isset( $_POST[ $this->plugin_slug . '_license_activate' ] ) ) {
			return;
		}

		if ( ! check_admin_referer( $this->plugin_slug . '_nonce', $this->plugin_slug . '_nonce' ) ) {
			return;
		}

		$license = trim( get_option( $this->license_option ) );
		if ( ! $license ) {
			$license = ! empty( $_POST[ $this->license_option ] ) ? sanitize_text_field( $_POST[ $this->license_option ] ) : '';
		}
		if ( ! $license ) {
			return;
		}

		$api_params = array(
			'edd_action'  => 'activate_license',
			'license'     => $license,
			'item_id'     => $this->item_id,
			'item_name'   => rawurlencode( $this->item_name ),
			'url'         => home_url(),
			'environment' => function_exists( 'wp_get_environment_type' ) ? wp_get_environment_type() : 'production',
		);

		$response = wp_remote_post(
			$this->api_url,
			array(
				'timeout'   => 15,
				'sslverify' => false,
				'body'      => $api_params,
			)
		);

		if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) {
			$message = is_wp_error( $response ) ? $response->get_error_message() : __( 'خطایی رخ داده است، لطفاً دوباره تلاش کنید.' );
		} else {
			$license_data = json_decode( wp_remote_retrieve_body( $response ) );

			if ( false === $license_data->success ) {
				switch ( $license_data->error ) {
					case 'expired':
						$message = 'لایسنس شما منقضی شده است.';
						break;
					case 'disabled':
					case 'revoked':
						$message = 'لایسنس شما غیر فعال شده است.';
						break;
					case 'missing':
						$message = 'لایسنس نامعتبر است.';
						break;
					case 'invalid':
					case 'site_inactive':
						$message = 'لایسنس شما برای این دامنه فعال نیست.';
						break;
					case 'item_name_mismatch':
						$message = 'این لایسنس برای ' . $this->item_name . ' معتبر نیست.';
						break;
					case 'no_activations_left':
						$message = 'تعداد فعال‌سازی لایسنس شما به حد مجاز رسیده است.';
						break;
					default:
						$message = 'خطایی رخ داده است، لطفاً دوباره تلاش کنید.';
						break;
				}
			}
		}

		if ( ! empty( $message ) ) {
			$redirect = add_query_arg(
				array(
					'page'    => $this->admin_page_slug,
					'type'    => 'error',
					'message' => rawurlencode( $message ),
				),
				$this->get_admin_url()
			);
			wp_safe_redirect( $redirect );
			exit();
		}

		if ( 'valid' === $license_data->license ) {
			update_option( $this->license_option, $license );
			update_option( $this->status_option, $license_data->license );
			$redirect = add_query_arg(
				array(
					'page'    => $this->admin_page_slug,
					'type'    => 'success',
					'message' => rawurlencode( 'لایسنس با موفقیت فعال شد.' ),
				),
				$this->get_admin_url()
			);
		} else {
			update_option( $this->status_option, $license_data->license );
			$redirect = add_query_arg(
				array(
					'page'    => $this->admin_page_slug,
					'type'    => 'error',
					'message' => rawurlencode( 'لایسنس فعال نشد.' ),
				),
				$this->get_admin_url()
			);
		}

		wp_safe_redirect( $redirect );
		exit();
	}

	/**
	 * Manual check license status.
	 */
	public function manual_check_license() {
		if ( ! isset( $_POST[ $this->plugin_slug . '_check_license' ] ) ) {
			return;
		}
		
		if ( ! check_admin_referer( $this->plugin_slug . '_nonce', $this->plugin_slug . '_nonce' ) ) {
			return;
		}
		
		// فورس چک کردن وضعیت لایسنس
		$this->force_check_license_status();
		
		$status = get_option( $this->status_option );
		$message = '';
		$type = 'info';
		
		switch ( $status ) {
			case 'valid':
				$message = 'لایسنس شما معتبر و فعال است.';
				$type = 'success';
				break;
			case 'expired':
				$message = 'لایسنس شما منقضی شده است.';
				$type = 'error';
				break;
			case 'disabled':
			case 'revoked':
				$message = 'لایسنس شما غیر فعال شده است.';
				$type = 'error';
				break;
			case 'inactive':
			case 'site_inactive':
				$message = 'لایسنس شما برای این دامنه فعال نیست.';
				$type = 'error';
				break;
			case 'invalid':
				$message = 'لایسنس نامعتبر است.';
				$type = 'error';
				break;
			default:
				$message = 'وضعیت لایسنس بررسی شد.';
				$type = 'info';
				break;
		}
		
		$redirect = add_query_arg(
			array(
				'page'    => $this->admin_page_slug,
				'type'    => $type,
				'message' => rawurlencode( $message ),
			),
			$this->get_admin_url()
		);
		
		wp_safe_redirect( $redirect );
		exit();
	}
	
	/**
	 * Deactivate license.
	 */
	public function deactivate_license() {
		if ( ! isset( $_POST[ $this->plugin_slug . '_license_deactivate' ] ) ) {
			return;
		}

		if ( ! check_admin_referer( $this->plugin_slug . '_nonce', $this->plugin_slug . '_nonce' ) ) {
			return;
		}

		$license = trim( get_option( $this->license_option ) );

		$api_params = array(
			'edd_action'  => 'deactivate_license',
			'license'     => $license,
			'item_id'     => $this->item_id,
			'item_name'   => rawurlencode( $this->item_name ),
			'url'         => home_url(),
			'environment' => function_exists( 'wp_get_environment_type' ) ? wp_get_environment_type() : 'production',
		);

		$response = wp_remote_post(
			$this->api_url,
			array(
				'timeout'   => 15,
				'sslverify' => false,
				'body'      => $api_params,
			)
		);

		if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) {
			$message = is_wp_error( $response ) ? $response->get_error_message() : 'خطایی رخ داده است، لطفاً دوباره تلاش کنید.';
			$redirect = add_query_arg(
				array(
					'page'    => $this->admin_page_slug,
					'type'    => 'error',
					'message' => rawurlencode( $message ),
				),
				$this->get_admin_url()
			);
			wp_safe_redirect( $redirect );
			exit();
		}

		$license_data = json_decode( wp_remote_retrieve_body( $response ) );

		if ( 'deactivated' === $license_data->license ) {
			delete_option( $this->status_option );
			delete_option( $this->license_option );
			$message = 'لایسنس با موفقیت غیر فعال شد.';
			$type = 'success';
		} else {
			$message = 'خطا در غیر فعال کردن لایسنس.';
			$type = 'error';
		}

		$redirect = add_query_arg(
			array(
				'page'    => $this->admin_page_slug,
				'type'    => $type,
				'message' => rawurlencode( $message ),
			),
			$this->get_admin_url()
		);

		wp_safe_redirect( $redirect );
		exit();
	}

	/**
	 * Maybe check license status from server.
	 * Checks license status periodically to ensure it's still valid on the server.
	 */
	public function maybe_check_license_status() {
		// فقط اگر لایسنس فعال باشد بررسی کنیم
		if ( 'valid' !== get_option( $this->status_option ) ) {
			return;
		}
		
		// بررسی آخرین زمان چک کردن
		$last_check = get_option( $this->plugin_slug . '_license_last_check', 0 );
		$check_interval = apply_filters( $this->plugin_slug . '_license_check_interval', 24 * HOUR_IN_SECONDS ); // هر 24 ساعت
		
		if ( ( time() - $last_check ) > $check_interval ) {
			$this->check_license_status();
		}
	}
	
	/**
	 * Check license status from server.
	 */
	public function check_license_status() {
		$license = get_option( $this->license_option );
		
		if ( empty( $license ) ) {
			return;
		}
		
		$api_params = array(
			'edd_action'  => 'check_license',
			'license'     => $license,
			'item_id'     => $this->item_id,
			'item_name'   => rawurlencode( $this->item_name ),
			'url'         => home_url(),
			'environment' => function_exists( 'wp_get_environment_type' ) ? wp_get_environment_type() : 'production',
		);
		
		$response = wp_remote_post(
			$this->api_url,
			array(
				'timeout'   => 15,
				'sslverify' => false,
				'body'      => $api_params,
			)
		);
		
		// به‌روزرسانی زمان آخرین بررسی
		update_option( $this->plugin_slug . '_license_last_check', time() );
		
		if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) {
			// در صورت خطا در اتصال، وضعیت فعلی را حفظ می‌کنیم
			return;
		}
		
		$license_data = json_decode( wp_remote_retrieve_body( $response ) );
		
		if ( ! $license_data ) {
			return;
		}
		
		// اگر لایسنس در سرور غیر فعال، منقضی یا حذف شده باشد
		if ( in_array( $license_data->license, array( 'inactive', 'disabled', 'expired', 'revoked', 'invalid', 'site_inactive' ) ) ) {
			// لایسنس محلی را غیر فعال کن
			update_option( $this->status_option, $license_data->license );
			
			// اگر لایسنس کاملاً حذف شده، غیر فعال شده یا برای این دامنه فعال نیست، کلید را هم پاک کن
			if ( in_array( $license_data->license, array( 'invalid', 'disabled', 'revoked', 'inactive', 'site_inactive' ) ) ) {
				delete_option( $this->license_option );
			}
		} else {
			// وضعیت لایسنس را به‌روزرسانی کن
			update_option( $this->status_option, $license_data->license );
		}
	}
	
	/**
	 * Check if license is valid.
	 */
	public function is_license_valid() {
		return 'valid' === get_option( $this->status_option );
	}

	/**
	 * Get license key.
	 */
	public function get_license_key() {
		return get_option( $this->license_option );
	}

	/**
	 * Get license status.
	 */
	public function get_license_status() {
		return get_option( $this->status_option );
	}
	
	/**
	 * Force check license status from server.
	 * This method bypasses the time interval check.
	 */
	public function force_check_license_status() {
		$this->check_license_status();
	}
	
	/**
	 * Get last license check time.
	 */
	public function get_last_check_time() {
		return get_option( $this->plugin_slug . '_license_last_check', 0 );
	}

	/**
	 * Display license notice.
	 */
	public function license_notice() {
		$status = get_option( $this->status_option );
		
		// فقط در صورتی که لایسنس معتبر نباشد نوتیس نمایش داده شود
		if ( 'valid' !== $status ) {
			$admin_url = $this->get_admin_url();
			?>
			<div class="notice notice-error" style="display: flex; align-items: center; padding: 12px 15px; border-radius: 4px; border-right: 4px solid #d63638;">
				<span class="dashicons dashicons-warning" style="font-size: 24px; margin-left: 10px; color: #d63638;"></span>
				<p style="margin: 0; font-size: 14px;">برای فعال‌سازی کامل <?php echo esc_html( $this->plugin_name ); ?> و دسترسی به تمامی امکانات، لطفا کلید لایسنس خود را وارد نمایید.</p>
				<a href="<?php echo esc_url( $admin_url ); ?>?page=<?php echo esc_attr( $this->admin_page_slug ); ?>" class="button button-primary" style="margin-right: 15px; height: 30px; line-height: 28px; padding: 0 12px; font-weight: 500;">فعال‌سازی لایسنس</a>
			</div>
			<?php
		}
	}
}