<?php
/**
 * Plugin Name: WC Product Surveys
 * Description: Adds 3 surveys (longevity, value, diffusion) to WooCommerce products.
 * Version: 1.0
 * Author: Developer
 */

if (!defined('ABSPATH')) exit;

class WC_Product_Surveys {

    public function __construct() {
        register_activation_hook(__FILE__, [$this, 'create_table']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_assets']);
        add_action('wcps_render_surveys', [$this, 'render_surveys']);
        add_action('wp_ajax_wcps_submit', [$this, 'submit_vote']);
        add_action('wp_ajax_nopriv_wcps_submit', [$this, 'submit_vote']);
    }

    // -----------------------------
    // 1) CREATE TABLE
    // -----------------------------
    public function create_table() {
        global $wpdb;
        $table = $wpdb->prefix . "product_surveys";

        $charset = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE $table (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            product_id BIGINT UNSIGNED NOT NULL,
            survey_key VARCHAR(50) NOT NULL,
            rating_value TINYINT NOT NULL,
            user_id BIGINT UNSIGNED NULL,
            device_hash VARCHAR(255) NULL,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP
        ) $charset;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    // -----------------------------
    // 2) LOAD JS + CSS
    // -----------------------------
    public function enqueue_assets() {
        wp_enqueue_style('wcps-style', plugin_dir_url(__FILE__) . 'style.css');
        wp_enqueue_script('wcps-js', plugin_dir_url(__FILE__) . 'script.js', ['jquery'], false, true);

        wp_localize_script('wcps-js', 'WCPS', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'device_hash' => $this->get_device_hash()
        ]);
    }

    // -----------------------------
    // 3) DEVICE HASH
    // -----------------------------
    private function get_device_hash() {
        if (!isset($_COOKIE['wcps_device'])) {
            $hash = wp_hash(uniqid('', true));
            setcookie('wcps_device', $hash, time() + 3600*24*365, "/");
            return $hash;
        }
        return sanitize_text_field($_COOKIE['wcps_device']);
    }

    // -----------------------------
    // 4) RENDER SURVEYS
    // -----------------------------
    public function render_surveys() {
        global $product;
        if (!$product) return;

        $surveys = [
            'longevity' => 'ماندگاری عطر',
            'value'     => 'ارزش نسبت به قیمت',
            'diffusion' => 'پخش بو'
        ];

        echo '<div class="wcps-wrapper"><div class="container"><div class="row">';

        foreach ($surveys as $key => $label) {
            echo '<div class="col-12 col-md-4">';
            $stats = $this->get_stats($product->get_id(), $key);

            echo '<div class="wcps-box">';
            echo "<h3 class='wcps-box_title'>$label</h3>";

            echo '<div class="wcps-row_wrapper">';
                foreach ([5=>'خیلی خوب',4=>'خوب',3=>'متوسط',2=>'ضعیف',1=>'خیلی ضعیف'] as $val=>$txt) {
                    $count = isset($stats[$val]) ? $stats[$val] : 0;
                    echo "<div class='wcps-row'>
                            <span class='text-start'>$txt</span>
                            <div class='wcps-bar'><div style='width:" . ($count*2) . "px'></div></div>
                            <span class='text-end'>$count نظر</span>
                          </div>";
                }
            echo '</div>';

            $options = [
                1 => ['emoji' => '😡', 'label' => 'خیلی ضعیف'],
                2 => ['emoji' => '🙁', 'label' => 'ضعیف'],
                3 => ['emoji' => '😐', 'label' => 'متوسط'],
                4 => ['emoji' => '🙂', 'label' => 'خوب'],
                5 => ['emoji' => '😍', 'label' => 'خیلی خوب'],
            ];

            // گزینه‌های رأی
            echo "<div class='wcps-options_wrapper  d-flex flex-column align-items-start' data-product='{$product->get_id()}' data-key='$key'>";
            echo "<span class='wcps-options_label'>نظر شما؟</span>";
//            foreach ([1,2,3,4,5] as $v) {
//                echo "<span class='wcps-rate' data-value='$v'></span>";
//            }
            echo "<div class='wcps-options'>";
                foreach ($options as $value => $opt) {
                    echo "<span class='wcps-rate d-flex flex-column align-items-center' data-value='$value'>
                        <span class='wcps-emoji'>{$opt['emoji']}</span>
                        <span class='wcps-label'>{$opt['label']}</span>
                      </span>";
                }
            echo '</div>';

            echo "</div>";

            echo '</div>';
            echo '</div>';
        }

        echo '</div></div></div>';
    }

    // -----------------------------
    // 5) GET STATS
    // -----------------------------
    private function get_stats($product_id, $survey_key) {
        global $wpdb;
        $table = $wpdb->prefix . "product_surveys";

        $rows = $wpdb->get_results($wpdb->prepare(
            "SELECT rating_value, COUNT(*) as total 
             FROM $table 
             WHERE product_id=%d AND survey_key=%s 
             GROUP BY rating_value",
            $product_id, $survey_key
        ));

        $stats = [];
        foreach ($rows as $r) {
            $stats[$r->rating_value] = $r->total;
        }
        return $stats;
    }

    // -----------------------------
    // 6) SUBMIT VOTE
    // -----------------------------
    public function submit_vote() {
        global $wpdb;
        $table = $wpdb->prefix . "product_surveys";

        $product_id = intval($_POST['product_id']);
        $survey_key = sanitize_text_field($_POST['survey_key']);
        $rating     = intval($_POST['rating']);
        $device     = sanitize_text_field($_POST['device_hash']);
        $user_id    = get_current_user_id();

        // جلوگیری از رأی تکراری
        $exists = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM $table 
             WHERE product_id=%d AND survey_key=%s 
             AND (user_id=%d OR device_hash=%s)",
            $product_id, $survey_key, $user_id, $device
        ));

        if ($exists) {
            wp_send_json(['status'=>'error','msg'=>'قبلاً رأی داده‌اید']);
        }

        // ذخیره رأی
        $wpdb->insert($table, [
            'product_id'  => $product_id,
            'survey_key'  => $survey_key,
            'rating_value'=> $rating,
            'user_id'     => $user_id ?: null,
            'device_hash' => $device
        ]);

        wp_send_json(['status'=>'success']);
    }
}

new WC_Product_Surveys();
