<?php

namespace Modules\Post\Http\Controllers\Admin;

use Artesaos\SEOTools\Traits\SEOTools;
use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Modules\Category\Entities\Category;
use Modules\Post\Entities\Post;
use Modules\Post\Http\Requests\CreatePost;

class PostController extends Controller
{
    use SEOTools;
    public function __construct()
    {
        $this->middleware('can:manage-posts');
    }
    /**
     * Display a listing of the resource.
     * @return Renderable
     */
    public function index()
    {
        $this->seo()->setTitle('همه پست ها');

        $posts = Post::query();
        if ($key = \request('search')) {
            $posts = $posts->where('title', 'like', "%$key%");
        }
        $posts = $posts->latest()->simplePaginate(15);
        return view('post::backend.all', compact('posts'));
    }


    public function create()
    {
        $this->seo()->setTitle('ایجاد پست جدید');

        $categories = Category::with(['posts', 'childrenRecursive'])
            ->where('category_type' , Post::class)
            ->whereNull('parent_id')
            ->get();
        return view('post::backend.create', ['categories' => $categories]);
    }


    public function store(Request $request)
    {

        try {
            $validData = $request->validate([
                'title' => 'required',
                'slug' => 'unique:posts|nullable',
                'body' => 'required',
                'categories' => 'nullable',
                'meta_title' => 'nullable',
                'meta_keywords' => 'nullable',
                'meta_description' => 'nullable',
                'meta_type' => 'nullable',
                'status' => 'nullable',
                'rating_count' => 'nullable'
            ]);
            $post = new Post();
            $post->title = $validData['title'];
            $post->body = $validData['body'];
            $post->meta_title = $validData['meta_title'];
            $post->meta_keywords = $validData['meta_keywords'];
            $post->meta_description = $validData['meta_description'];
            $post->meta_type = $validData['meta_type'];
            $post->rating_count = $validData['rating_count'];
            $post->image = $request->image;
            if (array_key_exists('status', $validData)) {
                $validData['status'] = 1;
                $post->status = $validData['status'];
            } else {
                $post->status = 0;
            }
            if ($request->input('slug') != "") {
                $post->slug = Str::slug($request->input('slug'), '-');
            } else {
                $post->slug = Str::slug($request->input('title'), '-');
            }
            $post->save();
            if ($request->categories != null) {
                $categories = Category::find($validData['categories']);
                foreach ($categories as $category) {
                    $post->categories()->attach($category->id);
                }
            }
            alert()->success('پست با موفقیت ایجاد شد', 'موفق');
            return redirect(route('admin.posts.index'));
        } catch (\Illuminate\Database\QueryException $e) {
            $errorCode = $e->errorInfo[1];
            if ($errorCode == '1062') {
                alert()->error('لینک مورد نظر قبلا در سیستم ثبت شده است', 'خطا');
                return back();
            }
        }

    }


    public function show($id)
    {
        return view('post::show');
    }

    public function edit(Post $post)
    {
        $this->seo()->setTitle("ویرایش پست $post->title");
        $categories = Category::with('posts')
            ->where('category_type' , Post::class)
            ->whereNull('parent_id')
            ->get();
        return view('post::backend.edit', ['post' => $post, 'categories' => $categories]);
    }


    public function update(Request $request, Post $post)
    {
        $validData = $request->validate([
            'title' => 'required',
            'slug' => ['nullable', Rule::unique('posts')->ignore($post)],
            'body' => 'required',
            'meta_title' => 'nullable',
            'meta_keywords' => 'nullable',
            'meta_description' => 'nullable',
            'meta_type' => 'nullable',
            'status' => 'nullable',
            'rating_count' => 'nullable'
        ]);
        //if status exists
        if (array_key_exists('status', $validData)) {
            $validData['status'] = 1;
            $post->update([
                'title' => $validData['title'],
                'body' => $validData['body'],
                'slug' => Str::slug($validData['slug'],'-'),
                'meta_title' => $validData['meta_title'],
                'meta_keywords' => $validData['meta_keywords'],
                'meta_description' => $validData['meta_description'],
                'meta_type' => $validData['meta_type'],
                'status' => $validData['status'],
                'rating_count' => $validData['rating_count'],
                'image' => $request->image
            ]);
        }else{
            //if status does not exists
            $post->update([
                'title' => $validData['title'],
                'body' => $validData['body'],
                'slug' => Str::slug($validData['slug'],'-'),
                'meta_title' => $validData['meta_title'],
                'meta_keywords' => $validData['meta_keywords'],
                'meta_description' => $validData['meta_description'],
                'meta_type' => $validData['meta_type'],
                'rating_count' => $validData['rating_count'],
                'status' => 0,
                'image' => $request->image
            ]);
        }

        if ($request->categories != null) {
            $categories = Category::find($request->categories);
            $cat_ids = $categories->mapWithKeys(function ($item) {
                return [$item['id'] => ['category_id' => $item['id']]];
            });
            $post->categories()->sync($cat_ids);
        }
        alert()->success('پست با موفقیت ویرایش شد', 'موفق');
        return back();
    }


    public function destroy(Post $post)
    {
        $post->delete();
        alert()->success(" پست  $post->title با موفقیت حذف شد." , 'موفق');
        return redirect(route('admin.posts.index'));
    }
}
