<?php

namespace Modules\Course\Http\Controllers\Admin;

use Artesaos\SEOTools\Traits\SEOTools;
use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Modules\Category\Entities\Category;
use Modules\Course\Entities\Course;
use Modules\Post\Entities\Post;

class CourseController extends Controller
{
    use SEOTools;
    public function index()
    {
        $this->seo()->setTitle('همه آموزش ها');

        $courses = Course::query();
        if ($key = \request('search')) {
            $courses = $courses->where('title' , 'like' , "%$key%");
        }

        $courses = $courses->latest()->simplePaginate(15);
        return view('course::backend.all' , compact('courses'));
    }

    /**
     * Show the form for creating a new resource.
     * @return Renderable
     */
    public function create()
    {
        $this->seo()->setTitle('ایجاد آموزش جدید');

        $categories = Category::with(['courses', 'childrenRecursive'])
            ->where('category_type' , Course::class)
            ->whereNull('parent_id')
            ->get();
        return view('course::backend.create', ['categories' => $categories]);
    }

    /**
     * Store a newly created resource in storage.
     * @param Request $request
     * @return Renderable
     */
    public function store(Request $request)
    {
        try {
            $validData = $request->validate([
                'title' => 'required',
                'slug' => 'unique:courses|nullable',
                'body' => 'required',
                'categories' => 'nullable',
                'meta_title' => 'nullable',
                'meta_keywords' => 'nullable',
                'meta_desc' => 'nullable',
                'status' => 'nullable',
                'teacher' => 'nullable',
                'time' => 'nullable',
                'price' => 'nullable',
                'video_cover' => 'nullable',
                'rating_count' => 'nullable',

            ]);
            $course = new Course();
            $course->title = $validData['title'];
            $course->body = $validData['body'];
            $course->meta_title = $validData['meta_title'];
            $course->meta_keywords = $validData['meta_keywords'];
            $course->meta_desc = $validData['meta_desc'];
            $course->time = $validData['time'];
            $course->price = $validData['price'];
            $course->teacher = $validData['teacher'];
            $course->image = $request->image;
            $course->video_cover = $validData['video_cover'];
            $course->rating_count = $validData['rating_count'];
            if (array_key_exists('status', $validData)) {
                $validData['status'] = 1;
                $course->status = $validData['status'];
            } else {
                $course->status = 0;
            }
            if ($request->input('slug') != "") {
                $course->slug = Str::slug($request->input('slug'), '-');
            } else {
                $course->slug = Str::slug($request->input('title'), '-');
            }
            $course->save();
            if ($request->categories != null) {
                $categories = Category::find($validData['categories']);
                foreach ($categories as $category) {
                    $course->categories()->attach($category->id);
                }
            }
            alert()->success('پست با موفقیت ایجاد شد', 'موفق');
            return redirect(route('admin.courses.index'));
        } catch (\Illuminate\Database\QueryException $e) {
            $errorCode = $e->errorInfo[1];
            if ($errorCode == '1062') {
                alert()->error('لینک مورد نظر قبلا در سیستم ثبت شده است', 'خطا');
                return back();
            }
        }
    }

    public function show(Course $course)
    {
        //
    }


    public function edit(Course $course)
    {
        $this->seo()->setTitle("ویرایش $course->title");

        $categories = Category::with('courses')
            ->whereNull('parent_id')
            ->where('category_type' , Course::class)
            ->get();
        return view('course::backend.edit' , compact(['course' , 'categories']));
    }

    public function update(Request $request, Course $course)
    {
        $validData = $request->validate([
            'title' => 'required',
            'slug' => ['nullable', Rule::unique('courses')->ignore($course)],
            'body' => 'required',
            'categories' => 'nullable',
            'meta_title' => 'nullable',
            'meta_keywords' => 'nullable',
            'meta_desc' => 'nullable',
            'status' => 'nullable',
            'teacher' => 'nullable',
            'time' => 'nullable',
            'price' => 'nullable',
            'video_cover' => 'nullable',
            'rating_count' => 'nullable',
        ]);
        //if status exists
        if (array_key_exists('status', $validData)) {
            $validData['status'] = 1;
            $course->update([
                'title' => $validData['title'],
                'body' => $validData['body'],
                'slug' => Str::slug($validData['slug'],'-'),
                'meta_title' => $validData['meta_title'],
                'meta_keywords' => $validData['meta_keywords'],
                'meta_desc' => $validData['meta_desc'],
                'status' => $validData['status'],
                'teacher' => $validData['teacher'],
                'price' => $validData['price'],
                'time' => $validData['time'],
                'image' => $request->image,
                'video_cover' => $validData['video_cover'],
                'rating_count' => $validData['rating_count'],

            ]);
        }else{
            //if status does not exists
            $course->update([
                'title' => $validData['title'],
                'body' => $validData['body'],
                'slug' => Str::slug($validData['slug'],'-'),
                'meta_title' => $validData['meta_title'],
                'meta_keywords' => $validData['meta_keywords'],
                'meta_desc' => $validData['meta_desc'],
                'status' => 0,
                'image' => $request->image,
                'teacher' => $validData['teacher'],
                'price' => $validData['price'],
                'time' => $validData['time'],
                'video_cover' => $validData['video_cover'],
                'rating_count' => $validData['rating_count'],

            ]);
        }

        if ($request->categories != null) {
            $categories = Category::find($request->categories);
            $cat_ids = $categories->mapWithKeys(function ($item) {
                return [$item['id'] => ['category_id' => $item['id']]];
            });
            $course->categories()->sync($cat_ids);
        }
        alert()->success('دوره با موفقیت ویرایش شد', 'موفق');
        return back();
    }


    public function destroy(Course $course)
    {
        $course->delete();
        alert()->success(" دوره  $course->title با موفقیت حذف شد." , 'موفق');
        return redirect(route('admin.courses.index'));
    }
}
