<?php

namespace Modules\User\Http\Controllers\Admin;

use Artesaos\SEOTools\Traits\SEOTools;
use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Route;
use Illuminate\Validation\Rule;
use Modules\ACL\Entities\Role;
use Modules\User\Entities\User;

class UserController extends Controller
{
    use SEOTools;
    public function __construct()
    {
        $this->middleware('can:manage-users');
    }

    /**
     * Display a listing of the resource.
     * @return Renderable
     */
    public function index()
    {
        $this->seo()->setTitle('همه ی کاربران');
        $users = User::query();
        if ($key = \request('search')) {
            $users = User::where('name', 'like', "%$key%")
                ->orWhere('phone', 'like', "%$key%")->orWhere('id',$key);
        }
        $users = $users->latest()->simplePaginate(15);
        return view('user::backend.all', compact('users'));
    }


    public function create()
    {
        $this->seo()->setTitle('ایجاد کاربر جدید');
        $roles = Role::all();
        return view('user::backend.create', compact('roles'));
    }


    public function store(Request $request)
    {
        $validData = $request->validate([
            'name' => 'required|min:3',
            'email' => 'nullable|email',
            'phone' => [
                'required',
                'unique:users'
            ],
            'h_phone' => 'nullable',
            'identity_number' => [
                'nullable',
                'unique:users'
            ],
            'address' => 'nullable',
            'role' => 'required'
        ]);
        $user = User::create($validData);
        $user->roles()->attach($validData['role']);
        alert()->success('کاربر با موفقیت ساخته شد.', 'موفق');
        return back();
    }


    public function show($id)
    {
        return view('user::show');
    }


    public function edit(User $user)
    {
        $this->seo()->setTitle("ویرایش کاربر $user->name");
        $roles = Role::all();
        return view('user::backend.edit', compact(['user', 'roles']));
    }


    public function update(Request $request, User $user)
    {
        $validData = $request->validate([
            'name' => 'required|min:3',
            'email' => 'nullable|email',
            'phone' => [
                'required',
                Rule::unique('users')->ignore($user->id)
            ],
            'h_phone' => 'nullable',
            'identity_number' => [
                'nullable',
                Rule::unique('users')->ignore($user->id)
            ],
            'address' => 'nullable',
            'role' => 'required'
        ]);

        $user->update($validData);
        $user->roles()->sync($request->input('role'));
        alert()->success("ویرایش کاربر  $user->name با موفقیت انجام شد.", 'موفق');
        return back();
    }


    public function destroy(User $user)
    {
        $user->delete();
        alert()->success(" کاربر  $user->name با موفقیت حذف شد.", 'موفق');
        return redirect(route('admin.users.index'));
    }
}
