<?php

namespace Modules\Portfolio\Http\Controllers\Admin;

use Artesaos\SEOTools\Traits\SEOTools;
use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Modules\Category\Entities\Category;
use Modules\Portfolio\Entities\Portfolio;

class PortfolioController extends Controller
{
    use SEOTools;
    public function index()
    {
        $this->seo()->setTitle('همه نمونه کارها');
        $portfolios = Portfolio::query();
        if ($key = \request('search')) {
            $portfolios = $portfolios->where('title', 'like', "%$key%");
        }
        $portfolios = $portfolios->orderBy('order_id')->simplePaginate(15);
        return view('portfolio::backend.all' , compact('portfolios'));
    }

    /**
     * Show the form for creating a new resource.
     * @return Renderable
     */
    public function create()
    {
        $this->seo()->setTitle('ایجاد نمونه کار جدید');

        $categories = Category::with('portfolios')
            ->where('category_type' , Portfolio::class)
            ->whereNull('parent_id')
            ->get();
        return view('portfolio::backend.create' , compact('categories'));
    }

    /**
     * Store a newly created resource in storage.
     * @param Request $request
     * @return Renderable
     */
    public function store(Request $request)
    {
        try {
            $validData = $request->validate([
                'title' => 'required',
                'slug' => 'unique:portfolios|nullable',
                'desc' => 'required',
                'categories' => 'nullable',
                'meta_title' => 'nullable',
                'keywords' => 'nullable',
                'meta_disc' => 'nullable',
                'meta_type' => 'nullable',
                'status' => 'nullable',
                'image' => 'nullable',
                'techs' => 'nullable',
                'project_running_date' => 'nullable',
                'project_progress' => 'nullable',
                'link' => 'required',
                'colors' => 'required',
                'order_id' => 'required',
                'logo' => 'nullable',
            ]);
            $portfolio = new Portfolio();
            $portfolio->desc = $validData['desc'];
            $portfolio->meta_title = $validData['meta_title'];
            $portfolio->keywords = $validData['keywords'];
            $portfolio->meta_disc = $validData['meta_disc'];
            $portfolio->meta_type = $validData['meta_type'];
            $portfolio->image = $validData['image'];
            $portfolio->techs = serialize($validData['techs']);
            $portfolio->project_running_date = $validData['project_running_date'];
            $portfolio->project_progress = $validData['project_progress'];
            $portfolio->link = $validData['link'];
            $portfolio->colors = $validData['colors'];
            $portfolio->order_id = $validData['order_id'];
            $portfolio->logo = $validData['logo'];
            $portfolio->title = $validData['title'];
            if (array_key_exists('status', $validData)) {
                $validData['status'] = 1;
                $portfolio->status = $validData['status'];
            } else {
                $portfolio->status = 0;
            }
            if ($request->input('slug') != "") {
                $portfolio->slug = Str::slug($request->input('slug'), '-');
            } else {
                $portfolio->slug = Str::slug($request->input('title'), '-');
            }
            $portfolio->save();
            if ($request->categories != null) {
                $categories = Category::find($validData['categories']);
                foreach ($categories as $category) {
                    $portfolio->categories()->attach($category->id);
                }
            }
            alert()->success('نمونه کار با موفقیت ایجاد شد', 'موفق');
            return redirect(route('admin.portfolios.index'));
        }  catch (\Illuminate\Database\QueryException $e) {
            $errorCode = $e->errorInfo[1];
            if ($errorCode == '1062') {
                alert()->error('لینک مورد نظر قبلا در سیستم ثبت شده است', 'خطا');
                return back();
            }
        }

    }

    /**
     * Show the specified resource.
     * @param int $id
     * @return Renderable
     */
    public function show($id)
    {
        return view('portfolio::show');
    }

    public function edit(Portfolio $portfolio)
    {
        $this->seo()->setTitle("ویرایش نمونه کار $portfolio->title");

        $key = unserialize($portfolio->techs);
        $categories = Category::with('portfolios')
            ->where('category_type' , Portfolio::class)
            ->whereNull('parent_id')
            ->get();
        return view('portfolio::backend.edit' , compact(['portfolio','categories']));
    }


    public function update(Request $request, Portfolio $portfolio)
    {
        $validData = $request->validate([
            'title' => 'required',
            'slug' => ['nullable', Rule::unique('portfolios')->ignore($portfolio)],
            'desc' => 'required',
            'categories' => 'nullable',
            'meta_title' => 'nullable',
            'keywords' => 'nullable',
            'meta_disc' => 'nullable',
            'meta_type' => 'nullable',
            'status' => 'nullable',
            'image' => 'nullable',
            'techs' => 'nullable',
            'project_running_date' => 'nullable',
            'project_progress' => 'nullable',
            'link' => 'required',
            'colors' => 'required',
            'order_id' => 'required',
            'logo' => 'nullable',
        ]);
        //if status exists

        if (array_key_exists('status', $validData)) {
            $validData['status'] = 1;
            $portfolio->update([
                'title' => $validData['title'],
                'desc' => $validData['desc'],
                'slug' => Str::slug($validData['slug'],'-'),
                'meta_title' => $validData['meta_title'],
                'keywords' => $validData['keywords'],
                'meta_disc' => $validData['meta_disc'],
                'meta_type' => $validData['meta_type'],
                'status' => $validData['status'],
                'image' => $request->image,
                'logo' => $request->logo,
                'techs' => serialize($validData['techs']),
                'project_running_date' => $validData['project_running_date'],
                'project_progress' => $validData['project_progress'],
                'link' => $validData['link'],
                'colors' => $validData['colors'],
                'order_id' => $validData['order_id'],
            ]);
        }else{
            //if status does not exists
            $portfolio->update([
                'title' => $validData['title'],
                'desc' => $validData['desc'],
                'slug' => Str::slug($validData['slug'],'-'),
                'meta_title' => $validData['meta_title'],
                'keywords' => $validData['keywords'],
                'meta_disc' => $validData['meta_disc'],
                'meta_type' => $validData['meta_type'],
                'status' => 0,
                'image' => $request->image,
                'logo' => $request->logo,
                'techs' => serialize($validData['techs']),
                'project_running_date' => $validData['project_running_date'],
                'project_progress' => $validData['project_progress'],
                'link' => $validData['link'],
                'colors' => $validData['colors'],
                'order_id' => $validData['order_id'],
            ]);
        }

        if ($request->categories != null) {
            $categories = Category::find($request->categories);
            $cat_ids = $categories->mapWithKeys(function ($item) {
                return [$item['id'] => ['category_id' => $item['id']]];
            });
            $portfolio->categories()->sync($cat_ids);
        }
        alert()->success('نمونه کار با موفقیت ویرایش شد', 'موفق');
        return back();
    }


    public function destroy(Portfolio $portfolio)
    {
        $portfolio->delete();
        alert()->success('عملیات با موفقیت انجام شد');
        return back();
    }
}
