import os
import datetime
from mysgi_api.settings.base import BASE_DIR
import environ
from rest_framework import status
from rest_framework.views import APIView
from rest_framework.response import Response
from rest_framework.permissions import IsAuthenticated
from .serializers import *
from permissions import CheckUserAccessLevel
from drf_spectacular.utils import extend_schema, OpenApiResponse, OpenApiExample, OpenApiParameter, OpenApiTypes
from rest_framework.decorators import action
from .models import ITEquipmentUnits, FurnitureUnits



env = environ.Env()
environ.Env.read_env(os.path.join(BASE_DIR, '.env'))


class ListTicketStatusView(APIView):
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = ListTicketStatusSerializer
    
    @extend_schema(
        request = ListTicketStatusSerializer,
        responses = {
            200: OpenApiResponse(response=FailedMessageDueToAccessSerializer, description='List Ticket Status'),
            406: OpenApiResponse(response=FailedMessageDueToAccessSerializer, description='List Ticket Status'),
        }
    )
    def get(self, request):
        serializer = ListTicketStatusSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.get_items()
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class AddTicketDepartmentView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = AddTicketDepartmentSerializer
    
    def post(self, request):
        serializer = AddTicketDepartmentSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.add_department(request.data, request.META)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)




class AddCustomFieldView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = AddCustomFieldSerializer
    
    def post(self, request):
        serializer = AddCustomFieldSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.add_customfield(request.data, request.META)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class UpdateCustomFieldView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = UpdateCustomFieldSerializer
    
    def put(self, request, cfid):
        serializer = UpdateCustomFieldSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.update_customfield(request.data, request.META, cfid)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class DeleteCustomFieldView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = DeleteCustomFieldSerializer
    
    def delete(self, request, cfid):
        serializer = DeleteCustomFieldSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.delete_customfield(request.data, request.META, cfid)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class GetCustomFieldView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetCustomFieldSerializer
    
    def get(self, request, dname):
        serializer = GetCustomFieldSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.get_customfield(request.data, request.META, dname)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)




class GetTicketTemplateView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetTicketTemplateSerializer
    
    @extend_schema(
        request = GetTicketTemplateSerializer,
        responses = {
            200: OpenApiResponse(response=TicketTemplateResponseSerializer, description='Ticket Template Fields'),
        }
    )
    def get(self, request, dname):
        serializer = GetTicketTemplateSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.get_template(request.data, request.META, dname)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)




class ListTicketView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = ListTicketSerializer

    @extend_schema(
        parameters=[
            OpenApiParameter(name='count', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Count Paginate Parameter', required=False),
            OpenApiParameter(name='page_number', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Page Number Paginate Parameter', required=False),
            OpenApiParameter(name='filter', location=OpenApiParameter.QUERY, type=OpenApiTypes.STR, description='Search Filter string', required=False)
        ],
        request = ListTicketSerializer,
        responses = {
            200: OpenApiResponse(response=ListTicketResponseSerializer, description='List Tickets'),
            406: OpenApiResponse(response=FailedMessageDueToAccessSerializer, description='List Tickets'),
        }
    )
    def get(self, request, dname):
        serializer = ListTicketSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.get_ticket_list(request.data, request.META, dname, request)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class OpenTicketView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = OpenTicketSerializer
    
    @extend_schema(
        request = OpenTicketSerializer,
        responses = {
            200: OpenApiResponse(response=OpenTicketResponseSerializer, description='Open Ticket Serializer'),
        }
    )
    def post(self, request):
        serializer = OpenTicketSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.open_ticket(request.data, request.META, request.FILES)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class AddReplyTicket(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = AddReplySerializer
    
    @extend_schema(
        request = AddReplySerializer,
        responses = {
            200: OpenApiResponse(response=AddReplyResponseSerializer, description='Add Reply Ticket'),
        }
    )
    def post(self, request, tid):
        serializer = AddReplySerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.add_reply(request.data, request.META, request.FILES, tid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class GetTicketDetailView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetTicketDetailSerializer
    
    def get(self, request, tid):
        serializer = GetTicketDetailSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.ticket_detail(request.data, request.META, tid)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class ChangeTicketStatusView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = ChangeTicketStatusSerializer
    
    @extend_schema(
        request = ChangeTicketStatusSerializer,
        responses={200: OpenApiResponse(response=ChangeTicketStatusResponseSerializer, description='Change Ticket Status')},
    )
    def post(self, request):
        serializer = ChangeTicketStatusSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.change_status(request.data, request.META)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class DeleteTicketView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = DeleteTicketSerializer
    
    @extend_schema(
        request = DeleteTicketSerializer,
        responses={200: OpenApiResponse(response=DeleteTicketResponseSerializer, description='Delete Ticket')},
    )
    def delete(self, request, tid):
        serializer = DeleteTicketSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.delete_ticket(request.data, request.META, tid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class UpdateTicketView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    
    @action(detail=False, methods=['GET'])
    def get(self, request, tid):
        serializer = UpdateTicketmethodGetSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.get_ticket(request.data, request.META, tid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    @extend_schema(
        request = UpdateTicketmethodUpdateSerializer,
        responses={200: OpenApiResponse(response=UpdateTicketmethodUpdateSerializer, description='Delete Ticket')},
    )
    @action(detail=False, methods=['POST'])
    def put(self, request, tid):
        serializer = UpdateTicketmethodUpdateSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.update_ticket_and_reply(request.data, request.META, tid, request.FILES)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class SaveResponseITRequestsView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = SaveResponseITRequestsSerializer
    
    def post(self, request, tid):
        serializer = SaveResponseITRequestsSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.save_response(request.data, request.META, tid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)





class ITEquipmentsListunitsView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = ITEquipmentsListunitsSerializer
    
    @extend_schema(
        responses={
            200: ITEquipmentsListunitsSerializer(many=True)
        }
    )
    def get(self, request):
        self.check_object_permissions(request, 'it_equipment_requests')
        units = ITEquipmentUnits.objects.all()
        serialized_data = ITEquipmentsListunitsSerializer(instance=units, many=True)
        return Response(data=serialized_data.data, status=status.HTTP_200_OK)


class ITEquipmentsListView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    
    @extend_schema(
        parameters=[
            OpenApiParameter(name='count', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Count Paginate Parameter', required=False),
            OpenApiParameter(name='page_number', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Page Number Paginate Parameter', required=False),
            OpenApiParameter(name='filter', location=OpenApiParameter.QUERY, type=OpenApiTypes.STR, description='Search Filter string', required=False)
        ],
        request = ITEquipmentsListSerializer,
        responses={
            200: OpenApiResponse(response=ITEquipmentsListResponseSerializer, description='IT Equipments List')
        },
    )
    def get(self, request):
        serializer = ITEquipmentsListSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'it_equipment_requests')
            result = serializer.get_items(request.data, request.META, request)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class ITEquipmentsAddView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = ITEquipmentsAddSerializer
    
    def post(self, request):
        serializer = ITEquipmentsAddSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'it_equipment_requests-create')
            result = serializer.add_item(request.data, request.META)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class ITEquipmentsDeleteView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = ITEquipmentsDeleteSerializer

    def delete(self, request, tid):
        serializer = ITEquipmentsDeleteSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'it_equipment_requests-destroy')
            result = serializer.delete_item(request.data, request.META, tid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class ITEquipmentsUpdateView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    
    @extend_schema(
        request = ITEquipmentsUpdateGetMethodSerializer,
        responses={200: OpenApiResponse(response=ITEquipmentsUpdateGetMethodResponseSerializer, description='Update IT Equipments')},
    )
    @action(detail=False, methods=['GET'])
    def get(self, request, tid):
        serializer = ITEquipmentsUpdateGetMethodSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'it_equipment_requests-edit')
            result = serializer.get_items(request.data, request.META, tid)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    @extend_schema(
        request = ITEquipmentsUpdateUpdateMethodSerializer,
        responses={200: OpenApiResponse(response=ITEquipmentsUpdateUpdateMethodSerializer, description='Update IT Equipments')},
    )
    @action(detail=False, methods=['POST'])
    def put(self, request, tid):
        serializer = ITEquipmentsUpdateUpdateMethodSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'it_equipment_requests-update')
            result = serializer.update_items(request.data, request.META, tid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
    

class ITEquipmentsChangeStatusView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = ITEquipmentsChangeStatusSerializer

    @extend_schema(
        request = ITEquipmentsChangeStatusSerializer,
        responses={200: OpenApiResponse(response=ITEquipmentsChangeStatusSerializer, description='Change Status')},
    )
    def post(self, request):
        serializer = ITEquipmentsChangeStatusSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'it_equipment_requests-update')
            result = serializer.change_status(request.data, request.META)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class ITEquipmentsAdminResponseView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = ITEquipmentsAdminResponseSerializer

    @extend_schema(
        responses={200: OpenApiResponse(response=ITEquipmentsAdminResponseSerializer, description='Change Status')},
    )
    def post(self, request, tid):
        serializer = ITEquipmentsAdminResponseSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'it_equipment_requests-manage')
            result = serializer.save_response(request.data, request.META, tid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)





class MeetingRequestContentTypeView(APIView):
    
    permission_classes = [IsAuthenticated]
    serializer_class = MeetingRequestContentTypeSerializer
    
    @extend_schema(
        responses={
            200: MeetingRequestContentTypeSerializer()
        }
    )
    def get(self, request):
        serializer = MeetingRequestContentTypeSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.get_items()
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class MeetingRequestLocationView(APIView):
    permission_classes = [IsAuthenticated]
    serializer_class = MeetingRequestLocationSerializer
    
    @extend_schema(
        responses={
            200: MeetingRequestLocationSerializer()
        }
    )
    def get(self, request):
        serializer = MeetingRequestLocationSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.get_items()
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class MeetingRequestServeView(APIView):
    permission_classes = [IsAuthenticated]
    serializer_class = MeetingRequestServeSerializer
    
    @extend_schema(
        responses={
            200: MeetingRequestServeSerializer()
        }
    )
    def get(self, request):
        serializer = MeetingRequestServeSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.get_items()
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
    

class MeetingRequestTypeView(APIView):
    permission_classes = [IsAuthenticated]
    serializer_class = MeetingRequestTypeSerializer
    
    @extend_schema(
        responses={
            200: MeetingRequestTypeSerializer()
        }
    )
    def get(self, request):
        serializer = MeetingRequestTypeSerializer(data=request.data)
        if serializer.is_valid():
            result = serializer.get_items()
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class GetMeetingRequestsView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = MeetingRequestGetSerializer

    @extend_schema(
        request = MeetingRequestGetSerializer,
        responses={200: OpenApiResponse(response=MessageDoneSerializer, description='Get Meeting Requests')},
    )
    def post(self, request, lid):
        serializer = MeetingRequestGetSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'meeting_requests')
            result = serializer.get_meetings(request.data, request.META, lid)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class ListMeetingRequests(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = MeetingRequestListSerializer

    @extend_schema(
        parameters=[
            OpenApiParameter(name='count', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Count Paginate Parameter', required=False),
            OpenApiParameter(name='page_number', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Page Number Paginate Parameter', required=False),
            OpenApiParameter(name='filter', location=OpenApiParameter.QUERY, type=OpenApiTypes.STR, description='Search Filter string', required=False)
        ],
        request = MeetingRequestListSerializer,
        responses={200: OpenApiResponse(response=MessageDoneSerializer, description='List Meeting Requests')},
    )
    def get(self, request):
        serializer = MeetingRequestListSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'meeting_requests')
            result = serializer.list_requests(request.data, request.META, request)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class AddMeetingRequestsView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = MeetingRequestAddSerializer

    @extend_schema(
        request = MeetingRequestAddSerializer,
        responses={200: OpenApiResponse(response=MessageDoneSerializer, description='Add Meeting Requests')},
    )
    def post(self, request):
        serializer = MeetingRequestAddSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'meeting_requests-create')
            result = serializer.add_meet(request.data, request.META)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class EditMeetingRequestsView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    
    @extend_schema(
        request = MeetingRequestDetailSerializer,
        responses={200: OpenApiResponse(response=MessageDoneSerializer, description='Edit Meeting Requests')},
    )
    def get(self, request, id):
        serializer = MeetingRequestDetailSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'meeting_requests-edit')
            result = serializer.get_item(request.data, request.META, id)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    @extend_schema(
        request = MeetingRequestEditSerializer,
        responses={200: OpenApiResponse(response=MessageDoneSerializer, description='Edit Meeting Requests')},
    )
    def post(self, request,id):
        serializer = MeetingRequestEditSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'meeting_requests-update')
            result = serializer.edit_meet(request.data, request.META,id)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class MeetingRequestChangeStatusView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = MeetingRequestChangeStatusSerializer

    @extend_schema(
        request = MeetingRequestChangeStatusSerializer,
        responses={200: OpenApiResponse(response=MeetingRequestChangeStatusSerializer, description='Change Status')},
    )
    def post(self, request):
        serializer = MeetingRequestChangeStatusSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'meeting_requests-update')
            result = serializer.change_status(request.data, request.META)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class DeleteMeetingRequestsView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = MeetingRequestDeleteSerializer

    @extend_schema(
        request = MeetingRequestDeleteSerializer,
        responses={200: OpenApiResponse(response=MessageDoneSerializer, description='Delete Meeting Requests')},
    )
    def delete(self, request, id):
        serializer = MeetingRequestDeleteSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'meeting_requests-destroy')
            result = serializer.delete_meet(request.data, request.META, id)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)





class FurnitureListunitsView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = FurnitureListunitsSerializer
    
    @extend_schema(
        responses={
            200: FurnitureListunitsSerializer(many=True)
        }
    )
    def get(self, request):
        self.check_object_permissions(request, 'furniture_requests')
        units = FurnitureUnits.objects.all()
        serialized_data = FurnitureListunitsSerializer(instance=units, many=True)
        return Response(data=serialized_data.data, status=status.HTTP_200_OK)


class FurnitureListView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    
    @extend_schema(
        parameters=[
            OpenApiParameter(name='count', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Count Paginate Parameter', required=False),
            OpenApiParameter(name='page_number', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Page Number Paginate Parameter', required=False),
            OpenApiParameter(name='filter', location=OpenApiParameter.QUERY, type=OpenApiTypes.STR, description='Search Filter string', required=False)
        ],
        request = FurnitureListSerializer,
        responses={
            200: OpenApiResponse(response=FurnitureListResponseSerializer, description='Furniture List')
        },
    )
    def get(self, request):
        serializer = FurnitureListSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'furniture_requests')
            result = serializer.get_items(request.data, request.META, request)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class FurnitureAddView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = FurnitureAddSerializer
    
    def post(self, request):
        serializer = FurnitureAddSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'furniture_requests-create')
            result = serializer.add_item(request.data, request.META)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class FurnitureDeleteView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = FurnitureDeleteSerializer

    def delete(self, request, tid):
        serializer = FurnitureDeleteSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'furniture_requests-destroy')
            result = serializer.delete_item(request.data, request.META, tid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class FurnitureUpdateView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    
    @extend_schema(
        request = FurnitureUpdateGetMethodSerializer,
        responses={200: OpenApiResponse(response=FurnitureUpdateGetMethodResponseSerializer, description='Update Furniture')},
    )
    @action(detail=False, methods=['GET'])
    def get(self, request, tid):
        serializer = FurnitureUpdateGetMethodSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'furniture_requests-edit')
            result = serializer.get_items(request.data, request.META, tid)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    @extend_schema(
        request = FurnitureUpdateUpdateMethodSerializer,
        responses = {
            200: OpenApiResponse(response=FurnitureUpdateUpdateMethodSerializer, description='Update Furniture'),
            406: OpenApiResponse(response=FailedMessageDueToAccessSerializer, description='Update Furniture')
        },
    )
    @action(detail=False, methods=['POST'])
    def put(self, request, tid):
        serializer = FurnitureUpdateUpdateMethodSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'furniture_requests-update')
            result = serializer.update_items(request.data, request.META, tid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
    

class FurnitureChangeStatusView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = FurnitureChangeStatusSerializer

    @extend_schema(
        request = FurnitureChangeStatusSerializer,
        responses={
            200: OpenApiResponse(response=FurnitureChangeStatusSerializer, description='Change Status'),
            406: OpenApiResponse(response=FailedMessageDueToAccessSerializer, description='Change Status')
        },
    )
    def post(self, request):
        serializer = FurnitureChangeStatusSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'furniture_requests-edit')
            result = serializer.change_status(request.data, request.META)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class FurnitureAdminResponseView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = FurnitureAdminResponseSerializer

    @extend_schema(
        request = FurnitureAdminResponseSerializer,
        responses = {
            200: OpenApiResponse(response=FurnitureAdminResponseSerializer, description='Furniture Admin Response'),
            406: OpenApiResponse(response=FailedMessageDueToAccessSerializer, description='Furniture Admin Response')
        },
    )
    def post(self, request, tid):
        serializer = FurnitureAdminResponseSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'furniture_requests-edit')
            result = serializer.save_response(request.data, request.META, tid)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)





class GamingRoomListTypesView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GamingRoomListTypesSerializer

    @extend_schema(
        parameters=[
            OpenApiParameter(name='count', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Count Paginate Parameter', required=False),
            OpenApiParameter(name='page_number', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Page Number Paginate Parameter', required=False),
            OpenApiParameter(name='filter', location=OpenApiParameter.QUERY, type=OpenApiTypes.STR, description='Search Filter string', required=False)
        ],
        request=GamingRoomListTypesSerializer,
        responses={
            200: OpenApiResponse(response=FurnitureListResponseSerializer, description='Gaming Room Request List Types')
        },
    )
    def get(self, request):
        serializer = GamingRoomListTypesSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'gaming_room_requests')
            result = serializer.get_types(request.data, request.META)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
    

class GamingRoomListUsersView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    
    @extend_schema(
        parameters=[
            OpenApiParameter(name='count', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Count Paginate Parameter', required=False),
            OpenApiParameter(name='page_number', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Page Number Paginate Parameter', required=False),
            OpenApiParameter(name='filter', location=OpenApiParameter.QUERY, type=OpenApiTypes.STR, description='Search Filter string', required=False)
        ]
    )
    def get(self, request):
        self.check_object_permissions(request, 'gaming_room_requests')
        if request.GET.get('filter') == None:
            users = Users.objects.all().order_by("created_at").annotate(
                member_name=Concat('first_name', Value(' '), F('last_name'), output_field=CharField())
            ).values('user_name', 'first_name', 'last_name', 'member_name')
        else:
            users = Users.objects.filter(
                    Q(user_name__icontains=request.GET.get('filter')) |
                    Q(first_name__icontains=request.GET.get('filter')) |
                    Q(last_name__icontains=request.GET.get('filter')) |
                    Q(personnel_code__icontains=request.GET.get('filter')) |
                    Q(mobile__icontains=request.GET.get('filter')) |
                    Q(email__icontains=request.GET.get('filter')) |
                    Q(job_title__icontains=request.GET.get('filter')) |
                    Q(department__icontains=request.GET.get('filter')) |
                    Q(office_name__icontains=request.GET.get('filter')) |
                    Q(national_code__icontains=request.GET.get('filter'))
                ).order_by("created_at").annotate(
                    member_name=Concat('first_name', Value(' '), F('last_name'), output_field=CharField())
                ).values('user_name', 'first_name', 'last_name', 'member_name')
        datas = custom_pagination(users, request.GET.get('count', 100000), request.GET.get('page_number', 1))
        return Response(datas, status=status.HTTP_200_OK)
    

class GamingRoomListRequestView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GamingRoomListRequestSerializer
    
    @extend_schema(
        parameters=[
            OpenApiParameter(name='count', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Count Paginate Parameter', required=False),
            OpenApiParameter(name='page_number', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Page Number Paginate Parameter', required=False),
            OpenApiParameter(name='filter', location=OpenApiParameter.QUERY, type=OpenApiTypes.STR, description='Search Filter string', required=False)
        ],
        request=GamingRoomListRequestSerializer,
        responses={
            200: OpenApiResponse(response=FurnitureListResponseSerializer, description='Gaming Room Request List')
        },
    )
    def get(self, request):
        serializer = GamingRoomListRequestSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'gaming_room_requests')
            result = serializer.get_requests(request.data, request.META, request)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class GamingRoomAddRequestView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GamingRoomAddRequestSerializer

    def post(self, request):
        serializer = GamingRoomAddRequestSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'gaming_room_requests-create')
            result = serializer.add_requests(request.data, request.META)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class GamingRoomGetTimeRangesView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GamingRoomGetTimeRangesSerializer
    
    @extend_schema(
        parameters=[
            OpenApiParameter(name='count', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Count Paginate Parameter', required=False),
            OpenApiParameter(name='page_number', location=OpenApiParameter.QUERY, type=OpenApiTypes.INT, description='Page Number Paginate Parameter', required=False),
            OpenApiParameter(name='filter', location=OpenApiParameter.QUERY, type=OpenApiTypes.STR, description='Search Filter string', required=False)
        ],
        request=GamingRoomGetTimeRangesSerializer,
        responses={
            200: OpenApiResponse(response=FurnitureListResponseSerializer, description='Gaming Room List Time Ranges')
        },
    )
    def post(self, request):
        serializer = GamingRoomGetTimeRangesSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'gaming_room_requests')
            result = serializer.get_times(request.data, request.META)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

 
class GamingRoomDeleteRequestView(APIView):
     
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GamingRoomDeleteRequestSerializer

    @extend_schema(
        request = GamingRoomDeleteRequestSerializer,
        responses={200: OpenApiResponse(response=MessageDoneSerializer, description='Delete gaming room Requests')},
    )
    def delete(self, request, id):
        serializer = GamingRoomDeleteRequestSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'gaming_room_requests-destroy')
            result = serializer.delete_room_request(request.data, request.META, id)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class GamingRoomEditRequestView(APIView):
     
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    
    @extend_schema(
        request = GamingRoomRequestDetailSerializer,
        responses={200: OpenApiResponse(response=MessageDoneSerializer, description='Edit Gaming Room Requests')},
    )
    def get(self, request, id):
        serializer = GamingRoomRequestDetailSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'gaming_room_requests-edit')
            result = serializer.get_item(request.data, request.META, id)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    @extend_schema(
        request = GamingRoomRequestEditSerializer,
        responses={200: OpenApiResponse(response=MessageDoneSerializer, description='Edit Gaming Room Requests')},
    )
    def post(self, request,id):
        serializer = GamingRoomRequestEditSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'gaming_room_requests-update')
            result = serializer.edit_room(request.data, request.META,id)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class GamingRoomChangeStatusView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GamingRoomChangeStatusSerializer

    @extend_schema(
        request = GamingRoomChangeStatusSerializer,
        responses={200: OpenApiResponse(response=GamingRoomChangeStatusSerializer, description='Change Status')},
    )
    def post(self, request):
        serializer = GamingRoomChangeStatusSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'gaming_room_requests-change_status')
            result = serializer.change_status(request.data, request.META)
            return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)




class GetVersioningView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = GetVersioningSerializer
    
    @extend_schema(
        responses={
            200: GetVersioningSerializer(many=True)
        }
    )
    def get(self, request):
        serializer = GetVersioningSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'gaming_room_requests')
            result = serializer.get_versions(request.data, request.META)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class AddVersioningView(APIView):
    
    permission_classes = [IsAuthenticated, CheckUserAccessLevel]
    serializer_class = AddVersioningSerializer

    def post(self, request):
        serializer = AddVersioningSerializer(data=request.data)
        if serializer.is_valid():
            self.check_object_permissions(request, 'panel_versioning-create')
            result = serializer.add_versions(request.data, request.META)
            if result[0] == True:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_200_OK)
            else:
                return Response({"result": result[0], "message": result[1]}, status=status.HTTP_406_NOT_ACCEPTABLE)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


