<?php
// upload.php

// 1. تنظیمات زمان اجرا و حافظه (در خط اول)
set_time_limit(0);  // غیرمحدود
ini_set('memory_limit', '2048M');  // 2GB حافظه
ini_set('max_input_time', '0');
ini_set('max_execution_time', '0');
ini_set('post_max_size', '100M');
ini_set('upload_max_filesize', '100M');

ignore_user_abort(true);
ob_implicit_flush(true);

// 2. حالت خطاگیری را فعال کنید (برای دیباگ)
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once 'config.php';
require_once 'vendor/autoload.php';

use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Shared\Date as PhpSpreadsheetDate;
use Morilog\Jalali\Jalalian;

// 3. توابع کمکی بهینه‌شده
function convertJalaliToGregorian($jalaliDate) {
    if (empty($jalaliDate) || $jalaliDate === null) return null;
    
    $jalaliDate = trim($jalaliDate);
    
    // اگر قبلاً تاریخ میلادی است
    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $jalaliDate)) {
        return $jalaliDate;
    }
    
    // اگر عدد اکسل است (مثلاً 45105)
    if (is_numeric($jalaliDate) && $jalaliDate > 1000) {
        try {
            return PhpSpreadsheetDate::excelToDateTimeObject($jalaliDate)->format('Y-m-d');
        } catch (Exception $e) {
            return null;
        }
    }
    
    // ***** افزودن اعتبارسنجی دقیق‌تر *****
    if (!preg_match('/^(\d{4})[\/\-\.](\d{1,2})[\/\-\.](\d{1,2})$/', $jalaliDate, $matches)) {
        return null;
    }
    
    $year = (int)$matches[1];
    $month = (int)$matches[2];
    $day = (int)$matches[3];
    
    // ***** اعتبارسنجی اولیه *****
    if ($year < 1300 || $year > 1500 || $month < 1 || $month > 12 || $day < 1 || $day > 31) {
        return null;
    }
    
    // ***** بررسی تعداد روزهای هر ماه شمسی *****
    $daysInMonth = [
        1 => 31, 2 => 31, 3 => 31, 4 => 31, 5 => 31, 6 => 31,
        7 => 30, 8 => 30, 9 => 30, 10 => 30, 11 => 30, 12 => 29
    ];
    
    // ***** بررسی کبیسه بودن سال برای اسفند *****
    if ($month == 12) {
        try {
            // بررسی کبیسه بودن
            $isLeapYear = false;
            // یک روش ساده برای بررسی کبیسه بودن (الگوریتم تقریبی)
            $leapYears = [1, 5, 9, 13, 17, 22, 26, 30]; // باقیمانده تقسیم بر 33
            $remainder = $year % 33;
            if (in_array($remainder, $leapYears)) {
                $isLeapYear = true;
                $daysInMonth[12] = 30;
            }
        } catch (Exception $e) {
            // در صورت خطا، حالت ایمن
            $daysInMonth[12] = 29;
        }
    }
    
    // ***** اصلاح روز اگر بیشتر از روزهای ماه باشد *****
    if ($day > $daysInMonth[$month]) {
        $day = $daysInMonth[$month];
    }
    
    // ***** ساخت تاریخ معتبر *****
    $validJalaliDate = sprintf('%04d/%02d/%02d', $year, $month, $day);
    
    // ***** استفاده از try-catch جامع‌تر *****
    try {
        // پیدا کردن جداکننده
        $separator = preg_match('/^(\d{4})([\/\-\.])(\d{1,2})\2(\d{1,2})$/', $jalaliDate, $sepMatches) ? $sepMatches[2] : '/';
        
        // ساخت فرمت بر اساس جداکننده
        $format = $separator == '/' ? 'Y/m/d' : ($separator == '-' ? 'Y-m-d' : 'Y.m.d');
        
        // ***** استفاده مستقیم از کلاس Jalalian *****
        $jalalian = Jalalian::fromFormat($format, $validJalaliDate);
        return $jalalian->toCarbon()->format('Y-m-d');
        
    } catch (Exception $e) {
        // ***** لاگ خطا برای دیباگ *****
        error_log("Error converting Jalali date '$jalaliDate' to Gregorian: " . $e->getMessage());
        
        // ***** تلاش با روش جایگزین *****
        try {
            // استفاده از strtotime فارسی (اگر extension نصب باشد)
            if (function_exists('jDateTime::toGregorian')) {
                list($gYear, $gMonth, $gDay) = \jDateTime::toGregorian($year, $month, $day);
                return sprintf('%04d-%02d-%02d', $gYear, $gMonth, $gDay);
            }
            
            // روش محاسباتی ساده (تقریبی)
            // تبدیل شمسی به میلادی با فرمول تقریبی
            $jd = \Morilog\Jalali\CalendarUtils::toJalali($year, $month, $day);
            $gregorian = \Morilog\Jalali\CalendarUtils::toGregorian($jd[0], $jd[1], $jd[2]);
            return sprintf('%04d-%02d-%02d', $gregorian[0], $gregorian[1], $gregorian[2]);
            
        } catch (Exception $ex) {
            error_log("Alternative conversion also failed for '$jalaliDate': " . $ex->getMessage());
            return null;
        }
    }
}

// 4. تابع isDuplicate بهینه‌شده با کش
function isDuplicate($pdo, $unique_id, &$cache = []) {
    if (isset($cache[$unique_id])) {
        return true;
    }
    
    static $existing_ids = null;
    
    // فقط یک بار همه IDها را از دیتابیس بگیریم
    if ($existing_ids === null) {
        $stmt = $pdo->prepare("SELECT شماره_درخواست FROM fuel_requests");
        $stmt->execute();
        $existing_ids = array_flip($stmt->fetchAll(PDO::FETCH_COLUMN));
    }
    
    $is_duplicate = isset($existing_ids[$unique_id]);
    
    if ($is_duplicate) {
        $cache[$unique_id] = true;
    }
    
    return $is_duplicate;
}

// 5. تابع mapColumns بهینه‌شده
function mapColumns($headerRow, $column_mapping) {
    $column_order = [];
    $used_db_columns = [];
    
    // ایجاد کلمات کلیدی برای هر هدر دیتابیس
    $db_keywords = [];
    foreach ($column_mapping as $excel_header => $db_column) {
        $db_keywords[$db_column] = preg_replace('/[_\s]/', '', strtolower($db_column));
    }
    
    foreach ($headerRow as $index => $header) {
        $header_clean = trim($header);
        $header_lower = preg_replace('/[_\s]/', '', strtolower($header_clean));
        $found = false;
        
        // جستجوی مستقیم
        if (isset($column_mapping[$header_clean])) {
            $db_column = $column_mapping[$header_clean];
            if (!in_array($db_column, $used_db_columns)) {
                $column_order[$index] = [
                    'excel_header' => $header_clean,
                    'db_column' => $db_column,
                    'matched_header' => $header_clean,
                    'similarity' => 100
                ];
                $used_db_columns[] = $db_column;
                $found = true;
            }
        }
        
        // جستجو با similarity
        if (!$found) {
            $best_match = null;
            $best_similarity = 0;
            
            foreach ($column_mapping as $excel_header => $db_column) {
                if (in_array($db_column, $used_db_columns)) continue;
                
                similar_text($header_lower, preg_replace('/[_\s]/', '', strtolower($excel_header)), $percent);
                
                if ($percent > $best_similarity && $percent > 95) {
                    $best_similarity = $percent;
                    $best_match = ['excel_header' => $excel_header, 'db_column' => $db_column];
                }
            }
            
            if ($best_match) {
                $column_order[$index] = [
                    'excel_header' => $header_clean,
                    'db_column' => $best_match['db_column'],
                    'matched_header' => $best_match['excel_header'],
                    'similarity' => $best_similarity
                ];
                $used_db_columns[] = $best_match['db_column'];
                $found = true;
            }
        }
        
        if (!$found) {
            $column_order[$index] = [
                'excel_header' => $header_clean,
                'db_column' => null,
                'matched_header' => null,
                'similarity' => 0
            ];
        }
    }
    
    return $column_order;
}

// 6. تابع insertBatch برای وارد کردن دسته‌ای
function insertBatch($pdo, $data_batch, $columns) {
    if (empty($data_batch)) return 0;
    
    $values = [];
    $placeholders = [];
    $params = [];
    
    foreach ($data_batch as $row) {
        $row_placeholders = [];
        foreach ($columns as $col) {
            $row_placeholders[] = '?';
            $params[] = $row[$col] ?? null;
        }
        $placeholders[] = '(' . implode(', ', $row_placeholders) . ')';
    }
    
    $columns_str = implode(', ', $columns);
    $placeholders_str = implode(', ', $placeholders);
    
    $sql = "INSERT IGNORE INTO fuel_requests ($columns_str) VALUES $placeholders_str";
    
    try {
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->rowCount();
    } catch (PDOException $e) {
        // اگر batch ناموفق بود، تک‌تک وارد کنیم
        error_log("Batch insert failed, trying one by one: " . $e->getMessage());
        $success_count = 0;
        foreach ($data_batch as $row) {
            try {
                $cols = array_keys($row);
                $vals = array_values($row);
                $sql_single = "INSERT IGNORE INTO fuel_requests (" . implode(', ', $cols) . ") VALUES (" . str_repeat('?,', count($vals)-1) . "?)";
                $stmt = $pdo->prepare($sql_single);
                $stmt->execute($vals);
                $success_count += $stmt->rowCount();
            } catch (Exception $ex) {
                error_log("Single insert failed: " . $ex->getMessage());
                continue;
            }
        }
        return $success_count;
    }
}

// 7. مرحله ۱: آپلود فایل و نمایش ستون‌ها
// ***** ابتدا بررسی کنید که آیا فرم آپلود ارسال شده است *****
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_FILES['excel_file']) && !isset($_POST['confirm_import'])) {
    // ***** بررسی خطاهای آپلود *****
    if (!isset($_FILES['excel_file']) || $_FILES['excel_file']['error'] !== UPLOAD_ERR_OK) {
        $error_message = "خطا در آپلود فایل: ";
        switch ($_FILES['excel_file']['error'] ?? 4) {
            case UPLOAD_ERR_INI_SIZE:
                $error_message .= "حجم فایل بیشتر از حد مجاز است!";
                break;
            case UPLOAD_ERR_FORM_SIZE:
                $error_message .= "حجم فایل بیشتر از حد مجاز فرم است!";
                break;
            case UPLOAD_ERR_PARTIAL:
                $error_message .= "فایل به طور کامل آپلود نشد!";
                break;
            case UPLOAD_ERR_NO_FILE:
                $error_message .= "هیچ فایلی انتخاب نشده است!";
                break;
            case UPLOAD_ERR_NO_TMP_DIR:
                $error_message .= "پوشه موقت وجود ندارد!";
                break;
            case UPLOAD_ERR_CANT_WRITE:
                $error_message .= "نوشتن فایل بر روی دیسک با خطا مواجه شد!";
                break;
            default:
                $error_message .= "خطای ناشناخته!";
        }
        die("<div class='alert alert-danger'>$error_message</div>");
    }
    
    $file = $_FILES['excel_file'];
    
    if ($file['error'] === UPLOAD_ERR_OK) {
        $tmp_name = $file['tmp_name'];
        
        try {
            // ذخیره فایل موقت برای مرحله بعد
            $temp_file = tempnam(sys_get_temp_dir(), 'excel_');
            move_uploaded_file($tmp_name, $temp_file);
            
            $spreadsheet = IOFactory::load($temp_file);
            $worksheet = $spreadsheet->getActiveSheet();
            $rows = $worksheet->toArray();
            
            $headerRow = array_shift($rows);
            $column_order = mapColumns($headerRow, $column_mapping);
            $sample_data = array_slice($rows, 0, 3); // 3 ردیف نمونه
            
            // نمایش فرم تأیید ستون‌ها
            ?>
            <!DOCTYPE html>
            <html lang="fa" dir="rtl">
            <head>
                <meta charset="UTF-8">
                <meta name="viewport" content="width=device-width, initial-scale=1.0">
                <title>تأیید ستون‌ها - آپلود فایل اکسل</title>
                <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
                <style>
                    body { font-family: 'Tahoma', 'Arial', sans-serif; background-color: #f8f9fa; }
                    .container { max-width: 1200px; }
                    .matched { background-color: #d4edda !important; }
                    .unmatched { background-color: #f8d7da !important; }
                    .column-mapping { max-height: 600px; overflow-y: auto; }
                    .sample-data { font-size: 0.85em; }
                </style>
            </head>
            <body>
                <div class="container mt-4">
                    <div class="card shadow">
                        <div class="card-header bg-primary text-white text-center">
                            <h4 class="mb-0">تأیید ارتباط ستون‌ها</h4>
                        </div>
                        <div class="card-body">
                            <form method="POST" id="confirmForm">
                                <input type="hidden" name="temp_file" value="<?= htmlspecialchars($temp_file) ?>">
                                <input type="hidden" name="confirm_import" value="1">
                                
                                <div class="alert alert-info">
                                    <h6>لطفاً ارتباط ستون‌ها را بررسی و تأیید کنید:</h6>
                                    <p class="mb-0">ستون‌های سبز رنگ به طور خودکار شناسایی شده‌اند. ستون‌های قرمز رنگ نیاز به انتخاب دستی دارند.</p>
                                </div>
                                
                                <div class="table-responsive column-mapping mb-4">
                                    <table class="table table-bordered table-hover">
                                        <thead class="table-dark">
                                            <tr>
                                                <th width="50">#</th>
                                                <th width="300">ستون در فایل اکسل</th>
                                                <th width="300">ستون متناظر در دیتابیس</th>
                                                <th width="150">درصد تطابق</th>
                                                <th width="200">انتخاب دستی</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($column_order as $index => $column): ?>
                                            <tr class="<?= $column['db_column'] ? 'matched' : 'unmatched' ?>">
                                                <td class="text-center fw-bold"><?= $index + 1 ?></td>
                                                <td>
                                                    <span class="fw-bold"><?= htmlspecialchars($column['excel_header']) ?></span>
                                                    <?php if ($column['db_column']): ?>
                                                    <br><small class="text-muted">تشخیص: <?= htmlspecialchars($column['matched_header']) ?></small>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <?php if ($column['db_column']): ?>
                                                    <span class="text-success fw-bold"><?= $column['db_column'] ?></span>
                                                    <?php else: ?>
                                                    <span class="text-danger">شناسایی نشد</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td class="text-center">
                                                    <?php if ($column['db_column']): ?>
                                                    <span class="badge bg-success"><?= number_format($column['similarity'], 1) ?>%</span>
                                                    <?php else: ?>
                                                    <span class="badge bg-danger">۰%</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <select class="form-select form-select-sm manual-select" name="manual_mapping[<?= $index ?>]" data-original-value="<?= $column['db_column'] ?>">
                                                        <option value="">-- انتخاب ستون --</option>
                                                        <?php foreach ($column_mapping as $excel_header => $db_column): ?>
                                                        <option value="<?= $db_column ?>" <?= $db_column === $column['db_column'] ? 'selected' : '' ?>>
                                                            <?= $db_column ?> (<?= $excel_header ?>)
                                                        </option>
                                                        <?php endforeach; ?>
                                                    </select>
                                                </td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                                
                                <!-- نمایش نمونه داده‌ها -->
                                <div class="card mb-4">
                                    <div class="card-header bg-secondary text-white">
                                        <h6 class="mb-0">نمونه داده‌ها (۳ ردیف اول)</h6>
                                    </div>
                                    <div class="card-body sample-data">
                                        <div class="table-responsive">
                                            <table class="table table-bordered table-sm">
                                                <thead class="table-light">
                                                    <tr>
                                                        <?php foreach ($headerRow as $header): ?>
                                                        <th><?= htmlspecialchars($header) ?></th>
                                                        <?php endforeach; ?>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                    <?php foreach ($sample_data as $row): ?>
                                                    <tr>
                                                        <?php foreach ($row as $cell): ?>
                                                        <td><?= htmlspecialchars($cell) ?></td>
                                                        <?php endforeach; ?>
                                                    </tr>
                                                    <?php endforeach; ?>
                                                </tbody>
                                            </table>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="row mb-3">
                                    <div class="col-md-6">
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" id="skip_errors" name="skip_errors" checked>
                                            <label class="form-check-label" for="skip_errors">رد کردن ردیف‌های دارای خطا</label>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" id="skip_duplicates" name="skip_duplicates" checked>
                                            <label class="form-check-label" for="skip_duplicates">رد کردن داده‌های تکراری</label>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="d-grid gap-2 d-md-flex justify-content-md-end">
                                    <button type="button" class="btn btn-secondary me-md-2" onclick="location.href='upload.php'">انصراف</button>
                                    <button type="submit" class="btn btn-success" id="confirmBtn">
                                        <i class="bi bi-check-circle"></i> تأیید و شروع وارد کردن داده‌ها
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
                
                <script>
                    // نمایش تغییرات دستی
                    document.querySelectorAll('.manual-select').forEach(select => {
                        select.addEventListener('change', function() {
                                            const originalValue = this.getAttribute('data-original-value');
                                            const row = this.closest('tr');
                                            
                                            if (this.value) {
                                                row.className = 'matched';
                                                row.cells[2].innerHTML = '<span class="text-success fw-bold">' + this.value + '</span>';
                                                row.cells[3].innerHTML = '<span class="badge bg-warning">دستی</span>';
                                            } else if (originalValue) {
                                                row.className = 'matched';
                                                row.cells[2].innerHTML = '<span class="text-success fw-bold">' + originalValue + '</span>';
                                                row.cells[3].innerHTML = '<span class="badge bg-success">اتوماتیک</span>';
                                            } else {
                                                row.className = 'unmatched';
                                                row.cells[2].innerHTML = '<span class="text-danger">شناسایی نشد</span>';
                                                row.cells[3].innerHTML = '<span class="badge bg-danger">۰%</span>';
                                            }
                                        });
                                    });
                    
                    document.getElementById('confirmForm').addEventListener('submit', function() {
                        document.getElementById('confirmBtn').innerHTML = '<span class="spinner-border spinner-border-sm"></span> در حال پردازش...';
                        document.getElementById('confirmBtn').disabled = true;
                    });
                </script>
                
                <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.8.0/font/bootstrap-icons.css">
                <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
            </body>
            </html>
            <?php
            exit;
            
        } catch (Exception $e) {
            echo "<div class='alert alert-danger'>خطا در بارگذاری فایل: " . $e->getMessage() . "</div>";
        }
    }
}

// 8. مرحله ۲: وارد کردن داده‌ها - بهینه‌شده
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['confirm_import'])) {
    session_write_close(); // برای پردازش طولانی، session را ببند
    
    // پیش‌پردازش شروع
    echo str_repeat(' ', 1024 * 64); // buffer flush
    echo '<!DOCTYPE html><html lang="fa" dir="rtl"><head>';
    echo '<meta charset="UTF-8"><meta name="viewport" content="width=device-width, initial-scale=1.0">';
    echo '<title>در حال پردازش...</title>';
    echo '<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">';
    echo '<style>body { padding: 20px; } .progress { height: 25px; }</style>';
    echo '</head><body>';
    echo '<div class="container">';
    echo '<div class="alert alert-info">در حال پردازش فایل، لطفا صبر کنید...</div>';
    echo '<div class="progress mb-3"><div id="progressBar" class="progress-bar progress-bar-striped progress-bar-animated" style="width: 0%"></div></div>';
    echo '<div id="status" class="alert alert-light"></div>';
    echo '</div>';
    echo '<script>
        function updateProgress(percent, message) {
            document.getElementById("progressBar").style.width = percent + "%";
            document.getElementById("progressBar").innerText = percent.toFixed(1) + "%";
            document.getElementById("status").innerHTML = message;
            window.scrollTo(0, document.body.scrollHeight);
        }
    </script>';
    flush();
    
    $temp_file = $_POST['temp_file'] ?? '';
    $skip_errors = isset($_POST['skip_errors']);
    $skip_duplicates = isset($_POST['skip_duplicates']);
    $manual_mapping = $_POST['manual_mapping'] ?? [];
    
    if (!file_exists($temp_file)) {
        die("<div class='alert alert-danger'>فایل موقت یافت نشد</div>");
    }
    
    try {
        // بارگذاری فایل با حافظه کمینه
        $inputFileType = IOFactory::identify($temp_file);
        $reader = IOFactory::createReader($inputFileType);
        $reader->setReadDataOnly(true);
        $reader->setReadEmptyCells(false);
        
        // فقط داده‌ها را بخوان
        $spreadsheet = $reader->load($temp_file);
        $worksheet = $spreadsheet->getActiveSheet();
        
        // خواندن سطر به سطر برای صرفه‌جویی در حافظه
        $rows = $worksheet->toArray();
        $headerRow = array_shift($rows);
        $totalRows = count($rows);
        
        $auto_mapping = mapColumns($headerRow, $column_mapping);
        
        // اعمال مپینگ دستی
        $final_mapping = [];
        $db_columns = [];
        foreach ($auto_mapping as $index => $column) {
            if (!empty($manual_mapping[$index])) {
                $db_col = $manual_mapping[$index];
                $final_mapping[$index] = $db_col;
                $db_columns[$index] = $db_col;
            } elseif ($column['db_column']) {
                $db_col = $column['db_column'];
                $final_mapping[$index] = $db_col;
                $db_columns[$index] = $db_col;
            }
        }
        
        // فیلتر کردن ستون‌های خالی
        $db_columns = array_filter($db_columns);
        
        $imported = 0;
        $skipped = 0;
        $duplicates = 0;
        $duplicate_data = [];
        $errors = [];
        $batch_size = 1000; // تعداد رکورد در هر batch
        $data_batch = [];
        $duplicate_cache = [];
        
        // لیست فیلدها
        $dateFields = ['تاریخ_تایید_درخواست_توسط_پخش'];
        $numericFields = [
            'مقداری_قابل_تحویل_از_فروشندگی', 'مقدار_تحویل_شده_درخواست_از_فروشندگی',
            'مقدار_سوخت_یارانه_ای_درخواستی', 'مقدار_درخواستی_متقاضی',
            'مقدار_تایید_شده_توسط_متولی_اول', 'مقدار_تایید_شده_توسط_پخش'
        ];
        
        foreach ($rows as $index => $row) {
            $row_number = $index + 2; // +2 برای هدر و شروع از 1
            
            try {
                // آماده‌سازی داده
                $data = [];
                foreach ($db_columns as $col_index => $db_column) {
                    $value = $row[$col_index] ?? null;
                    
                    // تبدیل تاریخ‌ها
                    if (in_array($db_column, $dateFields) && !empty($value)) {
                        $value = convertJalaliToGregorian($value);
                    }
                    
                    // تبدیل عددی
                    if (in_array($db_column, $numericFields) && $value !== null) {
                        if (is_string($value)) {
                            $value = preg_replace('/[^\d\.\-]/', '', $value);
                        }
                        $value = ($value !== '' && $value !== null) ? (float)$value : null;
                    }
                    
                    $data[$db_column] = $value;
                }
                
                // بررسی تکراری بودن
                $unique_id = $data['شماره_درخواست'] ?? null;
                
                if ($skip_duplicates && $unique_id) {
                    if (isDuplicate($pdo, $unique_id, $duplicate_cache)) {
                        $duplicates++;
                        $duplicate_data[] = [
                            'row' => $row_number,
                            'unique_id' => $unique_id,
                            'data' => $data
                        ];
                        error_log("Duplicate entry found at row $row_number: $unique_id");
                        continue;
                    }
                }
                
                // اضافه کردن به batch
                $data_batch[] = $data;
                
                // اگر batch پر شد، وارد دیتابیس کن
                if (count($data_batch) >= $batch_size) {
                    $batch_imported = insertBatch($pdo, $data_batch, array_keys($data_batch[0] ?? []));
                    $imported += $batch_imported;
                    $data_batch = []; // خالی کردن batch
                    
                    // به روزرسانی progress
                    $progress = min(100, ($index / $totalRows) * 100);
                    $message = "پردازش شده: $row_number از $totalRows ردیف | وارد شده: $imported | تکراری: $duplicates | خطا: $skipped";
                    echo "<script>updateProgress($progress, '$message');</script>";
                    flush();
                }
                
            } catch (Exception $e) {
                $skipped++;
                $errors[] = "ردیف $row_number: " . $e->getMessage();
                error_log("Error at row $row_number: " . $e->getMessage());
                
                if (!$skip_errors) {
                    $errors[] = "توقف در ردیف $row_number به دلیل خطا";
                    break;
                }
                continue;
            }
        }
        
        // وارد کردن باقی‌مانده batch
        if (!empty($data_batch)) {
            $batch_imported = insertBatch($pdo, $data_batch, array_keys($data_batch[0]));
            $imported += $batch_imported;
        }
        
        // پاک کردن حافظه
        unset($spreadsheet, $worksheet, $rows, $data_batch);
        gc_collect_cycles();
        
        // پاک کردن فایل موقت
        if (file_exists($temp_file)) {
            unlink($temp_file);
        }
        
        // نمایش نتایج نهایی
        echo "<script>updateProgress(100, 'پردازش کامل شد!');</script>";
        echo '<div class="alert alert-success mt-4"><h4>✅ عملیات تکمیل شد!</h4></div>';
        echo '<div class="row">';
        echo '<div class="col-md-3"><div class="alert alert-success">ردیف‌های وارد شده: <strong>' . $imported . '</strong></div></div>';
        echo '<div class="col-md-3"><div class="alert alert-warning">ردیف‌های تکراری: <strong>' . $duplicates . '</strong></div></div>';
        echo '<div class="col-md-3"><div class="alert alert-danger">ردیف‌های خطا: <strong>' . $skipped . '</strong></div></div>';
        echo '<div class="col-md-3"><div class="alert alert-info">کل ردیف‌ها: <strong>' . $totalRows . '</strong></div></div>';
        echo '</div>';
        
        // نمایش جزئیات خطاها
        if ($errors) {
            echo '<div class="alert alert-danger"><h5>خطاها:</h5><ul>';
            $display_errors = array_slice($errors, 0, 10);
            foreach ($display_errors as $error) {
                echo '<li>' . htmlspecialchars($error) . '</li>';
            }
            if (count($errors) > 10) {
                echo '<li>و ' . (count($errors) - 10) . ' خطای دیگر...</li>';
            }
            echo '</ul></div>';
        }
        
        // نمایش نمونه‌ای از داده‌های تکراری
        if ($duplicate_data) {
            echo '<div class="alert alert-warning"><h5>نمونه‌ای از داده‌های تکراری:</h5>';
            $sample_duplicates = array_slice($duplicate_data, 0, 5);
            echo '<table class="table table-sm table-bordered">';
            echo '<thead><tr><th>ردیف</th><th>شماره درخواست</th></tr></thead>';
            echo '<tbody>';
            foreach ($sample_duplicates as $dup) {
                echo '<tr><td>' . $dup['row'] . '</td><td>' . htmlspecialchars($dup['unique_id']) . '</td></tr>';
            }
            echo '</tbody></table>';
            if ($duplicates > 5) {
                echo '<p>و ' . ($duplicates - 5) . ' مورد دیگر...</p>';
            }
            echo '</div>';
        }
        
        echo '<div class="mt-4">';
        echo '<a href="upload.php" class="btn btn-primary me-2">آپلود فایل جدید</a>';
        echo '<a href="index.php" class="btn btn-success">مشاهده داده‌ها</a>';
        echo '</div>';
        
    } catch (Exception $e) {
        echo '<div class="alert alert-danger mt-4"><h4>خطا!</h4><p>' . htmlspecialchars($e->getMessage()) . '</p></div>';
        if (file_exists($temp_file)) {
            unlink($temp_file);
        }
        echo '<a href="upload.php" class="btn btn-primary">بازگشت</a>';
    }
    
    echo '</body></html>';
    exit;
}

// 9. فرم آپلود اولیه
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>آپلود فایل اکسل - نسخه پیشرفته</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body { font-family: 'Tahoma', 'Arial', sans-serif; background-color: #f8f9fa; }
        .container { max-width: 800px; }
        .upload-area { border: 2px dashed #dee2e6; border-radius: 10px; padding: 40px; text-align: center; background-color: #fff; }
        .upload-area:hover { border-color: #007bff; background-color: #f8f9fa; }
    </style>
</head>
<body>
    <div class="container mt-5">
        <div class="card shadow">
            <div class="card-header bg-primary text-white text-center">
                <h4 class="mb-0">آپلود فایل اکسل - نسخه پیشرفته</h4>
            </div>
            <div class="card-body">
                <!-- ***** تغییر: افزودن action و name برای دکمه ***** -->
                <form method="POST" enctype="multipart/form-data" action="<?php echo htmlspecialchars($_SERVER['PHP_SELF']); ?>">
                    
                    <!-- ***** افزودن یک hidden field برای تشخیص فرم ***** -->
                    <input type="hidden" name="upload_form" value="1">
                    
                    <div class="upload-area mb-4">
                        <i class="bi bi-cloud-arrow-up fs-1 text-muted mb-3"></i>
                        <h5 class="text-muted">فایل اکسل خود را اینجا رها کنید یا کلیک کنید</h5>
                        <p class="text-muted small">فرمت‌های مجاز: .xlsx, .xls</p>
                        
                        <input type="file" class="form-control d-none" id="excel_file" name="excel_file" accept=".xlsx,.xls" required>
                        
                        <button type="button" class="btn btn-outline-primary mt-3" onclick="document.getElementById('excel_file').click()">
                            انتخاب فایل
                        </button>
                        <div id="file-name" class="mt-2 text-success fw-bold"></div>
                    </div>
                    
                    <div class="alert alert-info">
                        <h6>ویژگی‌های جدید:</h6>
                        <ul class="mb-0 small">
                            <li>تشخیص هوشمند ستون‌ها با نمایش درصد تطابق</li>
                            <li>امکان تأیید دستی ارتباط ستون‌ها قبل از آپلود</li>
                            <li>نمایش نمونه داده‌ها برای بررسی</li>
                            <li>بررسی و حذف داده‌های تکراری</li>
                        </ul>
                    </div>
                    
                    <div class="d-grid gap-2">
                        <!-- ***** تغییر نام دکمه ***** -->
                        <button type="submit" name="upload_submit" class="btn btn-primary btn-lg">آپلود و بررسی ستون‌ها</button>
                        <a href="index.php" class="btn btn-outline-secondary">مشاهده داده ها</a>
                    </div>
                </form>
                
                <!-- ***** نمایش خطاها اگر فرم ارسال شده اما فایل نیست ***** -->
                <?php
                if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['upload_form'])) {
                    if (!isset($_FILES['excel_file']) || $_FILES['excel_file']['error'] !== UPLOAD_ERR_OK) {
                        echo '<div class="alert alert-danger mt-3">';
                        if (!isset($_FILES['excel_file'])) {
                            echo 'هیچ فایلی ارسال نشده است!';
                        } else {
                            switch ($_FILES['excel_file']['error']) {
                                case UPLOAD_ERR_INI_SIZE:
                                    echo 'حجم فایل بیشتر از حد مجاز است!';
                                    break;
                                case UPLOAD_ERR_FORM_SIZE:
                                    echo 'حجم فایل بیشتر از حد مجاز فرم است!';
                                    break;
                                case UPLOAD_ERR_PARTIAL:
                                    echo 'فایل به طور کامل آپلود نشد!';
                                    break;
                                case UPLOAD_ERR_NO_FILE:
                                    echo 'هیچ فایلی انتخاب نشده است!';
                                    break;
                                case UPLOAD_ERR_NO_TMP_DIR:
                                    echo 'پوشه موقت وجود ندارد!';
                                    break;
                                case UPLOAD_ERR_CANT_WRITE:
                                    echo 'نوشتن فایل بر روی دیسک با خطا مواجه شد!';
                                    break;
                                default:
                                    echo 'خطای ناشناخته در آپلود فایل!';
                            }
                        }
                        echo '</div>';
                    }
                }
                ?>
            </div>
        </div>
    </div>

    <script>
        document.getElementById('excel_file').addEventListener('change', function(e) {
            const fileName = e.target.files[0]?.name || 'هیچ فایلی انتخاب نشده';
            document.getElementById('file-name').textContent = fileName;
        });
        
        // جلوگیری از ارسال فرم اگر فایل انتخاب نشده
        document.querySelector('form').addEventListener('submit', function(e) {
            const fileInput = document.getElementById('excel_file');
            if (!fileInput.files.length) {
                e.preventDefault();
                alert('لطفا یک فایل انتخاب کنید!');
                return false;
            }
            
            // نمایش پیام در حال پردازش
            const submitBtn = this.querySelector('button[type="submit"]');
            submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> در حال پردازش...';
            submitBtn.disabled = true;
        });
    </script>

    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.8.0/font/bootstrap-icons.css">
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>