<?php
// regional_analysis.php
require_once 'config.php';
require_once 'vendor/autoload.php';

use Morilog\Jalali\Jalalian;
use Carbon\Carbon;

class RegionalAnalysis {
    private $pdo;
    private $column_mapping;

    public function __construct($pdo) {
        $this->pdo = $pdo;
        $this->column_mapping = [
            'بخش مصرف' => 'بخش_مصرف',
            'متولی اول' => 'متولی_اول',
            'گره کاربرد' => 'گره_کاربرد',
            'کاربرد' => 'کاربرد',
            'عنوان کاربرد کاربر' => 'عنوان_کاربرد_کاربر',
            'نام متقاضی' => 'نام_متقاضی',
            'کدملی/ شناسه ملی' => 'کدملی_شناسه_ملی',
            'تلفن همراه' => 'تلفن_همراه',
            'آدرس' => 'آدرس',
            'کد فروش' => 'کد_فروش',
            'استان' => 'استان',
            'شهرستان' => 'شهرستان',
            'شناسه یکتای کاربرد' => 'شناسه_یکتای_کاربرد',
             'شماره درخواست' => 'شماره_درخواست',
            'فراوده' => 'فراوده',
            'نوع تحویل سوخت' => 'نوع_تحویل_سوخت',
            'مقدار تایید شده توسط پخش' => 'مقدار_تایید_شده_توسط_پخش',
            'مقدار تایید شده توسط متولی اول' => 'مقدار_تایید_شده_توسط_متولی_اول',
            'نام فروشندگی انتخاب شده در درخواست' => 'نام_فروشندگی_انتخاب_شده_در_درخواست',
            'کد فروشندگی انتخاب شده در درخواست' => 'کد_فروشندگی_انتخاب_شده_در_درخواست',
            'تاریخ تایید درخواست توسط پخش' => 'تاریخ_تایید_درخواست_توسط_پخش',
            'نوع درخواست' => 'نوع_درخواست'
        ];
    }

    // توابع تبدیل تاریخ (همانند قبلی)
    public function getGregorianRangeFromShamsi($shamsiYear, $period = 'yearly', $period_number = null) {
        try {
            $start_shamsi = null;
            $end_shamsi = null;

            switch($period) {
                case 'monthly':
                    if ($period_number && $period_number >= 1 && $period_number <= 12) {
                        $start_shamsi = new Jalalian($shamsiYear, $period_number, 1);
                        $end_shamsi = (new Jalalian($shamsiYear, $period_number, 1))->addMonths(1)->addDays(-1);
                    }
                    break;

                case 'quarterly':
                    if ($period_number && $period_number >= 1 && $period_number <= 4) {
                        $start_month = (($period_number - 1) * 3) + 1;
                        $end_month = $start_month + 2;
                        $start_shamsi = new Jalalian($shamsiYear, $start_month, 1);
                        $end_shamsi = (new Jalalian($shamsiYear, $end_month, 1))->addMonths(1)->addDays(-1);
                    }
                    break;

                case 'halfyearly':
                    if ($period_number && $period_number >= 1 && $period_number <= 2) {
                        $start_month = $period_number == 1 ? 1 : 7;
                        $end_month = $period_number == 1 ? 6 : 12;
                        $start_shamsi = new Jalalian($shamsiYear, $start_month, 1);
                        $end_shamsi = (new Jalalian($shamsiYear, $end_month, 1))->addMonths(1)->addDays(-1);
                    }
                    break;

                case 'yearly':
                default:
                    $start_shamsi = new Jalalian($shamsiYear, 1, 1);
                    $end_shamsi = (new Jalalian($shamsiYear, 12, 1))->addMonths(1)->addDays(-1);
                    break;
            }

            if ($start_shamsi && $end_shamsi) {
                $start_gregorian = $start_shamsi->toCarbon();
                $end_gregorian = $end_shamsi->toCarbon();
                $end_gregorian->setTime(23, 59, 59);
                
                return [
                    'start' => $start_gregorian->format('Y-m-d H:i:s'),
                    'end' => $end_gregorian->format('Y-m-d H:i:s'),
                    'shamsi_period' => $this->getShamsiPeriodName($period, $period_number, $shamsiYear)
                ];
            }
        } catch (Exception $e) {
            return null;
        }
        return null;
    }

    public function getShamsiPeriodName($period, $period_number, $year) {
        $month_names = [
            1 => 'فروردین', 2 => 'اردیبهشت', 3 => 'خرداد', 4 => 'تیر',
            5 => 'مرداد', 6 => 'شهریور', 7 => 'مهر', 8 => 'آبان',
            9 => 'آذر', 10 => 'دی', 11 => 'بهمن', 12 => 'اسفند'
        ];
        
        $quarter_names = [1 => 'اول', 2 => 'دوم', 3 => 'سوم', 4 => 'چهارم'];
        $halfyear_names = [1 => 'اول', 2 => 'دوم'];
        
        switch($period) {
            case 'monthly':
                return $month_names[$period_number] . ' ' . $year;
            case 'quarterly':
                return 'فصل ' . $quarter_names[$period_number] . ' ' . $year;
            case 'halfyearly':
                return 'نیمه ' . $halfyear_names[$period_number] . ' ' . $year;
            case 'yearly':
                return 'سال ' . $year;
            default:
                return 'سال ' . $year;
        }
    }

    public function getAvailableShamsiYears() {
        $query = $this->pdo->query("
            SELECT DISTINCT تاریخ_تایید_درخواست_توسط_پخش 
            FROM fuel_requests 
            WHERE تاریخ_تایید_درخواست_توسط_پخش IS NOT NULL 
            ORDER BY تاریخ_تایید_درخواست_توسط_پخش DESC
        ");
        $dates = $query->fetchAll(PDO::FETCH_COLUMN);

        $years = [];
        foreach ($dates as $date) {
            try {
                $carbon = Carbon::parse($date);
                $jalali = Jalalian::fromCarbon($carbon);
                $shamsi_year = $jalali->getYear();
                if (!in_array($shamsi_year, $years)) {
                    $years[] = $shamsi_year;
                }
            } catch (Exception $e) {
                continue;
            }
        }

        rsort($years);
        return $years;
    }

    private function buildWhereClause($filters) {
        $where_conditions = [];
        $params = [];

        // فیلتر تاریخ
        if (!empty($filters['start_date']) && !empty($filters['end_date'])) {
            $where_conditions[] = "تاریخ_تایید_درخواست_توسط_پخش BETWEEN ? AND ?";
            $params[] = $filters['start_date'];
            $params[] = $filters['end_date'];
        }

        // فیلتر شهرستان
        if (!empty($filters['city']) && $filters['city'] !== 'all') {
            $where_conditions[] = "شهرستان = ?";
            $params[] = $filters['city'];
        }

        // فیلتر فرآورده
        if (!empty($filters['product']) && $filters['product'] !== 'all') {
            $where_conditions[] = "فراوده = ?";
            $params[] = $filters['product'];
        }

        // فیلتر بخش مصرف
        if (!empty($filters['usage_section']) && $filters['usage_section'] !== 'all') {
            $where_conditions[] = "بخش_مصرف = ?";
            $params[] = $filters['usage_section'];
        }

        $where_clause = "";
        if (!empty($where_conditions)) {
            $where_clause = "WHERE " . implode(" AND ", $where_conditions);
        }

        return ['where' => $where_clause, 'params' => $params];
    }

    // 1- مقایسه میزان مصرف کل فرآورده در کل ناحیه
    public function getRegionalTotalConsumption($filters, $comparison_type = 'continuous') {
        $where_data = $this->buildWhereClause($filters);
        
        $query = "
            SELECT 
                YEAR(تاریخ_تایید_درخواست_توسط_پخش) as gregorian_year,
                MONTH(تاریخ_تایید_درخواست_توسط_پخش) as gregorian_month,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_consumption
            FROM fuel_requests 
            {$where_data['where']}
            GROUP BY YEAR(تاریخ_تایید_درخواست_توسط_پخش), MONTH(تاریخ_تایید_درخواست_توسط_پخش)
            ORDER BY gregorian_year, gregorian_month
        ";

        $stmt = $this->pdo->prepare($query);
        $stmt->execute($where_data['params']);
        $data = $stmt->fetchAll(PDO::FETCH_ASSOC);

        return $this->organizeConsumptionData($data, $comparison_type);
    }

    // 2- مقایسه هر شهرستان در فواصل زمانی مختلف
    public function getCityConsumptionComparison($filters) {
        $where_data = $this->buildWhereClause($filters);
        
        $query = "
            SELECT 
                شهرستان,
                YEAR(تاریخ_تایید_درخواست_توسط_پخش) as year,
                MONTH(تاریخ_تایید_درخواست_توسط_پخش) as month,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_consumption
            FROM fuel_requests 
            {$where_data['where']}
            GROUP BY شهرستان, YEAR(تاریخ_تایید_درخواست_توسط_پخش), MONTH(تاریخ_تایید_درخواست_توسط_پخش)
            ORDER BY شهرستان, year, month
        ";

        $stmt = $this->pdo->prepare($query);
        $stmt->execute($where_data['params']);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // 3- مقایسه مصرف کل ناحیه در هر نوع فراورده
    public function getRegionalProductConsumption($filters) {
        $where_data = $this->buildWhereClause($filters);
        
        $query = "
            SELECT 
                فراوده,
                YEAR(تاریخ_تایید_درخواست_توسط_پخش) as year,
                MONTH(تاریخ_تایید_درخواست_توسط_پخش) as month,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_consumption
            FROM fuel_requests 
            {$where_data['where']}
            GROUP BY فراوده, YEAR(تاریخ_تایید_درخواست_توسط_پخش), MONTH(تاریخ_تایید_درخواست_توسط_پخش)
            ORDER BY فراوده, year, month
        ";

        $stmt = $this->pdo->prepare($query);
        $stmt->execute($where_data['params']);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // 4- مقایسه مصرف هر شهرستان در هر نوع فرآورده
    public function getCityProductConsumption($filters) {
        $where_data = $this->buildWhereClause($filters);
        
        $query = "
            SELECT 
                شهرستان,
                فراوده,
                YEAR(تاریخ_تایید_درخواست_توسط_پخش) as year,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_consumption
            FROM fuel_requests 
            {$where_data['where']}
            GROUP BY شهرستان, فراوده, YEAR(تاریخ_تایید_درخواست_توسط_پخش)
            ORDER BY شهرستان, فراوده, year
        ";

        $stmt = $this->pdo->prepare($query);
        $stmt->execute($where_data['params']);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // 5- مقایسه مصرف کل ناحیه بر اساس نوع بخش مصرف + تفاوت مقادیر
    public function getRegionalUsageSectionConsumption($filters) {
        $where_data = $this->buildWhereClause($filters);
        
        $query = "
            SELECT 
                بخش_مصرف,
                YEAR(تاریخ_تایید_درخواست_توسط_پخش) as year,
                MONTH(تاریخ_تایید_درخواست_توسط_پخش) as month,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_distribution,
                SUM(مقدار_تایید_شده_توسط_متولی_اول) as total_first_responsible,
                SUM(مقدار_تایید_شده_توسط_پخش - مقدار_تایید_شده_توسط_متولی_اول) as difference
            FROM fuel_requests 
            {$where_data['where']}
            GROUP BY بخش_مصرف, YEAR(تاریخ_تایید_درخواست_توسط_پخش), MONTH(تاریخ_تایید_درخواست_توسط_پخش)
            ORDER BY بخش_مصرف, year, month
        ";

        $stmt = $this->pdo->prepare($query);
        $stmt->execute($where_data['params']);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // 6- مقایسه مصرف کل شهرستان بر اساس نوع بخش مصرف + تفاوت مقادیر
    public function getCityUsageSectionConsumption($filters) {
        $where_data = $this->buildWhereClause($filters);
        
        $query = "
            SELECT 
                شهرستان,
                بخش_مصرف,
                YEAR(تاریخ_تایید_درخواست_توسط_پخش) as year,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_distribution,
                SUM(مقدار_تایید_شده_توسط_متولی_اول) as total_first_responsible,
                SUM(مقدار_تایید_شده_توسط_پخش - مقدار_تایید_شده_توسط_متولی_اول) as difference
            FROM fuel_requests 
            {$where_data['where']}
            GROUP BY شهرستان, بخش_مصرف, YEAR(تاریخ_تایید_درخواست_توسط_پخش)
            ORDER BY شهرستان, بخش_مصرف, year
        ";

        $stmt = $this->pdo->prepare($query);
        $stmt->execute($where_data['params']);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // 7- مقایسه مصرف هر بخش مصرف از هر نوع فرآورده
    public function getUsageSectionProductConsumption($filters) {
        $where_data = $this->buildWhereClause($filters);
        
        $query = "
            SELECT 
                بخش_مصرف,
                فراوده,
                YEAR(تاریخ_تایید_درخواست_توسط_پخش) as year,
                MONTH(تاریخ_تایید_درخواست_توسط_پخش) as month,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_consumption
            FROM fuel_requests 
            {$where_data['where']}
            GROUP BY بخش_مصرف, فراوده, YEAR(تاریخ_تایید_درخواست_توسط_پخش), MONTH(تاریخ_تایید_درخواست_توسط_پخش)
            ORDER BY بخش_مصرف, فراوده, year, month
        ";

        $stmt = $this->pdo->prepare($query);
        $stmt->execute($where_data['params']);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    private function organizeConsumptionData($data, $comparison_type) {
        $organized_data = [];

        foreach ($data as $row) {
            $year = $row['gregorian_year'];
            $month = $row['gregorian_month'];
            
            // تبدیل به تاریخ شمسی
            try {
                $carbon_date = Carbon::create($year, $month, 1);
                $jalali = Jalalian::fromCarbon($carbon_date);
                $shamsi_year = $jalali->getYear();
                $shamsi_month = $jalali->getMonth();
                
                if ($comparison_type === 'corresponding') {
                    // مقایسه ماه‌های متناظر
                    $key = $shamsi_month;
                } else {
                    // مقایسه پیوسته
                    $key = $shamsi_year . '-' . str_pad($shamsi_month, 2, '0', STR_PAD_LEFT);
                }
                
                if (!isset($organized_data[$key])) {
                    $organized_data[$key] = 0;
                }
                
                $organized_data[$key] += (float)$row['total_consumption'];
                
            } catch (Exception $e) {
                continue;
            }
        }

        return $organized_data;
    }

    public function generateAllReports($filters, $comparison_type = 'continuous') {
        $reports = [];
        
        $reports['regional_total'] = $this->getRegionalTotalConsumption($filters, $comparison_type);
        $reports['city_comparison'] = $this->getCityConsumptionComparison($filters);
        $reports['regional_product'] = $this->getRegionalProductConsumption($filters);
        $reports['city_product'] = $this->getCityProductConsumption($filters);
        $reports['regional_usage_section'] = $this->getRegionalUsageSectionConsumption($filters);
        $reports['city_usage_section'] = $this->getCityUsageSectionConsumption($filters);
        $reports['usage_section_product'] = $this->getUsageSectionProductConsumption($filters);
        
        return $reports;
    }
}

// استفاده از کلاس
try {
    $analysis = new RegionalAnalysis($pdo);
    
    // دریافت فیلترها از درخواست
    $filters = [
        'start_date' => $_POST['start_date'] ?? null,
        'end_date' => $_POST['end_date'] ?? null,
        'city' => $_POST['city'] ?? 'all',
        'product' => $_POST['product'] ?? 'all',
        'usage_section' => $_POST['usage_section'] ?? 'all',
        'years' => $_POST['years'] ?? [],
        'time_period' => $_POST['time_period'] ?? 'yearly'
    ];
    
    $comparison_type = $_POST['comparison_type'] ?? 'continuous';
    
    // تولید گزارشات
    $reports = $analysis->generateAllReports($filters, $comparison_type);
    
    // خروجی JSON برای استفاده در نمودارها
    header('Content-Type: application/json');
    echo json_encode([
        'success' => true,
        'reports' => $reports,
        'filters' => $filters
    ]);
    
} catch (Exception $e) {
    header('Content-Type: application/json');
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
?>