<?php
// index.php
require_once 'config.php';
require_once 'vendor/autoload.php';

use Morilog\Jalali\Jalalian;

// Get parameters
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$per_page = isset($_GET['per_page']) ? (int)$_GET['per_page'] : 20;
$sort_by = isset($_GET['sort_by']) ? $_GET['sort_by'] : 'شناسه_یکتای_کاربرد';
$sort_order = isset($_GET['sort_order']) ? $_GET['sort_order'] : 'ASC';
$city_filter = isset($_GET['city_filter']) ? $_GET['city_filter'] : '';
$product_filter = isset($_GET['product_filter']) ? $_GET['product_filter'] : '';
$unique_id_filter = isset($_GET['unique_id_filter']) ? $_GET['unique_id_filter'] : '';

// Validate sort parameters - only allow fields that exist in the table
$allowed_sorts = ['شناسه_یکتای_کاربرد', 'فراوده', 'مقدار_تایید_شده_توسط_پخش', 'تاریخ_تایید_درخواست_توسط_پخش', 'نام_متقاضی', 'استان', 'شهرستان'];
$sort_by = in_array($sort_by, $allowed_sorts) ? $sort_by : 'شناسه_یکتای_کاربرد';
$sort_order = in_array(strtoupper($sort_order), ['ASC', 'DESC']) ? strtoupper($sort_order) : 'ASC';

// Build WHERE clause
$where_conditions = [];
$params = [];

if (!empty($product_filter)) {
    $where_conditions[] = "فراوده LIKE ?";
    $params[] = "%$product_filter%";
}

if (!empty($unique_id_filter)) {
    $where_conditions[] = "شناسه_یکتای_کاربرد LIKE ?";
    $params[] = "%$unique_id_filter%";
}

if (!empty($city_filter)) {
    $where_conditions[] = "شهرستان LIKE ?";
    $params[] = "%$city_filter%";
}

$where_sql = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";

// Get total count
$count_sql = "SELECT COUNT(*) FROM fuel_requests $where_sql";
$stmt = $pdo->prepare($count_sql);
$stmt->execute($params);
$total_records = $stmt->fetchColumn();

// Calculate pagination
$total_pages = ceil($total_records / $per_page);
$offset = ($page - 1) * $per_page;

// Get data - select fields that actually exist in the table
$sql = "SELECT 
            شناسه_یکتای_کاربرد, 
            فراوده, 
            مقدار_تایید_شده_توسط_پخش, 
            تاریخ_تایید_درخواست_توسط_پخش,
            نام_متقاضی,
            استان,
            شهرستان
        FROM fuel_requests 
        $where_sql 
        ORDER BY $sort_by $sort_order 
        LIMIT $offset, $per_page";
        
$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$records = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get unique values for filters
$products = $pdo->query("SELECT DISTINCT فراوده FROM fuel_requests WHERE فراوده IS NOT NULL ORDER BY فراوده")->fetchAll(PDO::FETCH_COLUMN);
$cities = $pdo->query("SELECT DISTINCT شهرستان FROM fuel_requests WHERE شهرستان IS NOT NULL ORDER BY شهرستان")->fetchAll(PDO::FETCH_COLUMN);
?>

<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta http-equiv="Content-Type" content="text/html; charset=utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>مدیریت درخواست های سوخت</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.8.0/font/bootstrap-icons.css">
    <style>
        body {
            font-family: 'Tahoma', 'Arial', sans-serif;
        }
        .table {
            font-family: 'Tahoma', 'Arial', sans-serif;
        }
    </style>
</head>
<body>
    <div class="container mt-4">
        <h2 class="text-center mb-4">مدیریت درخواست های سوخت</h2>
        
        <!-- Filters -->
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="card-title">فیلتر و مرتب سازی</h5>
            </div>
            <div class="card-body">
                <form method="GET" class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">شناسه یکتا</label>
                        <input type="text" class="form-control" name="unique_id_filter" value="<?= htmlspecialchars($unique_id_filter) ?>" placeholder="جستجو بر اساس شناسه یکتا">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">فراورده</label>
                        <input type="text" class="form-control" name="product_filter" value="<?= htmlspecialchars($product_filter) ?>" list="products" placeholder="جستجو بر اساس فراورده">
                        <datalist id="products">
                            <?php foreach ($products as $product): ?>
                                <option value="<?= htmlspecialchars($product) ?>">
                            <?php endforeach; ?>
                        </datalist>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">شهرستان</label>
                        <input type="text" class="form-control" name="city_filter" value="<?= htmlspecialchars($city_filter) ?>" list="cities" placeholder="جستجو بر اساس شهرستان">
                        <datalist id="cities">
                            <?php foreach ($cities as $city): ?>
                                <option value="<?= htmlspecialchars($city) ?>">
                            <?php endforeach; ?>
                        </datalist>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">مرتب سازی بر اساس</label>
                        <select class="form-select" name="sort_by">
                            <option value="شناسه_یکتای_کاربرد" <?= $sort_by == 'شناسه_یکتای_کاربرد' ? 'selected' : '' ?>>شناسه یکتا</option>
                            <option value="فراوده" <?= $sort_by == 'فراوده' ? 'selected' : '' ?>>فراورده</option>
                            <option value="مقدار_تایید_شده_توسط_پخش" <?= $sort_by == 'مقدار_تایید_شده_توسط_پخش' ? 'selected' : '' ?>>تایید پخش</option>
                            <option value="تاریخ_تایید_درخواست_توسط_پخش" <?= $sort_by == 'تاریخ_تایید_درخواست_توسط_پخش' ? 'selected' : '' ?>>تاریخ تایید پخش</option>
                            <option value="نام_متقاضی" <?= $sort_by == 'نام_متقاضی' ? 'selected' : '' ?>>نام متقاضی</option>
                            <option value="شهرستان" <?= $sort_by == 'شهرستان' ? 'selected' : '' ?>>شهرستان</option>
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">ترتیب</label>
                        <select class="form-select" name="sort_order">
                            <option value="ASC" <?= $sort_order == 'ASC' ? 'selected' : '' ?>>صعودی</option>
                            <option value="DESC" <?= $sort_order == 'DESC' ? 'selected' : '' ?>>نزولی</option>
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">تعداد در هر صفحه</label>
                        <select class="form-select" name="per_page">
                            <option value="10" <?= $per_page == 10 ? 'selected' : '' ?>>10</option>
                            <option value="20" <?= $per_page == 20 ? 'selected' : '' ?>>20</option>
                            <option value="50" <?= $per_page == 50 ? 'selected' : '' ?>>50</option>
                            <option value="100" <?= $per_page == 100 ? 'selected' : '' ?>>100</option>
                        </select>
                    </div>
                    <div class="col-12">
                        <button type="submit" class="btn btn-primary">اعمال فیلترها</button>
                        <a href="index.php" class="btn btn-secondary">پاک کردن فیلترها</a>
                        <a href="export.php?<?= http_build_query($_GET) ?>" class="btn btn-success">
                            <i class="bi bi-download"></i> خروجی Excel
                        </a>
                        <a href="upload.php" class="btn btn-info">آپلود فایل جدید</a>
                        <a href="remove.php" class="btn btn-warning">حذف داده های دیتابیس</a>
                    </div>
                </form>
            </div>
        </div>

        <!-- لینک‌های سریع تحلیل داده‌ها -->
        <div class="card mt-4">
            <div class="card-header">
                <h5 class="mb-0">دسترسی سریع به صفحات تحلیل</h5>
            </div>
            <div class="card-body">
                <div class="d-grid gap-2 d-md-flex justify-content-md-start">
                    <a href="consumer_analysis.php" class="btn btn-primary me-md-2">
                        <i class="bi bi-people-fill"></i> تحلیل تغییرات مصرف‌کنندگان
                    </a>
                    <a href="product_distribution.php" class="btn btn-success me-md-2">
                        <i class="bi bi-pie-chart-fill"></i> توزیع فراورده
                    </a>
                    <a href="consumption_pattern.php" class="btn btn-warning">
                        <i class="bi bi-graph-up-arrow"></i> تحلیل الگوی مصرف
                    </a>
                </div>
            </div>
        </div>

        <!-- Data Table -->
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="card-title">داده ها</h5>
                <span class="badge bg-primary">تعداد کل: <?= number_format($total_records) ?></span>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-striped table-bordered">
                        <thead class="table-dark">
                            <tr>
                                <th>شناسه یکتا</th>
                                <th>فراورده</th>
                                <th>نام متقاضی</th>
                                <th>استان</th>
                                <th>شهرستان</th>
                                <th>تایید پخش</th>
                                <th>تاریخ تایید پخش</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($records as $record): ?>
                            <tr>
                                <td><?= htmlspecialchars($record['شناسه_یکتای_کاربرد'] ?? '') ?></td>
                                <td><?= htmlspecialchars($record['فراوده'] ?? '') ?></td>
                                <td><?= htmlspecialchars($record['نام_متقاضی'] ?? '') ?></td>
                                <td><?= htmlspecialchars($record['استان'] ?? '') ?></td>
                                <td><?= htmlspecialchars($record['شهرستان'] ?? '') ?></td>
                                <td><?= number_format($record['مقدار_تایید_شده_توسط_پخش'] ?? 0, 2) ?></td>
                                <td>
                                    <?php 
                                    if (!empty($record['تاریخ_تایید_درخواست_توسط_پخش'])) {
                                        try {
                                            $carbon = \Carbon\Carbon::parse($record['تاریخ_تایید_درخواست_توسط_پخش']);
                                            $jalali = Jalalian::fromCarbon($carbon);
                                            echo $jalali->format('Y/m/d');
                                        } catch (Exception $e) {
                                            echo htmlspecialchars($record['تاریخ_تایید_درخواست_توسط_پخش']);
                                        }
                                    } else {
                                        echo '-';
                                    }
                                    ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Pagination -->
                <?php if ($total_pages > 1): ?>
                <nav aria-label="Page navigation">
                    <ul class="pagination justify-content-center">
                        <?php if ($page > 1): ?>
                            <li class="page-item">
                                <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $page - 1])) ?>">قبلی</a>
                            </li>
                        <?php endif; ?>

                        <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                            <li class="page-item <?= $i == $page ? 'active' : '' ?>">
                                <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $i])) ?>"><?= $i ?></a>
                            </li>
                        <?php endfor; ?>

                        <?php if ($page < $total_pages): ?>
                            <li class="page-item">
                                <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $page + 1])) ?>">بعدی</a>
                            </li>
                        <?php endif; ?>
                    </ul>
                </nav>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>