<?php
// sales_analysis.php
require_once 'config.php';
require_once 'vendor/autoload.php';

use Morilog\Jalali\Jalalian;
use Carbon\Carbon;

class SalesAnalysis {
    private $pdo;
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
        // غیرفعال کردن ONLY_FULL_GROUP_BY
        $this->pdo->exec("SET SESSION sql_mode = (SELECT REPLACE(@@sql_mode, 'ONLY_FULL_GROUP_BY', ''))");
    }
    
    // دریافت لیست فروشندگی‌ها
    // دریافت لیست فروشندگی‌ها
public function getSalesPoints() {
    $query = "
        SELECT 
            DISTINCT کد_فروش,
            MIN(نام_فروشندگی_انتخاب_شده_در_درخواست) as نام_فروشندگی,  -- تغییر به MIN
            MIN(شهرستان) as شهرستان,  -- تغییر به MIN
            MIN(استان) as استان        -- تغییر به MIN
        FROM fuel_requests 
        WHERE کد_فروش IS NOT NULL 
        AND کد_فروش != ''
        GROUP BY کد_فروش
        ORDER BY نام_فروشندگی
    ";
    
    $stmt = $this->pdo->prepare($query);
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// و در سایر توابعی که از MAX یا ANY_VALUE استفاده کرده‌اید، به MIN تغییر دهید
    
    // دریافت سال‌های موجود
    public function getAvailableYears() {
        $query = "
            SELECT DISTINCT تاریخ_تایید_درخواست_توسط_پخش 
            FROM fuel_requests 
            WHERE تاریخ_تایید_درخواست_توسط_پخش IS NOT NULL 
            ORDER BY تاریخ_تایید_درخواست_توسط_پخش DESC
        ";
        
        $stmt = $this->pdo->prepare($query);
        $stmt->execute();
        $dates = $stmt->fetchAll(PDO::FETCH_COLUMN);

        $years = [];
        foreach ($dates as $date) {
            try {
                $carbon = Carbon::parse($date);
                $jalali = Jalalian::fromCarbon($carbon);
                $shamsi_year = $jalali->getYear();
                if (!in_array($shamsi_year, $years)) {
                    $years[] = $shamsi_year;
                }
            } catch (Exception $e) {
                continue;
            }
        }

        rsort($years);
        return $years;
    }
    
    // تابع تبدیل تاریخ شمسی به میلادی
    public function getGregorianRange($shamsiYear, $period = 'yearly', $period_number = null) {
        try {
            $start_shamsi = null;
            $end_shamsi = null;

            switch($period) {
                case 'monthly':
                    if ($period_number && $period_number >= 1 && $period_number <= 12) {
                        $start_shamsi = new Jalalian($shamsiYear, $period_number, 1);
                        $end_shamsi = (new Jalalian($shamsiYear, $period_number, 1))->addMonths(1)->addDays(-1);
                    }
                    break;

                case 'quarterly':
                    if ($period_number && $period_number >= 1 && $period_number <= 4) {
                        $start_month = (($period_number - 1) * 3) + 1;
                        $end_month = $start_month + 2;
                        $start_shamsi = new Jalalian($shamsiYear, $start_month, 1);
                        $end_shamsi = (new Jalalian($shamsiYear, $end_month, 1))->addMonths(1)->addDays(-1);
                    }
                    break;

                case 'halfyearly':
                    if ($period_number && $period_number >= 1 && $period_number <= 2) {
                        $start_month = $period_number == 1 ? 1 : 7;
                        $end_month = $period_number == 1 ? 6 : 12;
                        $start_shamsi = new Jalalian($shamsiYear, $start_month, 1);
                        $end_shamsi = (new Jalalian($shamsiYear, $end_month, 1))->addMonths(1)->addDays(-1);
                    }
                    break;

                case 'yearly':
                default:
                    $start_shamsi = new Jalalian($shamsiYear, 1, 1);
                    $end_shamsi = (new Jalalian($shamsiYear, 12, 1))->addMonths(1)->addDays(-1);
                    break;
            }

            if ($start_shamsi && $end_shamsi) {
                return [
                    'start' => $start_shamsi->toCarbon()->format('Y-m-d H:i:s'),
                    'end' => $end_shamsi->toCarbon()->setTime(23, 59, 59)->format('Y-m-d H:i:s')
                ];
            }
        } catch (Exception $e) {
            return null;
        }
        return null;
    }
    
    // تحلیل فروش بر اساس دوره‌های زمانی
    public function getSalesByPeriod($filters) {
        $where_conditions = [];
        $params = [];
        
        // فیلتر فروشندگی
        if (!empty($filters['sales_codes']) && $filters['sales_codes'] !== 'all') {
            if (is_array($filters['sales_codes'])) {
                $placeholders = str_repeat('?,', count($filters['sales_codes']) - 1) . '?';
                $where_conditions[] = "کد_فروش IN ($placeholders)";
                $params = array_merge($params, $filters['sales_codes']);
            } else {
                $where_conditions[] = "کد_فروش = ?";
                $params[] = $filters['sales_codes'];
            }
        }
        
        // فیلتر شهرستان
        if (!empty($filters['city']) && $filters['city'] !== 'all') {
            $where_conditions[] = "شهرستان = ?";
            $params[] = $filters['city'];
        }
        
        // فیلتر فرآورده
        if (!empty($filters['product']) && $filters['product'] !== 'all') {
            $where_conditions[] = "فراوده = ?";
            $params[] = $filters['product'];
        }
        
        // فیلتر بخش مصرف
        if (!empty($filters['usage_section']) && $filters['usage_section'] !== 'all') {
            $where_conditions[] = "بخش_مصرف = ?";
            $params[] = $filters['usage_section'];
        }
        
        $where_clause = "";
        if (!empty($where_conditions)) {
            $where_clause = "WHERE " . implode(" AND ", $where_conditions);
        }
        
        // ساخت کوئری بر اساس نوع مقایسه
        if ($filters['comparison_type'] === 'corresponding') {
            // مقایسه متناظر
            return $this->getCorrespondingComparison($where_clause, $params, $filters);
        } else {
            // مقایسه پیوسته
            return $this->getContinuousComparison($where_clause, $params, $filters);
        }
    }
    
    private function getCorrespondingComparison($where_clause, $params, $filters) {
        $period_type = $filters['period_type'];
        $years = $filters['years'];
        
        $data_by_period = [];
        
        foreach ($years as $year) {
            $year_data = $this->getYearData($year, $period_type, $where_clause, $params);
            $data_by_period[$year] = $year_data;
        }
        
        // سازماندهی برای مقایسه متناظر
        return $this->organizeCorrespondingData($data_by_period, $period_type);
    }
    
    private function getContinuousComparison($where_clause, $params, $filters) {
        $period_type = $filters['period_type'];
        $years = $filters['years'];
        
        $all_data = [];
        
        foreach ($years as $year) {
            $year_data = $this->getYearData($year, $period_type, $where_clause, $params);
            foreach ($year_data as $period_data) {
                $all_data[] = $period_data;
            }
        }
        
        // سازماندهی برای مقایسه پیوسته
        return $this->organizeContinuousData($all_data, $period_type);
    }
    
    private function getYearData($year, $period_type, $where_clause, $params) {
        $periods = $this->getPeriodsForYear($period_type);
        $year_data = [];
        
        foreach ($periods as $period_number => $period_info) {
            $range = $this->getGregorianRange($year, $period_type, $period_number);
            
            if ($range) {
                $period_query = $this->buildPeriodQuery($where_clause, $range, $params);
                $period_data = $this->executePeriodQuery($period_query['query'], $period_query['params']);
                
                if (!empty($period_data)) {
                    foreach ($period_data as &$data) {
                        $data['year'] = $year;
                        $data['period_number'] = $period_number;
                        $data['period_name'] = $period_info['name'];
                        $data['period_label'] = $period_info['name'] . ' ' . $year;
                    }
                    
                    $year_data = array_merge($year_data, $period_data);
                }
            }
        }
        
        return $year_data;
    }
    
    private function getPeriodsForYear($period_type) {
        $periods = [];
        
        switch($period_type) {
            case 'monthly':
                $month_names = [
                    1 => 'فروردین', 2 => 'اردیبهشت', 3 => 'خرداد', 
                    4 => 'تیر', 5 => 'مرداد', 6 => 'شهریور',
                    7 => 'مهر', 8 => 'آبان', 9 => 'آذر',
                    10 => 'دی', 11 => 'بهمن', 12 => 'اسفند'
                ];
                for ($i = 1; $i <= 12; $i++) {
                    $periods[$i] = ['name' => $month_names[$i], 'number' => $i];
                }
                break;
                
            case 'quarterly':
                $quarter_names = [
                    1 => 'فصل اول', 2 => 'فصل دوم', 
                    3 => 'فصل سوم', 4 => 'فصل چهارم'
                ];
                for ($i = 1; $i <= 4; $i++) {
                    $periods[$i] = ['name' => $quarter_names[$i], 'number' => $i];
                }
                break;
                
            case 'halfyearly':
                $halfyear_names = [1 => 'نیمه اول', 2 => 'نیمه دوم'];
                for ($i = 1; $i <= 2; $i++) {
                    $periods[$i] = ['name' => $halfyear_names[$i], 'number' => $i];
                }
                break;
                
            case 'yearly':
                $periods[1] = ['name' => 'سال', 'number' => 1];
                break;
        }
        
        return $periods;
    }
    
    private function buildPeriodQuery($where_clause, $range, $params) {
        $base_params = $params;
        
        if (strpos($where_clause, 'WHERE') !== false) {
            $where_clause .= " AND تاریخ_تایید_درخواست_توسط_پخش BETWEEN ? AND ?";
        } else {
            $where_clause = "WHERE تاریخ_تایید_درخواست_توسط_پخش BETWEEN ? AND ?";
        }
        
        $query_params = array_merge($base_params, [$range['start'], $range['end']]);
        
        $query = "
            SELECT 
                کد_فروش,
                MAX(نام_فروشندگی_انتخاب_شده_در_درخواست) as نام_فروشندگی,
                MAX(شهرستان) as شهرستان,
                COUNT(*) as total_requests,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_volume,
                AVG(مقدار_تایید_شده_توسط_پخش) as average_volume,
                MAX(مقدار_تایید_شده_توسط_پخش) as max_volume,
                MIN(مقدار_تایید_شده_توسط_پخش) as min_volume,
                COUNT(DISTINCT کدملی_شناسه_ملی) as unique_customers,
                COUNT(DISTINCT فراوده) as product_variety,
                COUNT(DISTINCT بخش_مصرف) as section_variety
            FROM fuel_requests 
            {$where_clause}
            AND کد_فروش IS NOT NULL
            AND کد_فروش != ''
            GROUP BY کد_فروش
            ORDER BY total_volume DESC
        ";
        
        return ['query' => $query, 'params' => $query_params];
    }
    
    private function executePeriodQuery($query, $params) {
        $stmt = $this->pdo->prepare($query);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    private function organizeCorrespondingData($data_by_year, $period_type) {
        $organized = [];
        
        // پیدا کردن همه فروشندگی‌ها
        $all_sales_points = [];
        foreach ($data_by_year as $year_data) {
            foreach ($year_data as $data) {
                $sales_code = $data['کد_فروش'];
                if (!in_array($sales_code, $all_sales_points)) {
                    $all_sales_points[] = $sales_code;
                }
            }
        }
        
        // سازماندهی داده‌ها بر اساس دوره‌های متناظر
        foreach ($all_sales_points as $sales_code) {
            $sales_data = ['کد_فروش' => $sales_code];
            
            // پیدا کردن نام فروشندگی
            foreach ($data_by_year as $year_data) {
                foreach ($year_data as $data) {
                    if ($data['کد_فروش'] == $sales_code) {
                        $sales_data['نام_فروشندگی'] = $data['نام_فروشندگی'];
                        $sales_data['شهرستان'] = $data['شهرستان'];
                        break 2;
                    }
                }
            }
            
            // جمع‌آوری داده‌های هر دوره
            $periods_data = [];
            foreach ($data_by_year as $year => $year_data) {
                foreach ($year_data as $data) {
                    if ($data['کد_فروش'] == $sales_code) {
                        $period_key = $data['period_number'];
                        
                        if (!isset($periods_data[$period_key])) {
                            $periods_data[$period_key] = [
                                'period_number' => $period_key,
                                'period_name' => $data['period_name'],
                                'years' => []
                            ];
                        }
                        
                        $periods_data[$period_key]['years'][$year] = [
                            'year' => $year,
                            'total_volume' => $data['total_volume'] ?? 0,
                            'total_requests' => $data['total_requests'] ?? 0,
                            'average_volume' => $data['average_volume'] ?? 0,
                            'unique_customers' => $data['unique_customers'] ?? 0
                        ];
                    }
                }
            }
            
            $sales_data['periods'] = array_values($periods_data);
            $organized[] = $sales_data;
        }
        
        return $organized;
    }
    
    private function organizeContinuousData($all_data, $period_type) {
        $organized = [];
        
        // گروه‌بندی بر اساس فروشندگی
        $sales_groups = [];
        foreach ($all_data as $data) {
            $sales_code = $data['کد_فروش'];
            if (!isset($sales_groups[$sales_code])) {
                $sales_groups[$sales_code] = [
                    'کد_فروش' => $sales_code,
                    'نام_فروشندگی' => $data['نام_فروشندگی'] ?? '',
                    'شهرستان' => $data['شهرستان'] ?? '',
                    'periods' => []
                ];
            }
            
            $period_label = $data['period_label'];
            $sales_groups[$sales_code]['periods'][$period_label] = [
                'period_label' => $period_label,
                'year' => $data['year'],
                'period_number' => $data['period_number'],
                'period_name' => $data['period_name'],
                'total_volume' => $data['total_volume'] ?? 0,
                'total_requests' => $data['total_requests'] ?? 0,
                'average_volume' => $data['average_volume'] ?? 0,
                'unique_customers' => $data['unique_customers'] ?? 0
            ];
        }
        
        // تبدیل به آرایه و مرتب‌سازی
        foreach ($sales_groups as $sales_code => $sales_data) {
            // مرتب‌سازی دوره‌ها
            uasort($sales_data['periods'], function($a, $b) {
                return $a['year'] <=> $b['year'] ?: $a['period_number'] <=> $b['period_number'];
            });
            
            $sales_data['periods'] = array_values($sales_data['periods']);
            $organized[] = $sales_data;
        }
        
        return $organized;
    }
    
    // تحلیل رتبه‌بندی فروشندگی‌ها
public function getSalesRanking($filters) {
    $where_conditions = [];
    $params = [];
    
    // فیلتر تاریخ
    if (!empty($filters['start_date']) && !empty($filters['end_date'])) {
        $where_conditions[] = "تاریخ_تایید_درخواست_توسط_پخش BETWEEN ? AND ?";
        $params[] = $filters['start_date'];
        $params[] = $filters['end_date'];
    }
    
    // فیلتر شهرستان
    if (!empty($filters['city']) && $filters['city'] !== 'all') {
        $where_conditions[] = "شهرستان = ?";
        $params[] = $filters['city'];
    }
    
    // همیشه شرط‌های مربوط به کد فروش را اضافه می‌کنیم
    $where_conditions[] = "کد_فروش IS NOT NULL";
    $where_conditions[] = "کد_فروش != ''";
    
    $where_clause = "";
    if (!empty($where_conditions)) {
        $where_clause = "WHERE " . implode(" AND ", $where_conditions);
    }
    
    // کوئری اصلی بدون RANK()
    $query = "
        SELECT 
            کد_فروش,
            MAX(نام_فروشندگی_انتخاب_شده_در_درخواست) as نام_فروشندگی,
            MAX(شهرستان) as شهرستان,
            COUNT(*) as total_requests,
            SUM(مقدار_تایید_شده_توسط_پخش) as total_volume,
            AVG(مقدار_تایید_شده_توسط_پخش) as average_volume,
            COUNT(DISTINCT کدملی_شناسه_ملی) as unique_customers,
            COUNT(DISTINCT فراوده) as product_variety
        FROM fuel_requests 
        {$where_clause}
        GROUP BY کد_فروش
        ORDER BY total_volume DESC
        LIMIT 50
    ";
    
    $stmt = $this->pdo->prepare($query);
    $stmt->execute($params);
    $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // محاسبه رتبه‌ها در PHP
    return $this->calculateRanks($results);
}

private function calculateRanks($results) {
    if (empty($results)) {
        return [];
    }
    
    // کپی آرایه برای مرتب‌سازی بر اساس معیارهای مختلف
    $by_volume = $results;
    $by_requests = $results;
    $by_customers = $results;
    
    // مرتب‌سازی بر اساس حجم (نزولی)
    usort($by_volume, function($a, $b) {
        $volume_a = (float)($a['total_volume'] ?? 0);
        $volume_b = (float)($b['total_volume'] ?? 0);
        return $volume_b <=> $volume_a;
    });
    
    // مرتب‌سازی بر اساس تعداد درخواست (نزولی)
    usort($by_requests, function($a, $b) {
        $requests_a = (int)($a['total_requests'] ?? 0);
        $requests_b = (int)($b['total_requests'] ?? 0);
        return $requests_b <=> $requests_a;
    });
    
    // مرتب‌سازی بر اساس تعداد مشتریان (نزولی)
    usort($by_customers, function($a, $b) {
        $customers_a = (int)($a['unique_customers'] ?? 0);
        $customers_b = (int)($b['unique_customers'] ?? 0);
        return $customers_b <=> $customers_a;
    });
    
    // ایجاد نقشه برای رتبه‌ها
    $volume_ranks = [];
    $request_ranks = [];
    $customer_ranks = [];
    
    // محاسبه رتبه بر اساس حجم (با در نظر گرفتن مقادیر مساوی)
    $current_rank = 1;
    $previous_volume = null;
    $skip_rank = 0;
    
    foreach ($by_volume as $index => $item) {
        $current_volume = (float)($item['total_volume'] ?? 0);
        
        if ($previous_volume !== null && $current_volume == $previous_volume) {
            // اگر حجم با قبلی برابر بود، رتبه یکسان بده
            $volume_ranks[$item['کد_فروش']] = $current_rank;
            $skip_rank++;
        } else {
            // رتبه جدید
            $current_rank = $current_rank + $skip_rank;
            $skip_rank = 0;
            $volume_ranks[$item['کد_فروش']] = $current_rank;
        }
        
        $previous_volume = $current_volume;
        $current_rank++;
    }
    
    // محاسبه رتبه بر اساس درخواست‌ها
    $current_rank = 1;
    $previous_requests = null;
    $skip_rank = 0;
    
    foreach ($by_requests as $index => $item) {
        $current_requests = (int)($item['total_requests'] ?? 0);
        
        if ($previous_requests !== null && $current_requests == $previous_requests) {
            $request_ranks[$item['کد_فروش']] = $current_rank;
            $skip_rank++;
        } else {
            $current_rank = $current_rank + $skip_rank;
            $skip_rank = 0;
            $request_ranks[$item['کد_فروش']] = $current_rank;
        }
        
        $previous_requests = $current_requests;
        $current_rank++;
    }
    
    // محاسبه رتبه بر اساس مشتریان
    $current_rank = 1;
    $previous_customers = null;
    $skip_rank = 0;
    
    foreach ($by_customers as $index => $item) {
        $current_customers = (int)($item['unique_customers'] ?? 0);
        
        if ($previous_customers !== null && $current_customers == $previous_customers) {
            $customer_ranks[$item['کد_فروش']] = $current_rank;
            $skip_rank++;
        } else {
            $current_rank = $current_rank + $skip_rank;
            $skip_rank = 0;
            $customer_ranks[$item['کد_فروش']] = $current_rank;
        }
        
        $previous_customers = $current_customers;
        $current_rank++;
    }
    
    // اضافه کردن رتبه‌ها به نتایج
    foreach ($results as &$result) {
        $sales_code = $result['کد_فروش'] ?? '';
        if (!empty($sales_code)) {
            $result['volume_rank'] = $volume_ranks[$sales_code] ?? null;
            $result['requests_rank'] = $request_ranks[$sales_code] ?? null;
            $result['customers_rank'] = $customer_ranks[$sales_code] ?? null;
        } else {
            $result['volume_rank'] = null;
            $result['requests_rank'] = null;
            $result['customers_rank'] = null;
        }
    }
    
    // مرتب‌سازی نهایی بر اساس حجم (نزولی)
    usort($results, function($a, $b) {
        $volume_a = (float)($a['total_volume'] ?? 0);
        $volume_b = (float)($b['total_volume'] ?? 0);
        return $volume_b <=> $volume_a;
    });
    
    return $results;
}
    
    // تحلیل فروش بر اساس فرآورده برای هر فروشندگی
    public function getSalesByProduct($sales_code, $filters) {
        $where_conditions = ["کد_فروش = ?"];
        $params = [$sales_code];
        
        // فیلتر تاریخ
        if (!empty($filters['start_date']) && !empty($filters['end_date'])) {
            $where_conditions[] = "تاریخ_تایید_درخواست_توسط_پخش BETWEEN ? AND ?";
            $params[] = $filters['start_date'];
            $params[] = $filters['end_date'];
        }
        
        $where_clause = "WHERE " . implode(" AND ", $where_conditions);
        
        $query = "
            SELECT 
                فراوده,
                COUNT(*) as total_requests,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_volume,
                AVG(مقدار_تایید_شده_توسط_پخش) as average_volume,
                MAX(مقدار_تایید_شده_توسط_پخش) as max_volume,
                MIN(مقدار_تایید_شده_توسط_پخش) as min_volume,
                COUNT(DISTINCT کدملی_شناسه_ملی) as unique_customers,
                GROUP_CONCAT(DISTINCT بخش_مصرف) as usage_sections
            FROM fuel_requests 
            {$where_clause}
            GROUP BY فراوده
            ORDER BY total_volume DESC
        ";
        
        $stmt = $this->pdo->prepare($query);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // دریافت شهرستان‌های موجود
    public function getAvailableCities() {
        $query = "
            SELECT DISTINCT شهرستان 
            FROM fuel_requests 
            WHERE شهرستان IS NOT NULL 
            ORDER BY شهرستان
        ";
        
        $stmt = $this->pdo->prepare($query);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_COLUMN);
    }
    
    // دریافت فرآورده‌های موجود
    public function getAvailableProducts() {
        $query = "
            SELECT DISTINCT فراوده 
            FROM fuel_requests 
            WHERE فراوده IS NOT NULL 
            ORDER BY فراوده
        ";
        
        $stmt = $this->pdo->prepare($query);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_COLUMN);
    }
    
    // دریافت بخش‌های مصرف موجود
    public function getAvailableUsageSections() {
        $query = "
            SELECT DISTINCT بخش_مصرف 
            FROM fuel_requests 
            WHERE بخش_مصرف IS NOT NULL 
            ORDER BY بخش_مصرف
        ";
        
        $stmt = $this->pdo->prepare($query);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_COLUMN);
    }
}

// ایجاد شیء تحلیل
$analysis = new SalesAnalysis($pdo);

// دریافت داده‌های موجود
$sales_points = $analysis->getSalesPoints();
$available_years = $analysis->getAvailableYears();
$available_cities = $analysis->getAvailableCities();
$available_products = $analysis->getAvailableProducts();
$available_sections = $analysis->getAvailableUsageSections();

// تنظیم فیلترهای پیش‌فرض
$current_year = $available_years[0] ?? 1400;

$filters = [
    'years' => $_POST['years'] ?? [$current_year],
    'period_type' => $_POST['period_type'] ?? 'monthly',
    'comparison_type' => $_POST['comparison_type'] ?? 'continuous',
    'sales_codes' => $_POST['sales_codes'] ?? 'all',
    'city' => $_POST['city'] ?? 'all',
    'product' => $_POST['product'] ?? 'all',
    'usage_section' => $_POST['usage_section'] ?? 'all',
    'ranking_limit' => $_POST['ranking_limit'] ?? 20
];

// تولید گزارشات
$reports = [];
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // تحلیل فروش بر اساس دوره
    $reports['sales_by_period'] = $analysis->getSalesByPeriod($filters);
    
    // رتبه‌بندی فروشندگی‌ها
    $reports['sales_ranking'] = $analysis->getSalesRanking([
        'city' => $filters['city'],
        'start_date' => null,
        'end_date' => null
    ]);
    
    // اگر یک فروشندگی خاص انتخاب شده، تحلیل جزئیات آن
    if ($filters['sales_codes'] !== 'all' && !is_array($filters['sales_codes'])) {
        $reports['sales_details'] = $analysis->getSalesByProduct($filters['sales_codes'], [
            'start_date' => null,
            'end_date' => null
        ]);
    }
}
?>

<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>سیستم تحلیل فروشندگی‌ها</title>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        body {
            font-family: 'Tahoma', sans-serif;
            background-color: #f5f5f5;
            margin: 0;
            padding: 20px;
        }
        .container {
            max-width: 1400px;
            margin: 0 auto;
            background: white;
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
        }
        .filters {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        .filter-group {
            margin-bottom: 15px;
        }
        .filter-row {
            display: flex;
            gap: 15px;
            flex-wrap: wrap;
        }
        .filter-column {
            flex: 1;
            min-width: 200px;
        }
        label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
        }
        select, input {
            width: 100%;
            padding: 8px;
            border: 1px solid #ddd;
            border-radius: 4px;
        }
        .btn {
            background: #007bff;
            color: white;
            padding: 10px 20px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 16px;
        }
        .btn:hover {
            background: #0056b3;
        }
        .btn-success {
            background: #28a745;
        }
        .btn-success:hover {
            background: #218838;
        }
        .btn-warning {
            background: #ffc107;
            color: #212529;
        }
        .btn-warning:hover {
            background: #e0a800;
        }
        .report-section {
            margin: 30px 0;
            padding: 20px;
            border: 1px solid #ddd;
            border-radius: 8px;
        }
        .report-section h3 {
            color: #333;
            border-bottom: 2px solid #007bff;
            padding-bottom: 10px;
        }
        .chart-container {
            margin: 20px 0;
            padding: 15px;
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            height: 400px;
        }
        canvas {
            width: 100% !important;
            height: 100% !important;
        }
        .data-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 15px;
            font-size: 14px;
        }
        .data-table th {
            background-color: #007bff;
            color: white;
            text-align: center;
            padding: 12px;
            position: sticky;
            top: 0;
        }
        .data-table td {
            border: 1px solid #ddd;
            padding: 8px;
            text-align: center;
        }
        .data-table tr:nth-child(even) {
            background-color: #f8f9fa;
        }
        .data-table tr:hover {
            background-color: #e9ecef;
        }
        .summary-card {
            background: #f8f9fa;
            border: 1px solid #ddd;
            border-radius: 8px;
            padding: 20px;
            margin: 20px 0;
        }
        .stat-card {
            background: white;
            border: 1px solid #ddd;
            border-radius: 8px;
            padding: 15px;
            margin: 10px;
            text-align: center;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        .stat-value {
            font-size: 24px;
            font-weight: bold;
            color: #007bff;
            display: block;
            margin: 10px 0;
        }
        .stat-label {
            color: #6c757d;
            font-size: 14px;
        }
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin: 20px 0;
        }
        .badge {
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 12px;
            font-weight: bold;
        }
        .badge-primary {
            background-color: #007bff;
            color: white;
        }
        .badge-success {
            background-color: #28a745;
            color: white;
        }
        .badge-warning {
            background-color: #ffc107;
            color: #212529;
        }
        .badge-info {
            background-color: #17a2b8;
            color: white;
        }
        .rank-1 { background-color: gold !important; color: #333 !important; }
        .rank-2 { background-color: silver !important; color: #333 !important; }
        .rank-3 { background-color: #cd7f32 !important; color: white !important; }
        .rank-badge { 
            width: 30px; 
            height: 30px; 
            border-radius: 50%; 
            display: inline-flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
        }
        .no-data {
            text-align: center;
            padding: 40px;
            color: #6c757d;
            font-size: 18px;
        }
        .sales-point-card {
            border: 1px solid #ddd;
            border-radius: 8px;
            padding: 15px;
            margin: 10px 0;
            background: #f8f9fa;
        }
        .period-comparison {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 15px;
            margin: 15px 0;
        }
        .period-card {
            border: 1px solid #ddd;
            border-radius: 6px;
            padding: 10px;
            background: white;
        }
        .period-header {
            background: #e9ecef;
            padding: 8px;
            border-radius: 4px;
            margin-bottom: 8px;
            font-weight: bold;
        }
        .year-data {
            display: flex;
            justify-content: space-between;
            padding: 5px 0;
            border-bottom: 1px solid #eee;
        }
        .year-label {
            font-weight: bold;
            color: #495057;
        }
        .year-value {
            color: #007bff;
            font-weight: bold;
        }
        .toggle-details {
            color: #007bff;
            cursor: pointer;
            text-decoration: underline;
            font-size: 14px;
            margin-top: 10px;
            display: inline-block;
        }
        .details-content {
            display: none;
            margin-top: 10px;
            padding: 10px;
            background: #f8f9fa;
            border-radius: 4px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>📊 سیستم تحلیل فروشندگی‌ها</h1>
        
        <!-- فیلترهای تحلیل -->
        <div class="filters">
            <form method="POST">
                <div class="filter-row">
                    <div class="filter-column">
                        <label>انتخاب سال‌ها:</label>
                        <select name="years[]" multiple style="height: 120px;">
                            <?php foreach ($available_years as $year): ?>
                                <option value="<?= $year ?>" <?= in_array($year, $filters['years']) ? 'selected' : '' ?>>
                                    <?= $year ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="filter-column">
                        <label>بازه زمانی تحلیل:</label>
                        <select name="period_type" id="periodType">
                            <option value="monthly" <?= $filters['period_type'] == 'monthly' ? 'selected' : '' ?>>ماهانه</option>
                            <option value="quarterly" <?= $filters['period_type'] == 'quarterly' ? 'selected' : '' ?>>فصلی</option>
                            <option value="halfyearly" <?= $filters['period_type'] == 'halfyearly' ? 'selected' : '' ?>>۶ ماهه</option>
                            <option value="yearly" <?= $filters['period_type'] == 'yearly' ? 'selected' : '' ?>>سالانه</option>
                        </select>
                    </div>
                    
                    <div class="filter-column">
                        <label>نوع مقایسه:</label>
                        <select name="comparison_type">
                            <option value="continuous" <?= $filters['comparison_type'] == 'continuous' ? 'selected' : '' ?>>پیوسته</option>
                            <option value="corresponding" <?= $filters['comparison_type'] == 'corresponding' ? 'selected' : '' ?>>متناظر</option>
                        </select>
                    </div>
                </div>
                
                <div class="filter-row" style="margin-top: 15px;">
                    <div class="filter-column">
                        <label>انتخاب فروشندگی‌ها:</label>
                        <select name="sales_codes" id="salesCodes" style="height: 120px;">
                            <option value="all">همه فروشندگی‌ها</option>
                            <?php foreach ($sales_points as $point): ?>
                                <option value="<?= $point['کد_فروش'] ?>" 
                                    <?= $filters['sales_codes'] == $point['کد_فروش'] ? 'selected' : '' ?>>
                                    <?= $point['کد_فروش'] ?> - <?= $point['نام_فروشندگی'] ?> (<?= $point['شهرستان'] ?>)
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <small>برای انتخاب چند مورد از Ctrl استفاده کنید</small>
                    </div>
                    
                    <div class="filter-column">
                        <label>شهرستان:</label>
                        <select name="city">
                            <option value="all">همه شهرستان‌ها</option>
                            <?php foreach ($available_cities as $city): ?>
                                <option value="<?= $city ?>" <?= $filters['city'] == $city ? 'selected' : '' ?>>
                                    <?= $city ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="filter-column">
                        <label>فرآورده:</label>
                        <select name="product">
                            <option value="all">همه فرآورده‌ها</option>
                            <?php foreach ($available_products as $product): ?>
                                <option value="<?= $product ?>" <?= $filters['product'] == $product ? 'selected' : '' ?>>
                                    <?= $product ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="filter-column">
                        <label>بخش مصرف:</label>
                        <select name="usage_section">
                            <option value="all">همه بخش‌ها</option>
                            <?php foreach ($available_sections as $section): ?>
                                <option value="<?= $section ?>" <?= $filters['usage_section'] == $section ? 'selected' : '' ?>>
                                    <?= $section ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                
                <button type="submit" class="btn">اجرای تحلیل</button>
                <button type="button" class="btn btn-success" onclick="printReport()">چاپ گزارش</button>
                <button type="button" class="btn btn-warning" onclick="exportToExcel()">خروجی اکسل</button>
            </form>
        </div>
        
        <?php if (!empty($reports)): ?>
            <!-- خلاصه آمار -->
            <div class="summary-card">
                <h3>📈 خلاصه آمار</h3>
                <div class="stats-grid">
                    <div class="stat-card">
                        <span class="stat-label">تعداد فروشندگی‌ها</span>
                        <span class="stat-value"><?= count($reports['sales_by_period']) ?></span>
                    </div>
                    <div class="stat-card">
                        <span class="stat-label">تعداد سال‌های تحلیل</span>
                        <span class="stat-value"><?= count($filters['years']) ?></span>
                    </div>
                    <div class="stat-card">
                        <span class="stat-label">نوع تحلیل</span>
                        <span class="stat-value">
                            <?= $filters['comparison_type'] == 'continuous' ? 'پیوسته' : 'متناظر' ?>
                        </span>
                    </div>
                    <div class="stat-card">
                        <span class="stat-label">بازه زمانی</span>
                        <span class="stat-value">
                            <?= 
                                $filters['period_type'] == 'monthly' ? 'ماهانه' : 
                                ($filters['period_type'] == 'quarterly' ? 'فصلی' : 
                                ($filters['period_type'] == 'halfyearly' ? '۶ ماهه' : 'سالانه')) 
                            ?>
                        </span>
                    </div>
                </div>
            </div>
            
            <!-- نمودار مقایسه فروشندگی‌ها -->
            <div class="report-section">
                <h3>📊 نمودار مقایسه حجم فروش</h3>
                <div class="chart-container">
                    <canvas id="salesComparisonChart"></canvas>
                </div>
            </div>
            
            <!-- رتبه‌بندی فروشندگی‌ها -->
            <div class="report-section">
                <h3>🏆 رتبه‌بندی فروشندگی‌ها</h3>
                <div class="chart-container">
                    <canvas id="rankingChart"></canvas>
                </div>
                
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>رتبه</th>
                            <th>کد فروش</th>
                            <th>نام فروشندگی</th>
                            <th>شهرستان</th>
                            <th>حجم کل (لیتر)</th>
                            <th>تعداد درخواست</th>
                            <th>متوسط حجم</th>
                            <th>تعداد مشتریان</th>
                            <th>تنوع فرآورده</th>
                            <th>رتبه حجم</th>
                            <th>رتبه درخواست</th>
                            <th>رتبه مشتری</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($reports['sales_ranking'] as $index => $ranking): ?>
                            <tr>
                                <td>
                                    <span class="rank-badge 
                                        <?= $ranking['volume_rank'] == 1 ? 'rank-1' : 
                                           ($ranking['volume_rank'] == 2 ? 'rank-2' : 
                                           ($ranking['volume_rank'] == 3 ? 'rank-3' : 'badge-info')) ?>">
                                        <?= $ranking['volume_rank'] ?>
                                    </span>
                                </td>
                                <td><?= $ranking['کد_فروش'] ?></td>
                                <td><?= $ranking['نام_فروشندگی'] ?></td>
                                <td><?= $ranking['شهرستان'] ?></td>
                                <td><?= number_format($ranking['total_volume']) ?></td>
                                <td><?= number_format($ranking['total_requests']) ?></td>
                                <td><?= number_format($ranking['average_volume'], 1) ?></td>
                                <td><?= number_format($ranking['unique_customers']) ?></td>
                                <td><?= $ranking['product_variety'] ?></td>
                                <td><span class="badge badge-primary"><?= $ranking['volume_rank'] ?></span></td>
                                <td><span class="badge badge-warning"><?= $ranking['requests_rank'] ?></span></td>
                                <td><span class="badge badge-success"><?= $ranking['customers_rank'] ?></span></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            
            <!-- تحلیل جزئیات فروشندگی‌ها -->
            <div class="report-section">
                <h3>📋 تحلیل جزئیات فروشندگی‌ها</h3>
                
                <?php if ($filters['comparison_type'] == 'corresponding'): ?>
                    <!-- نمایش برای مقایسه متناظر -->
                    <?php foreach ($reports['sales_by_period'] as $sales): ?>
                        <div class="sales-point-card">
                            <h4>
                                <?= $sales['کد_فروش'] ?> - <?= $sales['نام_فروشندگی'] ?>
                                <span class="badge badge-info"><?= $sales['شهرستان'] ?></span>
                            </h4>
                            
                            <div class="period-comparison">
                                <?php foreach ($sales['periods'] as $period): ?>
                                    <div class="period-card">
                                        <div class="period-header"><?= $period['period_name'] ?></div>
                                        <?php foreach ($period['years'] as $year_data): ?>
                                            <div class="year-data">
                                                <span class="year-label">سال <?= $year_data['year'] ?>:</span>
                                                <span class="year-value"><?= number_format($year_data['total_volume']) ?> لیتر</span>
                                            </div>
                                        <?php endforeach; ?>
                                        <?php if (count($period['years']) > 1): ?>
                                            <?php
                                            $years = array_keys($period['years']);
                                            $first_year = $years[0];
                                            $last_year = end($years);
                                            $first_volume = $period['years'][$first_year]['total_volume'] ?? 0;
                                            $last_volume = $period['years'][$last_year]['total_volume'] ?? 0;
                                            $change = $last_volume - $first_volume;
                                            $change_percent = $first_volume > 0 ? ($change / $first_volume) * 100 : 0;
                                            ?>
                                            <div class="year-data" style="background: #e9ecef; margin-top: 8px; padding: 8px; border-radius: 4px;">
                                                <span class="year-label">تغییرات:</span>
                                                <span class="year-value" style="color: <?= $change >= 0 ? '#28a745' : '#dc3545' ?>;">
                                                    <?= $change >= 0 ? '+' : '' ?><?= number_format($change) ?> لیتر
                                                    (<?= $change >= 0 ? '+' : '' ?><?= number_format($change_percent, 1) ?>%)
                                                </span>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                            
                            <span class="toggle-details" onclick="toggleDetails('details-<?= $sales['کد_فروش'] ?>')">
                                نمایش جزئیات بیشتر
                            </span>
                            <div class="details-content" id="details-<?= $sales['کد_فروش'] ?>">
                                <table class="data-table">
                                    <thead>
                                        <tr>
                                            <th>دوره</th>
                                            <?php foreach ($filters['years'] as $year): ?>
                                                <th>سال <?= $year ?></th>
                                            <?php endforeach; ?>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($sales['periods'] as $period): ?>
                                            <tr>
                                                <td><?= $period['period_name'] ?></td>
                                                <?php foreach ($filters['years'] as $year): ?>
                                                    <td>
                                                        <?php if (isset($period['years'][$year])): ?>
                                                            <?= number_format($period['years'][$year]['total_volume']) ?> لیتر<br>
                                                            <small><?= $period['years'][$year]['total_requests'] ?> درخواست</small>
                                                        <?php else: ?>
                                                            <span style="color: #6c757d;">-</span>
                                                        <?php endif; ?>
                                                    </td>
                                                <?php endforeach; ?>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    <?php endforeach; ?>
                    
                <?php else: ?>
                    <!-- نمایش برای مقایسه پیوسته -->
                    <?php foreach ($reports['sales_by_period'] as $sales): ?>
                        <div class="sales-point-card">
                            <h4>
                                <?= $sales['کد_فروش'] ?> - <?= $sales['نام_فروشندگی'] ?>
                                <span class="badge badge-info"><?= $sales['شهرستان'] ?></span>
                            </h4>
                            
                            <table class="data-table">
                                <thead>
                                    <tr>
                                        <th>دوره زمانی</th>
                                        <th>حجم فروش (لیتر)</th>
                                        <th>تعداد درخواست</th>
                                        <th>متوسط حجم</th>
                                        <th>تعداد مشتریان</th>
                                        <th>روند</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($sales['periods'] as $period): ?>
                                        <tr>
                                            <td><?= $period['period_label'] ?></td>
                                            <td><?= number_format($period['total_volume']) ?></td>
                                            <td><?= number_format($period['total_requests']) ?></td>
                                            <td><?= number_format($period['average_volume'], 1) ?></td>
                                            <td><?= number_format($period['unique_customers']) ?></td>
                                            <td>
                                                <?php 
                                                // محاسبه روند
                                                $current_index = array_search($period, $sales['periods']);
                                                if ($current_index > 0) {
                                                    $prev_period = $sales['periods'][$current_index - 1];
                                                    $change = $period['total_volume'] - $prev_period['total_volume'];
                                                    $change_percent = $prev_period['total_volume'] > 0 ? 
                                                        ($change / $prev_period['total_volume']) * 100 : 0;
                                                    
                                                    if ($change > 0) {
                                                        echo '<span style="color: #28a745;">▲ ' . number_format($change_percent, 1) . '%</span>';
                                                    } elseif ($change < 0) {
                                                        echo '<span style="color: #dc3545;">▼ ' . number_format($change_percent, 1) . '%</span>';
                                                    } else {
                                                        echo '<span style="color: #6c757d;">-</span>';
                                                    }
                                                } else {
                                                    echo '<span style="color: #6c757d;">-</span>';
                                                }
                                                ?>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
            
            <!-- تحلیل جزئیات فروشندگی انتخاب شده -->
            <?php if (isset($reports['sales_details'])): ?>
                <div class="report-section">
                    <h3>⛽ تحلیل فروش بر اساس فرآورده</h3>
                    <?php 
                    $selected_sales = $filters['sales_codes'] !== 'all' ? $filters['sales_codes'] : '';
                    $sales_info = array_filter($sales_points, function($point) use ($selected_sales) {
                        return $point['کد_فروش'] == $selected_sales;
                    });
                    $sales_info = reset($sales_info);
                    ?>
                    
                    <?php if ($sales_info): ?>
                        <div class="summary-card">
                            <h4>فروشندگی: <?= $sales_info['کد_فروش'] ?> - <?= $sales_info['نام_فروشندگی'] ?></h4>
                            <p>شهرستان: <?= $sales_info['شهرستان'] ?></p>
                        </div>
                        
                        <div class="chart-container">
                            <canvas id="productDistributionChart"></canvas>
                        </div>
                        
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>فرآورده</th>
                                    <th>حجم فروش (لیتر)</th>
                                    <th>تعداد درخواست</th>
                                    <th>متوسط حجم</th>
                                    <th>بیشترین حجم</th>
                                    <th>کمترین حجم</th>
                                    <th>تعداد مشتریان</th>
                                    <th>بخش‌های مصرف</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($reports['sales_details'] as $product): ?>
                                    <tr>
                                        <td><?= $product['فراوده'] ?></td>
                                        <td><?= number_format($product['total_volume']) ?></td>
                                        <td><?= number_format($product['total_requests']) ?></td>
                                        <td><?= number_format($product['average_volume'], 1) ?></td>
                                        <td><?= number_format($product['max_volume']) ?></td>
                                        <td><?= number_format($product['min_volume']) ?></td>
                                        <td><?= number_format($product['unique_customers']) ?></td>
                                        <td><?= $product['usage_sections'] ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
            
        <?php else: ?>
            <div class="no-data">
                <p>برای مشاهده گزارش تحلیل، فیلترها را تنظیم و دکمه "اجرای تحلیل" را کلیک کنید.</p>
            </div>
        <?php endif; ?>
    </div>

    <script>
        // تابع نمایش/مخفی کردن جزئیات
        function toggleDetails(id) {
            const element = document.getElementById(id);
            if (element.style.display === 'block') {
                element.style.display = 'none';
                event.target.textContent = 'نمایش جزئیات بیشتر';
            } else {
                element.style.display = 'block';
                event.target.textContent = 'مخفی کردن جزئیات';
            }
        }
        
        // ایجاد نمودارها
        document.addEventListener('DOMContentLoaded', function() {
            <?php if (!empty($reports)): ?>
                // نمودار مقایسه فروشندگی‌ها
                <?php 
                $top_sales = array_slice($reports['sales_ranking'], 0, 10);
                if (!empty($top_sales)): 
                ?>
                    const salesCtx = document.getElementById('salesComparisonChart');
                    if (salesCtx) {
                        new Chart(salesCtx.getContext('2d'), {
                            type: 'bar',
                            data: {
                                labels: <?= json_encode(array_column($top_sales, 'نام_فروشندگی')) ?>,
                                datasets: [{
                                    label: 'حجم فروش (لیتر)',
                                    data: <?= json_encode(array_column($top_sales, 'total_volume')) ?>,
                                    backgroundColor: 'rgba(54, 162, 235, 0.5)',
                                    borderColor: 'rgba(54, 162, 235, 1)',
                                    borderWidth: 2
                                }]
                            },
                            options: {
                                responsive: true,
                                maintainAspectRatio: false,
                                plugins: {
                                    title: {
                                        display: true,
                                        text: '۱۰ فروشندگی برتر از نظر حجم فروش'
                                    }
                                },
                                scales: {
                                    y: {
                                        beginAtZero: true,
                                        title: {
                                            display: true,
                                            text: 'لیتر'
                                        }
                                    },
                                    x: {
                                        title: {
                                            display: true,
                                            text: 'فروشندگی'
                                        }
                                    }
                                }
                            }
                        });
                    }
                <?php endif; ?>
                
                // نمودار رتبه‌بندی
                <?php if (!empty($reports['sales_ranking'])): ?>
                    const rankingCtx = document.getElementById('rankingChart');
                    if (rankingCtx) {
                        const rankingData = <?= json_encode(array_slice($reports['sales_ranking'], 0, 15)) ?>;
                        
                        new Chart(rankingCtx.getContext('2d'), {
                            type: 'line',
                            data: {
                                labels: rankingData.map((r, i) => i + 1),
                                datasets: [{
                                    label: 'حجم فروش',
                                    data: rankingData.map(r => r.total_volume),
                                    borderColor: 'rgba(255, 99, 132, 1)',
                                    backgroundColor: 'rgba(255, 99, 132, 0.2)',
                                    borderWidth: 3,
                                    tension: 0.1,
                                    fill: true,
                                    yAxisID: 'y'
                                }, {
                                    label: 'تعداد درخواست',
                                    data: rankingData.map(r => r.total_requests),
                                    borderColor: 'rgba(54, 162, 235, 1)',
                                    backgroundColor: 'rgba(54, 162, 235, 0.2)',
                                    borderWidth: 2,
                                    tension: 0.1,
                                    fill: false,
                                    yAxisID: 'y1'
                                }]
                            },
                            options: {
                                responsive: true,
                                maintainAspectRatio: false,
                                plugins: {
                                    title: {
                                        display: true,
                                        text: 'روند فروش بر اساس رتبه'
                                    }
                                },
                                scales: {
                                    y: {
                                        type: 'linear',
                                        display: true,
                                        position: 'left',
                                        title: {
                                            display: true,
                                            text: 'حجم (لیتر)'
                                        }
                                    },
                                    y1: {
                                        type: 'linear',
                                        display: true,
                                        position: 'right',
                                        title: {
                                            display: true,
                                            text: 'تعداد درخواست'
                                        },
                                        grid: {
                                            drawOnChartArea: false
                                        }
                                    }
                                }
                            }
                        });
                    }
                <?php endif; ?>
                
                // نمودار توزیع فرآورده
                <?php if (isset($reports['sales_details'])): ?>
                    const productCtx = document.getElementById('productDistributionChart');
                    if (productCtx) {
                        new Chart(productCtx.getContext('2d'), {
                            type: 'pie',
                            data: {
                                labels: <?= json_encode(array_column($reports['sales_details'], 'فراوده')) ?>,
                                datasets: [{
                                    label: 'توزیع فرآورده',
                                    data: <?= json_encode(array_column($reports['sales_details'], 'total_volume')) ?>,
                                    backgroundColor: [
                                        'rgba(255, 99, 132, 0.5)',
                                        'rgba(54, 162, 235, 0.5)',
                                        'rgba(255, 206, 86, 0.5)',
                                        'rgba(75, 192, 192, 0.5)',
                                        'rgba(153, 102, 255, 0.5)',
                                        'rgba(255, 159, 64, 0.5)'
                                    ],
                                    borderWidth: 2
                                }]
                            },
                            options: {
                                responsive: true,
                                maintainAspectRatio: false,
                                plugins: {
                                    title: {
                                        display: true,
                                        text: 'توزیع فروش بر اساس فرآورده'
                                    }
                                }
                            }
                        });
                    }
                <?php endif; ?>
                
            <?php endif; ?>
        });
        
        // تابع چاپ گزارش
        function printReport() {
            window.print();
        }
        
        // تابع خروجی اکسل
        function exportToExcel() {
            alert('این قابلیت در نسخه فعلی پیاده‌سازی نشده است.');
        }
    </script>
</body>
</html>