<?php
// consumer_analysis.php
require_once 'config.php';
require_once 'vendor/autoload.php';

use Morilog\Jalali\Jalalian;

// دریافت پارامترهای فیلتر
$time_period = $_GET['time_period'] ?? 'monthly'; // monthly, quarterly, yearly
$change_type = $_GET['change_type'] ?? 'all'; // all, increase, decrease
$min_change = $_GET['min_change'] ?? 10; // حداقل درصد تغییر
$city_filter = $_GET['city_filter'] ?? '';
$product_filter = $_GET['product_filter'] ?? '';

// ساخت شرط‌های WHERE
$where_conditions = [];
$params = [];

if (!empty($city_filter)) {
    $where_conditions[] = "شهرستان = ?";
    $params[] = $city_filter;
}

if (!empty($product_filter)) {
    $where_conditions[] = "فراوده = ?";
    $params[] = $product_filter;
}

$where_sql = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";

// کوئری‌های مختلف بر اساس دوره زمانی - استفاده از تاریخ تایید پخش
switch($time_period) {
    case 'monthly':
        // برای ماهانه، از تاریخ تایید پخش استفاده می‌کنیم
        $period_format = "DATE_FORMAT(تاریخ_تایید_درخواست_توسط_پخش, '%Y-%m')";
        $period_alias = "period";
        break;
    case 'quarterly':
        $period_format = "CONCAT(YEAR(تاریخ_تایید_درخواست_توسط_پخش), '-Q', QUARTER(تاریخ_تایید_درخواست_توسط_پخش))";
        $period_alias = "period";
        break;
    case 'yearly':
        $period_format = "YEAR(تاریخ_تایید_درخواست_توسط_پخش)";
        $period_alias = "period";
        break;
    default:
        $period_format = "DATE_FORMAT(تاریخ_تایید_درخواست_توسط_پخش, '%Y-%m')";
        $period_alias = "period";
}

// 1. تحلیل تغییرات تعداد مصرف‌کنندگان بر اساس شهرستان (نسخه سازگار)
$consumer_change_city = $pdo->prepare("
    SELECT 
        city_data.شهرستان,
        city_data.period,
        city_data.consumer_count,
        prev_data.consumer_count as prev_consumer_count,
        CASE 
            WHEN prev_data.consumer_count > 0 
            THEN ROUND(((city_data.consumer_count - prev_data.consumer_count) / prev_data.consumer_count) * 100, 2)
            ELSE NULL
        END as change_percent
    FROM (
        SELECT 
            شهرستان,
            $period_format as period,
            COUNT(DISTINCT شناسه_یکتای_کاربرد) as consumer_count
        FROM fuel_requests 
        $where_sql
        GROUP BY شهرستان, $period_format
    ) as city_data
    LEFT JOIN (
        SELECT 
            شهرستان,
            $period_format as period,
            COUNT(DISTINCT شناسه_یکتای_کاربرد) as consumer_count
        FROM fuel_requests 
        $where_sql
        GROUP BY شهرستان, $period_format
    ) as prev_data ON city_data.شهرستان = prev_data.شهرستان 
        AND prev_data.period = (
            SELECT MAX($period_format) 
            FROM fuel_requests 
            WHERE شهرستان = city_data.شهرستان 
            AND $period_format < city_data.period
            $where_sql
        )
    ORDER BY city_data.شهرستان, city_data.period
");
$consumer_change_city->execute($params);
$city_data = $consumer_change_city->fetchAll();

// 2. تحلیل تغییرات تعداد مصرف‌کنندگان بر اساس فراورده (نسخه سازگار)
$consumer_change_product = $pdo->prepare("
    SELECT 
        product_data.product,
        product_data.period,
        product_data.consumer_count,
        prev_data.consumer_count as prev_consumer_count,
        CASE 
            WHEN prev_data.consumer_count > 0 
            THEN ROUND(((product_data.consumer_count - prev_data.consumer_count) / prev_data.consumer_count) * 100, 2)
            ELSE NULL
        END as change_percent
    FROM (
        SELECT 
            فراوده as product,
            $period_format as period,
            COUNT(DISTINCT شناسه_یکتای_کاربرد) as consumer_count
        FROM fuel_requests 
        $where_sql
        GROUP BY فراوده, $period_format
    ) as product_data
    LEFT JOIN (
        SELECT 
            فراوده as product,
            $period_format as period,
            COUNT(DISTINCT شناسه_یکتای_کاربرد) as consumer_count
        FROM fuel_requests 
        $where_sql
        GROUP BY فراوده, $period_format
    ) as prev_data ON product_data.product = prev_data.product 
        AND prev_data.period = (
            SELECT MAX($period_format) 
            FROM fuel_requests 
            WHERE فراوده = product_data.product 
            AND $period_format < product_data.period
            $where_sql
        )
    ORDER BY product_data.product, product_data.period
");
$consumer_change_product->execute($params);
$product_data = $consumer_change_product->fetchAll();

// 3. تحلیل ترکیبی شهرستان و فراورده (نسخه ساده‌تر)
$consumer_change_combined = $pdo->prepare("
    SELECT 
        شهرستان,
        فراوده as product,
        $period_format as period,
        COUNT(DISTINCT شناسه_یکتای_کاربرد) as consumer_count
    FROM fuel_requests 
    $where_sql
    GROUP BY شهرستان, فراوده, $period_format
    ORDER BY شهرستان, فراوده, $period_format
");
$consumer_change_combined->execute($params);
$combined_data_raw = $consumer_change_combined->fetchAll();

// محاسبه تغییرات به صورت دستی
$combined_data = [];
$previous_data = [];

foreach ($combined_data_raw as $row) {
    $key = $row['شهرستان'] . '|' . $row['product'];
    $current_period = $row['period'];
    
    if (isset($previous_data[$key])) {
        $prev_count = $previous_data[$key]['consumer_count'];
        $change_percent = $prev_count > 0 ? round((($row['consumer_count'] - $prev_count) / $prev_count) * 100, 2) : null;
        
        $combined_data[] = [
            'شهرستان' => $row['شهرستان'],
            'product' => $row['product'],
            'period' => $current_period,
            'consumer_count' => $row['consumer_count'],
            'prev_consumer_count' => $prev_count,
            'change_percent' => $change_percent
        ];
    }
    
    $previous_data[$key] = [
        'consumer_count' => $row['consumer_count'],
        'period' => $current_period
    ];
}

// تابع تبدیل دوره به شمسی
function convertPeriodToShamsi($period, $time_period) {
    try {
        if ($time_period == 'yearly') {
            // تبدیل سال میلادی به شمسی
            $carbon = \Carbon\Carbon::createFromDate($period, 1, 1);
            $jalali = \Morilog\Jalali\Jalalian::fromCarbon($carbon);
            return $jalali->getYear();
        } elseif ($time_period == 'monthly') {
            // تبدیل YYYY-MM به شمسی
            if (preg_match('/^(\d{4})-(\d{2})$/', $period, $matches)) {
                $carbon = \Carbon\Carbon::createFromDate($matches[1], $matches[2], 1);
                $jalali = \Morilog\Jalali\Jalalian::fromCarbon($carbon);
                return $jalali->format('Y/m');
            }
        } elseif ($time_period == 'quarterly') {
            // تبدیل YYYY-QX به شمسی
            if (preg_match('/^(\d{4})-Q(\d)$/', $period, $matches)) {
                $month = ($matches[2] - 1) * 3 + 1;
                $carbon = \Carbon\Carbon::createFromDate($matches[1], $month, 1);
                $jalali = \Morilog\Jalali\Jalalian::fromCarbon($carbon);
                return $jalali->getYear() . '-Q' . $matches[2];
            }
        }
    } catch (Exception $e) {
        return $period;
    }
    return $period;
}

// دریافت لیست شهرستان‌ها و فراورده‌ها برای فیلتر
$cities = $pdo->query("SELECT DISTINCT شهرستان FROM fuel_requests WHERE شهرستان IS NOT NULL ORDER BY شهرستان")->fetchAll(PDO::FETCH_COLUMN);
$products = $pdo->query("SELECT DISTINCT فراوده FROM fuel_requests WHERE فراوده IS NOT NULL ORDER BY فراوده")->fetchAll(PDO::FETCH_COLUMN);
?>

<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>تحلیل تغییرات مصرف‌کنندگان</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        body { font-family: 'Tahoma', 'Arial', sans-serif; background-color: #f8f9fa; }
        .card { margin-bottom: 20px; }
        .increase { color: #dc3545; font-weight: bold; }
        .decrease { color: #28a745; font-weight: bold; }
        .table-hover tbody tr:hover { background-color: rgba(0,0,0,.075); }
        .collapsible-section { margin-bottom: 15px; }
        .section-header { 
            cursor: pointer; 
            padding: 15px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 8px;
            margin-bottom: 10px;
            transition: all 0.3s ease;
        }
        .section-header:hover {
            background: linear-gradient(135deg, #764ba2 0%, #667eea 100%);
        }
        .section-header h5 {
            margin: 0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .section-content {
            display: none;
            padding: 20px;
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .section-content.show {
            display: block;
        }
        .toggle-icon {
            transition: transform 0.3s ease;
        }
        .toggle-icon.rotated {
            transform: rotate(180deg);
        }
    </style>
</head>
<body>
    <div class="container-fluid mt-4">
        <div class="row">
            <div class="col-12">
                <div class="card shadow">
                    <div class="card-header bg-primary text-white">
                        <h4 class="mb-0">تحلیل تغییرات تعداد مصرف‌کنندگان</h4>
                    </div>
                    <div class="card-body">
                        <!-- فیلترها -->
                        <div class="collapsible-section">
                            <div class="section-header" onclick="toggleSection('filters-section')">
                                <h5>
                                    <span>فیلترها و تنظیمات</span>
                                    <i class="bi bi-chevron-down toggle-icon" id="filters-icon"></i>
                                </h5>
                            </div>
                            <div class="section-content show" id="filters-section">
                                <form method="GET" class="row g-3 mb-4">
                                    <div class="col-md-3">
                                        <label class="form-label">دوره زمانی</label>
                                        <select class="form-select" name="time_period">
                                            <option value="monthly" <?= $time_period == 'monthly' ? 'selected' : '' ?>>ماهانه</option>
                                            <option value="quarterly" <?= $time_period == 'quarterly' ? 'selected' : '' ?>>فصلی</option>
                                            <option value="yearly" <?= $time_period == 'yearly' ? 'selected' : '' ?>>سالانه</option>
                                        </select>
                                    </div>
                                    <div class="col-md-3">
                                        <label class="form-label">شهرستان</label>
                                        <select class="form-select" name="city_filter">
                                            <option value="">همه شهرستان‌ها</option>
                                            <?php foreach($cities as $city): ?>
                                                <option value="<?= $city ?>" <?= $city == $city_filter ? 'selected' : '' ?>><?= $city ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    <div class="col-md-3">
                                        <label class="form-label">فراورده</label>
                                        <select class="form-select" name="product_filter">
                                            <option value="">همه فراورده‌ها</option>
                                            <?php foreach($products as $product): ?>
                                                <option value="<?= $product ?>" <?= $product == $product_filter ? 'selected' : '' ?>><?= $product ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    <div class="col-md-3">
                                        <label class="form-label">حداقل درصد تغییر</label>
                                        <input type="number" class="form-control" name="min_change" value="<?= $min_change ?>" step="5">
                                    </div>
                                    <div class="col-12">
                                        <button type="submit" class="btn btn-primary">اعمال فیلترها</button>
                                        <a href="consumer_analysis.php" class="btn btn-secondary">پاک کردن فیلترها</a>
                                    </div>
                                </form>
                            </div>
                        </div>

                        <!-- آمار کلی -->
                        <div class="collapsible-section">
                            <div class="section-header" onclick="toggleSection('stats-section')">
                                <h5>
                                    <span>آمار کلی</span>
                                    <i class="bi bi-chevron-down toggle-icon" id="stats-icon"></i>
                                </h5>
                            </div>
                            <div class="section-content" id="stats-section">
                                <div class="row mb-4">
                                    <div class="col-md-4">
                                        <div class="card text-white bg-info">
                                            <div class="card-body text-center">
                                                <h4><?= count($city_data) ?></h4>
                                                <p class="mb-0">تعداد رکوردهای تحلیل شده</p>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="card text-white bg-success">
                                            <div class="card-body text-center">
                                                <h4><?= count(array_filter($city_data, function($item) { return $item['change_percent'] > 0; })) ?></h4>
                                                <p class="mb-0">موارد افزایش</p>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="card text-white bg-danger">
                                            <div class="card-body text-center">
                                                <h4><?= count(array_filter($city_data, function($item) { return $item['change_percent'] < 0; })) ?></h4>
                                                <p class="mb-0">موارد کاهش</p>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- نمودار تغییرات -->
                        <div class="collapsible-section">
                            <div class="section-header" onclick="toggleSection('charts-section')">
                                <h5>
                                    <span>نمودارهای تغییرات</span>
                                    <i class="bi bi-chevron-down toggle-icon" id="charts-icon"></i>
                                </h5>
                            </div>
                            <div class="section-content" id="charts-section">
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="card">
                                            <div class="card-header">تغییرات بر اساس شهرستان</div>
                                            <div class="card-body">
                                                <canvas id="cityChart" height="300"></canvas>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="card">
                                            <div class="card-header">تغییرات بر اساس فراورده</div>
                                            <div class="card-body">
                                                <canvas id="productChart" height="300"></canvas>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- جداول داده -->
                        <div class="collapsible-section">
                            <div class="section-header" onclick="toggleSection('tables-section')">
                                <h5>
                                    <span>جداول داده‌ها</span>
                                    <i class="bi bi-chevron-down toggle-icon" id="tables-icon"></i>
                                </h5>
                            </div>
                            <div class="section-content" id="tables-section">
                                <div class="row mt-4">
                                    <div class="col-md-6">
                                        <div class="card">
                                            <div class="card-header">تغییرات شهرستان‌ها</div>
                                            <div class="card-body">
                                                <div class="table-responsive">
                                                    <table class="table table-bordered table-hover">
                                                        <thead class="table-dark">
                                                            <tr>
                                                                <th>شهرستان</th>
                                                                <th>دوره (شمسی)</th>
                                                                <th>تعداد مصرف‌کننده</th>
                                                                <th>تعداد دوره قبل</th>
                                                                <th>درصد تغییر</th>
                                                            </tr>
                                                        </thead>
                                                        <tbody>
                                                            <?php foreach($city_data as $row): ?>
                                                                <?php if($row['change_percent'] !== null && abs($row['change_percent']) >= $min_change): ?>
                                                                <tr>
                                                                    <td><?= $row['شهرستان'] ?></td>
                                                                    <td><?= convertPeriodToShamsi($row['period'], $time_period) ?></td>
                                                                    <td><?= $row['consumer_count'] ?></td>
                                                                    <td><?= $row['prev_consumer_count'] ?? '-' ?></td>
                                                                    <td class="<?= $row['change_percent'] > 0 ? 'increase' : 'decrease' ?>">
                                                                        <?= $row['change_percent'] ?>%
                                                                    </td>
                                                                </tr>
                                                                <?php endif; ?>
                                                            <?php endforeach; ?>
                                                        </tbody>
                                                    </table>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="card">
                                            <div class="card-header">تغییرات فراورده‌ها</div>
                                            <div class="card-body">
                                                <div class="table-responsive">
                                                    <table class="table table-bordered table-hover">
                                                        <thead class="table-dark">
                                                            <tr>
                                                                <th>فراورده</th>
                                                                <th>دوره (شمسی)</th>
                                                                <th>تعداد مصرف‌کننده</th>
                                                                <th>تعداد دوره قبل</th>
                                                                <th>درصد تغییر</th>
                                                            </tr>
                                                        </thead>
                                                        <tbody>
                                                            <?php foreach($product_data as $row): ?>
                                                                <?php if($row['change_percent'] !== null && abs($row['change_percent']) >= $min_change): ?>
                                                                <tr>
                                                                    <td><?= $row['product'] ?></td>
                                                                    <td><?= convertPeriodToShamsi($row['period'], $time_period) ?></td>
                                                                    <td><?= $row['consumer_count'] ?></td>
                                                                    <td><?= $row['prev_consumer_count'] ?? '-' ?></td>
                                                                    <td class="<?= $row['change_percent'] > 0 ? 'increase' : 'decrease' ?>">
                                                                        <?= $row['change_percent'] ?>%
                                                                    </td>
                                                                </tr>
                                                                <?php endif; ?>
                                                            <?php endforeach; ?>
                                                        </tbody>
                                                    </table>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- داده‌های ترکیبی -->
                        <div class="collapsible-section">
                            <div class="section-header" onclick="toggleSection('combined-section')">
                                <h5>
                                    <span>داده‌های ترکیبی شهرستان و فراورده</span>
                                    <i class="bi bi-chevron-down toggle-icon" id="combined-icon"></i>
                                </h5>
                            </div>
                            <div class="section-content" id="combined-section">
                                <div class="row mt-4">
                                    <div class="col-12">
                                        <div class="card">
                                            <div class="card-header">داده‌های ترکیبی شهرستان و فراورده</div>
                                            <div class="card-body">
                                                <div class="table-responsive">
                                                    <table class="table table-bordered table-hover">
                                                        <thead class="table-dark">
                                                            <tr>
                                                                <th>شهرستان</th>
                                                                <th>فراورده</th>
                                                                <th>دوره (شمسی)</th>
                                                                <th>تعداد مصرف‌کننده</th>
                                                                <th>تعداد دوره قبل</th>
                                                                <th>درصد تغییر</th>
                                                            </tr>
                                                        </thead>
                                                        <tbody>
                                                            <?php foreach($combined_data as $row): ?>
                                                                <?php if($row['change_percent'] !== null && abs($row['change_percent']) >= $min_change): ?>
                                                                <tr>
                                                                    <td><?= $row['شهرستان'] ?></td>
                                                                    <td><?= $row['product'] ?></td>
                                                                    <td><?= convertPeriodToShamsi($row['period'], $time_period) ?></td>
                                                                    <td><?= $row['consumer_count'] ?></td>
                                                                    <td><?= $row['prev_consumer_count'] ?? '-' ?></td>
                                                                    <td class="<?= $row['change_percent'] > 0 ? 'increase' : 'decrease' ?>">
                                                                        <?= $row['change_percent'] ?>%
                                                                    </td>
                                                                </tr>
                                                                <?php endif; ?>
                                                            <?php endforeach; ?>
                                                        </tbody>
                                                    </table>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- اضافه کردن آیکون‌های Bootstrap Icons -->
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.8.0/font/bootstrap-icons.css">

    <script>
        // تابع toggle برای بخش‌ها
        function toggleSection(sectionId) {
            const section = document.getElementById(sectionId);
            const icon = document.getElementById(sectionId.replace('-section', '-icon'));
            
            section.classList.toggle('show');
            icon.classList.toggle('rotated');
        }

        // آماده‌سازی داده‌ها برای نمودارها
        const cityData = <?= json_encode($city_data) ?>;
        const productData = <?= json_encode($product_data) ?>;

        // گروه‌بندی داده‌های شهرستان
        const cityGroups = {};
        cityData.forEach(item => {
            if (!cityGroups[item.شهرستان]) {
                cityGroups[item.شهرستان] = [];
            }
            cityGroups[item.شهرستان].push(item);
        });

        // گروه‌بندی داده‌های فراورده
        const productGroups = {};
        productData.forEach(item => {
            if (!productGroups[item.product]) {
                productGroups[item.product] = [];
            }
            productGroups[item.product].push(item);
        });

        // نمودار شهرستان‌ها
        const cityCtx = document.getElementById('cityChart').getContext('2d');
        const cityChart = new Chart(cityCtx, {
            type: 'line',
            data: {
                labels: [...new Set(cityData.map(item => item.period))].sort(),
                datasets: Object.keys(cityGroups).map(city => {
                    const color = '#' + Math.floor(Math.random()*16777215).toString(16);
                    return {
                        label: city,
                        data: cityGroups[city].map(item => item.change_percent || 0),
                        borderColor: color,
                        backgroundColor: color + '20',
                        tension: 0.1,
                        fill: false
                    };
                })
            },
            options: {
                responsive: true,
                plugins: {
                    title: {
                        display: true,
                        text: 'درصد تغییرات تعداد مصرف‌کنندگان بر اساس شهرستان'
                    }
                },
                scales: {
                    y: {
                        title: {
                            display: true,
                            text: 'درصد تغییر'
                        }
                    }
                }
            }
        });

        // نمودار فراورده‌ها
        const productCtx = document.getElementById('productChart').getContext('2d');
        const productChart = new Chart(productCtx, {
            type: 'bar',
            data: {
                labels: [...new Set(productData.map(item => item.period))].sort(),
                datasets: Object.keys(productGroups).map(product => {
                    const color = '#' + Math.floor(Math.random()*16777215).toString(16);
                    return {
                        label: product,
                        data: productGroups[product].map(item => item.change_percent || 0),
                        backgroundColor: color,
                    };
                })
            },
            options: {
                responsive: true,
                plugins: {
                    title: {
                        display: true,
                        text: 'درصد تغییرات تعداد مصرف‌کنندگان بر اساس فراورده'
                    }
                },
                scales: {
                    y: {
                        title: {
                            display: true,
                            text: 'درصد تغییر'
                        }
                    }
                }
            }
        });

        // حالت پیش‌فرض: فقط فیلترها باز باشند
        document.addEventListener('DOMContentLoaded', function() {
            // بستن تمام بخش‌ها به جز فیلترها
            const sections = ['stats-section', 'charts-section', 'tables-section', 'combined-section'];
            sections.forEach(section => {
                const sectionElement = document.getElementById(section);
                const icon = document.getElementById(section.replace('-section', '-icon'));
                if (sectionElement && icon) {
                    sectionElement.classList.remove('show');
                    icon.classList.remove('rotated');
                }
            });
        });
    </script>
</body>
</html>