<?php
// upload_bulk.php

// 1. تنظیمات زمان اجرا و حافظه
set_time_limit(0);  // غیرمحدود
ini_set('memory_limit', '4096M');  // 2GB حافظه
ini_set('max_input_time', '0');
ini_set('max_execution_time', '0');

ignore_user_abort(true);
ob_implicit_flush(true);

// 2. حالت خطاگیری را فعال کنید
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once 'config.php';
require_once 'vendor/autoload.php';

use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Shared\Date as PhpSpreadsheetDate;
use Morilog\Jalali\Jalalian;

// 3. توابع کمکی (از فایل اصلی کپی شده)
function convertJalaliToGregorian($jalaliDate) {
    if (empty($jalaliDate) || $jalaliDate === null) return null;
    
    $jalaliDate = trim($jalaliDate);
    
    // اگر قبلاً تاریخ میلادی است
    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $jalaliDate)) {
        return $jalaliDate;
    }
    
    // اگر عدد اکسل است (مثلاً 45105)
    if (is_numeric($jalaliDate) && $jalaliDate > 1000) {
        try {
            return PhpSpreadsheetDate::excelToDateTimeObject($jalaliDate)->format('Y-m-d');
        } catch (Exception $e) {
            return null;
        }
    }
    
    // اعتبارسنجی دقیق‌تر
    if (!preg_match('/^(\d{4})[\/\-\.](\d{1,2})[\/\-\.](\d{1,2})$/', $jalaliDate, $matches)) {
        return null;
    }
    
    $year = (int)$matches[1];
    $month = (int)$matches[2];
    $day = (int)$matches[3];
    
    // اعتبارسنجی اولیه
    if ($year < 1300 || $year > 1500 || $month < 1 || $month > 12 || $day < 1 || $day > 31) {
        return null;
    }
    
    // بررسی تعداد روزهای هر ماه شمسی
    $daysInMonth = [
        1 => 31, 2 => 31, 3 => 31, 4 => 31, 5 => 31, 6 => 31,
        7 => 30, 8 => 30, 9 => 30, 10 => 30, 11 => 30, 12 => 29
    ];
    
    // بررسی کبیسه بودن سال برای اسفند
    if ($month == 12) {
        try {
            $leapYears = [1, 5, 9, 13, 17, 22, 26, 30];
            $remainder = $year % 33;
            if (in_array($remainder, $leapYears)) {
                $daysInMonth[12] = 30;
            }
        } catch (Exception $e) {
            $daysInMonth[12] = 29;
        }
    }
    
    // اصلاح روز اگر بیشتر از روزهای ماه باشد
    if ($day > $daysInMonth[$month]) {
        $day = $daysInMonth[$month];
    }
    
    // ساخت تاریخ معتبر
    $validJalaliDate = sprintf('%04d/%02d/%02d', $year, $month, $day);
    
    try {
        $separator = preg_match('/^(\d{4})([\/\-\.])(\d{1,2})\2(\d{1,2})$/', $jalaliDate, $sepMatches) ? $sepMatches[2] : '/';
        $format = $separator == '/' ? 'Y/m/d' : ($separator == '-' ? 'Y-m-d' : 'Y.m.d');
        
        $jalalian = Jalalian::fromFormat($format, $validJalaliDate);
        return $jalalian->toCarbon()->format('Y-m-d');
        
    } catch (Exception $e) {
        error_log("Error converting Jalali date '$jalaliDate' to Gregorian: " . $e->getMessage());
        return null;
    }
}

// 4. تابع isDuplicate بهینه‌شده با کش
function isDuplicate($pdo, $unique_id, &$cache = []) {
    if (isset($cache[$unique_id])) {
        return true;
    }
    
    static $existing_ids = null;
    
    // فقط یک بار همه IDها را از دیتابیس بگیریم
    if ($existing_ids === null) {
        $stmt = $pdo->prepare("SELECT شماره_درخواست FROM fuel_requests");
        $stmt->execute();
        $existing_ids = array_flip($stmt->fetchAll(PDO::FETCH_COLUMN));
    }
    
    $is_duplicate = isset($existing_ids[$unique_id]);
    
    if ($is_duplicate) {
        $cache[$unique_id] = true;
    }
    
    return $is_duplicate;
}

// 5. تابع mapColumns بهینه‌شده
function mapColumns($headerRow, $column_mapping) {
    $column_order = [];
    $used_db_columns = [];
    
    foreach ($headerRow as $index => $header) {
        $header_clean = trim($header);
        $header_lower = preg_replace('/[_\s]/', '', strtolower($header_clean));
        $found = false;
        
        // جستجوی مستقیم
        if (isset($column_mapping[$header_clean])) {
            $db_column = $column_mapping[$header_clean];
            if (!in_array($db_column, $used_db_columns)) {
                $column_order[$index] = $db_column;
                $used_db_columns[] = $db_column;
                $found = true;
            }
        }
        
        // جستجو با similarity
        if (!$found) {
            $best_match = null;
            $best_similarity = 0;
            
            foreach ($column_mapping as $excel_header => $db_column) {
                if (in_array($db_column, $used_db_columns)) continue;
                
                similar_text($header_lower, preg_replace('/[_\s]/', '', strtolower($excel_header)), $percent);
                
                if ($percent > $best_similarity && $percent > 95) {
                    $best_similarity = $percent;
                    $best_match = $db_column;
                }
            }
            
            if ($best_match) {
                $column_order[$index] = $best_match;
                $used_db_columns[] = $best_match;
                $found = true;
            }
        }
        
        if (!$found) {
            $column_order[$index] = null;
        }
    }
    
    return $column_order;
}

// 6. تابع insertBatch برای وارد کردن دسته‌ای
function insertBatch($pdo, $data_batch, $columns) {
    if (empty($data_batch)) return 0;
    
    $values = [];
    $placeholders = [];
    $params = [];
    
    foreach ($data_batch as $row) {
        $row_placeholders = [];
        foreach ($columns as $col) {
            $row_placeholders[] = '?';
            $params[] = $row[$col] ?? null;
        }
        $placeholders[] = '(' . implode(', ', $row_placeholders) . ')';
    }
    
    $columns_str = implode(', ', $columns);
    $placeholders_str = implode(', ', $placeholders);
    
    $sql = "INSERT IGNORE INTO fuel_requests ($columns_str) VALUES $placeholders_str";
    
    try {
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->rowCount();
    } catch (PDOException $e) {
        error_log("Batch insert failed: " . $e->getMessage());
        return 0;
    }
}

// 7. تابع پردازش یک فایل اکسل
function processExcelFile($file_path, $pdo, $skip_duplicates = true) {
    $results = [
        'filename' => basename($file_path),
        'total_rows' => 0,
        'imported' => 0,
        'duplicates' => 0,
        'errors' => [],
        'error_count' => 0
    ];
    
    try {
        // بارگذاری فایل
        $inputFileType = IOFactory::identify($file_path);
        $reader = IOFactory::createReader($inputFileType);
        $reader->setReadDataOnly(true);
        $reader->setReadEmptyCells(false);
        
        $spreadsheet = $reader->load($file_path);
        $worksheet = $spreadsheet->getActiveSheet();
        
        // خواندن داده‌ها
        $rows = $worksheet->toArray();
        $headerRow = array_shift($rows);
        $results['total_rows'] = count($rows);
        
        // مپ کردن ستون‌ها
        global $column_mapping;
        $column_order = mapColumns($headerRow, $column_mapping);
        
        // فیلتر کردن ستون‌های نامعتبر
        $db_columns = [];
        foreach ($column_order as $index => $db_column) {
            if ($db_column !== null) {
                $db_columns[$index] = $db_column;
            }
        }
        
        if (empty($db_columns)) {
            $results['errors'][] = "هیچ ستون معتبری شناسایی نشد";
            return $results;
        }
        
        $batch_size = 1000;
        $data_batch = [];
        $duplicate_cache = [];
        
        // لیست فیلدها
        $dateFields = ['تاریخ_تایید_درخواست_توسط_پخش'];
        $numericFields = [
            'مقداری_قابل_تحویل_از_فروشندگی', 'مقدار_تحویل_شده_درخواست_از_فروشندگی',
            'مقدار_سوخت_یارانه_ای_درخواستی', 'مقدار_درخواستی_متقاضی',
            'مقدار_تایید_شده_توسط_متولی_اول', 'مقدار_تایید_شده_توسط_پخش'
        ];
        
        // پردازش ردیف‌ها
        foreach ($rows as $index => $row) {
            $row_number = $index + 2;
            
            try {
                // آماده‌سازی داده
                $data = [];
                foreach ($db_columns as $col_index => $db_column) {
                    $value = $row[$col_index] ?? null;
                    
                    // تبدیل تاریخ‌ها
                    if (in_array($db_column, $dateFields) && !empty($value)) {
                        $value = convertJalaliToGregorian($value);
                    }
                    
                    // تبدیل عددی
                    if (in_array($db_column, $numericFields) && $value !== null) {
                        if (is_string($value)) {
                            $value = preg_replace('/[^\d\.\-]/', '', $value);
                        }
                        $value = ($value !== '' && $value !== null) ? (float)$value : null;
                    }
                    
                    $data[$db_column] = $value;
                }
                
                // بررسی تکراری بودن
                $unique_id = $data['شماره_درخواست'] ?? null;
                
                if ($skip_duplicates && $unique_id) {
                    if (isDuplicate($pdo, $unique_id, $duplicate_cache)) {
                        $results['duplicates']++;
                        continue;
                    }
                }
                
                // اضافه کردن به batch
                $data_batch[] = $data;
                
                // اگر batch پر شد، وارد دیتابیس کن
                if (count($data_batch) >= $batch_size) {
                    $batch_imported = insertBatch($pdo, $data_batch, array_keys($data_batch[0] ?? []));
                    $results['imported'] += $batch_imported;
                    $data_batch = [];
                }
                
            } catch (Exception $e) {
                $results['error_count']++;
                $results['errors'][] = "فایل: " . basename($file_path) . " - ردیف $row_number: " . $e->getMessage();
                
                // فقط 10 خطای اول را ذخیره کن
                if (count($results['errors']) > 10) {
                    $results['errors'] = array_slice($results['errors'], 0, 10);
                    $results['errors'][] = "... و " . ($results['error_count'] - 10) . " خطای دیگر";
                }
                continue;
            }
        }
        
        // وارد کردن باقی‌مانده batch
        if (!empty($data_batch)) {
            $batch_imported = insertBatch($pdo, $data_batch, array_keys($data_batch[0]));
            $results['imported'] += $batch_imported;
        }
        
        // پاک کردن حافظه
        unset($spreadsheet, $worksheet, $rows, $data_batch);
        gc_collect_cycles();
        
    } catch (Exception $e) {
        $results['errors'][] = "خطا در پردازش فایل: " . $e->getMessage();
        $results['error_count']++;
    }
    
    return $results;
}

// 8. تابع اسکن پوشه sample
function scanSampleFolder() {
    $sample_dir = __DIR__ . '/sample';
    
    if (!is_dir($sample_dir)) {
        return [
            'success' => false,
            'message' => "پوشه sample یافت نشد: " . $sample_dir,
            'files' => []
        ];
    }
    
    $excel_files = [];
    $allowed_extensions = ['xlsx', 'xls', 'csv'];
    
    // اسکن پوشه
    $files = scandir($sample_dir);
    
    foreach ($files as $file) {
        if ($file === '.' || $file === '..') continue;
        
        $file_path = $sample_dir . '/' . $file;
        $extension = strtolower(pathinfo($file, PATHINFO_EXTENSION));
        
        if (in_array($extension, $allowed_extensions) && is_file($file_path)) {
            $excel_files[] = [
                'path' => $file_path,
                'name' => $file,
                'size' => filesize($file_path),
                'extension' => $extension
            ];
        }
    }
    
    return [
        'success' => true,
        'message' => count($excel_files) . " فایل اکسل یافت شد",
        'files' => $excel_files,
        'directory' => $sample_dir
    ];
}

// 9. پردازش درخواست
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // 9.1. اگر دکمه اسکن زده شده
    if (isset($_POST['scan_folder'])) {
        $scan_result = scanSampleFolder();
        
        if (!$scan_result['success']) {
            die("<div class='alert alert-danger'>{$scan_result['message']}</div>");
        }
        
        if (empty($scan_result['files'])) {
            die("<div class='alert alert-warning'>هیچ فایل اکسلی در پوشه sample یافت نشد.</div>");
        }
        
        // نمایش لیست فایل‌ها برای تأیید
        ?>
        <!DOCTYPE html>
        <html lang="fa" dir="rtl">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>تأیید فایل‌ها برای آپلود دسته‌ای</title>
            <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
            <style>
                body { font-family: 'Tahoma', 'Arial', sans-serif; background-color: #f8f9fa; padding: 20px; }
                .container { max-width: 1200px; }
                .file-list { max-height: 500px; overflow-y: auto; }
                .file-item { border-left: 5px solid #0d6efd; }
                .file-item-success { border-left-color: #198754; }
                .file-item-warning { border-left-color: #ffc107; }
                .file-item-danger { border-left-color: #dc3545; }
            </style>
        </head>
        <body>
            <div class="container">
                <div class="card shadow mt-4">
                    <div class="card-header bg-primary text-white text-center">
                        <h4 class="mb-0">تأیید فایل‌ها برای آپلود دسته‌ای</h4>
                    </div>
                    <div class="card-body">
                        <div class="alert alert-info">
                            <i class="bi bi-info-circle"></i>
                            <strong>توجه:</strong> <?= $scan_result['message'] ?><br>
                            <small>پوشه: <?= htmlspecialchars($scan_result['directory']) ?></small>
                        </div>
                        
                        <form method="POST" id="bulkUploadForm">
                            <input type="hidden" name="start_import" value="1">
                            
                            <div class="file-list mb-4">
                                <div class="table-responsive">
                                    <table class="table table-hover table-bordered">
                                        <thead class="table-dark">
                                            <tr>
                                                <th width="50"><input type="checkbox" id="selectAll" checked></th>
                                                <th width="50">#</th>
                                                <th width="300">نام فایل</th>
                                                <th width="150">نوع فایل</th>
                                                <th width="150">حجم (KB)</th>
                                                <th width="200">وضعیت</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($scan_result['files'] as $index => $file): ?>
                                            <tr class="file-item">
                                                <td class="text-center">
                                                    <input type="checkbox" name="selected_files[]" value="<?= htmlspecialchars($file['path']) ?>" checked>
                                                </td>
                                                <td class="text-center fw-bold"><?= $index + 1 ?></td>
                                                <td>
                                                    <i class="bi bi-file-earmark-excel text-success"></i>
                                                    <strong><?= htmlspecialchars($file['name']) ?></strong>
                                                </td>
                                                <td class="text-center">
                                                    <span class="badge bg-secondary"><?= strtoupper($file['extension']) ?></span>
                                                </td>
                                                <td class="text-center">
                                                    <?= number_format($file['size'] / 1024, 2) ?> KB
                                                </td>
                                                <td class="text-center">
                                                    <span class="badge bg-success">آماده آپلود</span>
                                                </td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                            
                            <div class="row mb-4">
                                <div class="col-md-6">
                                    <div class="card">
                                        <div class="card-header bg-secondary text-white">
                                            <h6 class="mb-0">تنظیمات آپلود</h6>
                                        </div>
                                        <div class="card-body">
                                            <div class="form-check mb-3">
                                                <input class="form-check-input" type="checkbox" id="skip_duplicates" name="skip_duplicates" checked>
                                                <label class="form-check-label" for="skip_duplicates">
                                                    رد کردن داده‌های تکراری
                                                </label>
                                                <small class="d-block text-muted">(اگر فعال باشد، رکوردهای تکراری وارد نمی‌شوند)</small>
                                            </div>
                                            
                                            <div class="form-check mb-3">
                                                <input class="form-check-input" type="checkbox" id="skip_errors" name="skip_errors" checked>
                                                <label class="form-check-label" for="skip_errors">
                                                    ادامه پردازش در صورت خطا
                                                </label>
                                                <small class="d-block text-muted">(اگر غیرفعال باشد، در اولین خطا متوقف می‌شود)</small>
                                            </div>
                                            
                                            <div class="form-check">
                                                <input class="form-check-input" type="checkbox" id="delete_after_import" name="delete_after_import">
                                                <label class="form-check-label" for="delete_after_import">
                                                    حذف فایل‌ها پس از آپلود موفق
                                                </label>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="col-md-6">
                                    <div class="card">
                                        <div class="card-header bg-info text-white">
                                            <h6 class="mb-0">خلاصه</h6>
                                        </div>
                                        <div class="card-body">
                                            <ul class="list-group list-group-flush">
                                                <li class="list-group-item d-flex justify-content-between">
                                                    <span>تعداد فایل‌ها:</span>
                                                    <span class="badge bg-primary"><?= count($scan_result['files']) ?></span>
                                                </li>
                                                <li class="list-group-item d-flex justify-content-between">
                                                    <span>کل حجم:</span>
                                                    <span class="badge bg-secondary">
                                                        <?= number_format(array_sum(array_column($scan_result['files'], 'size')) / 1024 / 1024, 2) ?> MB
                                                    </span>
                                                </li>
                                                <li class="list-group-item d-flex justify-content-between">
                                                    <span>پیش‌بینی زمان:</span>
                                                    <span class="badge bg-warning"><?= count($scan_result['files']) * 2 ?> ثانیه</span>
                                                </li>
                                            </ul>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="d-grid gap-2 d-md-flex justify-content-md-end">
                                <button type="button" class="btn btn-secondary me-md-2" onclick="location.href='upload_bulk.php'">انصراف</button>
                                <button type="submit" class="btn btn-success" id="startImportBtn">
                                    <i class="bi bi-play-circle"></i> شروع آپلود دسته‌ای
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
            
            <script>
                // انتخاب همه/هیچ
                document.getElementById('selectAll').addEventListener('change', function() {
                    const checkboxes = document.querySelectorAll('input[name="selected_files[]"]');
                    checkboxes.forEach(checkbox => {
                        checkbox.checked = this.checked;
                    });
                });
                
                // تأیید قبل از ارسال
                document.getElementById('bulkUploadForm').addEventListener('submit', function(e) {
                    const selectedFiles = document.querySelectorAll('input[name="selected_files[]"]:checked');
                    
                    if (selectedFiles.length === 0) {
                        e.preventDefault();
                        alert('لطفا حداقل یک فایل را انتخاب کنید!');
                        return false;
                    }
                    
                    // نمایش پیام در حال پردازش
                    const submitBtn = document.getElementById('startImportBtn');
                    submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> در حال شروع...';
                    submitBtn.disabled = true;
                    
                    return true;
                });
            </script>
            
            <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.8.0/font/bootstrap-icons.css">
            <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
        </body>
        </html>
        <?php
        exit;
    }
    
    // 9.2. اگر دکمه شروع آپلود زده شده
    if (isset($_POST['start_import'])) {
        session_write_close();
        
        // پیش‌پردازش شروع
        echo str_repeat(' ', 1024 * 64);
        echo '<!DOCTYPE html><html lang="fa" dir="rtl"><head>';
        echo '<meta charset="UTF-8"><meta name="viewport" content="width=device-width, initial-scale=1.0">';
        echo '<title>در حال آپلود دسته‌ای...</title>';
        echo '<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">';
        echo '<style>body { padding: 20px; } .progress { height: 30px; } .file-progress { height: 20px; }</style>';
        echo '</head><body>';
        echo '<div class="container">';
        echo '<div class="alert alert-info">';
        echo '<h4><i class="bi bi-upload"></i> در حال آپلود دسته‌ای فایل‌ها</h4>';
        echo '<p class="mb-0">لطفا تا پایان عملیات صبر کنید...</p>';
        echo '</div>';
        
        // نوار پیشرفت اصلی
        echo '<div class="progress mb-4">';
        echo '<div id="mainProgress" class="progress-bar progress-bar-striped progress-bar-animated" style="width: 0%"></div>';
        echo '</div>';
        
        // نوار پیشرفت فایل جاری
        echo '<div class="progress mb-2">';
        echo '<div id="fileProgress" class="progress-bar bg-success" style="width: 0%"></div>';
        echo '</div>';
        
        echo '<div id="fileStatus" class="alert alert-light mb-3"></div>';
        echo '<div id="results" class="mb-4"></div>';
        echo '</div>';
        
        echo '<script>
            function updateMainProgress(percent, message) {
                document.getElementById("mainProgress").style.width = percent + "%";
                document.getElementById("mainProgress").innerText = percent.toFixed(1) + "%";
                if (message) {
                    document.getElementById("fileStatus").innerHTML = message;
                }
                window.scrollTo(0, document.body.scrollHeight);
            }
            
            function updateFileProgress(percent) {
                document.getElementById("fileProgress").style.width = percent + "%";
            }
            
            function addResult(message, type = "info") {
                const resultsDiv = document.getElementById("results");
                const alertDiv = document.createElement("div");
                alertDiv.className = "alert alert-" + type;
                alertDiv.innerHTML = message;
                resultsDiv.appendChild(alertDiv);
                window.scrollTo(0, document.body.scrollHeight);
            }
        </script>';
        flush();
        
        // دریافت تنظیمات
        $selected_files = $_POST['selected_files'] ?? [];
        $skip_duplicates = isset($_POST['skip_duplicates']);
        $skip_errors = isset($_POST['skip_errors']);
        $delete_after_import = isset($_POST['delete_after_import']);
        
        if (empty($selected_files)) {
            echo '<div class="alert alert-danger">هیچ فایلی انتخاب نشده است!</div>';
            echo '<a href="upload_bulk.php" class="btn btn-primary">بازگشت</a>';
            exit;
        }
        
        $total_files = count($selected_files);
        $processed_files = 0;
        $total_imported = 0;
        $total_duplicates = 0;
        $total_errors = 0;
        $all_errors = [];
        
        // پردازش هر فایل
        foreach ($selected_files as $index => $file_path) {
            $file_number = $index + 1;
            $filename = basename($file_path);
            
            // به روزرسانی وضعیت
            $file_progress = ($index / $total_files) * 100;
            $message = "در حال پردازش فایل $file_number از $total_files: <strong>" . htmlspecialchars($filename) . "</strong>";
            echo "<script>updateMainProgress($file_progress, '$message'); updateFileProgress(0);</script>";
            flush();
            
            // پردازش فایل
            $result = processExcelFile($file_path, $pdo, $skip_duplicates);
            
            // به روزرسانی پیشرفت فایل
            echo "<script>updateFileProgress(100);</script>";
            flush();
            
            // جمع‌آوری نتایج
            $processed_files++;
            $total_imported += $result['imported'];
            $total_duplicates += $result['duplicates'];
            $total_errors += $result['error_count'];
            
            // نمایش نتیجه فایل
            $result_class = $result['error_count'] > 0 ? 'warning' : 'success';
            $result_message = "<strong>{$result['filename']}:</strong> ";
            $result_message .= "{$result['imported']} وارد شده، ";
            $result_message .= "{$result['duplicates']} تکراری، ";
            $result_message .= "{$result['error_count']} خطا";
            
            echo "<script>addResult('$result_message', '$result_class');</script>";
            flush();
            
            // جمع‌آوری خطاها
            if (!empty($result['errors'])) {
                $all_errors = array_merge($all_errors, $result['errors']);
            }
            
            // حذف فایل اگر درخواست شده باشد
            if ($delete_after_import && $result['error_count'] == 0) {
                if (file_exists($file_path)) {
                    unlink($file_path);
                    echo "<script>addResult('فایل " . htmlspecialchars($filename) . " پس از آپلود موفق حذف شد', 'info');</script>";
                    flush();
                }
            }
            
            // توقف کوتاه برای نمایش بهتر
            usleep(500000); // 0.5 ثانیه
        }
        
        // به روزرسانی پیشرفت نهایی
        echo "<script>updateMainProgress(100, 'آپلود کامل شد!');</script>";
        flush();
        
        // نمایش نتایج نهایی
        echo '<div class="alert alert-success mt-4">';
        echo '<h4><i class="bi bi-check-circle"></i> آپلود دسته‌ای تکمیل شد!</h4>';
        echo '<div class="row mt-3">';
        echo '<div class="col-md-3"><div class="alert alert-success">فایل‌های پردازش شده: <strong>' . $processed_files . '</strong></div></div>';
        echo '<div class="col-md-3"><div class="alert alert-success">ردیف‌های وارد شده: <strong>' . $total_imported . '</strong></div></div>';
        echo '<div class="col-md-3"><div class="alert alert-warning">ردیف‌های تکراری: <strong>' . $total_duplicates . '</strong></div></div>';
        echo '<div class="col-md-3"><div class="alert alert-danger">خطاها: <strong>' . $total_errors . '</strong></div></div>';
        echo '</div>';
        echo '</div>';
        
        // نمایش خطاهای جمع‌آوری شده
        if (!empty($all_errors)) {
            echo '<div class="alert alert-danger">';
            echo '<h5><i class="bi bi-exclamation-triangle"></i> خطاهای جمع‌آوری شده:</h5>';
            echo '<div style="max-height: 300px; overflow-y: auto;">';
            echo '<ul class="mb-0">';
            $display_errors = array_slice($all_errors, 0, 20);
            foreach ($display_errors as $error) {
                echo '<li>' . htmlspecialchars($error) . '</li>';
            }
            if (count($all_errors) > 20) {
                echo '<li>... و ' . (count($all_errors) - 20) . ' خطای دیگر</li>';
            }
            echo '</ul>';
            echo '</div>';
            echo '</div>';
        }
        
        echo '<div class="mt-4">';
        echo '<a href="upload_bulk.php" class="btn btn-primary me-2">آپلود مجدد</a>';
        echo '<a href="upload.php" class="btn btn-secondary me-2">آپلود دستی</a>';
        echo '<a href="index.php" class="btn btn-success">مشاهده داده‌ها</a>';
        echo '</div>';
        
        echo '</body></html>';
        exit;
    }
}

// 10. فرم اصلی
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>آپلود دسته‌ای فایل‌های اکسل</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body { font-family: 'Tahoma', 'Arial', sans-serif; background-color: #f8f9fa; }
        .container { max-width: 800px; }
        .feature-card { border-left: 5px solid #0d6efd; }
        .feature-card-success { border-left-color: #198754; }
        .feature-card-warning { border-left-color: #ffc107; }
    </style>
</head>
<body>
    <div class="container mt-5">
        <div class="card shadow">
            <div class="card-header bg-primary text-white text-center">
                <h4 class="mb-0"><i class="bi bi-upload"></i> آپلود دسته‌ای فایل‌های اکسل</h4>
            </div>
            <div class="card-body">
                <div class="alert alert-info">
                    <h6><i class="bi bi-info-circle"></i> راهنمای آپلود دسته‌ای:</h6>
                    <p class="mb-0">این سیستم به طور خودکار تمام فایل‌های اکسل موجود در پوشه <code>sample</code> را خوانده و وارد دیتابیس می‌کند.</p>
                </div>
                
                <div class="row mb-4">
                    <div class="col-md-6">
                        <div class="card feature-card h-100">
                            <div class="card-body">
                                <h5 class="card-title text-primary"><i class="bi bi-folder-check"></i> پوشه sample</h5>
                                <p class="card-text">فایل‌های اکسل خود را در پوشه زیر قرار دهید:</p>
                                <code class="d-block p-2 bg-light rounded mb-3"><?= __DIR__ ?>/sample/</code>
                                <p class="card-text"><small class="text-muted">فرمت‌های مجاز: .xlsx, .xls, .csv</small></p>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-6">
                        <div class="card feature-card-success h-100">
                            <div class="card-body">
                                <h5 class="card-title text-success"><i class="bi bi-lightning-charge"></i> ویژگی‌ها</h5>
                                <ul class="mb-0">
                                    <li>تشخیص خودکار ستون‌ها</li>
                                    <li>حذف داده‌های تکراری</li>
                                    <li>پردازش دسته‌ای</li>
                                    <li>گزارش کامل خطاها</li>
                                    <li>پیشرفت زنده</li>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
                
                <form method="POST">
                    <div class="d-grid gap-2">
                        <button type="submit" name="scan_folder" class="btn btn-primary btn-lg">
                            <i class="bi bi-search"></i> اسکن پوشه و شروع آپلود
                        </button>
                        
                        <div class="d-flex gap-2">
                            <a href="upload.php" class="btn btn-outline-secondary flex-fill">
                                <i class="bi bi-upload"></i> آپلود دستی تک فایل
                            </a>
                            <a href="index.php" class="btn btn-outline-success flex-fill">
                                <i class="bi bi-eye"></i> مشاهده داده‌ها
                            </a>
                        </div>
                    </div>
                </form>
                
                <?php
                // نمایش وضعیت پوشه sample
                $sample_dir = __DIR__ . '/sample';
                if (is_dir($sample_dir)) {
                    $files = scandir($sample_dir);
                    $excel_files = array_filter($files, function($file) {
                        $ext = strtolower(pathinfo($file, PATHINFO_EXTENSION));
                        return in_array($ext, ['xlsx', 'xls', 'csv']) && $file !== '.' && $file !== '..';
                    });
                    
                    $file_count = count($excel_files);
                    
                    if ($file_count > 0) {
                        echo '<div class="alert alert-success mt-3">';
                        echo '<i class="bi bi-check-circle"></i> ';
                        echo "<strong>$file_count فایل اکسل</strong> در پوشه sample یافت شد.";
                        echo '<div class="mt-2">';
                        foreach (array_slice($excel_files, 0, 5) as $file) {
                            echo '<span class="badge bg-secondary me-1">' . htmlspecialchars($file) . '</span>';
                        }
                        if ($file_count > 5) {
                            echo '<span class="badge bg-info">و ' . ($file_count - 5) . ' فایل دیگر</span>';
                        }
                        echo '</div>';
                        echo '</div>';
                    } else {
                        echo '<div class="alert alert-warning mt-3">';
                        echo '<i class="bi bi-exclamation-triangle"></i> ';
                        echo 'هیچ فایل اکسل در پوشه sample یافت نشد.';
                        echo '</div>';
                    }
                } else {
                    echo '<div class="alert alert-danger mt-3">';
                    echo '<i class="bi bi-exclamation-octagon"></i> ';
                    echo 'پوشه sample یافت نشد! لطفا پوشه ای با این نام ایجاد کنید.';
                    echo '</div>';
                }
                ?>
            </div>
        </div>
    </div>

    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.8.0/font/bootstrap-icons.css">
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>