<?php
namespace handler;

use handler\basic_element;

use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;

if ( ! defined( 'ABSPATH' ) ) { exit; } // Exit if accessed directly


trait product_query {
	use basic_element;
	/**
	 * Register Query Controls
	 */
	protected function register_query_builder_controls() {

		$this->start_controls_section( 'section_post_query_builder', [
			'label' => esc_html__( 'Query', THEME_NAME ) ,
			'tab'   => Controls_Manager::TAB_CONTENT,
		] );

		$this->add_control(
			'posts_source',
			[
				'label'   => esc_html__( 'Posts Source', THEME_NAME ),
				'type'    => Controls_Manager::SELECT,
				'options' => $this->get_posts_source(),
				'default' => 'post',
			]
		);

		$this->add_control(
			'current_query_note',
			[
				'label'             => '',
				'type'              => Controls_Manager::RAW_HTML,
				'raw'               => esc_html__( 'Use it for archive template.', THEME_NAME ),
				'content_classes'   => 'elementor-panel-alert elementor-panel-alert-info',
				'condition' => [
					'posts_source' => 'current_query',
				],
			]
		);

		$this->add_control(
			'related_posts_note',
			[
				'label'             => '',
				'type'              => Controls_Manager::RAW_HTML,
				'raw'               => esc_html__( 'Use it for related posts template.', THEME_NAME ),
				'content_classes'   => 'elementor-panel-alert elementor-panel-alert-info',
				'condition' => [
					'posts_source' => 'related',
				],
			]
		);

		$this->add_control( 'posts_per_page', [
			'label'     => esc_html__( 'Posts Per Page', THEME_NAME ),
			'type'      => Controls_Manager::NUMBER,
			'default'   => '3',
			'condition' => [
				'posts_source!' => ['current_query'],
			]
		] );

		$this->start_controls_tabs(
			'tabs_posts_include_exclude',
			[
				'condition' => [
					'posts_source!' => ['current_query', 'related'],
				]
			]
		);

		$this->start_controls_tab(
			'tab_posts_include',
			[
				'label'     => esc_html__( 'Include', THEME_NAME ),
				'condition' => [
					'posts_source!' => ['current_query', 'related'],
				]
			]
		);

		$this->add_control(
			'posts_include_by',
			[
				'label'       => esc_html__( 'Include By', THEME_NAME ),
				'type'        => Controls_Manager::SELECT2,
				'multiple'    => true,
				'label_block' => true,
				'options'     => [
					'taxonomy' => esc_html__( 'Taxonomy', THEME_NAME ),
					'authors' => esc_html__( 'Authors', THEME_NAME ),
					'current_author' => esc_html__( 'Current Author', THEME_NAME ),
					'manual_selection' => esc_html__( 'Manual Selection', THEME_NAME ),
				],
				'condition'   => [
					'posts_source!' => ['current_query', 'related'],
				]
			]
		);

		$this->add_control(
			'posts_include_taxonomy_ids',
			array(
				'label'       => esc_html__( 'Select Taxonomy', THEME_NAME ),
				'description'       => esc_html__( 'Select Categories, Tags, Post Format or custom taxonomies.', THEME_NAME ),
				'label_block' => true,
				'type'        => 'theme-select2',
				'multiple' => true,
				'source_type' => 'taxonomy',
				'condition'   => [
					'posts_include_by' => 'taxonomy',
					'posts_source!' => ['current_query', 'related'],
				]
			)
		);

		$this->add_control(
			'posts_include_author_ids',
			array(
				'label'       => esc_html__( 'Select Authors', THEME_NAME ),
				'label_block' => true,
				'type'        => 'theme-select2',
				'multiple' => true,
				'source_type' => 'author',
				'condition'   => [
					'posts_include_by' => 'authors',
					'posts_source!' => ['current_query', 'related'],
				]
			)
		);

		$this->add_control(
			'posts_include_post_ids',
			array(
				'label'       => esc_html__( 'Select Posts', THEME_NAME ),
				'label_block' => true,
				'type'        => 'theme-select2',
				'multiple' => true,
				'source_type' => 'post',
				'condition'   => [
					'posts_include_by' => 'manual_selection',
					'posts_source!' => ['current_query', 'related'],
				]
			)
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_posts_exclude',
			[
				'label'     => esc_html__( 'Exclude', THEME_NAME ),
				'condition' => [
					'posts_source!' => ['current_query', 'related'],
				]
			]
		);

		$this->add_control(
			'posts_exclude_by',
			[
				'label'       => esc_html__( 'Exclude By', THEME_NAME ),
				'type'        => Controls_Manager::SELECT2,
				'multiple'    => true,
				'label_block' => true,
				'options'     => [
					'taxonomy' => esc_html__( 'Taxonomy', THEME_NAME ),
					'authors' => esc_html__( 'Authors', THEME_NAME ),
					'current_author' => esc_html__( 'Current Author', THEME_NAME ),
					'manual_selection' => esc_html__( 'Manual Selection', THEME_NAME ),
				],
				'condition'   => [
					'posts_source!' => ['current_query', 'related'],
				]
			]
		);

		$this->add_control(
			'posts_exclude_taxonomy_ids',
			array(
				'label'       => esc_html__( 'Select Taxonomy', THEME_NAME ),
				'description'       => esc_html__( 'Select Categories, Tags, Post Format or custom taxonomies.', THEME_NAME ),
				'label_block' => true,
				'type'        => 'theme-select2',
				'multiple' => true,
				'source_type' => 'taxonomy',
				'condition'   => [
					'posts_exclude_by' => 'taxonomy',
					'posts_source!' => ['current_query', 'related'],
				]
			)
		);

		$this->add_control(
			'posts_exclude_author_ids',
			array(
				'label'       => esc_html__( 'Select Authors', THEME_NAME ),
				'label_block' => true,
				'type'        => 'theme-select2',
				'multiple' => true,
				'source_type' => 'author',
				'condition'   => [
					'posts_exclude_by' => 'authors',
					'posts_source!' => ['current_query', 'related'],
				]
			)
		);

		$this->add_control(
			'posts_exclude_post_ids',
			array(
				'label'       => esc_html__( 'Select Posts', THEME_NAME ),
				'label_block' => true,
				'type'        => 'theme-select2',
				'multiple' => true,
				'source_type' => 'post',
				'condition'   => [
					'posts_exclude_by' => 'manual_selection',
					'posts_source!' => ['current_query', 'related'],
				]
			)
		);

		$this->end_controls_tab();
		$this->end_controls_tabs();

		$this->add_control(
			'posts_divider',
			[
				'type' => Controls_Manager::DIVIDER,
			]
		);

		$this->add_control(
			'posts_select_date',
			[
				'label'     => esc_html__( 'Date', THEME_NAME ),
				'type'      => Controls_Manager::SELECT,
				'default'   => 'anytime',
				'options'   => [
					'anytime' => esc_html__( 'All', THEME_NAME ),
					'today'   => esc_html__( 'Past Day', THEME_NAME ),
					'week'    => esc_html__( 'Past Week', THEME_NAME ),
					'month'   => esc_html__( 'Past Month', THEME_NAME ),
					'quarter' => esc_html__( 'Past Quarter', THEME_NAME ),
					'year'    => esc_html__( 'Past Year', THEME_NAME ),
					'exact'   => esc_html__( 'Custom', THEME_NAME ),
				],
				'condition' => [
					'posts_source!' => 'current_query',
				]
			]
		);

		$this->add_control(
			'posts_date_before',
			[
				'label'       => esc_html__( 'Before', THEME_NAME ),
				'type'        => Controls_Manager::DATE_TIME,
				'description' => esc_html__( 'Setting a ‘Before’ date will show all the posts published until the chosen date (inclusive).', THEME_NAME ),
				'condition'   => [
					'posts_select_date' => 'exact',
					'posts_source!' => 'current_query',
				]
			]
		);

		$this->add_control(
			'posts_date_after',
			[
				'label'       => esc_html__( 'After', THEME_NAME ),
				'type'        => Controls_Manager::DATE_TIME,
				'description' => esc_html__( 'Setting an ‘After’ date will show all the posts published since the chosen date (inclusive).', THEME_NAME ),
				'condition'   => [
					'posts_select_date' => 'exact',
					'posts_source!' => 'current_query',
				]
			]
		);

		$this->add_control(
			'posts_orderby',
			[
				'label'   => esc_html__( 'Order By', THEME_NAME ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'date',
				'options' => [
					'title'         => esc_html__( 'Title', THEME_NAME ),
					'date'          => esc_html__( 'Date', THEME_NAME ),
					'views_count'   => esc_html__( 'Views Count', THEME_NAME ),
					'modified'      => esc_html__( 'Last Modified Date', THEME_NAME ),
					'author'        => esc_html__( 'Author', THEME_NAME ),
					'comment_count' => esc_html__( 'Comment Count', THEME_NAME ),
					'menu_order'    => esc_html__( 'Menu Order', THEME_NAME ),
					'rand'          => esc_html__( 'Random', THEME_NAME ),
				],
				'condition'    => [
					'posts_source!' => 'current_query',
				]
			]
		);

		$this->add_control(
			'posts_order',
			[
				'label'   => esc_html__( 'Order', THEME_NAME ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'desc',
				'options' => [
					'asc'  => esc_html__( 'ASC', THEME_NAME ),
					'desc' => esc_html__( 'DESC', THEME_NAME ),
				],
				'condition'    => [
					'posts_source!' => 'current_query',
				]
			]
		);

		$this->add_control(
			'posts_ignore_sticky_posts',
			[
				'label'        => esc_html__( 'Ignore Sticky Posts', THEME_NAME ),
				'type'         => Controls_Manager::SWITCHER,
				'condition'    => [
					'posts_source'     => 'post',
				]
			]
		);

		$this->add_control(
			'posts_ignore_current_post',
			[
				'label'        => esc_html__( 'Ignore Current Post', THEME_NAME ),
				'type'         => Controls_Manager::SWITCHER,
				'condition'    => [
					'posts_source!' => 'current_query',
				]
			]
		);

		$this->add_control(
			'posts_only_with_featured_image',
			[
				'label'        => esc_html__( 'Only Featured Image Post', THEME_NAME ),
				'description'  => esc_html__( 'Enable to display posts only when featured image is present.', THEME_NAME ),
				'type'         => Controls_Manager::SWITCHER,
				'condition'    => [
					'posts_source!' => 'current_query',
				]
			]
		);

		$this->add_control(
			'posts_offset',
			[
				'label'        => esc_html__( 'Offset', THEME_NAME ),
				'description'  => esc_html__( 'number of posts to skip them.', THEME_NAME ),
				'type'         => Controls_Manager::NUMBER,
				'min' => 0,
				'max' => 10,
				'condition'    => [
					'posts_source!' => 'current_query',
				]
			]
		);

		$this->end_controls_section();

		$this->start_controls_section( 
			'section_pagination', [ 
			'label' => esc_html__( 'Pagination', THEME_NAME ),
			'tab' => Controls_Manager::TAB_CONTENT,
			'condition' => [ 
				'layout!' => 'carousel',
				'posts_source!' => 'current_query',
			],
		] );

		$this->add_control(
			'pagination_type',
			[ 
				'label' => esc_html__( 'Pagination', THEME_NAME ),
				'type' => Controls_Manager::SELECT,
				'default' => 'none',
				'options' => [ 
					'none' => esc_html__( 'None', THEME_NAME ),
					'numbers' => esc_html__( 'Numbers', THEME_NAME ),
					'prev_next' => esc_html__( 'Previous/Next', THEME_NAME ),
					'numbers_and_prev_next' => esc_html__( 'Numbers + Previous/Next', THEME_NAME ),
				],
			]
		);

		$this->add_control(
			'pagination_page_limit',
			[ 
				'label' => esc_html__( 'Page Limit', THEME_NAME ),
				'type' => Controls_Manager::NUMBER,
				'condition' => [ 
					'pagination_type!' => 'none',
				],
			]
		);

		$this->add_control(
			'pagination_prev_label',
			[ 
				'label' => esc_html__( 'Previous Label', THEME_NAME ),
				'default' => esc_html__( 'Previous', THEME_NAME ),
				'condition' => [ 
					'pagination_type' => [ 'numbers_and_prev_next', 'prev_next' ],
				],
			]
		);

		$this->add_control(
			'pagination_next_label',
			[ 
				'label' => esc_html__( 'Next Label', THEME_NAME ),
				'default' => esc_html__( 'Next', THEME_NAME ),
				'condition' => [ 
					'pagination_type' => [ 'numbers_and_prev_next', 'prev_next' ],
				],
			]
		);

		$this->add_control(
			'pagination_numbers_shorten',
			[ 
				'label' => esc_html__( 'Shorten', THEME_NAME ),
				'type' => Controls_Manager::SWITCHER,
				'default' => '',
				'condition' => [ 
					'pagination_type' => [ 'numbers', 'numbers_and_prev_next' ],
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_pagination_style',
			[ 
				'label' => esc_html__( 'Pagination', THEME_NAME ),
				'tab' => Controls_Manager::TAB_STYLE,
				'condition' => [ 
					'pagination_type!' => 'none',
					'layout!' => 'carousel',
				],
			]
		);

		$this->register_container_style( 'box', '.posts-pagination-wrap' );

		$this->start_controls_tabs( 'tabs_pagination' );

		$this->start_controls_tab(
			'tab_pagination_normal',
			[ 
				'label' => esc_html__( 'Normal', THEME_NAME ),
			]
		);
		$this->register_text_style( 'normal', '.posts-pagination .page-numbers, {{WRAPPER}} .posts-pagination a',  $align=false);

		$this->add_control(
			'next-sep',
			[ 
				'label' => esc_html__( 'next', THEME_NAME ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->register_text_style( 'next', '.posts-pagination a.next', $align = false );

		$this->add_control(
			'Previous-sep',
			[ 
				'label' => esc_html__( 'Previous', THEME_NAME ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->register_text_style( 'Previous', '.posts-pagination a.Previous', $align = false );

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_pagination_hover',
			[ 
				'label' => esc_html__( 'Hover', THEME_NAME ),
			]
		);

		$this->register_text_style( 'hover', '.posts-pagination a:hover', $align = false);

		$this->add_control(
			'next-sep-hover',
			[ 
				'label' => esc_html__( 'next', THEME_NAME ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->register_text_style( 'next-hover', '.posts-pagination a.next:hover', $align = false );

		$this->add_control(
			'Previous-sep-hover',
			[ 
				'label' => esc_html__( 'Previous', THEME_NAME ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->register_text_style( 'Previous-hover', '.posts-pagination a.Previous:hover', $align = false );


		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_pagination_active',
			[ 
				'label' => esc_html__( 'Active', THEME_NAME ),

			]
		);

		$this->register_text_style( 'active', '.posts-pagination .page-numbers.current', $align = false );

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();
	}

	protected function render_pagination() {
		$settings = $this->get_settings_for_display();

		$pagination_type = $settings['pagination_type'];
		$page_limit = $settings['pagination_page_limit'];
		$pagination_shorten = $settings['pagination_numbers_shorten'];

		if ( 'none' === $pagination_type ) {
			return;
		}

		if ( $settings['layout'] == 'carousel' ) {
			return;
		}
		$current_page = $this->get_paged();

		$query = $this->get_query_result();
		$total_pages = $query->max_num_pages;

		if ( $page_limit ) {
			$total_pages = min( $page_limit, $total_pages );
		}

		if ( 2 > $total_pages ) {
			return;
		}

		$has_prev_next = in_array( $pagination_type, [ 'prev_next', 'numbers_and_prev_next' ] );

		$paginate_args = array(
			'type' => 'array',
			'current' => $current_page,
			'total' => $total_pages,
			'prev_next' => false,
			'show_all' => 'yes' !== $pagination_shorten,
		);

		if ( $has_prev_next ) {
			$prev_label = $settings['pagination_prev_label'];
			$next_label = $settings['pagination_next_label'];

			$paginate_args['prev_next'] = true;

			if ( $prev_label ) {
				$paginate_args['prev_text'] = $prev_label;
			}
			if ( $next_label ) {
				$paginate_args['next_text'] = $next_label;
			}
		}

		$paginate_args['format'] = '?page=%#%';
		$links = paginate_links( $paginate_args );

		if ( $pagination_type == 'prev_next' ) { 
			$prev_next_links = [];
			if ( $current_page != 1 ) {
				$prev_next_links[] = $links[0];
			}

			if ( $current_page != $total_pages ) {
				$prev_next_links[] = end( $links );
			}

			$links = $prev_next_links;
		}

		?>
		<div class="posts-pagination-wrap d-flex justify-content-center">
			<nav class="posts-pagination d-flex" data-total="<?php echo esc_html( $total_pages ); ?>">
				<?php echo implode( PHP_EOL, $links ); ?>
			</nav>
		</div>
		<?php

	}

	/**
	 * Get All Post Source
	 * @return array
	 */
	protected function get_posts_source(){
		$post_types = get_post_types(['public' => true, 'show_in_nav_menus' => true], 'objects');
		$post_types = wp_list_pluck($post_types, 'label', 'name');

		$include = [
			'related' => esc_html__( 'Related Posts', THEME_NAME ),
			'current_query' => esc_html__( 'Current Query', THEME_NAME ),
		];

		$post_types = array_merge($post_types, $include);
		unset($post_types['elementor_library'], $post_types['page'], $post_types['e-landing-page']);

		return $post_types;
	}

	/**
	 * @param array $term_ids
	 *
	 * @return array
	 */
	private function group_terms_by_taxonomy( $term_ids = [] ) {

		if(!$term_ids)
			return [];

		$terms = get_terms(
			[
				'include' => $term_ids,
				'hide_empty'       => false,
			] );

		$group_terms = [];

		foreach ( $terms as $term ) {
			$group_terms[ $term->taxonomy ][] = $term->term_id;
		}

		return $group_terms;
	}

	/**
	 * @return array
	 */
	private function get_meta_args() {

		$args = [];
		$settings = $this->get_settings_for_display();


		/**
		 * Order
		 */
		$args['order']   = $settings[ 'posts_order' ];

		if($settings[ 'posts_orderby' ] == 'views_count') {
			$args['orderby']  = 'meta_value_num';
			$args['meta_key'] = 'post_views';
		}
		else {
			$args['orderby'] = $settings[ 'posts_orderby' ];
		}


		/**
		 * Set Feature Images
		 */
		if ( $settings[ 'posts_only_with_featured_image' ] ) {
			$args['meta_key'] = '_thumbnail_id';
		}


		/**
		 * Set Ignore Sticky
		 */
		if ( $settings['posts_source'] == 'post' && $settings[ 'posts_ignore_sticky_posts' ] ) {
			$args['ignore_sticky_posts'] = true;
		}


		/**
		 * Set Date
		 */

		$selected_date = $settings['posts_select_date'];

		if ( $selected_date ) {
			$date_query = [];

			switch ( $selected_date ) {
				case 'today':
					$date_query['after'] = '-1 day';
					break;

				case 'week':
					$date_query['after'] = '-1 week';
					break;

				case 'month':
					$date_query['after'] = '-1 month';
					break;

				case 'quarter':
					$date_query['after'] = '-3 month';
					break;

				case 'year':
					$date_query['after'] = '-1 year';
					break;

				case 'exact':
					$after_date = $settings['posts_date_after'];
					if ( $after_date ) {
						$date_query['after'] = $after_date;
					}

					$before_date = $settings['posts_date_before'];
					if ( $before_date ) {
						$date_query['before'] = $before_date;
					}

					$date_query['inclusive'] = true;
					break;
			}

			if ( $date_query ) {
				$args['date_query'] = $date_query;
			}
		}

		return $args;

	}

	/**
	 * Returns the paged number for the query.
	 *
	 * @return int
	 */
	protected function get_paged() {
		$settings = $this->get_settings_for_display();
		$pagination_type = isset($settings['pagination_type'])? $settings['pagination_type'] : 'none';

		return max( 1, get_query_var( 'paged' ), get_query_var( 'page' ) );
	}

	/**
	 * Get Query Args
	 *
	 * @return array
	 */
	protected function get_query_args() {

		$args = [];
		$settings = $this->get_settings_for_display();
		$pagination_type = isset($settings['pagination_type'])? $settings['pagination_type'] : 'none';

		/**
		 * Set paged
		 * Disable args['paged'] for widgets that have not pagination to prevent paginate them by mistake.
		 */
		$paged = ( $pagination_type != 'none' )? $this->get_paged() : '';
		$args['paged'] = $paged;


		/**
		 * Current Query
		 */
		if( $settings['posts_source'] == 'current_query' ) {

			if (\Elementor\Plugin::$instance->editor->is_edit_mode() || \Elementor\Plugin::$instance->preview->is_preview_mode()) {
				$args = [
					'post_type'         => 'product',
					'post_status'       => 'publish',
					'posts_per_page'    => intval(get_option( 'posts_per_page' )),
				];
			}
			else {

				$args = $GLOBALS['wp_query']->query_vars;

				$args['paged'] = $paged; // Fix Load more paged
				$args['post_status']         = 'publish';

				$args = apply_filters( 'core/Traits/get_query_args/current_query', $args );
			}

			return $args;
		}


		/**
		 * Posts Per Page
		 */
		$args['posts_per_page'] = $settings['posts_per_page'];


		/**
		 * Single Post Id
		 */
		$single_post_id = ( is_singular() && get_queried_object_id() )? intval(get_queried_object_id()) : 0;

		/**
		 * Post Type
		 */
		$post_type = 'product';
		if( $settings['posts_source'] == 'related' ) {

			if (\Elementor\Plugin::$instance->editor->is_edit_mode() || \Elementor\Plugin::$instance->preview->is_preview_mode()) {
				$post_type = 'product';
			}
			elseif($single_post_id) {
				$post_type = get_post_type($single_post_id);
			}
		}
		else {
			$post_type = $settings['posts_source'];
		}

		$args['post_status']         = 'publish';
		$args['suppress_filters']    = false;
		$args['post_type']           = $post_type;


		if( $settings['posts_source'] != 'related' ) {

			/**
			 * Authors
			 */
			$author__in = [];
			$author__not_in = [];

			if( in_array( 'authors', (array)$settings['posts_include_by'] )) {
				$author__in = wp_parse_id_list( $settings['posts_include_author_ids'] );
			}

			if( in_array( 'authors', (array)$settings['posts_exclude_by'] )) {
				$author__not_in = wp_parse_id_list( $settings['posts_exclude_author_ids'] );
			}

			if( $single_post_id && in_array( 'current_author', (array)$settings['posts_include_by'] ) ) {
				$single_author_id = get_post_field( 'post_author', $single_post_id );
				$author__in[] = intval($single_author_id);
			}

			if($single_post_id && in_array( 'current_author', (array)$settings['posts_exclude_by'] ) ) {
				$single_author_id = get_post_field( 'post_author', $single_post_id );
				$author__not_in[] = intval($single_author_id);
			}

			if($author__in) {
				$args['author__in'] = $author__in;
			}
			if($author__not_in) {
				$args['author__not_in'] = $author__not_in;
			}

		}


		/**
		 * Manual Selection
		 */
		$post__in = [];
		$post__not_in = [];

		if( $settings['posts_source'] != 'related' ) {

			if( in_array( 'manual_selection', (array)$settings['posts_include_by'] )) {
				$post__in = wp_parse_id_list( $settings['posts_include_post_ids'] );
			}

			if( in_array( 'manual_selection', (array)$settings['posts_exclude_by'] )) {
				$post__not_in = wp_parse_id_list( $settings['posts_exclude_post_ids'] );
			}

		}

		if( $single_post_id && $settings['posts_ignore_current_post'] ) {
			$post__not_in[] = $single_post_id;
		}

		if($post__in) {
			$args['post__in'] = $post__in;
		}
		if($post__not_in) {
			$args['post__not_in'] = $post__not_in;
		}



		/**
		 * Taxonomy
		 */
		$include_term_ids = [];
		$exclude_term_ids = [];

		if( $settings['posts_source'] == 'related' ) {
			if($single_post_id && !(\Elementor\Plugin::$instance->editor->is_edit_mode() || \Elementor\Plugin::$instance->preview->is_preview_mode())) {
				$taxonomies = get_object_taxonomies( $post_type );
				if($post_type == 'product') { // Remove post_format taxonomy from post
					$taxonomies = array_diff( $taxonomies, ['post_format'] );
				}

				if($taxonomies) {
					$include_term_ids = wp_get_post_terms($single_post_id, $taxonomies, ['fields' => 'ids', 'orderby' => 'count', 'order' => 'DESC', 'number' => 6]);
				}

			}
		}
		else {
			if( in_array( 'taxonomy', (array)$settings['posts_include_by'] ) ) {
				$include_term_ids = wp_parse_id_list( $settings['posts_include_taxonomy_ids'] );
			}

			if( in_array( 'taxonomy', (array)$settings['posts_exclude_by'] ) ) {
				$exclude_term_ids = wp_parse_id_list( $settings['posts_exclude_taxonomy_ids'] );
				$exclude_term_ids = array_diff( $exclude_term_ids, $include_term_ids );
			}
		}

		$include_taxonomy_terms = $this->group_terms_by_taxonomy($include_term_ids);
		$exclude_taxonomy_terms = $this->group_terms_by_taxonomy($exclude_term_ids);

		$terms_query = [];

		if ( $include_taxonomy_terms ) {

			foreach ( $include_taxonomy_terms as $tax => $terms ) {
				$terms_query[] = [
					'taxonomy' => $tax,
					'field'    => 'term_id',
					'terms'    => $terms,
					'operator' => 'IN',
				];
			}
		}

		if ( $exclude_taxonomy_terms ) {

			foreach ( $exclude_taxonomy_terms as $tax => $terms ) {
				$terms_query[] = [
					'taxonomy' => $tax,
					'field'    => 'term_id',
					'terms'    => $terms,
					'operator' => 'NOT IN',
				];
			}
		}


		if ( $terms_query ) {
			$args['tax_query']             = $terms_query;
			$args['tax_query']['relation'] = 'AND';
		}


		$args = array_merge($args, $this->get_meta_args());

		return $args;

	}

	/**
	 * Get Query
	 * @return array
	 */
	protected function get_query_result() {
		$settings = $this->get_settings_for_display();

		$query_args = $this->get_query_args();

		$offset_control = $settings['posts_offset'];

		if ( 0 < $offset_control ) {
			/**
			 * Offset break the pagination. Using WordPress's work around
			 *
			 * @see https://codex.wordpress.org/Making_Custom_Queries_using_Offset_and_Pagination
			 */
			add_action( 'pre_get_posts', [ $this, 'fix_query_offset' ], 1 );
			add_filter( 'found_posts', [ $this, 'fix_query_found_posts' ], 1 );
		}

		$query_result = new \WP_Query( $query_args );

		remove_action( 'pre_get_posts', [ $this, 'fix_query_offset' ], 1 );
		remove_filter( 'found_posts', [ $this, 'fix_query_found_posts' ], 1 );

		return $query_result;

	}

	/**
	 * @param \WP_Query $query
	 */
	public function fix_query_offset( &$query ) {
		$settings = $this->get_settings_for_display();
		$offset = $settings['posts_offset'];

		if ( $query->is_paged ) {
			$post_offset = $offset + ( ( intval($query->query_vars['paged']) - 1 ) * intval($query->query_vars['posts_per_page']) );
		} else {
			$post_offset = $offset;
		}

		$query->set('offset', $post_offset);
	}

	/**
	 * @param int       $found_posts
	 *
	 * @return int
	 */
	public function fix_query_found_posts( $found_posts ) {
		$settings = $this->get_settings_for_display();
		$offset = $settings['posts_offset'];

		if ( $offset ) {
			$found_posts -= $offset;
		}

		return $found_posts;
	}

}

