<?php 
/**
 * Dashboard Page (No Plugins Version)
 *
 * @package Siaracorporatebusiness
 */

if ( ! class_exists( 'Siaracorporatebusiness_Dashboard' ) ) {
	class Siaracorporatebusiness_Dashboard {

		private static $instance;
		private $theme_name;
		private $theme_version;
		private $theme_slug;
		private $page_slug;
		private $review_url;
		private $redirect_template_url;
		private $errors = array();
		private $tab_sections = array();
		private $theme_url = '';
		private $doc_url = '';
		private $support_url = '';

		public static function get_instance() {
			if ( ! isset( self::$instance ) ) {
				self::$instance = new self();
			}
			return self::$instance;
		}

		public function __construct() {
			$theme = wp_get_theme();

			$this->theme_name    = $theme->get( 'Name' );
			$this->theme_version = $theme->get( 'Version' );
			$this->theme_slug    = $theme->get_template();
			$this->page_slug     = $this->theme_slug . '-dashboard';
			$this->review_url    = 'https://wordpress.org/support/theme/' . $this->theme_slug . '/reviews/?rate=5#new-post';

			$this->redirect_template_url = admin_url( 'themes.php?page=' . $this->page_slug . '&section=starter-templates' );

			$this->tab_sections = array(
				'getting-started'   => __( 'Getting Started', 'siara-corporate-business' ),
				'starter-templates' => __( 'Starter Templates', 'siara-corporate-business' ),
			);

			$this->errors = array(
				'permission' => __( 'Sorry, you do not have permission to perform this action.', 'siara-corporate-business' ),
				'nonce'      => __( 'Could not verify the nonce.', 'siara-corporate-business' ),
				'default'    => __( 'Oops! something went wrong.', 'siara-corporate-business' ),
				'invalid'    => __( 'Not found!', 'siara-corporate-business' ),
			);

			if ( current_user_can( 'manage_options' ) ) {
				add_action( 'admin_notices', array( $this, 'display_admin_notice' ), 99 );
				add_action( 'admin_menu', array( $this, 'add_menu' ) );
				add_action( 'admin_bar_menu', array( $this, 'add_bar_menu' ), 999 );
			}

			add_action( 'admin_init', array( $this, 'init_reminder' ) );
			add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
			add_action( 'wp_ajax_siaracorporatebusiness_dismiss_notice', array( $this, 'dismiss_notice' ) );
			add_action( 'switch_theme', array( $this, 'reset_notices' ) );
			add_action( 'after_switch_theme', array( $this, 'reset_notices' ) );
		}

		public function init_reminder() {
			if ( ! get_option( 'siaracorporatebusiness_reminder_time' ) ) {
				update_option( 'siaracorporatebusiness_reminder_time', time() + 14 * DAY_IN_SECONDS );
			}
		}

		public function reset_notices() {
			delete_option( 'siaracorporatebusiness_reminder_time' );
			delete_option( 'siaracorporatebusiness_dismissed_notices' );
		}

		public function display_admin_notice() {
			if ( $this->can_display_notice() && ! $this->is_dismissed( 'welcome' ) ) {
				$this->display_welcome_notice();
			}

			$reminder_time = get_option( 'siaracorporatebusiness_reminder_time' );
			if ( $this->can_display_notice() && ! $this->is_dismissed( 'review' ) && ! empty( $reminder_time ) && time() > $reminder_time ) {
				$this->display_review_notice();
			}
		}

		public function display_welcome_notice() {
			$user = wp_get_current_user();
			?>
			<div id="siaracorporatebusiness-welcome-notice" class="siaracorporatebusiness-dashboard-wrapper notice siaracorporatebusiness-notice is-dismissible">
				<div class="siaracorporatebusiness-notice-wrapper">
					<div class="siaracorporatebusiness-notice-intro">
						<div class="intro__hello">
							<?php esc_html_e( '🤝🏻', 'siara-corporate-business' ); ?>
							<?php esc_html_e( 'Welcome, ', 'siara-corporate-business' ); ?>
							<?php echo esc_html( ucfirst( $user->display_name ) ); ?>
						</div>
						<div class="intro__thanks">
							<?php printf( esc_html__( 'Thank you for using %s', 'siara-corporate-business' ), $this->theme_name ); ?>
							<sup><?php echo esc_html( $this->theme_version ); ?></sup>
						</div>
						<div class="intro__desc">
							<?php
							printf(
								/* translators: %s: Theme Name. */
								esc_html__( '%s is now installed and ready for use. Next, you can speed up your development process by using amazing starter templates or visit the theme dashboard for more information and settings.', 'siara-corporate-business' ),
								$this->theme_name
							);
							?>
						</div>
					</div>
					<div class="siaracorporatebusiness-notice-message">
						<div class="siaracorporatebusiness-notice-image">
							<img src="<?php echo esc_url( get_stylesheet_directory_uri() . '/screenshot.png' ); ?>" alt="<?php esc_attr_e( 'Siaracorporatebusiness Screenshot', 'siara-corporate-business' ); ?>">
						</div>
						<div class="siaracorporatebusiness-notice-template-import">
							<h3><?php esc_html_e( 'Starter Templates', 'siara-corporate-business' ); ?></h3>
							<p><?php esc_html_e( 'Make use of awesome starter website templates & get your site up & ready in no time.', 'siara-corporate-business' ); ?></p>
							<p><a href="<?php echo esc_url( $this->redirect_template_url ); ?>" class="button button-primary"><?php esc_html_e( 'Get Started', 'siara-corporate-business' ); ?></a></p>
						</div>
						<div class="siaracorporatebusiness-notice-theme-dashboard">
							<h3><?php esc_html_e( 'Dashboard', 'siara-corporate-business' ); ?></h3>
							<p><?php esc_html_e( 'To fully take advantage of the best our theme can offer, please make sure to visit the theme dashboard page.', 'siara-corporate-business' ); ?></p>
							<p><a href="<?php echo esc_url( admin_url( 'themes.php?page=' . $this->page_slug ) ); ?>" class="button button-secondary"><?php esc_html_e( 'Theme Dashboard', 'siara-corporate-business' ); ?></a></p>
						</div>
					</div>
				</div>
				<button type="button" class="notice-dismiss siaracorporatebusiness-remove-notice" data-dismiss="welcome">
					<span class="screen-reader-text"><?php esc_html_e( 'Dismiss this notice.', 'siara-corporate-business' ); ?></span>
				</button>
			</div>
			<?php
		}

		public function display_review_notice() {
			?>
			<div id="siaracorporatebusiness-review-notice" class="notice siaracorporatebusiness-notice is-dismissible">
				<div class="siaracorporatebusiness-notice-wrapper">
					<div class="siaracorporatebusiness-notice-message">
						<div class="siaracorporatebusiness-notice-review-content">
							<div class="intro__thanks">
								<?php printf( esc_html__( 'Thank you for using %s', 'siara-corporate-business' ), $this->theme_name ); ?>
							</div>
							<div class="intro__desc">
								<?php printf( esc_html__( 'We hope you love it! Please consider %1$sgiving us a 5-star review%2$s.', 'siara-corporate-business' ), '<a href="' . esc_url( $this->review_url ) . '" target="_blank">', '</a>' ); ?>
							</div>
						</div>
					</div>
				</div>
				<button type="button" class="notice-dismiss siaracorporatebusiness-remove-notice" data-dismiss="review">
					<span class="screen-reader-text"><?php esc_html_e( 'Dismiss this notice.', 'siara-corporate-business' ); ?></span>
				</button>
			</div>
			<?php
		}

		protected function can_display_notice() {
			$screen = get_current_screen();
			if ( empty( $screen ) ) {
				return false;
			}
			return in_array( $screen->id, array( 'dashboard', 'themes' ) );
		}

		public static function is_dismissed( $notice ) {
			$dismissed = get_option( 'siaracorporatebusiness_dismissed_notices', array() );
			return in_array( $notice, $dismissed );
		}

		public static function dismiss( $notice ) {
			$dismissed = get_option( 'siaracorporatebusiness_dismissed_notices', array() );
			if ( ! in_array( $notice, $dismissed ) ) {
				$dismissed[] = $notice;
				update_option( 'siaracorporatebusiness_dismissed_notices', array_unique( $dismissed ) );
			}
		}

		public function add_menu() {
			add_theme_page( __( 'Theme Dashboard', 'siara-corporate-business' ), __( 'Theme Dashboard', 'siara-corporate-business' ), 'manage_options', $this->page_slug, array( $this, 'render_main_page' ), 1 );
		}

		public function add_bar_menu() {
			global $wp_admin_bar;
			if ( ! is_super_admin() || ! is_admin_bar_showing() ) {
				return;
			}
			$wp_admin_bar->add_menu( array(
				'parent' => 'site-name',
				'id'     => $this->page_slug,
				'title'  => esc_html__( 'Theme Dashboard', 'siara-corporate-business' ),
				'href'   => admin_url( 'themes.php?page=' . $this->page_slug ),
			) );
		}

		public function render_main_page() {
			$tabs = $this->tab_sections;
			?>
			<div class="siaracorporatebusiness-dashboard-wrapper">
				<?php require_once get_template_directory() . '/inc/dashboard/sections/header.php'; ?>
				<?php
				$section = isset( $_GET['section'] ) ? sanitize_text_field( wp_unslash( $_GET['section'] ) ) : '';
				switch ( $section ) {
					case 'starter-templates':
						require get_template_directory() . '/inc/dashboard/sections/starter-templates.php';
						break;
					default:
						require get_template_directory() . '/inc/dashboard/sections/getting-started.php';
				}
				?>
			</div>
			<?php
		}

		public function enqueue_scripts( $hook ) {
			if ( ! in_array( $hook, array( 'index.php', 'themes.php', 'appearance_page_' . $this->page_slug ) ) ) {
				return;
			}
			$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
			wp_enqueue_style( 'siaracorporatebusiness-dashboard-style', get_template_directory_uri() . '/inc/dashboard/css/style' . $min . '.css', array(), $this->theme_version );
			wp_enqueue_script( 'siaracorporatebusiness-dashboard-script', get_template_directory_uri() . '/inc/dashboard/js/build.js', array( 'jquery' ), $this->theme_version, true );
			wp_localize_script( 'siaracorporatebusiness-dashboard-script', 'siaracorporatebusinessDashboard', array(
				'notice_dismiss_nonce' => wp_create_nonce( 'siaracorporatebusiness-dismiss-nonce' ),
			) );
		}

		public function dismiss_notice() {
			$response_data = array( 'message' => $this->get_error_msg( 'permission' ) );
			if ( ! current_user_can( 'manage_options' ) ) {
				wp_send_json_error( $response_data );
			}
			if ( empty( $_POST ) ) {
				$response_data = array( 'message' => $this->get_error_msg( 'invalid' ) );
				wp_send_json_error( $response_data );
			}
			if ( ! check_ajax_referer( 'siaracorporatebusiness-dismiss-nonce', 'nonce', false ) ) {
				$response_data = array( 'message' => $this->get_error_msg( 'nonce' ) );
				wp_send_json_error( $response_data );
			}
			$dismiss = isset( $_POST['dismiss'] ) ? sanitize_text_field( wp_unslash( $_POST['dismiss'] ) ) : '';
			if ( ! $dismiss ) {
				wp_send_json_error( array( 'success' => false, 'message' => __( 'No dismissable notice found.', 'siara-corporate-business' ) ) );
			}
			$dismiss_type = isset( $_POST['dismiss_type'] ) ? sanitize_text_field( wp_unslash( $_POST['dismiss_type'] ) ) : '';
			if ( $dismiss_type && 'remind' === $dismiss_type ) {
				update_option( 'siaracorporatebusiness_reminder_time', time() + 14 * DAY_IN_SECONDS );
			} else {
				self::dismiss( $dismiss );
			}
			wp_send_json_success( array( 'success' => true, 'message' => __( 'Notice Dismissed Successfully', 'siara-corporate-business' ) ) );
		}
	}
}

add_action( 'after_setup_theme', function() {
	Siaracorporatebusiness_Dashboard::get_instance();
} );
