<?php
/**
 * Custom Functions
 * @package Outdoor Adventure
 * @since 1.0.0
 */

if( !function_exists('outdoor_adventure_site_logo') ):

    /**
     * Logo & Description
     */
    /**
     * Displays the site logo, either text or image.
     *
     * @param array $outdoor_adventure_args Arguments for displaying the site logo either as an image or text.
     * @param boolean $outdoor_adventure_echo Echo or return the HTML.
     *
     * @return string $outdoor_adventure_html Compiled HTML based on our arguments.
     */
    function outdoor_adventure_site_logo( $outdoor_adventure_args = array(), $outdoor_adventure_echo = true ){
        $outdoor_adventure_logo = get_custom_logo();
        $outdoor_adventure_site_title = get_bloginfo('name');
        $outdoor_adventure_contents = '';
        $outdoor_adventure_classname = '';
        $outdoor_adventure_defaults = array(
            'logo' => '%1$s<span class="screen-reader-text">%2$s</span>',
            'logo_class' => 'site-logo site-branding',
            'title' => '<a href="%1$s" class="custom-logo-name">%2$s</a>',
            'title_class' => 'site-title',
            'home_wrap' => '<h1 class="%1$s">%2$s</h1>',
            'single_wrap' => '<div class="%1$s">%2$s</div>',
            'condition' => (is_front_page() || is_home()) && !is_page(),
        );
        $outdoor_adventure_args = wp_parse_args($outdoor_adventure_args, $outdoor_adventure_defaults);
        /**
         * Filters the arguments for `outdoor_adventure_site_logo()`.
         *
         * @param array $outdoor_adventure_args Parsed arguments.
         * @param array $outdoor_adventure_defaults Function's default arguments.
         */
        $outdoor_adventure_args = apply_filters('outdoor_adventure_site_logo_args', $outdoor_adventure_args, $outdoor_adventure_defaults);
        
        $outdoor_adventure_show_logo  = get_theme_mod('outdoor_adventure_display_logo', false);
        $outdoor_adventure_show_title = get_theme_mod('outdoor_adventure_display_title', true);

        if ( has_custom_logo() && $outdoor_adventure_show_logo ) {
            $outdoor_adventure_contents .= sprintf($outdoor_adventure_args['logo'], $outdoor_adventure_logo, esc_html($outdoor_adventure_site_title));
            $outdoor_adventure_classname = $outdoor_adventure_args['logo_class'];
        }

        if ( $outdoor_adventure_show_title ) {
            $outdoor_adventure_contents .= sprintf($outdoor_adventure_args['title'], esc_url(get_home_url(null, '/')), esc_html($outdoor_adventure_site_title));
            // If logo isn't shown, fallback to title class for wrapper.
            if ( !$outdoor_adventure_show_logo ) {
                $outdoor_adventure_classname = $outdoor_adventure_args['title_class'];
            }
        }

        // If nothing is shown (logo or title both disabled), exit early
        if ( empty($outdoor_adventure_contents) ) {
            return;
        }

        $outdoor_adventure_wrap = $outdoor_adventure_args['condition'] ? 'home_wrap' : 'single_wrap';
        // $outdoor_adventure_wrap = 'home_wrap';
        $outdoor_adventure_html = sprintf($outdoor_adventure_args[$outdoor_adventure_wrap], $outdoor_adventure_classname, $outdoor_adventure_contents);
        /**
         * Filters the arguments for `outdoor_adventure_site_logo()`.
         *
         * @param string $outdoor_adventure_html Compiled html based on our arguments.
         * @param array $outdoor_adventure_args Parsed arguments.
         * @param string $outdoor_adventure_classname Class name based on current view, home or single.
         * @param string $outdoor_adventure_contents HTML for site title or logo.
         */
        $outdoor_adventure_html = apply_filters('outdoor_adventure_site_logo', $outdoor_adventure_html, $outdoor_adventure_args, $outdoor_adventure_classname, $outdoor_adventure_contents);
        if (!$outdoor_adventure_echo) {
            return $outdoor_adventure_html;
        }
        echo $outdoor_adventure_html; //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

    }

endif;

if( !function_exists('outdoor_adventure_site_description') ):

    /**
     * Displays the site description.
     *
     * @param boolean $outdoor_adventure_echo Echo or return the html.
     *
     * @return string $outdoor_adventure_html The HTML to display.
     */
    function outdoor_adventure_site_description($outdoor_adventure_echo = true){

        if ( get_theme_mod('outdoor_adventure_display_header_text', false) == true ) :
        $outdoor_adventure_description = get_bloginfo('description');
        if (!$outdoor_adventure_description) {
            return;
        }
        $outdoor_adventure_wrapper = '<div class="site-description"><span>%s</span></div><!-- .site-description -->';
        $outdoor_adventure_html = sprintf($outdoor_adventure_wrapper, esc_html($outdoor_adventure_description));
        /**
         * Filters the html for the site description.
         *
         * @param string $outdoor_adventure_html The HTML to display.
         * @param string $outdoor_adventure_description Site description via `bloginfo()`.
         * @param string $outdoor_adventure_wrapper The format used in case you want to reuse it in a `sprintf()`.
         * @since 1.0.0
         *
         */
        $outdoor_adventure_html = apply_filters('outdoor_adventure_site_description', $outdoor_adventure_html, $outdoor_adventure_description, $outdoor_adventure_wrapper);
        if (!$outdoor_adventure_echo) {
            return $outdoor_adventure_html;
        }
        echo $outdoor_adventure_html; //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
        endif;
    }

endif;

if( !function_exists('outdoor_adventure_posted_on') ):

    /**
     * Prints HTML with meta information for the current post-date/time.
     */
    function outdoor_adventure_posted_on( $outdoor_adventure_icon = true, $outdoor_adventure_animation_class = '' ){

        $outdoor_adventure_default = outdoor_adventure_get_default_theme_options();
        $outdoor_adventure_post_date = absint( get_theme_mod( 'outdoor_adventure_post_date',$outdoor_adventure_default['outdoor_adventure_post_date'] ) );

        if( $outdoor_adventure_post_date ){

            $outdoor_adventure_time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';
            if (get_the_time('U') !== get_the_modified_time('U')) {
                $outdoor_adventure_time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time><time class="updated" datetime="%3$s">%4$s</time>';
            }

            $outdoor_adventure_time_string = sprintf($outdoor_adventure_time_string,
                esc_attr(get_the_date(DATE_W3C)),
                esc_html(get_the_date()),
                esc_attr(get_the_modified_date(DATE_W3C)),
                esc_html(get_the_modified_date())
            );

            $outdoor_adventure_year = get_the_date('Y');
            $outdoor_adventure_month = get_the_date('m');
            $outdoor_adventure_day = get_the_date('d');
            $outdoor_adventure_link = get_day_link($outdoor_adventure_year, $outdoor_adventure_month, $outdoor_adventure_day);

            $outdoor_adventure_posted_on = '<a href="' . esc_url($outdoor_adventure_link) . '" rel="bookmark">' . $outdoor_adventure_time_string . '</a>';

            echo '<div class="entry-meta-item entry-meta-date">';
            echo '<div class="entry-meta-wrapper '.esc_attr( $outdoor_adventure_animation_class ).'">';

            if( $outdoor_adventure_icon ){

                echo '<span class="entry-meta-icon calendar-icon"> ';
                outdoor_adventure_the_theme_svg('calendar');
                echo '</span>';

            }

            echo '<span class="posted-on">' . $outdoor_adventure_posted_on . '</span>'; // phpcs:ignore Standard.Category.SniffName.ErrorCode
            echo '</div>';
            echo '</div>';

        }

    }

endif;

if( !function_exists('outdoor_adventure_posted_by') ) :

    /**
     * Prints HTML with meta information for the current author.
     */
    function outdoor_adventure_posted_by( $outdoor_adventure_icon = true, $outdoor_adventure_animation_class = '' ){   

        $outdoor_adventure_default = outdoor_adventure_get_default_theme_options();
        $outdoor_adventure_post_author = absint( get_theme_mod( 'outdoor_adventure_post_author',$outdoor_adventure_default['outdoor_adventure_post_author'] ) );

        if( $outdoor_adventure_post_author ){

            echo '<div class="entry-meta-item entry-meta-author">';
            echo '<div class="entry-meta-wrapper '.esc_attr( $outdoor_adventure_animation_class ).'">';

            if( $outdoor_adventure_icon ){
            
                echo '<span class="entry-meta-icon author-icon"> ';
                outdoor_adventure_the_theme_svg('user');
                echo '</span>';
                
            }

            $outdoor_adventure_byline = '<span class="author vcard"><a class="url fn n" href="' . esc_url( get_author_posts_url( get_the_author_meta('ID') ) ) . '">' . esc_html(get_the_author()) . '</a></span>';
            echo '<span class="byline"> ' . $outdoor_adventure_byline . '</span>'; // phpcs:ignore Standard.Category.SniffName.ErrorCode
            echo '</div>';
            echo '</div>';

        }

    }

endif;


if( !function_exists('outdoor_adventure_posted_by_avatar') ) :

    /**
     * Prints HTML with meta information for the current author.
     */
    function outdoor_adventure_posted_by_avatar( $outdoor_adventure_date = false ){

        $outdoor_adventure_default = outdoor_adventure_get_default_theme_options();
        $outdoor_adventure_post_author = absint( get_theme_mod( 'outdoor_adventure_post_author',$outdoor_adventure_default['outdoor_adventure_post_author'] ) );

        if( $outdoor_adventure_post_author ){



            echo '<div class="entry-meta-left">';
            echo '<div class="entry-meta-item entry-meta-avatar">';
            echo wp_kses_post( get_avatar( get_the_author_meta( 'ID' ) ) );
            echo '</div>';
            echo '</div>';

            echo '<div class="entry-meta-right">';

            $outdoor_adventure_byline = '<span class="author vcard"><a class="url fn n" href="' . esc_url( get_author_posts_url( get_the_author_meta('ID') ) ) . '">' . esc_html(get_the_author()) . '</a></span>';

            echo '<div class="entry-meta-item entry-meta-byline"> ' . $outdoor_adventure_byline . '</div>';

            if( $outdoor_adventure_date ){
                outdoor_adventure_posted_on($outdoor_adventure_icon = false);
            }
            echo '</div>';

        }

    }

endif;

if( !function_exists('outdoor_adventure_entry_footer') ):

    /**
     * Prints HTML with meta information for the categories, tags and comments.
     */
    function outdoor_adventure_entry_footer( $outdoor_adventure_cats = true, $outdoor_adventure_tags = true, $outdoor_adventure_edits = true){   

        $outdoor_adventure_default = outdoor_adventure_get_default_theme_options();
        $outdoor_adventure_post_category = absint( get_theme_mod( 'outdoor_adventure_post_category',$outdoor_adventure_default['outdoor_adventure_post_category'] ) );
        $outdoor_adventure_post_tags = absint( get_theme_mod( 'outdoor_adventure_post_tags',$outdoor_adventure_default['outdoor_adventure_post_tags'] ) );

        // Hide category and tag text for pages.
        if ('post' === get_post_type()) {

            if( $outdoor_adventure_cats && $outdoor_adventure_post_category ){

                /* translators: used between list items, there is a space after the comma */
                $outdoor_adventure_categories = get_the_category();
                if ($outdoor_adventure_categories) {
                    echo '<div class="entry-meta-item entry-meta-categories">';
                    echo '<div class="entry-meta-wrapper">';
                
                    /* translators: 1: list of categories. */
                    echo '<span class="cat-links">';
                    foreach( $outdoor_adventure_categories as $outdoor_adventure_category ){

                        $outdoor_adventure_cat_name = $outdoor_adventure_category->name;
                        $outdoor_adventure_cat_slug = $outdoor_adventure_category->slug;
                        $outdoor_adventure_cat_url = get_category_link( $outdoor_adventure_category->term_id );
                        ?>

                        <a class="twp_cat_<?php echo esc_attr( $outdoor_adventure_cat_slug ); ?>" href="<?php echo esc_url( $outdoor_adventure_cat_url ); ?>" rel="category tag"><?php echo esc_html( $outdoor_adventure_cat_name ); ?></a>

                    <?php }
                    echo '</span>'; // phpcs:ignore Standard.Category.SniffName.ErrorCode
                    echo '</div>';
                    echo '</div>';
                }

            }

            if( $outdoor_adventure_tags && $outdoor_adventure_post_tags ){
                /* translators: used between list items, there is a space after the comma */
                $outdoor_adventure_tags_list = get_the_tag_list('', esc_html_x(', ', 'list item separator', 'outdoor-adventure'));
                if( $outdoor_adventure_tags_list ){

                    echo '<div class="entry-meta-item entry-meta-tags">';
                    echo '<div class="entry-meta-wrapper">';

                    /* translators: 1: list of tags. */
                    echo '<span class="tags-links">';
                    echo wp_kses_post($outdoor_adventure_tags_list) . '</span>'; // phpcs:ignore Standard.Category.SniffName.ErrorCode
                    echo '</div>';
                    echo '</div>';

                }

            }

            if( $outdoor_adventure_edits ){

                edit_post_link(
                    sprintf(
                        wp_kses(
                        /* translators: %s: Name of current post. Only visible to screen readers */
                            __('Edit <span class="screen-reader-text">%s</span>', 'outdoor-adventure'),
                            array(
                                'span' => array(
                                    'class' => array(),
                                ),
                            )
                        ),
                        get_the_title()
                    ),
                    '<span class="edit-link">',
                    '</span>'
                );
            }

        }
    }

endif;

if ( ! function_exists( 'outdoor_adventure_post_thumbnail' ) ) :

    /**
     * Displays an optional post thumbnail.
     *
     * Shows background style image with height class on archive/search/front,
     * and a normal inline image on single post/page views.
     */
    function outdoor_adventure_post_thumbnail( $outdoor_adventure_image_size = 'full' ) {

        if ( post_password_required() || is_attachment() ) {
            return;
        }

        // Fallback image path
        $outdoor_adventure_default_image = get_template_directory_uri() . '/inc/homepage-setup/assets/homepage-setup-images/post-img4.png';

        // Image size → height class map
        $outdoor_adventure_size_class_map = array(
            'full'      => 'data-bg-large',
            'large'     => 'data-bg-big',
            'medium'    => 'data-bg-medium',
            'small'     => 'data-bg-small',
            'xsmall'    => 'data-bg-xsmall',
            'thumbnail' => 'data-bg-thumbnail',
        );

        $outdoor_adventure_class = isset( $outdoor_adventure_size_class_map[ $outdoor_adventure_image_size ] )
            ? $outdoor_adventure_size_class_map[ $outdoor_adventure_image_size ]
            : 'data-bg-medium';

        if ( is_singular() ) {
            the_post_thumbnail();
        } else {
            // 🔵 On archives → use background image style
            $outdoor_adventure_image = has_post_thumbnail()
                ? wp_get_attachment_image_src( get_post_thumbnail_id(), $outdoor_adventure_image_size )
                : array( $outdoor_adventure_default_image );

            $outdoor_adventure_bg_image = isset( $outdoor_adventure_image[0] ) ? $outdoor_adventure_image[0] : $outdoor_adventure_default_image;
            ?>
            <div class="post-thumbnail data-bg <?php echo esc_attr( $outdoor_adventure_class ); ?>"
                 data-background="<?php echo esc_url( $outdoor_adventure_bg_image ); ?>">
                <a href="<?php the_permalink(); ?>" class="theme-image-responsive" tabindex="0"></a>
            </div>
            <?php
        }
    }

endif;

if( !function_exists('outdoor_adventure_is_comment_by_post_author') ):

    /**
     * Comments
     */
    /**
     * Check if the specified comment is written by the author of the post commented on.
     *
     * @param object $outdoor_adventure_comment Comment data.
     *
     * @return bool
     */
    function outdoor_adventure_is_comment_by_post_author($outdoor_adventure_comment = null){

        if (is_object($outdoor_adventure_comment) && $outdoor_adventure_comment->user_id > 0) {
            $outdoor_adventure_user = get_userdata($outdoor_adventure_comment->user_id);
            $post = get_post($outdoor_adventure_comment->comment_post_ID);
            if (!empty($outdoor_adventure_user) && !empty($post)) {
                return $outdoor_adventure_comment->user_id === $post->post_author;
            }
        }
        return false;
    }

endif;

if( !function_exists('outdoor_adventure_breadcrumb') ) :

    /**
     * Outdoor Adventure Breadcrumb
     */
    function outdoor_adventure_breadcrumb($outdoor_adventure_comment = null){

        echo '<div class="entry-breadcrumb">';
        outdoor_adventure_breadcrumb_trail();
        echo '</div>';

    }

endif;