<?php

namespace Modules\User\Http\Controllers\Admin;

use Artesaos\SEOTools\Traits\SEOTools;
use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Route;
use Illuminate\Validation\Rule;
use Modules\ACL\Entities\Role;
use Modules\User\Entities\User;

class UserController extends Controller
{
    use SEOTools;
    public function __construct()
    {
        $this->middleware('can:manage-users');
    }

    /**
     * Display a listing of the resource.
     * @return Renderable
     */
    public function index()
    {
        $this->seo()->setTitle('همه ی کاربران');
        $users = User::query();
        if ($key = \request('search')) {
            $users = User::where('name', 'like', "%$key%")
                ->orWhere('phone', 'like', "%$key%")->orWhere('id',$key);
        }
        $users = $users->latest()->simplePaginate(15);
        return view('user::backend.all', compact('users'));
    }


    public function create()
    {
        $this->seo()->setTitle('ایجاد کاربر جدید');
        $roles = Role::all();
        return view('user::backend.create', compact('roles'));
    }


    public function store(Request $request)
    {
        $validData = $request->validate([
            'name' => 'required|min:3',
            'phone' => [
                'required',
                'unique:users',
                'regex:/^(?:0|98|\+98|\+980|0098|098|00980)?(9\d{9})$/'
            ],
            'password' => 'nullable',
            'role' => 'required'
        ]);
        if( empty( $validData['password'] ) ){
            $user = User::create([
                'name' => $validData['name'],
                'phone' => $validData['phone'],
            ]);
        }else {
            $user = User::create([
                'name' => $validData['name'],
                'phone' => $validData['phone'],
                'password' => Hash::make($validData['password']),
            ]);
        }
        if ($validData['role'] == 1) {
            $user->update([
               'is_admin' => 1
            ]);
        }
        $user->roles()->attach($validData['role']);
        $user->save();
        alert()->success('کاربر با موفقیت ساخته شد.', 'موفق');
        return back();
    }


    public function show($id)
    {
        return view('user::show');
    }


    public function edit(User $user)
    {
        $this->seo()->setTitle("ویرایش کاربر $user->name");
        $roles = Role::all();
        return view('user::backend.edit', compact(['user', 'roles']));
    }


    public function update(Request $request, User $user)
    {
        $validData = $request->validate([
            'name' => 'required|min:3',
            'phone' => [
                'required',
                Rule::unique('users')->ignore($user->id)
            ],
            'password' => 'nullable',
            'role' => 'required'
        ]);

        if( empty( $validData['password'] ) ){
            $user->update([
                'name' => $validData['name'],
                'phone' => $validData['phone'],
                ]);
        } else {
            $user->update([
                'name' => $validData['name'],
                'phone' => $validData['phone'],
                'password' => Hash::make($validData['password']),
            ]);
        }

        if ($validData['role'] == 1) {
            $user->update([
               'is_admin' => 1
            ]);
        }else {
            $user->update([
                'is_admin' => 0
            ]);
        }
        $user->roles()->sync($request->input('role'));
        alert()->success("ویرایش کاربر  $user->name با موفقیت انجام شد.", 'موفق');
        return back();
    }


    public function destroy(User $user)
    {
        $user->delete();
        alert()->success(" کاربر  $user->name با موفقیت حذف شد.", 'موفق');
        return redirect(route('admin.users.index'));
    }
}
