<?php

namespace Modules\Product\Http\Controllers\Admin;

use Artesaos\SEOTools\Traits\SEOTools;
use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Modules\Attribute\Entities\Attribute;
use Modules\Category\Entities\Category;
use Modules\Photo\Entities\Photo;
use Modules\Price\Entities\Price;
use Modules\Product\Entities\Product;

class ProductController extends Controller
{
    use SEOTools;

    public function __construct()
    {
        $this->middleware('can:manage-products');
    }

    /**
     * Display a listing of the resource.
     * @return Renderable
     */
    public function index()
    {
        $this->seo()->setTitle('همه محصولات');

        $products = Product::query();
        if ($key = \request('search')) {
            $products = $products->where('title', 'like', "%$key%")->orWhere('sku' , 'like' , "%$key%");
        }
        $products = $products->latest()->simplePaginate(15);
        return view('product::backend.all', compact('products'));
    }

    /**
     * Show the form for creating a new resource.
     * @return Renderable
     */
    public function create()
    {
        $this->seo()->setTitle('ایجاد محصول جدید');

        $categories = Category::with(['products', 'childrenRecursive'])
            ->where('category_type', Product::class)
            ->whereNull('parent_id')
            ->get();
        $prices = Price::all();
        return view('product::backend.create', ['categories' => $categories , 'prices' => $prices]);
    }


    public function store(Request $request)
    {

        $validData = $request->validate([
            'title' => 'required',
            'slug' => 'unique:products|nullable',
            'long_description' => 'nullable',
            'short_description' => 'nullable',
            'price' => 'nullable',
            'sale_price' => 'nullable',
            'inventory' => 'nullable',
            'is_single_sell' => 'nullable',
            'sku' => 'nullable',
            'brand' => 'nullable',
            'categories' => 'nullable',
            'meta_title' => 'nullable',
            'meta_keywords' => 'nullable',
            'meta_description' => 'nullable',
            'status' => 'nullable',
            'english_title' => 'nullable',
            'min_order_quantity' => 'nullable',
            'pin_count' => 'nullable',
            'attributes' => 'array|nullable',
            'price_type' => 'nullable',
            'photos.*' => [
                'nullable',
                'image'
            ]
        ]);
        //create product
        $product = new Product();
        $product->title = $validData['title'];
        $product->english_title = $validData['english_title'];
        $product->long_description = $validData['long_description'];
        $product->short_description = $validData['short_description'];
        if ($validData['price_type'] != 'normal') {
            $priceType = Price::whereId($validData['price_type'])->first();
            $price = $priceType->price * $validData['pin_count'];
            $product->price = $price;
            $product->price_id = $validData['price_type'];
            $product->pin_count = $validData['pin_count'];
        }else {
            $product->price = $validData['price'];
            $product->sale_price = $validData['sale_price'];
        }
        $product->inventory = $validData['inventory'];
        $product->is_single_sell = $validData['is_single_sell'];
        $product->sku = $validData['sku'];
        $product->brand = $validData['brand'];
        $product->meta_title = $validData['meta_title'];
        $product->meta_keywords = $validData['meta_keywords'];
        $product->meta_description = $validData['meta_description'];
        $product->single_image = $request->image;
        $product->user_id = auth()->user()->id;
        if($validData['min_order_quantity'] == 0 || is_null($validData['min_order_quantity'])) {
            $product->min_order_quantity = 1;
        }else{
        	$product->min_order_quantity = $validData['min_order_quantity'];
        }
        if (array_key_exists('status', $validData)) {
            $validData['status'] = 1;
            $product->status = $validData['status'];
        } else {
            $product->status = 0;
        }
        if ($request->input('slug') != "") {

            $product->slug = str_slug($validData['slug']);
        } else {
            $product->slug = str_slug($validData['title']);
        }

        $product->save();


        //upload photos
        if ($files = $request->file('photos')) {
            foreach ($request->file('photos') as $file) {
                $name = time() . $file->getClientOriginalName();
                $path = $file->move('images/products/' . $product->id . '/', $name);
                $photo = new Photo();
                $photo->path = 'images/products/' . $product->id . '/' . $name;
                $photo->save();
                $photo->products()->attach($product->id);
            }
        }

        //attach categories
        if ($request->categories != null) {
            $categories = Category::find($validData['categories']);
            foreach ($categories as $category) {
                $product->categories()->attach($category->id);
            }
        }

        if (isset($validData['attributes'])) {
            //add or edit attributes
            $this->attachAttributesToProduct($validData['attributes'], $product);
        }
        alert()->success('محصول با موفقیت ایجاد شد', 'موفق');
        return redirect(route('admin.products.index'));

    }

    /**
     * Show the specified resource.
     * @param int $id
     * @return Renderable
     */
    public function show($id)
    {
        return view('product::show');
    }

    /**
     * Show the form for editing the specified resource.
     * @param int $id
     * @return Renderable
     */
    public function edit(Product $product)
    {
        $this->seo()->setTitle("ویرایش محصول $product->title");
        $categories = Category::with('posts')
            ->where('category_type', Product::class)
            ->whereNull('parent_id')
            ->get();
        $prices = Price::all();

        return view('product::backend.edit', [
            'product' => $product,
            'categories' => $categories,
            'prices' => $prices
            ]);
    }


    public function update(Request $request, Product $product)
    {

        $validData = productUpdateValidator($request, $product);
        //if status exists
        if (array_key_exists('status', $validData)) {
            $validData['status'] = 1;
            productUpdate($product, $validData, $request, $validData['status']);
        } else {
            //if status does not exists
            productUpdate($product, $validData, $request, 0);
        }

        if ($request->categories != null) {
            $categories = Category::find($request->categories);
            $cat_ids = $categories->mapWithKeys(function ($item) {
                return [$item['id'] => ['category_id' => $item['id']]];
            });
            $product->categories()->sync($cat_ids);
        }

        //get new and old product photos
        if ($product->photos) {
            $oldProductPhotos = [];
            foreach ($product->photos as $photo) {
                $oldProductPhotos[] = $photo->id;
            }
            $newProductPhotos = explode(',', $validData['new-product-photos']);

            $photosIntersect = array_intersect($oldProductPhotos, $newProductPhotos);
            $product->photos()->sync($photosIntersect);
        }


        //upload photos
        if ($files = $request->file('photos')) {
            foreach ($request->file('photos') as $file) {
                $name = time() . $file->getClientOriginalName();
                $path = $file->move('images/products/' . $product->id . '/', $name);
                $photo = new Photo();
                $photo->path = 'images/products/' . $product->id . '/' . $name;
                $photo->save();
                $photo->products()->sync($product->id);
            }
        }



        if (isset($validData['attributes'])) {
            //add or edit attributes
            $product->attributes()->detach();
            $this->attachAttributesToProduct($validData['attributes'], $product);
        }

        alert()->success('محصول با موفقیت ویرایش شد', 'موفق');
        return back();
    }

    /**
     * Remove the specified resource from storage.
     * @param int $id
     * @return Renderable
     */
    public function destroy(Product $product)
    {
        $product->delete();
        alert()->success('محصول با موفقیت حذف شد.');
        return back();
    }

    /**
     * @param $attributes1
     * @param Product $product
     * @return void
     */
    protected function attachAttributesToProduct($attributes1, Product $product): void
    {
        $attributes = collect($attributes1);
        $attributes->each(function ($item) use ($product) {
            if (is_null($item['name']) || is_null($item['value'])) return;

            $attr = Attribute::firstOrCreate([
                'name' => $item['name']
            ]);

            $attr_value = $attr->values()->firstOrCreate([
                'value' => $item['value']
            ]);

            $product->attributes()->attach($attr->id, ['value_id' => $attr_value->id]);
        });
    }

    public function getCat(Request $request)
    {
        if($request->ajax()) {
            $category = Category::whereId($request->id)->first();
            if (!is_null($request->id)) {
                if (!is_null($category->parent_id)) {
                    $parent = Category::whereId($category->parent_id)->first();
                    return response()->json([
                        'data' => $parent,
                        'parent' => false
                    ]);
                }else {
                    return response()->json([
                        'data' => $category,
                        'parent' => true
                    ]);
                }
            }
            return response()->json([
                'data' => 'قیمت پین را وارد کنید',
                'parent' => null
            ]);
        }
    }

}
