<?php
/**
 * API Queue Processor
 * 
 * This script runs for 1 minute and continuously processes queue items
 * It should be run via a cron job every minute
 */

// Initialize WHMCS
require_once(__DIR__ . '/../../../init.php');
require_once(__DIR__ . '/hooks.php');

use WHMCS\Database\Capsule;

// Set time limit to 60 seconds
set_time_limit(60);

// Set script start time
$startTime = time();
$endTime = $startTime + 55; // Run for 55 seconds to allow for script overhead

// Initialize counters
$processed = 0;
$success = 0;
$failed = 0;

// Log start of processing
hourlyLog("API Queue Processor started");

// Process queue items until time limit is reached
while (time() < $endTime) {
    // Get pending queue items
    $queueItems = Capsule::table('mod_HourlyAddonSeller_api_queue')
        ->where('status', '=', 'pending')
        ->orWhere(function ($query) {
            $query->where('status', '=', 'failed')
                  ->where('attempts', '<', 3);
        })
        ->orderBy('created_at', 'asc')
        ->limit(5)
        ->get();
    
    // If no items found, wait a bit and check again
    if (count($queueItems) == 0) {
        sleep(2);
        continue;
    }
    
    // Process each item
    foreach ($queueItems as $item) {
        $processed++;
        
        // Process the item
        $result = processApiQueueItem((array) $item);
        
        if ($result) {
            $success++;
        } else {
            $failed++;
        }
        
        // Check if we're approaching the time limit
        if (time() >= $endTime) {
            break;
        }
    }
    
    // Small pause between batches
    usleep(200000); // 0.2 seconds
}

// Log results
hourlyLog("API Queue Processor finished: Processed {$processed} items - Success: {$success}, Failed: {$failed}");

/**
 * Process a single API queue item
 * 
 * @param array $queueItem Queue item to process
 * @return bool Success status
 */
function processApiQueueItem($queueItem)
{
    try {
        // Mark item as processing
        Capsule::table('mod_HourlyAddonSeller_api_queue')
            ->where('id', '=', $queueItem['id'])
            ->update([
                'status' => 'processing',
                'attempts' => $queueItem['attempts'] + 1,
                'last_attempt' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            ]);
        
        // Parse request data
        $requestData = json_decode($queueItem['request_data'], true);
        if (!$requestData || !isset($requestData['action'])) {
            throw new Exception("Invalid request data");
        }
        
        // Get API key
        $apiKey = Hook_HourlyAddonSeller_getApiKey();
        if (empty($apiKey)) {
            throw new Exception("API key not configured");
        }
        
        // API endpoint URL
        $apiUrl = "https://cl.keloncloud.com/api/hourlyseller/action.php";
        
        // API authentication
        $headers = [
            'Authorization: Bearer ' . $apiKey,
            'Content-Type: application/json'
        ];
        
        // Send request
        $ch = curl_init($apiUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($requestData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        
        curl_close($ch);
        
        // Check for cURL errors
        if ($error) {
            throw new Exception("cURL Error: " . $error);
        }
        
        // Check HTTP response code
        if ($httpCode < 200 || $httpCode >= 300) {
            throw new Exception("API Error: HTTP Code " . $httpCode . " - " . $response);
        }
        
        // Parse response
        $responseData = json_decode($response, true);
        if (!$responseData || !isset($responseData['status'])) {
            throw new Exception("Invalid API response: " . $response);
        }
        
        // Check API response status
        if ($responseData['status'] != 'success') {
            $errorMessage = isset($responseData['message']) ? $responseData['message'] : 'Unknown error';
            throw new Exception("API Error: " . $errorMessage);
        }
        
        // Mark item as completed
        // Capsule::table('mod_HourlyAddonSeller_api_queue')
        //     ->where('id', '=', $queueItem['id'])
        //     ->update([
        //         'status' => 'completed',
        //         'updated_at' => date('Y-m-d H:i:s')
        //     ]);
        Capsule::table('mod_HourlyAddonSeller_api_queue')
            ->where('id', '=', $queueItem['id'])
            ->delete();

        hourlyLog("API Request processed successfully: " . $requestData['action'] . " (Queue ID: " . $queueItem['id'] . ")");
        
        return true;
    } catch (Exception $e) {
        // Mark item as failed
        Capsule::table('mod_HourlyAddonSeller_api_queue')
            ->where('id', '=', $queueItem['id'])
            ->update([
                'status' => 'failed',
                'error' => $e->getMessage(),
                'updated_at' => date('Y-m-d H:i:s')
            ]);
        
        hourlyLog("API Request failed: " . $e->getMessage() . " (Queue ID: " . $queueItem['id'] . ")", 'error');
        
        return false;
    }
}