<?php
require_once(__DIR__ . '/../../../init.php');

use \WHMCS\Service\Service;
use \WHMCS\Database\Capsule;
use WHMCS\View\Menu\Item as MenuItem;

if (!defined('WHMCS')) {
    die('This hook should not be run directly');
}

// Global logging control
$GLOBALS['hourlyAddonLoggingEnabled'] = true;

/**
 * Custom logging function that can be enabled/disabled
 * 
 * @param string $message Message to log
 * @param string $level Log level (debug, info, notice, warning, error)
 * @return void
 */
function hourlyLog($message, $level = 'info')
{
    if (!isset($GLOBALS['hourlyAddonLoggingEnabled']) || $GLOBALS['hourlyAddonLoggingEnabled'] === true) {
        logActivity("[HourlyAddonSeller] " . $message);
    }
}

/**
 * Enable or disable logging
 * 
 * @param bool $enabled Whether logging should be enabled
 * @return void
 */
function setHourlyLogging($enabled = true)
{
    $GLOBALS['hourlyAddonLoggingEnabled'] = (bool) $enabled;
}

/**
 * Add API request to queue
 * @param array $params Parameters for the API request
 * @return int Queue ID
 */
function addToApiQueue($params)
{
    // Set default action if not provided
    if (!isset($params['action'])) {
        $params['action'] = 'unknown';
    }

    // Add timestamp if not provided
    if (!isset($params['timestamp'])) {
        $params['timestamp'] = time();
    }

    // Insert request into queue
    $values = [
        'request_data' => json_encode($params),
        'status' => 'pending',
        'attempts' => 0,
        'created_at' => date('Y-m-d H:i:s'),
        'updated_at' => date('Y-m-d H:i:s')
    ];

    $queueId = insert_query('mod_HourlyAddonSeller_api_queue', $values);

    // Log queue addition
    hourlyLog("API Request queued: {$params['action']} - Queue ID: {$queueId}");

    return $queueId;
}

/**
 * Send API request directly (not using queue)
 * @param array $params Parameters for the API request
 * @return array Response data
 */
function sendApiRequest($params)
{
    try {
        // Add to queue instead of sending directly
        return ['queue_id' => addToApiQueue($params)];
    } catch (Exception $e) {
        hourlyLog("Error queueing API request: " . $e->getMessage(), 'error');
        return ['error' => $e->getMessage()];
    }
}

/**
 * Get API key from database
 * @return string API key
 */
function Hook_HourlyAddonSeller_getApiKey()
{
    $apiKey = Capsule::table('mod_HourlyAddonSeller_apikey')
        ->value('api_key');

    return $apiKey ?: '';
}

add_hook('AfterModuleSuspend', 1, function (array $vars) {
    $sql = "select * from mod_HourlyAddonSeller_services where service_id=" . $vars["params"]["serviceid"];
    $result = full_query($sql);
    if (mysql_num_rows($result) > 0) {
        $values = [
            "status" => 'Suspended',
            "updated_at" => date("Y-m-d H:i:s")
        ];
        Capsule::table('mod_HourlyAddonSeller_services')
            ->where('service_id', '=', $vars["params"]["serviceid"])
            ->update($values);

        // Send API request for service suspension
        $apiParams = [
            'action' => 'suspend',
            'service_id' => $vars["params"]["serviceid"],
            'product_id' => $vars["params"]["pid"],
            'client_id' => $vars["params"]["userid"],
            'domain' => $vars["params"]["domain"],
            'server_ip' => $vars["params"]["dedicatedip"] ?? '',
            'status' => 'Suspended',
            'suspend_reason' => $vars["params"]["suspendreason"] ?? ''
        ];
        sendApiRequest($apiParams);
    }
});

add_hook('AfterModuleTerminate', 1, function (array $vars) {
    $sql = "select * from mod_HourlyAddonSeller_services where service_id=" . $vars["params"]["serviceid"];
    $result = full_query($sql);
    if (mysql_num_rows($result) > 0) {
        $values = [
            "status" => 'Terminated',
            "updated_at" => date("Y-m-d H:i:s")
        ];
        Capsule::table('mod_HourlyAddonSeller_services')
            ->where('service_id', '=', $vars["params"]["serviceid"])
            ->update($values);

        // Send API request for service termination
        $apiParams = [
            'action' => 'terminate',
            'service_id' => $vars["params"]["serviceid"],
            'product_id' => $vars["params"]["pid"],
            'client_id' => $vars["params"]["userid"],
            'domain' => $vars["params"]["domain"],
            'server_ip' => $vars["params"]["dedicatedip"] ?? '',
            'status' => 'Terminated'
        ];
        sendApiRequest($apiParams);
    }
});

add_hook('AfterModuleUnsuspend', 1, function (array $vars) {
    $sql = "select * from mod_HourlyAddonSeller_services where service_id=" . $vars["params"]["serviceid"];
    $result = full_query($sql);
    if (mysql_num_rows($result) > 0) {
        $values = [
            "status" => 'Active',
            "updated_at" => date("Y-m-d H:i:s")
        ];
        Capsule::table('mod_HourlyAddonSeller_services')
            ->where('service_id', '=', $vars["params"]["serviceid"])
            ->update($values);

        // Send API request for service unsuspension
        $apiParams = [
            'action' => 'unsuspend',
            'service_id' => $vars["params"]["serviceid"],
            'product_id' => $vars["params"]["pid"],
            'client_id' => $vars["params"]["userid"],
            'domain' => $vars["params"]["domain"],
            'server_ip' => $vars["params"]["dedicatedip"] ?? '',
            'status' => 'Active'
        ];
        sendApiRequest($apiParams);
    }
});

add_hook('AdminClientServicesTabFields', 1, function ($vars) {
    $productID = Capsule::table('tblhosting')
        ->where("id", "=", $vars["id"])
        ->value('packageid');

    $HourlyIsActive = Capsule::table('mod_HourlyAddonSeller_products')
        ->where("product_id", "=", $productID)
        ->first();
    if (count($HourlyIsActive) > 0) {
        $HourlyServiceIsActive = Capsule::table('mod_HourlyAddonSeller_services')
            ->where("service_id", "=", $vars["id"])
            ->first();
        if (count($HourlyServiceIsActive) > 0) {
            //  $output = "<a class='btn btn-danger'>Deactive Hourly Billing</a>";
        } else {
            // $output = "<a class='btn btn-danger'>Deactive Hourly Billing</a>";
        }
        //return ["aa" => $output];
    }
});

// Also add to the client services page
add_hook('AdminAreaClientServicesTabFields', 1, function ($vars) {
    $userId = $vars['userid'];

    // Get user's balance
    try {
        $balance = Capsule::table('mod_HourlyAddonSeller_balance')
            ->where('userid', '=', $userId)
            ->value('balance');

        if ($balance === null) {
            $balance = 0;
        }

        // Format balance with thousand separator
        $formattedBalance = number_format($balance);

        return [
            'Hourly Credit Balance' => '<strong>' . $formattedBalance . '</strong> <a href="addonmodules.php?module=HourlyAddonSeller&action=balances&userid=' . $userId . '" class="btn btn-xs btn-default">Manage</a>'
        ];
    } catch (\Exception $e) {
        hourlyLog("Error displaying hourly credit balance: " . $e->getMessage(), 'error');
        return [];
    }
});

// Add to client summary page
add_hook('AdminAreaClientSummaryPage', 1, function ($vars) {
    $userId = $vars['userid'];

    // Get user's balance
    try {
        $balance = Capsule::table('mod_HourlyAddonSeller_balance')
            ->where('userid', '=', $userId)
            ->value('balance');

        if ($balance === null) {
            $balance = 0;
        }

        // Format balance with thousand separator
        $formattedBalance = number_format($balance);

        // Create the HTML for the balance display
        $balanceHtml = <<<HTML
        <div class="client-summary-panel panel panel-default">
            <div class="panel-heading">
                <h3 class="panel-title">Hourly Service Credit</h3>
            </div>
            <div class="panel-body">
                <div class="summary-value">{$formattedBalance}</div>
                <div class="text-center">
                    <a href="addonmodules.php?module=AddonSeller&action=balances&userid={$userId}" 
                       class="btn btn-default btn-sm">
                        Manage Balance
                    </a>
                </div>
            </div>
        </div>
HTML;

        return $balanceHtml;
    } catch (\Exception $e) {
        hourlyLog("Error displaying hourly credit balance: " . $e->getMessage(), 'error');
        return '';
    }
});

/**
 * Process a single API queue item
 * 
 * @param array $queueItem Queue item to process
 * @return bool Success status
 */
function processApiQueueItem($queueItem)
{
    try {
        // Mark item as processing
        Capsule::table('mod_HourlyAddonSeller_api_queue')
            ->where('id', '=', $queueItem['id'])
            ->update([
                'status' => 'processing',
                'attempts' => $queueItem['attempts'] + 1,
                'last_attempt' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            ]);

        // Parse request data
        $requestData = json_decode($queueItem['request_data'], true);
        if (!$requestData || !isset($requestData['action'])) {
            throw new Exception("Invalid request data");
        }

        // Get API key
        $apiKey = Hook_HourlyAddonSeller_getApiKey();
        if (empty($apiKey)) {
            throw new Exception("API key not configured");
        }

        // API endpoint URL
        $apiUrl = "https://cl.keloncloud.com/api/hourlyseller/action.php";

        // API authentication
        $headers = [
            'Authorization: Bearer ' . $apiKey,
            'Content-Type: application/json'
        ];

        // Send request
        $ch = curl_init($apiUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($requestData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);

        curl_close($ch);

        // Check for cURL errors
        if ($error) {
            throw new Exception("cURL Error: " . $error);
        }

        // Check HTTP response code
        if ($httpCode < 200 || $httpCode >= 300) {
            throw new Exception("API Error: HTTP Code " . $httpCode . " - " . $response);
        }

        // Parse response
        $responseData = json_decode($response, true);
        if (!$responseData || !isset($responseData['status'])) {
            throw new Exception("Invalid API response: " . $response);
        }

        // Check API response status
        if ($responseData['status'] != 'success') {
            $errorMessage = isset($responseData['message']) ? $responseData['message'] : 'Unknown error';
            throw new Exception("API Error: " . $errorMessage);
        }

        // Mark item as completed
        // Capsule::table('mod_HourlyAddonSeller_api_queue')
        //     ->where('id', '=', $queueItem['id'])
        //     ->update([
        //         'status' => 'completed',
        //         'updated_at' => date('Y-m-d H:i:s')
        //     ]);
        Capsule::table('mod_HourlyAddonSeller_api_queue')
            ->where('id', '=', $queueItem['id'])
            ->delete();
        hourlyLog("API Request processed successfully: " . $requestData['action'] . " (Queue ID: " . $queueItem['id'] . ")");

        return true;
    } catch (Exception $e) {
        // Mark item as failed
        Capsule::table('mod_HourlyAddonSeller_api_queue')
            ->where('id', '=', $queueItem['id'])
            ->update([
                'status' => 'failed',
                'error' => $e->getMessage(),
                'updated_at' => date('Y-m-d H:i:s')
            ]);

        hourlyLog("API Request failed: " . $e->getMessage() . " (Queue ID: " . $queueItem['id'] . ")", 'error');

        return false;
    }
}