<?php

namespace classes\Export;

use classes\Export\UMS_Csv_User;

defined('ABSPATH') || exit;
class UMS_Csv_Export
{

    /**
     * Send HTTP headers for download CSV file.
     */
    private function send_file_stream_http_headers()
    {
        header('Content-Encoding: UTF-8');
        header('Content-Type: text/csv; charset=UTF-8');
        header('Content-Disposition: attachment; filename=' . date('Y-m-d-H-i') . '-users.csv');
        header('Pragma: no-cache');
        header('Expires: 0');
    }

    /**
     * Save and generate CSV. Called on form submit.
     */
    public function save($columns, $roles, $user)
    {

        $allowlist = ums_get_all_columns();
        /** SECURITY: Do not export sensitive data, even if they are on the allow list. */
        $denylist = array(
            'user_pass',
            'user_activation_key',
            'session_tokens',
            'wp_user-settings',
            'wp_user-settings-time',
            'wp_capabilities',
            'community-events-location',
            'ums_new_email_activated',
            'ufpp_fields_profile',
        );

        /** Get selected columns. */
        $columns = $columns ? $columns : $allowlist;
        /** Empty = All. */

        /**
         * Delimiter / Enclosure.
         */
        $delimiter_char      = null;
        $enclosure_char      = null;

        /** Get selected users (ids). */
        $ids   = $roles ? $this->get_user_ids_by_roles($roles) : $this->get_user_ids();
        if (empty($ids)) {
            wp_die('هیچ کاربری با این نقش یافت نشد!');
        }
        /** Execute this block on a try-catch because CSV lib can throw exceptions. */
        try {
            $csv = (new UMS_Csv_User())
                ->set_filename('php://output')
                ->set_columns($columns)
                ->set_delimiter($delimiter_char)
                ->set_enclosure($enclosure_char)
                ->set_allowlist($allowlist)
                ->set_denylist($denylist)
                ->check_errors();

            /**
             * Tell browser that response is a file-stream.
             */
            $this->send_file_stream_http_headers();

            if ($user) {
                $data     = $this->get_users_data(null, $user);
                $csv->write($data);
            } else {
                /** Process in batches of 1k users */
                $page_size  = 1000;
                $page_count = floor((count($ids) - 1) / $page_size) + 1;
                for ($i = 0; $i < $page_count; $i++) {
                    $ids_page = array_splice($ids, 0, $page_size);
                    $data     = $this->get_users_data($ids_page, null);
                    $csv->write($data);
                }
            }

            /**
             * Close stream and quit.
             */
            $csv->close();
            exit();
        } catch (\Exception $e) {
            $this->wc->add_error($e->getMessage());
        }
    }

    /**
     * Load users and their meta.

     * @param  string[] $ids User ids.
     *
     * @return int A database handler that returns a list of users and their data. Example: [
     *   [fname => John, lname => Snow], [fname => Jane, lname => Doe]
     * ].
     */
    public function get_users_data($ids, $user_id = null)
    {
        global $wpdb;


        if ($user_id) {
            /** Get user of user_id. */
            $sql          = $wpdb->prepare("SELECT * FROM {$wpdb->users} WHERE id =%d", $user_id);
            $user_records = $wpdb->get_results($sql);

            /** Get meta data of user_id. */
            $sql          = $wpdb->prepare("SELECT * FROM {$wpdb->usermeta} WHERE user_id =%d", $user_id);
            $meta_records = $wpdb->get_results($sql);
        } else {
            /** A string "%d, %d, %d, ..." to be used on "id IN(...)". */
            $array_d = implode(',', array_fill(0, count($ids), '%d'));

            /** Get users of $ids. */
            $sql          = "SELECT * FROM {$wpdb->users} WHERE id IN ({$array_d})";
            $query        = call_user_func_array(array($wpdb, 'prepare'), array_merge(array($sql), $ids));
            $user_records = $wpdb->get_results($query);

            /** Get meta data of $ids. */
            $sql          = "SELECT * FROM {$wpdb->usermeta} WHERE user_id in ({$array_d})";
            $query        = call_user_func_array(array($wpdb, 'prepare'), array_merge(array($sql), $ids));
            $meta_records = $wpdb->get_results($query);
        }
        /** Final array. */
        $user_rows = array();

        /** Set key = user ID. */
        foreach ($user_records as $record) {
            $user_rows[$record->ID] = (array) $record;
        }

        /** Add meta info */
        foreach ($meta_records as $record) {
            $value = is_serialized($record->meta_value) ? unserialize($record->meta_value) : $record->meta_value;

            if (is_array($value)) {
                $string_parts = [];
                foreach ($value as $v) {
                    $string_parts[] = is_array($v) ? json_encode($v) : $v;
                }
                $user_rows[$record->user_id][$record->meta_key] = implode(' - ', $string_parts);
            } else {
                $user_rows[$record->user_id][$record->meta_key] = $value;
            }
        }

        return $user_rows;
    }

    /**
     * Get all user IDs from a given set of roles.
     *
     * @param  string[] $roles Filter users with these roles.
     *
     * @return int[] An array of user ids.
     */
    public function get_user_ids_by_roles($roles)
    {
        global $wpdb;

        /** Get user IDs by their roles. */
        $role_statements = array();
        foreach ($roles as $role) {
            $value             = serialize(esc_sql($role));
            $role_statements[] = "meta_value LIKE '%{$value}%'";
        }
        $role_statements = join(' OR ', $role_statements);
        $sql = "
			SELECT user_id
			FROM {$wpdb->usermeta}
			WHERE meta_key = '{$wpdb->prefix}capabilities' AND ({$role_statements})
		";
        $ids = $wpdb->get_col($sql, 0);
        return array_map('intval', $ids);
    }

    /**
     * Get all user ids.
     *
     * @return int[] An array of user ids.
     */
    public function get_user_ids()
    {
        global $wpdb;
        $sql = "SELECT ID FROM {$wpdb->users}";
        $ids = $wpdb->get_col($sql, 0);
        return array_map('intval', $ids);
    }
}
