/**
 * WordPress Quick Setup JavaScript - RankMath Style
 * Clean, modern interactions matching RankMath design
 */

jQuery(document).ready(function ($) {
  "use strict";

  let currentStep = 1;
  let totalSteps = $(".progress-step").length;
  let isProcessing = false;

  // Initialize
  init();

  function init() {
    bindEvents();
    updateUI();
    updateProgressSteps();
    addNoteBoxes();
  }

  function bindEvents() {
    // Next button
    $("#next-btn").on("click", function (e) {
      e.preventDefault();
      if (!isProcessing) {
        processStep("next");
      }
    });

    // Previous button
    $("#prev-btn").on("click", function (e) {
      e.preventDefault();
      if (!isProcessing) {
        processStep("prev");
      }
    });

    // Skip button
    $("#skip-btn").on("click", function (e) {
      e.preventDefault();
      if (!isProcessing) {
        showSkipConfirmation();
      }
    });

    // Complete button
    $("#complete-btn").on("click", function (e) {
      e.preventDefault();
      if (!isProcessing) {
        completeSetup();
      }
    });
  }

  function showSkipConfirmation() {
    const stepTitles = {
      1: "انتخاب نوع راه‌اندازی",
      2: "تنظیمات اولیه",
      3: "انتخاب ویژگی‌ها",
      4: "آماده‌سازی نهایی",
    };

    const currentStepTitle = stepTitles[currentStep] || "این مرحله";

    const modal = $(`
            <div class="skip-confirmation-modal">
                <div class="modal-content">
                    <div class="modal-header">
                        <span class="modal-icon dashicons dashicons-warning"></span>
                        <h3>رد کردن مرحله</h3>
                    </div>
                    <div class="modal-body">
                        <p>آیا مطمئن هستید که می‌خواهید "${currentStepTitle}" را رد کنید?</p>
                        <p class="modal-description">در صورت رد کردن، تنظیمات پیش‌فرض اعمال خواهد شد.</p>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary cancel-skip">انصراف</button>
                        <button type="button" class="btn btn-skip confirm-skip">بله، رد کن</button>
                    </div>
                </div>
            </div>
        `);

    $("body").append(modal);
    modal.fadeIn(200);

    // Handle modal actions
    modal.find(".cancel-skip").on("click", function () {
      modal.fadeOut(200, function () {
        $(this).remove();
      });
    });

    modal.find(".confirm-skip").on("click", function () {
      modal.fadeOut(200, function () {
        $(this).remove();
      });
      processStep("skip");
    });

    // Close on backdrop click
    modal.on("click", function (e) {
      if (e.target === this) {
        modal.fadeOut(200, function () {
          $(this).remove();
        });
      }
    });
  }

  function processStep(direction) {
    if (isProcessing) return;

    // Skip validation for skip action
    if (
      direction !== "skip" &&
      direction === "next" &&
      !validateCurrentStep()
    ) {
      showNotification("لطفاً تمام فیلدهای ضروری را پر کنید.", "error");
      return;
    }

    isProcessing = true;
    showLoading(true);

    const formData = direction === "skip" ? {} : collectFormData();

    $.ajax({
      url: quickSetupAjax.ajax_url,
      type: "POST",
      data: {
        action: "quick_setup_step",
        step: currentStep,
        direction: direction,
        form_data: formData,
        nonce: quickSetupAjax.nonce,
      },
      success: function (response) {
        if (response.success) {
          // Show skip notification if step was skipped
          if (response.data.skipped) {
            showNotification(
              "مرحله رد شد و تنظیمات پیش‌فرض اعمال شد",
              "warning"
            );
          }

          // Smooth transition
          $(".step-content").fadeOut(200, function () {
            $(this).html(response.data.content).fadeIn(300);
            addNoteBoxes();
          });

          currentStep = response.data.step;
          updateUI();
          updateProgressSteps();
        } else {
          showNotification(response.data || "خطا در انجام عملیات", "error");
        }
      },
      error: function () {
        showNotification("خطا در ارتباط با سرور", "error");
      },
      complete: function () {
        isProcessing = false;
        showLoading(false);
      },
    });
  }
  jQuery(document).delegate(".install-page", "click", function (e) {
    e.preventDefault();
    $.ajax({
      url: quickSetupAjax.ajax_url,
      type: "POST",
      data: {
        action: "quick_setup_install_page",
        slug: jQuery(this).attr("data-install"),
        nonce: quickSetupAjax.nonce,
      },
      success: function (response) {
        if (response.result) {
          jQuery(".install-page-login").hide();
          jQuery(".ums-setup-wizard-installed").show();
        } else {
          showNotification(response.data || "خطا در انجام عملیات", "error");
        }
      },
      error: function () {
        showNotification("خطا در ارتباط با سرور", "error");
      },
      complete: function () {},
    });
  }),
    function completeSetup() {
      if (isProcessing) return;

      if (!validateCurrentStep()) {
        showNotification("لطفاً تمام فیلدهای ضروری را پر کنید.", "error");
        return;
      }

      isProcessing = true;
      showLoading(true);

      const formData = collectFormData();

      $.ajax({
        url: quickSetupAjax.ajax_url,
        type: "POST",
        data: {
          action: "quick_setup_complete",
          form_data: formData,
          nonce: quickSetupAjax.nonce,
        },
        success: function (response) {
          if (response.success) {
            // Mark all steps as completed
            $(".progress-step").removeClass("active").addClass("completed");

            // Hide navigation
            $(".quick-setup-footer .btn").hide();

            showNotification("راه‌اندازی با موفقیت تکمیل شد!", "success");
          } else {
            showNotification(
              response.data || "خطا در تکمیل راه‌اندازی",
              "error"
            );
          }
        },
        error: function () {
          showNotification("خطا در ارتباط با سرور", "error");
        },
        complete: function () {
          isProcessing = false;
          showLoading(false);
        },
      });
    };

  function collectFormData() {
    const formData = {};

    // Collect form inputs
    $('.step-content input, .step-content input[type="hidden"]').each(
      function () {
        const $input = $(this);
        const name = $input.attr("name");
        const type = $input.attr("type");

        if (name) {
          if (type === "checkbox") {
            if ($input.is(":checked")) {
              if (name.endsWith("[]")) {
                const arrayName = name.slice(0, -2);
                if (!formData[arrayName]) formData[arrayName] = [];
                formData[arrayName].push($input.val());
              } else {
                formData[name] = $input.val();
              }
            }
          } else if (type === "radio") {
            if ($input.is(":checked")) {
              formData[name] = $input.val();
            }
          } else {
            formData[name] = $input.val();
          }
        }
      }
    );

    return formData;
  }

  function validateCurrentStep() {
    let isValid = true;

    // Remove previous error states
    $(".form-input.error").removeClass("error");

    // Check required fields
    $(".step-content .form-input[required]").each(function () {
      console.log($(this));
      return;
      if (!validateField($(this))) {
        isValid = false;
      }
    });

    // Check if at least one option is selected
    // if (currentStep === 1) {
    //   if ($(".option-card.selected").length === 0) {
    //     showNotification("لطفاً یکی از گزینه‌ها را انتخاب کنید.", "error");
    //     isValid = false;
    //   }
    // }

    return isValid;
  }

  function validateField($input) {
    const value = $input.val().trim();
    const type = $input.attr("type");
    let isValid = true;

    // Required field check
    if ($input.prop("required") && !value) {
      isValid = false;
    }

    // Email validation
    if (type === "email" && value) {
      const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
      if (!emailRegex.test(value)) {
        isValid = false;
      }
    }

    // Update field appearance
    if (isValid) {
      $input.removeClass("error");
    } else {
      $input.addClass("error");
    }

    return isValid;
  }

  function updateUI() {
    // Update navigation buttons
    if (currentStep === 1) {
      $("#prev-btn").hide();
    } else {
      $("#prev-btn").show();
    }

    // Show/hide skip button based on step
    // Hide skip button on first step (welcome) and last step (completion)
    if (currentStep === 1 || currentStep === totalSteps) {
      $("#skip-btn").hide();
    } else {
      $("#skip-btn").show();
    }

    if (currentStep === totalSteps) {
      $("#next-btn").hide();
      $("#complete-btn").show();
    } else {
      $("#next-btn").show();
      $("#complete-btn").hide();
    }

    // Update skip button text based on step
    const skipTexts = {
      2: "رد کردن تنظیمات",
      3: "رد کردن ویژگی‌ها",
      4: "رد کردن آماده‌سازی",
    };

    if (skipTexts[currentStep]) {
      $("#skip-btn").find("span:first").text(skipTexts[currentStep]);
    }
  }

  function updateProgressSteps() {
    $(".progress-step").each(function (index) {
      const stepNumber = index + 1;
      const $step = $(this);
      const $label = $(".step-label").eq(index);

      // Reset classes
      $step.removeClass("completed active");
      $label.removeClass("active");

      if (stepNumber < currentStep) {
        $step.addClass("completed");
      } else if (stepNumber === currentStep) {
        $step.addClass("active");
        $label.addClass("active");
      }
    });
  }

  function showLoading(show) {
    if (show) {
      $(".footer-right .btn").prop("disabled", true);
      $(".loading-indicator").show();
    } else {
      $(".footer-right .btn").prop("disabled", false);
      $(".loading-indicator").hide();
    }
  }

  function showNotification(message, type = "info") {
    // Remove existing notifications
    $(".notification-error").remove();

    // Icon mapping
    const icons = {
      success: "dashicons-yes-alt",
      error: "dashicons-dismiss",
      warning: "dashicons-warning",
      info: "dashicons-info",
    };

    const notification = $(`
            <div class="notification notification-${type}" style="display: none;">
                <span class="dashicons ${icons[type] || icons.info}"></span>
                <span>${message}</span>
            </div>
        `);

    // Add to page
    $(".step-content").prepend(notification);
    notification.slideDown(300);

    // Auto hide
    setTimeout(function () {
      notification.slideUp(300, function () {
        $(this).remove();
      });
    }, 4000);
  }

  function addNoteBoxes() {
    // Add note box for step 1 if it doesn't exist
    if (currentStep === 1 && $(".setup-note").length === 0) {
      const noteBox = $(`
                <div class="setup-note">
                    <span class="note-icon dashicons dashicons-info"></span>
                    <div>
                        <strong>نکته:</strong> می‌توانید در هر زمان بین حالت‌ها تغییر دهید.
                    </div>
                </div>
            `);
      $(".status-message").after(noteBox);
    }
  }

  function saveProgress() {
    const progressData = {
      currentStep: currentStep,
      formData: collectFormData(),
      timestamp: Date.now(),
    };

    try {
      localStorage.setItem(
        "wp_quick_setup_progress",
        JSON.stringify(progressData)
      );
      showQuickNotification("پیشرفت ذخیره شد");
    } catch (e) {
      console.warn("Could not save progress:", e);
    }
  }

  function loadProgress() {
    try {
      const saved = localStorage.getItem("wp_quick_setup_progress");
      if (saved) {
        const data = JSON.parse(saved);
        const timeDiff = Date.now() - data.timestamp;

        // Only restore if less than 1 hour old
        if (timeDiff < 3600000) {
          currentStep = data.currentStep;

          updateUI();
          updateProgressSteps();
          return true;
        }
      }
    } catch (e) {
      console.warn("Could not load progress:", e);
    }
    return false;
  }

  function clearProgress() {
    try {
      localStorage.removeItem("wp_quick_setup_progress");
    } catch (e) {
      console.warn("Could not clear progress:", e);
    }
  }

  function showQuickNotification(message) {
    const notification = $(`
            <div class="quick-notification">
                <span class="dashicons dashicons-saved"></span>
                ${message}
            </div>
        `);

    $("body").append(notification);

    setTimeout(function () {
      notification.addClass("show");
    }, 10);

    setTimeout(function () {
      notification.removeClass("show");
      setTimeout(function () {
        notification.remove();
      }, 300);
    }, 2000);
  }

  // Keyboard shortcuts
  $(document).keydown(function (e) {
    if (isProcessing) return;

    // Ctrl/Cmd + Arrow keys
    if (e.ctrlKey || e.metaKey) {
      switch (e.which) {
        case 37: // Left Arrow
          e.preventDefault();
          if (currentStep > 1) {
            processStep("prev");
          }
          break;

        case 39: // Right Arrow
          e.preventDefault();
          if (currentStep < totalSteps) {
            processStep("next");
          } else {
            completeSetup();
          }
          break;
      }
    }
  });

  // Enhanced error handling
  $(document).ajaxError(function (event, xhr, settings, thrownError) {
    if (settings.url && settings.url.includes("quick_setup")) {
      console.error("Setup AJAX Error:", {
        status: xhr.status,
        statusText: xhr.statusText,
        responseText: xhr.responseText,
        thrownError: thrownError,
      });

      if (xhr.status === 403) {
        showNotification("دسترسی غیرمجاز. لطفاً دوباره وارد شوید.", "error");
      } else if (xhr.status === 0) {
        showNotification(
          "اتصال اینترنت قطع است. لطفاً اتصال خود را بررسی کنید.",
          "error"
        );
      } else {
        showNotification(
          "خطا در ارتباط با سرور. لطفاً دوباره تلاش کنید.",
          "error"
        );
      }

      isProcessing = false;
      showLoading(false);
    }
  });

  // Initialize tooltips
  function initTooltips() {
    $("[data-tooltip]").each(function () {
      const $element = $(this);
      const tooltipText = $element.attr("data-tooltip");

      $element.hover(
        function () {
          const tooltip = $(`<div class="tooltip">${tooltipText}</div>`);
          $("body").append(tooltip);

          const elementOffset = $element.offset();
          const tooltipWidth = tooltip.outerWidth();
          const tooltipHeight = tooltip.outerHeight();

          tooltip
            .css({
              top: elementOffset.top - tooltipHeight - 10,
              left:
                elementOffset.left +
                $element.outerWidth() / 2 -
                tooltipWidth / 2,
              opacity: 0,
            })
            .animate({ opacity: 1 }, 200);
        },
        function () {
          $(".tooltip").fadeOut(200, function () {
            $(this).remove();
          });
        }
      );
    });
  }

  // Responsive handling
  let resizeTimer;
  $(window).on("resize", function () {
    clearTimeout(resizeTimer);
    resizeTimer = setTimeout(function () {
      updateResponsiveLayout();
    }, 250);
  });

  function updateResponsiveLayout() {
    const width = $(window).width();

    if (width < 768) {
      $(".quick-setup-footer").addClass("mobile");
      $(".progress-steps").addClass("mobile");
    } else {
      $(".quick-setup-footer").removeClass("mobile");
      $(".progress-steps").removeClass("mobile");
    }
  }

  // Initialize everything
  initTooltips();
  updateResponsiveLayout();
  loadProgress();

  // Public API
  window.QuickSetup = {
    getCurrentStep: function () {
      return currentStep;
    },
    getTotalSteps: function () {
      return totalSteps;
    },
    goToStep: function (step) {
      if (step >= 1 && step <= totalSteps && step !== currentStep) {
        currentStep = step;
        updateUI();
        updateProgressSteps();
        return true;
      }
      return false;
    },
    validateCurrentStep: validateCurrentStep,
    showNotification: showNotification,
    isProcessing: function () {
      return isProcessing;
    },
    saveProgress: saveProgress,
    clearProgress: clearProgress,
  };

  // Trigger ready event
  $(document).trigger("quickSetupReady", {
    currentStep: currentStep,
    totalSteps: totalSteps,
  });
});

// Additional CSS for dynamic elements
const additionalCSS = `
<style>
.notification {
    display: flex;
    align-items: center;
    gap: 10px;
    padding: 12px 16px;
    border-radius: 6px;
    margin-bottom: 20px;
    font-size: 13px;
    font-weight: 500;
    animation: slideInDown 0.3s ease-out;
}

@keyframes slideInDown {
    from {
        opacity: 0;
        transform: translateY(-20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.notification-success {
    background: #f0fdf4;
    color: #166534;
    border: 1px solid #bbf7d0;
}

.notification-error {
    background: #fef2f2;
    color: #dc2626;
    border: 1px solid #fecaca;
}

.notification-warning {
    background: #fffbeb;
    color: #92400e;
    border: 1px solid #fed7aa;
}

.notification-info {
    background: #eff6ff;
    color: #1d4ed8;
    border: 1px solid #bfdbfe;
}

.option-card.hover-effect {
    border-color: #cbd5e0;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.06);
    transform: translateY(-1px);
}

.form-group.focused .form-label {
    color: #3b82f6;
}

.quick-notification {
    position: fixed;
    bottom: 20px;
    right: 20px;
    background: #1f2937;
    color: white;
    padding: 8px 16px;
    border-radius: 6px;
    font-size: 13px;
    font-weight: 500;
    display: flex;
    align-items: center;
    gap: 6px;
    opacity: 0;
    transform: translateY(20px);
    transition: all 0.3s ease;
    z-index: 10000;
}

.quick-notification.show {
    opacity: 1;
    transform: translateY(0);
}

.quick-notification .dashicons {
    font-size: 16px;
    color: #10b981;
}

.tooltip {
    position: absolute;
    background: rgba(0, 0, 0, 0.9);
    color: white;
    padding: 6px 10px;
    border-radius: 4px;
    font-size: 12px;
    max-width: 200px;
    z-index: 10000;
    pointer-events: none;
    word-wrap: break-word;
}

.tooltip::after {
    content: '';
    position: absolute;
    top: 100%;
    left: 50%;
    transform: translateX(-50%);
    border: 4px solid transparent;
    border-top-color: rgba(0, 0, 0, 0.9);
}

@media (max-width: 768px) {
    .quick-notification {
        bottom: 10px;
        right: 10px;
        left: 10px;
        text-align: center;
    }
    
    .progress-steps.mobile {
        margin: 0 10px 15px;
    }
    
    .quick-setup-footer.mobile .footer-left,
    .quick-setup-footer.mobile .footer-right {
        width: 100%;
        justify-content: center;
    }
}

</style>
`;

// Inject CSS
if (document.head) {
  document.head.insertAdjacentHTML("beforeend", additionalCSS);
}
