<?php
// personal_consumption_analysis.php
require_once 'config.php';
require_once 'vendor/autoload.php';

use Morilog\Jalali\Jalalian;
use Carbon\Carbon;

class PersonalConsumptionAnalysis {
    private $pdo;
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
        // غیرفعال کردن ONLY_FULL_GROUP_BY
        $this->pdo->exec("SET SESSION sql_mode = (SELECT REPLACE(@@sql_mode, 'ONLY_FULL_GROUP_BY', ''))");
    }
    
    // دریافت اطلاعات اولیه شخص
    public function getPersonInfo($national_code) {
        $query = "
            SELECT 
                کدملی_شناسه_ملی,
                MAX(نام_متقاضی) as نام_متقاضی,
                MAX(تلفن_همراه) as تلفن_همراه,
                MAX(آدرس) as آدرس,
                MAX(شهرستان) as شهرستان,
                MAX(استان) as استان,
                GROUP_CONCAT(DISTINCT شناسه_یکتای_کاربرد) as identifiers,
                COUNT(DISTINCT شناسه_یکتای_کاربرد) as identifier_count,
                GROUP_CONCAT(DISTINCT فراوده) as products,
                GROUP_CONCAT(DISTINCT بخش_مصرف) as usage_sections
            FROM fuel_requests 
            WHERE کدملی_شناسه_ملی = ?
            GROUP BY کدملی_شناسه_ملی
        ";
        
        $stmt = $this->pdo->prepare($query);
        $stmt->execute([$national_code]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    // دریافت تاریخ‌های موجود برای این شخص
    public function getAvailableYearsForPerson($national_code) {
        $query = "
            SELECT DISTINCT تاریخ_تایید_درخواست_توسط_پخش 
            FROM fuel_requests 
            WHERE کدملی_شناسه_ملی = ? 
            AND تاریخ_تایید_درخواست_توسط_پخش IS NOT NULL 
            ORDER BY تاریخ_تایید_درخواست_توسط_پخش DESC
        ";
        
        $stmt = $this->pdo->prepare($query);
        $stmt->execute([$national_code]);
        $dates = $stmt->fetchAll(PDO::FETCH_COLUMN);

        $years = [];
        foreach ($dates as $date) {
            try {
                $carbon = Carbon::parse($date);
                $jalali = Jalalian::fromCarbon($carbon);
                $shamsi_year = $jalali->getYear();
                if (!in_array($shamsi_year, $years)) {
                    $years[] = $shamsi_year;
                }
            } catch (Exception $e) {
                continue;
            }
        }

        rsort($years);
        return $years;
    }
    
    // تحلیل مصرف ماهانه شخص
    public function getMonthlyConsumption($national_code, $year = null) {
        $where_conditions = ["کدملی_شناسه_ملی = ?"];
        $params = [$national_code];
        
        if ($year) {
            $where_conditions[] = "YEAR(تاریخ_تایید_درخواست_توسط_پخش) = ?";
            $params[] = $this->shamsiToGregorianYear($year);
        }
        
        $where_clause = "WHERE " . implode(" AND ", $where_conditions);
        
        $query = "
            SELECT 
                YEAR(تاریخ_تایید_درخواست_توسط_پخش) as gregorian_year,
                MONTH(تاریخ_تایید_درخواست_توسط_پخش) as gregorian_month,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_consumption,
                COUNT(*) as request_count,
                AVG(مقدار_تایید_شده_توسط_پخش) as average_per_request,
                MAX(مقدار_تایید_شده_توسط_پخش) as max_consumption,
                MIN(مقدار_تایید_شده_توسط_پخش) as min_consumption
            FROM fuel_requests 
            {$where_clause}
            GROUP BY YEAR(تاریخ_تایید_درخواست_توسط_پخش), MONTH(تاریخ_تایید_درخواست_توسط_پخش)
            ORDER BY gregorian_year DESC, gregorian_month DESC
        ";
        
        $stmt = $this->pdo->prepare($query);
        $stmt->execute($params);
        $data = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // تبدیل به تاریخ شمسی و سازماندهی
        return $this->organizeMonthlyData($data);
    }
    
    // تحلیل مصرف سه ماهه
    public function getThreeMonthConsumption($national_code, $year = null) {
        $where_conditions = ["کدملی_شناسه_ملی = ?"];
        $params = [$national_code];
        
        if ($year) {
            $where_conditions[] = "YEAR(تاریخ_تایید_درخواست_توسط_پخش) = ?";
            $params[] = $this->shamsiToGregorianYear($year);
        }
        
        $where_clause = "WHERE " . implode(" AND ", $where_conditions);
        
        $query = "
            SELECT 
                YEAR(تاریخ_تایید_درخواست_توسط_پخش) as gregorian_year,
                CASE 
                    WHEN MONTH(تاریخ_تایید_درخواست_توسط_پخش) BETWEEN 1 AND 3 THEN 1
                    WHEN MONTH(تاریخ_تایید_درخواست_توسط_پخش) BETWEEN 4 AND 6 THEN 2
                    WHEN MONTH(تاریخ_تایید_درخواست_توسط_پخش) BETWEEN 7 AND 9 THEN 3
                    WHEN MONTH(تاریخ_تایید_درخواست_توسط_پخش) BETWEEN 10 AND 12 THEN 4
                END as quarter,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_consumption,
                COUNT(*) as request_count,
                AVG(مقدار_تایید_شده_توسط_پخش) as average_per_request,
                MAX(مقدار_تایید_شده_توسط_پخش) as max_consumption,
                MIN(مقدار_تایید_شده_توسط_پخش) as min_consumption,
                GROUP_CONCAT(DISTINCT MONTH(تاریخ_تایید_درخواست_توسط_پخش)) as months
            FROM fuel_requests 
            {$where_clause}
            GROUP BY YEAR(تاریخ_تایید_درخواست_توسط_پخش), quarter
            ORDER BY gregorian_year DESC, quarter
        ";
        
        $stmt = $this->pdo->prepare($query);
        $stmt->execute($params);
        $data = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        return $this->organizeQuarterlyData($data);
    }
    
    // تحلیل مصرف فصلی
    public function getSeasonalConsumption($national_code, $year = null) {
        $where_conditions = ["کدملی_شناسه_ملی = ?"];
        $params = [$national_code];
        
        if ($year) {
            $where_conditions[] = "YEAR(تاریخ_تایید_درخواست_توسط_پخش) = ?";
            $params[] = $this->shamsiToGregorianYear($year);
        }
        
        $where_clause = "WHERE " . implode(" AND ", $where_conditions);
        
        $query = "
            SELECT 
                YEAR(تاریخ_تایید_درخواست_توسط_پخش) as gregorian_year,
                CASE 
                    WHEN MONTH(تاریخ_تایید_درخواست_توسط_پخش) BETWEEN 1 AND 3 THEN 1
                    WHEN MONTH(تاریخ_تایید_درخواست_توسط_پخش) BETWEEN 4 AND 6 THEN 2
                    WHEN MONTH(تاریخ_تایید_درخواست_توسط_پخش) BETWEEN 7 AND 9 THEN 3
                    WHEN MONTH(تاریخ_تایید_درخواست_توسط_پخش) BETWEEN 10 AND 12 THEN 4
                END as season,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_consumption,
                COUNT(*) as request_count,
                AVG(مقدار_تایید_شده_توسط_پخش) as average_per_request,
                GROUP_CONCAT(DISTINCT فراوده) as products
            FROM fuel_requests 
            {$where_clause}
            GROUP BY YEAR(تاریخ_تایید_درخواست_توسط_پخش), season
            ORDER BY gregorian_year DESC, season
        ";
        
        $stmt = $this->pdo->prepare($query);
        $stmt->execute($params);
        $data = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        return $this->organizeSeasonalData($data);
    }
    
    // تحلیل مصرف بر اساس فرآورده
    public function getConsumptionByProduct($national_code, $year = null) {
        $where_conditions = ["کدملی_شناسه_ملی = ?"];
        $params = [$national_code];
        
        if ($year) {
            $where_conditions[] = "YEAR(تاریخ_تایید_درخواست_توسط_پخش) = ?";
            $params[] = $this->shamsiToGregorianYear($year);
        }
        
        $where_clause = "WHERE " . implode(" AND ", $where_conditions);
        
        $query = "
            SELECT 
                فراوده,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_consumption,
                COUNT(*) as request_count,
                AVG(مقدار_تایید_شده_توسط_پخش) as average_per_request,
                MAX(مقدار_تایید_شده_توسط_پخش) as max_consumption,
                MIN(مقدار_تایید_شده_توسط_پخش) as min_consumption,
                GROUP_CONCAT(DISTINCT بخش_مصرف) as usage_sections
            FROM fuel_requests 
            {$where_clause}
            GROUP BY فراوده
            ORDER BY total_consumption DESC
        ";
        
        $stmt = $this->pdo->prepare($query);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // تحلیل مصرف بر اساس بخش مصرف
    public function getConsumptionByUsageSection($national_code, $year = null) {
        $where_conditions = ["کدملی_شناسه_ملی = ?"];
        $params = [$national_code];
        
        if ($year) {
            $where_conditions[] = "YEAR(تاریخ_تایید_درخواست_توسط_پخش) = ?";
            $params[] = $this->shamsiToGregorianYear($year);
        }
        
        $where_clause = "WHERE " . implode(" AND ", $where_conditions);
        
        $query = "
            SELECT 
                بخش_مصرف,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_consumption,
                COUNT(*) as request_count,
                AVG(مقدار_تایید_شده_توسط_پخش) as average_per_request,
                MAX(مقدار_تایید_شده_توسط_پخش) as max_consumption,
                MIN(مقدار_تایید_شده_توسط_پخش) as min_consumption,
                GROUP_CONCAT(DISTINCT فراوده) as products
            FROM fuel_requests 
            {$where_clause}
            GROUP BY بخش_مصرف
            ORDER BY total_consumption DESC
        ";
        
        $stmt = $this->pdo->prepare($query);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // تحلیل مصرف بر اساس فرآورده و بخش مصرف
    public function getConsumptionByProductAndSection($national_code, $year = null) {
        $where_conditions = ["کدملی_شناسه_ملی = ?"];
        $params = [$national_code];
        
        if ($year) {
            $where_conditions[] = "YEAR(تاریخ_تایید_درخواست_توسط_پخش) = ?";
            $params[] = $this->shamsiToGregorianYear($year);
        }
        
        $where_clause = "WHERE " . implode(" AND ", $where_conditions);
        
        $query = "
            SELECT 
                فراوده,
                بخش_مصرف,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_consumption,
                COUNT(*) as request_count,
                AVG(مقدار_تایید_شده_توسط_پخش) as average_per_request
            FROM fuel_requests 
            {$where_clause}
            GROUP BY فراوده, بخش_مصرف
            ORDER BY total_consumption DESC
        ";
        
        $stmt = $this->pdo->prepare($query);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // مقایسه مصرف سالانه
    public function getYearlyComparison($national_code, $years = []) {
        $where_conditions = ["کدملی_شناسه_ملی = ?"];
        $params = [$national_code];
        
        if (!empty($years)) {
            $year_conditions = [];
            foreach ($years as $year) {
                $year_conditions[] = "YEAR(تاریخ_تایید_درخواست_توسط_پخش) = ?";
                $params[] = $this->shamsiToGregorianYear($year);
            }
            $where_conditions[] = "(" . implode(" OR ", $year_conditions) . ")";
        }
        
        $where_clause = "WHERE " . implode(" AND ", $where_conditions);
        
        $query = "
            SELECT 
                YEAR(تاریخ_تایید_درخواست_توسط_پخش) as gregorian_year,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_consumption,
                COUNT(*) as request_count,
                AVG(مقدار_تایید_شده_توسط_پخش) as average_per_request,
                MAX(مقدار_تایید_شده_توسط_پخش) as max_consumption,
                MIN(مقدار_تایید_شده_توسط_پخش) as min_consumption,
                GROUP_CONCAT(DISTINCT فراوده) as products,
                GROUP_CONCAT(DISTINCT بخش_مصرف) as usage_sections
            FROM fuel_requests 
            {$where_clause}
            GROUP BY YEAR(تاریخ_تایید_درخواست_توسط_پخش)
            ORDER BY gregorian_year DESC
        ";
        
        $stmt = $this->pdo->prepare($query);
        $stmt->execute($params);
        $data = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // تبدیل به سال شمسی
        foreach ($data as &$row) {
            $row['shamsi_year'] = $this->gregorianToShamsiYear($row['gregorian_year']);
        }
        
        return $data;
    }
    
    // تحلیل روند مصرف (افزایش/کاهش)
    public function getConsumptionTrend($national_code, $base_year, $compare_year) {
        $base_data = $this->getYearlyComparison($national_code, [$base_year]);
        $compare_data = $this->getYearlyComparison($national_code, [$compare_year]);
        
        $base_consumption = $base_data[0]['total_consumption'] ?? 0;
        $compare_consumption = $compare_data[0]['total_consumption'] ?? 0;
        
        if ($base_consumption > 0) {
            $change_amount = $compare_consumption - $base_consumption;
            $change_percentage = ($change_amount / $base_consumption) * 100;
        } else {
            $change_amount = $compare_consumption;
            $change_percentage = 100;
        }
        
        return [
            'base_year' => $base_year,
            'base_consumption' => $base_consumption,
            'compare_year' => $compare_year,
            'compare_consumption' => $compare_consumption,
            'change_amount' => $change_amount,
            'change_percentage' => $change_percentage,
            'trend' => $change_amount > 0 ? 'افزایش' : ($change_amount < 0 ? 'کاهش' : 'ثابت')
        ];
    }
    
    // تحلیل تفاوت مقدار تایید شده
    public function getApprovalDifferenceAnalysis($national_code, $year = null) {
        $where_conditions = ["کدملی_شناسه_ملی = ?"];
        $params = [$national_code];
        
        if ($year) {
            $where_conditions[] = "YEAR(تاریخ_تایید_درخواست_توسط_پخش) = ?";
            $params[] = $this->shamsiToGregorianYear($year);
        }
        
        $where_clause = "WHERE " . implode(" AND ", $where_conditions);
        
        $query = "
            SELECT 
                بخش_مصرف,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_distribution,
                SUM(مقدار_تایید_شده_توسط_متولی_اول) as total_first_responsible,
                SUM(مقدار_تایید_شده_توسط_پخش - مقدار_تایید_شده_توسط_متولی_اول) as total_difference,
                AVG(مقدار_تایید_شده_توسط_پخش - مقدار_تایید_شده_توسط_متولی_اول) as average_difference,
                COUNT(*) as request_count
            FROM fuel_requests 
            {$where_clause}
            AND بخش_مصرف IN ('بخش ۵', 'بخش ۶')
            GROUP BY بخش_مصرف
            ORDER BY total_difference DESC
        ";
        
        $stmt = $this->pdo->prepare($query);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // تحلیل کامل شخص
    public function getCompleteAnalysis($national_code, $filters = []) {
        $analysis = [];
        
        // اطلاعات اولیه
        $analysis['person_info'] = $this->getPersonInfo($national_code);
        
        // سال‌های موجود
        $analysis['available_years'] = $this->getAvailableYearsForPerson($national_code);
        
        // تحلیل بر اساس نوع انتخاب شده
        $analysis_type = $filters['analysis_type'] ?? 'monthly';
        
        switch ($analysis_type) {
            case 'three_month':
                $analysis['three_month_consumption'] = $this->getThreeMonthConsumption(
                    $national_code, 
                    $filters['year'] ?? null
                );
                break;
                
            case 'seasonal':
                $analysis['seasonal_consumption'] = $this->getSeasonalConsumption(
                    $national_code, 
                    $filters['year'] ?? null
                );
                break;
                
            case 'product':
                $analysis['by_product'] = $this->getConsumptionByProduct(
                    $national_code, 
                    $filters['year'] ?? null
                );
                break;
                
            case 'section':
                $analysis['by_usage_section'] = $this->getConsumptionByUsageSection(
                    $national_code, 
                    $filters['year'] ?? null
                );
                break;
                
            case 'product_section':
                $analysis['by_product_section'] = $this->getConsumptionByProductAndSection(
                    $national_code, 
                    $filters['year'] ?? null
                );
                break;
                
            case 'yearly':
                $analysis['yearly_comparison'] = $this->getYearlyComparison(
                    $national_code, 
                    $filters['comparison_years'] ?? []
                );
                break;
                
            case 'monthly':
            default:
                $analysis['monthly_consumption'] = $this->getMonthlyConsumption(
                    $national_code, 
                    $filters['year'] ?? null
                );
                break;
        }
        
        // مصرف ماهانه (همیشه برای خلاصه)
        $analysis['monthly_consumption_summary'] = $this->getMonthlyConsumption(
            $national_code, 
            $filters['year'] ?? null
        );
        
        // تحلیل روند
        if (!empty($filters['base_year']) && !empty($filters['compare_year'])) {
            $analysis['consumption_trend'] = $this->getConsumptionTrend(
                $national_code, 
                $filters['base_year'], 
                $filters['compare_year']
            );
        }
        
        // تحلیل تفاوت تایید
        $analysis['approval_difference'] = $this->getApprovalDifferenceAnalysis(
            $national_code, 
            $filters['year'] ?? null
        );
        
        // آمار کلی
        $analysis['summary'] = $this->getConsumptionSummary($national_code, $filters);
        
        return $analysis;
    }
    
    private function getConsumptionSummary($national_code, $filters) {
        $where_conditions = ["کدملی_شناسه_ملی = ?"];
        $params = [$national_code];
        
        if (!empty($filters['year'])) {
            $where_conditions[] = "YEAR(تاریخ_تایید_درخواست_توسط_پخش) = ?";
            $params[] = $this->shamsiToGregorianYear($filters['year']);
        }
        
        $where_clause = "WHERE " . implode(" AND ", $where_conditions);
        
        $query = "
            SELECT 
                COUNT(*) as total_requests,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_consumption,
                AVG(مقدار_تایید_شده_توسط_پخش) as average_consumption,
                MAX(مقدار_تایید_شده_توسط_پخش) as max_consumption,
                MIN(مقدار_تایید_شده_توسط_پخش) as min_consumption,
                COUNT(DISTINCT فراوده) as product_variety,
                COUNT(DISTINCT بخش_مصرف) as section_variety,
                COUNT(DISTINCT شناسه_یکتای_کاربرد) as identifier_count,
                MAX(تاریخ_تایید_درخواست_توسط_پخش) as last_request_date,
                MIN(تاریخ_تایید_درخواست_توسط_پخش) as first_request_date
            FROM fuel_requests 
            {$where_clause}
        ";
        
        $stmt = $this->pdo->prepare($query);
        $stmt->execute($params);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    private function organizeMonthlyData($data) {
        $organized = [];
        
        foreach ($data as $row) {
            try {
                $carbon_date = Carbon::create($row['gregorian_year'], $row['gregorian_month'], 1);
                $jalali = Jalalian::fromCarbon($carbon_date);
                
                $shamsi_year = $jalali->getYear();
                $shamsi_month = $jalali->getMonth();
                $month_name = $this->getMonthName($shamsi_month);
                
                $organized[] = [
                    'shamsi_year' => $shamsi_year,
                    'shamsi_month' => $shamsi_month,
                    'month_name' => $month_name,
                    'period' => $month_name . ' ' . $shamsi_year,
                    'total_consumption' => $row['total_consumption'],
                    'request_count' => $row['request_count'],
                    'average_per_request' => $row['average_per_request'],
                    'max_consumption' => $row['max_consumption'],
                    'min_consumption' => $row['min_consumption']
                ];
            } catch (Exception $e) {
                continue;
            }
        }
        
        return $organized;
    }
    
    private function organizeQuarterlyData($data) {
        $organized = [];
        $quarter_names = ['سه‌ماهه اول', 'سه‌ماهه دوم', 'سه‌ماهه سوم', 'سه‌ماهه چهارم'];
        $month_ranges = [
            'فروردین-خرداد',
            'تیر-شهریور',
            'مهر-آذر',
            'دی-اسفند'
        ];
        
        foreach ($data as $row) {
            try {
                $carbon_date = Carbon::create($row['gregorian_year'], 1, 1);
                $jalali = Jalalian::fromCarbon($carbon_date);
                $shamsi_year = $jalali->getYear();
                
                $quarter_num = (int)$row['quarter'];
                $organized[] = [
                    'shamsi_year' => $shamsi_year,
                    'quarter_number' => $quarter_num,
                    'quarter_name' => $quarter_names[$quarter_num - 1],
                    'month_range' => $month_ranges[$quarter_num - 1],
                    'period' => $quarter_names[$quarter_num - 1] . ' ' . $shamsi_year,
                    'total_consumption' => $row['total_consumption'],
                    'request_count' => $row['request_count'],
                    'average_per_request' => $row['average_per_request'],
                    'max_consumption' => $row['max_consumption'],
                    'min_consumption' => $row['min_consumption'],
                    'months' => $row['months']
                ];
            } catch (Exception $e) {
                continue;
            }
        }
        
        return $organized;
    }
    
    private function organizeSeasonalData($data) {
        $organized = [];
        $season_names = ['فصل اول', 'فصل دوم', 'فصل سوم', 'فصل چهارم'];
        $season_ranges = [
            'فروردین-خرداد',
            'تیر-شهریور',
            'مهر-آذر',
            'دی-اسفند'
        ];
        
        foreach ($data as $row) {
            try {
                $carbon_date = Carbon::create($row['gregorian_year'], 1, 1);
                $jalali = Jalalian::fromCarbon($carbon_date);
                $shamsi_year = $jalali->getYear();
                
                $season_num = (int)$row['season'];
                $organized[] = [
                    'shamsi_year' => $shamsi_year,
                    'season_number' => $season_num,
                    'season_name' => $season_names[$season_num - 1],
                    'season_range' => $season_ranges[$season_num - 1],
                    'period' => $season_names[$season_num - 1] . ' ' . $shamsi_year,
                    'total_consumption' => $row['total_consumption'],
                    'request_count' => $row['request_count'],
                    'average_per_request' => $row['average_per_request'],
                    'products' => $row['products']
                ];
            } catch (Exception $e) {
                continue;
            }
        }
        
        return $organized;
    }
    
    private function getMonthName($month) {
        $months = [
            1 => 'فروردین', 2 => 'اردیبهشت', 3 => 'خرداد',
            4 => 'تیر', 5 => 'مرداد', 6 => 'شهریور',
            7 => 'مهر', 8 => 'آبان', 9 => 'آذر',
            10 => 'دی', 11 => 'بهمن', 12 => 'اسفند'
        ];
        return $months[$month] ?? 'نامشخص';
    }
    
    private function shamsiToGregorianYear($shamsi_year) {
        // تبدیل تقریبی سال شمسی به میلادی
        return $shamsi_year + 621;
    }
    
    private function gregorianToShamsiYear($gregorian_year) {
        // تبدیل تقریبی سال میلادی به شمسی
        return $gregorian_year - 621;
    }
}

// دریافت کدملی از درخواست
$national_code = $_GET['national_code'] ?? $_POST['national_code'] ?? '';

// ایجاد شیء تحلیل
$analysis = new PersonalConsumptionAnalysis($pdo);

// تنظیم فیلترها
$filters = [
    'year' => $_POST['year'] ?? null,
    'comparison_years' => $_POST['comparison_years'] ?? [],
    'base_year' => $_POST['base_year'] ?? null,
    'compare_year' => $_POST['compare_year'] ?? null,
    'analysis_type' => $_POST['analysis_type'] ?? 'monthly'
];

// دریافت اطلاعات شخص
$person_info = $analysis->getPersonInfo($national_code);

// اگر کدملی معتبر بود، تحلیل را انجام بده
$reports = [];
$available_years = [];

if ($person_info && !empty($national_code)) {
    $available_years = $analysis->getAvailableYearsForPerson($national_code);
    
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $reports = $analysis->getCompleteAnalysis($national_code, $filters);
    }
}
?>

<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>تحلیل مصرف شخصی سوخت</title>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        body {
            font-family: 'Tahoma', sans-serif;
            background-color: #f5f5f5;
            margin: 0;
            padding: 20px;
        }
        .container {
            max-width: 1400px;
            margin: 0 auto;
            background: white;
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
        }
        .search-box {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
            text-align: center;
        }
        .search-input {
            width: 300px;
            padding: 10px;
            border: 2px solid #007bff;
            border-radius: 4px;
            font-size: 16px;
            text-align: center;
        }
        .search-btn {
            background: #007bff;
            color: white;
            padding: 10px 20px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 16px;
            margin-right: 10px;
        }
        .person-info {
            background: #e9ecef;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        .person-info h3 {
            color: #007bff;
            margin-top: 0;
        }
        .info-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 15px;
            margin-top: 15px;
        }
        .info-item {
            background: white;
            padding: 10px;
            border-radius: 4px;
            border: 1px solid #ddd;
        }
        .info-label {
            font-weight: bold;
            color: #495057;
            display: block;
            margin-bottom: 5px;
        }
        .filters {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        .filter-group {
            margin-bottom: 15px;
        }
        .filter-row {
            display: flex;
            gap: 15px;
            flex-wrap: wrap;
        }
        .filter-column {
            flex: 1;
            min-width: 200px;
        }
        label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
        }
        select, input {
            width: 100%;
            padding: 8px;
            border: 1px solid #ddd;
            border-radius: 4px;
        }
        .btn {
            background: #007bff;
            color: white;
            padding: 10px 20px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 16px;
        }
        .btn:hover {
            background: #0056b3;
        }
        .btn-success {
            background: #28a745;
        }
        .btn-success:hover {
            background: #218838;
        }
        .btn-warning {
            background: #ffc107;
            color: #212529;
        }
        .btn-warning:hover {
            background: #e0a800;
        }
        .report-section {
            margin: 30px 0;
            padding: 20px;
            border: 1px solid #ddd;
            border-radius: 8px;
        }
        .report-section h3 {
            color: #333;
            border-bottom: 2px solid #007bff;
            padding-bottom: 10px;
        }
        .chart-container {
            margin: 20px 0;
            padding: 15px;
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            height: 400px;
        }
        canvas {
            width: 100% !important;
            height: 100% !important;
        }
        .data-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 15px;
            font-size: 14px;
        }
        .data-table th {
            background-color: #007bff;
            color: white;
            text-align: center;
            padding: 12px;
            position: sticky;
            top: 0;
        }
        .data-table td {
            border: 1px solid #ddd;
            padding: 8px;
            text-align: center;
        }
        .data-table tr:nth-child(even) {
            background-color: #f8f9fa;
        }
        .data-table tr:hover {
            background-color: #e9ecef;
        }
        .summary-card {
            background: #f8f9fa;
            border: 1px solid #ddd;
            border-radius: 8px;
            padding: 20px;
            margin: 20px 0;
        }
        .stat-card {
            background: white;
            border: 1px solid #ddd;
            border-radius: 8px;
            padding: 15px;
            margin: 10px;
            text-align: center;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        .stat-value {
            font-size: 24px;
            font-weight: bold;
            color: #007bff;
            display: block;
            margin: 10px 0;
        }
        .stat-label {
            color: #6c757d;
            font-size: 14px;
        }
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin: 20px 0;
        }
        .badge {
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 12px;
            font-weight: bold;
        }
        .badge-success {
            background-color: #28a745;
            color: white;
        }
        .badge-danger {
            background-color: #dc3545;
            color: white;
        }
        .badge-warning {
            background-color: #ffc107;
            color: #212529;
        }
        .badge-info {
            background-color: #17a2b8;
            color: white;
        }
        .trend-up {
            color: #dc3545;
            font-weight: bold;
        }
        .trend-down {
            color: #28a745;
            font-weight: bold;
        }
        .trend-stable {
            color: #6c757d;
            font-weight: bold;
        }
        .no-data {
            text-align: center;
            padding: 40px;
            color: #6c757d;
            font-size: 18px;
        }
        .chart-type-selector {
            display: flex;
            gap: 10px;
            margin: 10px 0;
            flex-wrap: wrap;
        }
        .chart-type-btn {
            padding: 8px 15px;
            border: 1px solid #ddd;
            border-radius: 4px;
            background: white;
            cursor: pointer;
        }
        .chart-type-btn.active {
            background: #007bff;
            color: white;
            border-color: #007bff;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>سیستم تحلیل مصرف شخصی سوخت</h1>
        
        <!-- جستجوی کدملی -->
        <div class="search-box">
            <form method="GET">
                <input 
                    type="text" 
                    name="national_code" 
                    class="search-input" 
                    placeholder="کدملی / شناسه ملی را وارد کنید"
                    value="<?= htmlspecialchars($national_code) ?>"
                    required
                >
                <button type="submit" class="search-btn">جستجو</button>
            </form>
        </div>
        
        <?php if ($person_info): ?>
            <!-- نمایش اطلاعات شخص -->
            <div class="person-info">
                <h3>📋 اطلاعات شخص</h3>
                <div class="info-grid">
                    <div class="info-item">
                        <span class="info-label">کدملی / شناسه ملی:</span>
                        <span><?= $person_info['کدملی_شناسه_ملی'] ?></span>
                    </div>
                    <div class="info-item">
                        <span class="info-label">نام متقاضی:</span>
                        <span><?= $person_info['نام_متقاضی'] ?? 'نامشخص' ?></span>
                    </div>
                    <div class="info-item">
                        <span class="info-label">شهرستان:</span>
                        <span><?= $person_info['شهرستان'] ?? 'نامشخص' ?></span>
                    </div>
                    <div class="info-item">
                        <span class="info-label">تلفن همراه:</span>
                        <span><?= $person_info['تلفن_همراه'] ?? 'نامشخص' ?></span>
                    </div>
                    <div class="info-item">
                        <span class="info-label">تعداد شناسه‌های یکتا:</span>
                        <span class="badge <?= $person_info['identifier_count'] > 1 ? 'badge-danger' : 'badge-success' ?>">
                            <?= $person_info['identifier_count'] ?>
                        </span>
                    </div>
                    <div class="info-item">
                        <span class="info-label">فرآورده‌های مصرفی:</span>
                        <span><?= $person_info['products'] ?? 'نامشخص' ?></span>
                    </div>
                    <div class="info-item">
                        <span class="info-label">بخش‌های مصرف:</span>
                        <span><?= $person_info['usage_sections'] ?? 'نامشخص' ?></span>
                    </div>
                </div>
            </div>
            
            <!-- فیلترهای تحلیل -->
            <div class="filters">
                <form method="POST" id="analysisForm">
                    <input type="hidden" name="national_code" value="<?= $national_code ?>">
                    
                    <div class="filter-row">
                        <div class="filter-column">
                            <label>انتخاب سال برای تحلیل:</label>
                            <select name="year" id="yearSelect">
                                <option value="">همه سال‌ها</option>
                                <?php foreach ($available_years as $year): ?>
                                    <option value="<?= $year ?>" <?= $filters['year'] == $year ? 'selected' : '' ?>>
                                        <?= $year ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="filter-column">
                            <label>سال پایه برای مقایسه:</label>
                            <select name="base_year">
                                <option value="">انتخاب نشده</option>
                                <?php foreach ($available_years as $year): ?>
                                    <option value="<?= $year ?>" <?= $filters['base_year'] == $year ? 'selected' : '' ?>>
                                        <?= $year ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="filter-column">
                            <label>سال مقایسه:</label>
                            <select name="compare_year">
                                <option value="">انتخاب نشده</option>
                                <?php foreach ($available_years as $year): ?>
                                    <option value="<?= $year ?>" <?= $filters['compare_year'] == $year ? 'selected' : '' ?>>
                                        <?= $year ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    
                    <div class="filter-row" style="margin-top: 15px;">
                        <div class="filter-column">
                            <label>انتخاب چند سال برای مقایسه:</label>
                            <select name="comparison_years[]" multiple style="height: 120px;">
                                <?php foreach ($available_years as $year): ?>
                                    <option value="<?= $year ?>" <?= in_array($year, $filters['comparison_years']) ? 'selected' : '' ?>>
                                        <?= $year ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="filter-column">
                            <label>نوع تحلیل:</label>
                            <select name="analysis_type" id="analysisType">
                                <option value="monthly" <?= $filters['analysis_type'] == 'monthly' ? 'selected' : '' ?>>تحلیل ماهانه</option>
                                <option value="three_month" <?= $filters['analysis_type'] == 'three_month' ? 'selected' : '' ?>>تحلیل سه‌ماهه</option>
                                <option value="seasonal" <?= $filters['analysis_type'] == 'seasonal' ? 'selected' : '' ?>>تحلیل فصلی</option>
                                <option value="yearly" <?= $filters['analysis_type'] == 'yearly' ? 'selected' : '' ?>>تحلیل سالانه</option>
                                <option value="product" <?= $filters['analysis_type'] == 'product' ? 'selected' : '' ?>>تحلیل بر اساس فرآورده</option>
                                <option value="section" <?= $filters['analysis_type'] == 'section' ? 'selected' : '' ?>>تحلیل بر اساس بخش مصرف</option>
                                <option value="product_section" <?= $filters['analysis_type'] == 'product_section' ? 'selected' : '' ?>>تحلیل بر اساس فرآورده و بخش مصرف</option>
                            </select>
                        </div>
                    </div>
                    
                    <button type="submit" class="btn">اجرای تحلیل</button>
                    <button type="button" class="btn btn-success" onclick="printReport()">چاپ گزارش</button>
                    <button type="button" class="btn btn-warning" onclick="exportToPDF()">خروجی PDF</button>
                </form>
            </div>
            
            <?php if (!empty($reports)): ?>
                <!-- خلاصه آمار -->
                <div class="summary-card">
                    <h3>📊 خلاصه آمار مصرف</h3>
                    <div class="stats-grid">
                        <div class="stat-card">
                            <span class="stat-label">مجموع مصرف</span>
                            <span class="stat-value"><?= number_format($reports['summary']['total_consumption'] ?? 0) ?> لیتر</span>
                        </div>
                        <div class="stat-card">
                            <span class="stat-label">تعداد درخواست‌ها</span>
                            <span class="stat-value"><?= number_format($reports['summary']['total_requests'] ?? 0) ?></span>
                        </div>
                        <div class="stat-card">
                            <span class="stat-label">متوسط مصرف هر درخواست</span>
                            <span class="stat-value"><?= number_format($reports['summary']['average_consumption'] ?? 0, 1) ?> لیتر</span>
                        </div>
                        <div class="stat-card">
                            <span class="stat-label">تنوع فرآورده</span>
                            <span class="stat-value"><?= $reports['summary']['product_variety'] ?? 0 ?></span>
                        </div>
                        <div class="stat-card">
                            <span class="stat-label">بیشترین مصرف</span>
                            <span class="stat-value"><?= number_format($reports['summary']['max_consumption'] ?? 0) ?> لیتر</span>
                        </div>
                        <div class="stat-card">
                            <span class="stat-label">کمترین مصرف</span>
                            <span class="stat-value"><?= number_format($reports['summary']['min_consumption'] ?? 0) ?> لیتر</span>
                        </div>
                    </div>
                </div>
                
                <!-- انتخاب نوع نمودار -->
                <div class="chart-type-selector">
                    <button class="chart-type-btn active" data-chart-type="bar">نمودار میله‌ای</button>
                    <button class="chart-type-btn" data-chart-type="line">نمودار خطی</button>
                    <button class="chart-type-btn" data-chart-type="pie">نمودار دایره‌ای</button>
                </div>
                
                <?php 
                $analysis_type = $filters['analysis_type'];
                ?>
                
                <!-- تحلیل سه‌ماهه -->
                <?php if ($analysis_type == 'three_month' && !empty($reports['three_month_consumption'])): ?>
                    <div class="report-section">
                        <h3>📊 تحلیل مصرف سه‌ماهه</h3>
                        <div class="chart-container">
                            <canvas id="threeMonthChart"></canvas>
                        </div>
                        
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>ردیف</th>
                                    <th>سه‌ماهه</th>
                                    <th>سال</th>
                                    <th>محدوده ماه‌ها</th>
                                    <th>مصرف کل (لیتر)</th>
                                    <th>تعداد درخواست</th>
                                    <th>متوسط هر درخواست</th>
                                    <th>بیشترین مصرف</th>
                                    <th>کمترین مصرف</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($reports['three_month_consumption'] as $index => $quarter): ?>
                                    <tr>
                                        <td><?= $index + 1 ?></td>
                                        <td><?= $quarter['quarter_name'] ?></td>
                                        <td><?= $quarter['shamsi_year'] ?></td>
                                        <td><?= $quarter['month_range'] ?></td>
                                        <td><?= number_format($quarter['total_consumption']) ?></td>
                                        <td><?= $quarter['request_count'] ?></td>
                                        <td><?= number_format($quarter['average_per_request'], 1) ?></td>
                                        <td><?= number_format($quarter['max_consumption']) ?></td>
                                        <td><?= number_format($quarter['min_consumption']) ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
                
                <!-- تحلیل فصلی -->
                <?php if ($analysis_type == 'seasonal' && !empty($reports['seasonal_consumption'])): ?>
                    <div class="report-section">
                        <h3>🍂 تحلیل مصرف فصلی</h3>
                        <div class="chart-container">
                            <canvas id="seasonalChart"></canvas>
                        </div>
                        
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>ردیف</th>
                                    <th>فصل</th>
                                    <th>سال</th>
                                    <th>محدوده ماه‌ها</th>
                                    <th>مصرف کل (لیتر)</th>
                                    <th>تعداد درخواست</th>
                                    <th>متوسط هر درخواست</th>
                                    <th>فرآورده‌ها</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($reports['seasonal_consumption'] as $index => $season): ?>
                                    <tr>
                                        <td><?= $index + 1 ?></td>
                                        <td><?= $season['season_name'] ?></td>
                                        <td><?= $season['shamsi_year'] ?></td>
                                        <td><?= $season['season_range'] ?></td>
                                        <td><?= number_format($season['total_consumption']) ?></td>
                                        <td><?= $season['request_count'] ?></td>
                                        <td><?= number_format($season['average_per_request'], 1) ?></td>
                                        <td><?= $season['products'] ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
                
                <!-- تحلیل بر اساس فرآورده -->
                <?php if ($analysis_type == 'product' && !empty($reports['by_product'])): ?>
                    <div class="report-section">
                        <h3>⛽ تحلیل مصرف بر اساس فرآورده</h3>
                        <div class="chart-container">
                            <canvas id="productChart"></canvas>
                        </div>
                        
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>ردیف</th>
                                    <th>فرآورده</th>
                                    <th>مصرف کل (لیتر)</th>
                                    <th>تعداد درخواست</th>
                                    <th>متوسط هر درخواست</th>
                                    <th>بیشترین مصرف</th>
                                    <th>کمترین مصرف</th>
                                    <th>بخش‌های مصرف</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($reports['by_product'] as $index => $product): ?>
                                    <tr>
                                        <td><?= $index + 1 ?></td>
                                        <td><?= $product['فراوده'] ?></td>
                                        <td><?= number_format($product['total_consumption']) ?></td>
                                        <td><?= $product['request_count'] ?></td>
                                        <td><?= number_format($product['average_per_request'], 1) ?></td>
                                        <td><?= number_format($product['max_consumption']) ?></td>
                                        <td><?= number_format($product['min_consumption']) ?></td>
                                        <td><?= $product['usage_sections'] ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
                
                <!-- تحلیل بر اساس بخش مصرف -->
                <?php if ($analysis_type == 'section' && !empty($reports['by_usage_section'])): ?>
                    <div class="report-section">
                        <h3>🏢 تحلیل مصرف بر اساس بخش مصرف</h3>
                        <div class="chart-container">
                            <canvas id="sectionChart"></canvas>
                        </div>
                        
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>ردیف</th>
                                    <th>بخش مصرف</th>
                                    <th>مصرف کل (لیتر)</th>
                                    <th>تعداد درخواست</th>
                                    <th>متوسط هر درخواست</th>
                                    <th>بیشترین مصرف</th>
                                    <th>کمترین مصرف</th>
                                    <th>فرآورده‌ها</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($reports['by_usage_section'] as $index => $section): ?>
                                    <tr>
                                        <td><?= $index + 1 ?></td>
                                        <td><?= $section['بخش_مصرف'] ?></td>
                                        <td><?= number_format($section['total_consumption']) ?></td>
                                        <td><?= $section['request_count'] ?></td>
                                        <td><?= number_format($section['average_per_request'], 1) ?></td>
                                        <td><?= number_format($section['max_consumption']) ?></td>
                                        <td><?= number_format($section['min_consumption']) ?></td>
                                        <td><?= $section['products'] ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
                
                <!-- تحلیل بر اساس فرآورده و بخش مصرف -->
                <?php if ($analysis_type == 'product_section' && !empty($reports['by_product_section'])): ?>
                    <div class="report-section">
                        <h3>📊 تحلیل مصرف بر اساس فرآورده و بخش مصرف</h3>
                        <div class="chart-container">
                            <canvas id="productSectionChart"></canvas>
                        </div>
                        
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>ردیف</th>
                                    <th>فرآورده</th>
                                    <th>بخش مصرف</th>
                                    <th>مصرف کل (لیتر)</th>
                                    <th>تعداد درخواست</th>
                                    <th>متوسط هر درخواست</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($reports['by_product_section'] as $index => $item): ?>
                                    <tr>
                                        <td><?= $index + 1 ?></td>
                                        <td><?= $item['فراوده'] ?></td>
                                        <td><?= $item['بخش_مصرف'] ?></td>
                                        <td><?= number_format($item['total_consumption']) ?></td>
                                        <td><?= $item['request_count'] ?></td>
                                        <td><?= number_format($item['average_per_request'], 1) ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
                
                <!-- تحلیل سالانه -->
                <?php if ($analysis_type == 'yearly' && !empty($reports['yearly_comparison'])): ?>
                    <div class="report-section">
                        <h3>📅 تحلیل مصرف سالانه</h3>
                        <div class="chart-container">
                            <canvas id="yearlyChart"></canvas>
                        </div>
                        
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>سال شمسی</th>
                                    <th>مصرف کل (لیتر)</th>
                                    <th>تعداد درخواست</th>
                                    <th>متوسط هر درخواست</th>
                                    <th>بیشترین مصرف</th>
                                    <th>کمترین مصرف</th>
                                    <th>فرآورده‌ها</th>
                                    <th>بخش‌های مصرف</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($reports['yearly_comparison'] as $year): ?>
                                    <tr>
                                        <td><?= $year['shamsi_year'] ?></td>
                                        <td><?= number_format($year['total_consumption']) ?></td>
                                        <td><?= $year['request_count'] ?></td>
                                        <td><?= number_format($year['average_per_request'], 1) ?></td>
                                        <td><?= number_format($year['max_consumption']) ?></td>
                                        <td><?= number_format($year['min_consumption']) ?></td>
                                        <td><?= $year['products'] ?></td>
                                        <td><?= $year['usage_sections'] ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
                
                <!-- تحلیل روند مصرف -->
                <?php if (!empty($reports['consumption_trend'])): ?>
                    <div class="report-section">
                        <h3>📈 تحلیل روند مصرف</h3>
                        <div class="chart-container">
                            <canvas id="trendChart"></canvas>
                        </div>
                        
                        <div class="summary-card">
                            <h4>نتایج مقایسه</h4>
                            <p>
                                سال پایه (<?= $reports['consumption_trend']['base_year'] ?>): 
                                <strong><?= number_format($reports['consumption_trend']['base_consumption']) ?> لیتر</strong>
                            </p>
                            <p>
                                سال مقایسه (<?= $reports['consumption_trend']['compare_year'] ?>): 
                                <strong><?= number_format($reports['consumption_trend']['compare_consumption']) ?> لیتر</strong>
                            </p>
                            <p>
                                تغییر مقدار: 
                                <span class="<?= $reports['consumption_trend']['change_amount'] > 0 ? 'trend-up' : ($reports['consumption_trend']['change_amount'] < 0 ? 'trend-down' : 'trend-stable') ?>">
                                    <?= $reports['consumption_trend']['change_amount'] > 0 ? '+' : '' ?><?= number_format($reports['consumption_trend']['change_amount']) ?> لیتر
                                </span>
                            </p>
                            <p>
                                تغییر درصد: 
                                <span class="<?= $reports['consumption_trend']['change_percentage'] > 0 ? 'trend-up' : ($reports['consumption_trend']['change_percentage'] < 0 ? 'trend-down' : 'trend-stable') ?>">
                                    <?= $reports['consumption_trend']['change_percentage'] > 0 ? '+' : '' ?><?= number_format($reports['consumption_trend']['change_percentage'], 1) ?>%
                                </span>
                            </p>
                            <p>
                                روند: 
                                <span class="badge <?= $reports['consumption_trend']['trend'] == 'افزایش' ? 'badge-danger' : ($reports['consumption_trend']['trend'] == 'کاهش' ? 'badge-success' : 'badge-info') ?>">
                                    <?= $reports['consumption_trend']['trend'] ?>
                                </span>
                            </p>
                        </div>
                    </div>
                <?php endif; ?>
                
                <!-- تحلیل تفاوت تایید -->
                <?php if (!empty($reports['approval_difference'])): ?>
                    <div class="report-section">
                        <h3>⚖️ تحلیل تفاوت مقادیر تایید شده</h3>
                        <div class="chart-container">
                            <canvas id="approvalChart"></canvas>
                        </div>
                        
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>بخش مصرف</th>
                                    <th>مجموع تایید پخش (لیتر)</th>
                                    <th>مجموع تایید متولی اول (لیتر)</th>
                                    <th>تفاوت کل (لیتر)</th>
                                    <th>متوسط تفاوت هر درخواست</th>
                                    <th>تعداد درخواست</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($reports['approval_difference'] as $item): ?>
                                    <tr>
                                        <td><?= $item['بخش_مصرف'] ?></td>
                                        <td><?= number_format($item['total_distribution']) ?></td>
                                        <td><?= number_format($item['total_first_responsible']) ?></td>
                                        <td>
                                            <span class="<?= $item['total_difference'] > 0 ? 'trend-up' : 'trend-down' ?>">
                                                <?= number_format($item['total_difference']) ?>
                                            </span>
                                        </td>
                                        <td><?= number_format($item['average_difference'], 1) ?></td>
                                        <td><?= $item['request_count'] ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
                
            <?php else: ?>
                <div class="no-data">
                    <p>برای مشاهده گزارش تحلیل، فیلترها را تنظیم و دکمه "اجرای تحلیل" را کلیک کنید.</p>
                </div>
            <?php endif; ?>
            
        <?php elseif (!empty($national_code)): ?>
            <div class="no-data">
                <p>❌ شخصی با کدملی <?= htmlspecialchars($national_code) ?> یافت نشد.</p>
            </div>
        <?php endif; ?>
    </div>

    <script>
        // مدیریت تغییر نوع نمودار
        document.querySelectorAll('.chart-type-btn').forEach(btn => {
            btn.addEventListener('click', function() {
                document.querySelectorAll('.chart-type-btn').forEach(b => b.classList.remove('active'));
                this.classList.add('active');
                
                const chartType = this.dataset.chartType;
                updateAllCharts(chartType);
            });
        });
        
        // تابع برای به‌روزرسانی همه نمودارها
        function updateAllCharts(type) {
            // این تابع همه نمودارهای فعال را با نوع جدید به‌روزرسانی می‌کند
            Chart.helpers.each(Chart.instances, function(instance) {
                instance.config.type = type;
                instance.update();
            });
        }
        
        // ایجاد نمودارها
        document.addEventListener('DOMContentLoaded', function() {
            <?php if (!empty($reports)): ?>
                <?php $analysis_type = $filters['analysis_type']; ?>
                
                // نمودار سه‌ماهه
                <?php if ($analysis_type == 'three_month' && !empty($reports['three_month_consumption'])): ?>
                    const threeMonthCtx = document.getElementById('threeMonthChart');
                    if (threeMonthCtx) {
                        new Chart(threeMonthCtx.getContext('2d'), {
                            type: 'bar',
                            data: {
                                labels: <?= json_encode(array_column($reports['three_month_consumption'], 'period')) ?>,
                                datasets: [{
                                    label: 'مصرف سه‌ماهه (لیتر)',
                                    data: <?= json_encode(array_column($reports['three_month_consumption'], 'total_consumption')) ?>,
                                    backgroundColor: 'rgba(54, 162, 235, 0.5)',
                                    borderColor: 'rgba(54, 162, 235, 1)',
                                    borderWidth: 2
                                }]
                            },
                            options: {
                                responsive: true,
                                maintainAspectRatio: false,
                                plugins: {
                                    title: {
                                        display: true,
                                        text: 'مصرف سه‌ماهه'
                                    }
                                },
                                scales: {
                                    y: {
                                        beginAtZero: true,
                                        title: {
                                            display: true,
                                            text: 'لیتر'
                                        }
                                    },
                                    x: {
                                        title: {
                                            display: true,
                                            text: 'سه‌ماهه'
                                        }
                                    }
                                }
                            }
                        });
                    }
                <?php endif; ?>
                
                // نمودار فصلی
                <?php if ($analysis_type == 'seasonal' && !empty($reports['seasonal_consumption'])): ?>
                    const seasonalCtx = document.getElementById('seasonalChart');
                    if (seasonalCtx) {
                        new Chart(seasonalCtx.getContext('2d'), {
                            type: 'bar',
                            data: {
                                labels: <?= json_encode(array_column($reports['seasonal_consumption'], 'period')) ?>,
                                datasets: [{
                                    label: 'مصرف فصلی (لیتر)',
                                    data: <?= json_encode(array_column($reports['seasonal_consumption'], 'total_consumption')) ?>,
                                    backgroundColor: 'rgba(255, 159, 64, 0.5)',
                                    borderColor: 'rgba(255, 159, 64, 1)',
                                    borderWidth: 2
                                }]
                            },
                            options: {
                                responsive: true,
                                maintainAspectRatio: false,
                                plugins: {
                                    title: {
                                        display: true,
                                        text: 'مصرف فصلی'
                                    }
                                },
                                scales: {
                                    y: {
                                        beginAtZero: true,
                                        title: {
                                            display: true,
                                            text: 'لیتر'
                                        }
                                    },
                                    x: {
                                        title: {
                                            display: true,
                                            text: 'فصل'
                                        }
                                    }
                                }
                            }
                        });
                    }
                <?php endif; ?>
                
                // نمودار فرآورده
                <?php if ($analysis_type == 'product' && !empty($reports['by_product'])): ?>
                    const productCtx = document.getElementById('productChart');
                    if (productCtx) {
                        new Chart(productCtx.getContext('2d'), {
                            type: 'pie',
                            data: {
                                labels: <?= json_encode(array_column($reports['by_product'], 'فراوده')) ?>,
                                datasets: [{
                                    label: 'مصرف بر اساس فرآورده',
                                    data: <?= json_encode(array_column($reports['by_product'], 'total_consumption')) ?>,
                                    backgroundColor: [
                                        'rgba(255, 99, 132, 0.5)',
                                        'rgba(54, 162, 235, 0.5)',
                                        'rgba(255, 206, 86, 0.5)',
                                        'rgba(75, 192, 192, 0.5)',
                                        'rgba(153, 102, 255, 0.5)',
                                        'rgba(255, 159, 64, 0.5)',
                                        'rgba(201, 203, 207, 0.5)'
                                    ],
                                    borderWidth: 2
                                }]
                            },
                            options: {
                                responsive: true,
                                maintainAspectRatio: false,
                                plugins: {
                                    title: {
                                        display: true,
                                        text: 'توزیع مصرف بر اساس فرآورده'
                                    }
                                }
                            }
                        });
                    }
                <?php endif; ?>
                
                // نمودار بخش مصرف
                <?php if ($analysis_type == 'section' && !empty($reports['by_usage_section'])): ?>
                    const sectionCtx = document.getElementById('sectionChart');
                    if (sectionCtx) {
                        new Chart(sectionCtx.getContext('2d'), {
                            type: 'doughnut',
                            data: {
                                labels: <?= json_encode(array_column($reports['by_usage_section'], 'بخش_مصرف')) ?>,
                                datasets: [{
                                    label: 'مصرف بر اساس بخش مصرف',
                                    data: <?= json_encode(array_column($reports['by_usage_section'], 'total_consumption')) ?>,
                                    backgroundColor: [
                                        'rgba(255, 99, 132, 0.5)',
                                        'rgba(54, 162, 235, 0.5)',
                                        'rgba(255, 206, 86, 0.5)',
                                        'rgba(75, 192, 192, 0.5)',
                                        'rgba(153, 102, 255, 0.5)'
                                    ],
                                    borderWidth: 2
                                }]
                            },
                            options: {
                                responsive: true,
                                maintainAspectRatio: false,
                                plugins: {
                                    title: {
                                        display: true,
                                        text: 'توزیع مصرف بر اساس بخش مصرف'
                                    }
                                }
                            }
                        });
                    }
                <?php endif; ?>
                
                // نمودار فرآورده و بخش مصرف
                <?php if ($analysis_type == 'product_section' && !empty($reports['by_product_section'])): ?>
                    const productSectionCtx = document.getElementById('productSectionChart');
                    if (productSectionCtx) {
                        // گروه‌بندی داده‌ها بر اساس فرآورده
                        const labels = <?= json_encode(array_unique(array_column($reports['by_product_section'], 'فراوده'))) ?>;
                        const sections = <?= json_encode(array_unique(array_column($reports['by_product_section'], 'بخش_مصرف'))) ?>;
                        
                        const datasets = sections.map((section, index) => {
                            const data = labels.map(label => {
                                const item = <?= json_encode($reports['by_product_section']) ?>.find(i => 
                                    i.فراوده === label && i.بخش_مصرف === section
                                );
                                return item ? item.total_consumption : 0;
                            });
                            
                            const colors = [
                                'rgba(255, 99, 132, 0.5)',
                                'rgba(54, 162, 235, 0.5)',
                                'rgba(255, 206, 86, 0.5)',
                                'rgba(75, 192, 192, 0.5)',
                                'rgba(153, 102, 255, 0.5)'
                            ];
                            
                            return {
                                label: section,
                                data: data,
                                backgroundColor: colors[index % colors.length],
                                borderColor: colors[index % colors.length].replace('0.5', '1'),
                                borderWidth: 2
                            };
                        });
                        
                        new Chart(productSectionCtx.getContext('2d'), {
                            type: 'bar',
                            data: {
                                labels: labels,
                                datasets: datasets
                            },
                            options: {
                                responsive: true,
                                maintainAspectRatio: false,
                                plugins: {
                                    title: {
                                        display: true,
                                        text: 'مصرف بر اساس فرآورده و بخش مصرف'
                                    }
                                },
                                scales: {
                                    y: {
                                        beginAtZero: true,
                                        title: {
                                            display: true,
                                            text: 'لیتر'
                                        }
                                    },
                                    x: {
                                        title: {
                                            display: true,
                                            text: 'فرآورده'
                                        }
                                    }
                                }
                            }
                        });
                    }
                <?php endif; ?>
                
                // نمودار سالانه
                <?php if ($analysis_type == 'yearly' && !empty($reports['yearly_comparison'])): ?>
                    const yearlyCtx = document.getElementById('yearlyChart');
                    if (yearlyCtx) {
                        new Chart(yearlyCtx.getContext('2d'), {
                            type: 'bar',
                            data: {
                                labels: <?= json_encode(array_column($reports['yearly_comparison'], 'shamsi_year')) ?>,
                                datasets: [{
                                    label: 'مصرف سالانه',
                                    data: <?= json_encode(array_column($reports['yearly_comparison'], 'total_consumption')) ?>,
                                    backgroundColor: 'rgba(75, 192, 192, 0.5)',
                                    borderColor: 'rgba(75, 192, 192, 1)',
                                    borderWidth: 2
                                }]
                            },
                            options: {
                                responsive: true,
                                maintainAspectRatio: false,
                                plugins: {
                                    title: {
                                        display: true,
                                        text: 'مقایسه مصرف سالانه'
                                    }
                                },
                                scales: {
                                    y: {
                                        beginAtZero: true,
                                        title: {
                                            display: true,
                                            text: 'لیتر'
                                        }
                                    },
                                    x: {
                                        title: {
                                            display: true,
                                            text: 'سال شمسی'
                                        }
                                    }
                                }
                            }
                        });
                    }
                <?php endif; ?>
                
                // نمودار روند مصرف
                <?php if (!empty($reports['consumption_trend'])): ?>
                    const trendCtx = document.getElementById('trendChart');
                    if (trendCtx) {
                        new Chart(trendCtx.getContext('2d'), {
                            type: 'line',
                            data: {
                                labels: ['سال <?= $reports['consumption_trend']['base_year'] ?>', 'سال <?= $reports['consumption_trend']['compare_year'] ?>'],
                                datasets: [{
                                    label: 'مصرف سالانه',
                                    data: [
                                        <?= $reports['consumption_trend']['base_consumption'] ?>,
                                        <?= $reports['consumption_trend']['compare_consumption'] ?>
                                    ],
                                    borderColor: 'rgba(255, 99, 132, 1)',
                                    backgroundColor: 'rgba(255, 99, 132, 0.2)',
                                    borderWidth: 3,
                                    tension: 0.1,
                                    fill: true
                                }]
                            },
                            options: {
                                responsive: true,
                                maintainAspectRatio: false,
                                plugins: {
                                    title: {
                                        display: true,
                                        text: 'روند تغییرات مصرف'
                                    }
                                },
                                scales: {
                                    y: {
                                        beginAtZero: true,
                                        title: {
                                            display: true,
                                            text: 'لیتر'
                                        }
                                    }
                                }
                            }
                        });
                    }
                <?php endif; ?>
                
            <?php endif; ?>
        });
        
        // تابع چاپ گزارش
        function printReport() {
            window.print();
        }
        
        // تابع خروجی PDF
        function exportToPDF() {
            alert('این قابلیت در نسخه فعلی پیاده‌سازی نشده است.');
        }
    </script>
</body>
</html>