<?php
// consumption_pattern.php
require_once 'config.php';
require_once 'vendor/autoload.php';

use Morilog\Jalali\Jalalian;

// دریافت پارامترهای فیلتر
$change_threshold = $_GET['change_threshold'] ?? 10;
$city_filter = $_GET['city_filter'] ?? '';
$product_filter = $_GET['product_filter'] ?? '';
$year = $_GET['year'] ?? date('Y');
$change_direction = $_GET['change_direction'] ?? 'both'; // both, increase, decrease

// تبدیل سال شمسی به میلادی اگر لازم باشد
$year_gregorian = $year;
if (preg_match('/^\d{4}$/', $year) && $year > 1300 && $year < 1500) {
    // احتمالاً سال شمسی است
    try {
        $jalali = Jalalian::fromFormat('Y/m/d', $year . '/01/01');
        $year_gregorian = $jalali->toCarbon()->format('Y');
    } catch (Exception $e) {
        // اگر تبدیل نشد، همان سال استفاده می‌شود
    }
}

// ساخت شرط‌های WHERE برای سال جاری - استفاده از تاریخ تایید پخش
$where_conditions_current = [];
$params_current = [$year_gregorian];

$where_conditions_previous = [];
$params_previous = [$year_gregorian - 1];

if (!empty($city_filter)) {
    $where_conditions_current[] = "شهرستان = ?";
    $where_conditions_previous[] = "شهرستان = ?";
    $params_current[] = $city_filter;
    $params_previous[] = $city_filter;
}

if (!empty($product_filter)) {
    $where_conditions_current[] = "فراوده = ?";
    $where_conditions_previous[] = "فراوده = ?";
    $params_current[] = $product_filter;
    $params_previous[] = $product_filter;
}

$where_sql_current = !empty($where_conditions_current) ? "AND " . implode(" AND ", $where_conditions_current) : "";
$where_sql_previous = !empty($where_conditions_previous) ? "AND " . implode(" AND ", $where_conditions_previous) : "";

// تحلیل تغییرات الگوی مصرف (نسخه سازگار بدون CTE)
$consumption_pattern = $pdo->prepare("
        SELECT 
            c.شناسه_یکتای_کاربرد,
            c.شهرستان,
            c.product,
            c.month,
            c.current_year_consumption,
            p.previous_year_consumption,
        CASE 
            WHEN p.previous_year_consumption > 0 
            THEN ROUND(((c.current_year_consumption - p.previous_year_consumption) / p.previous_year_consumption) * 100, 2)
            ELSE NULL
        END as change_percent,
        CASE 
            WHEN p.previous_year_consumption > 0 
            THEN c.current_year_consumption - p.previous_year_consumption
            ELSE NULL
        END as change_amount
    FROM (
        SELECT 
            شناسه_یکتای_کاربرد,
            شهرستان,
            فراوده as product,
            MONTH(تاریخ_تایید_درخواست_توسط_پخش) as month,
            SUM(مقدار_تایید_شده_توسط_پخش) as current_year_consumption
        FROM fuel_requests 
        WHERE YEAR(تاریخ_تایید_درخواست_توسط_پخش) = ? 
        $where_sql_current
        GROUP BY شناسه_یکتای_کاربرد, شهرستان, فراوده, MONTH(تاریخ_تایید_درخواست_توسط_پخش)
    ) as c
    LEFT JOIN (
        SELECT 
            شناسه_یکتای_کاربرد,
            شهرستان,
            فراوده as product,
            MONTH(تاریخ_تایید_درخواست_توسط_پخش) as month,
            SUM(مقدار_تایید_شده_توسط_پخش) as previous_year_consumption
        FROM fuel_requests 
        WHERE YEAR(تاریخ_تایید_درخواست_توسط_پخش) = ? 
        $where_sql_previous
        GROUP BY شناسه_یکتای_کاربرد, شهرستان, فراوده, MONTH(تاریخ_تایید_درخواست_توسط_پخش)
    ) as p ON c.شناسه_یکتای_کاربرد = p.شناسه_یکتای_کاربرد 
        AND c.شهرستان = p.شهرستان 
        AND c.product = p.product 
        AND c.month = p.month
    HAVING change_percent IS NOT NULL 
        AND ABS(change_percent) >= ?
        " . ($change_direction == 'increase' ? " AND change_percent > 0" : "") . "
        " . ($change_direction == 'decrease' ? " AND change_percent < 0" : "") . "
    ORDER BY ABS(change_percent) DESC
    LIMIT 1000
");

$params = array_merge($params_current, $params_previous, [$change_threshold]);
$consumption_pattern->execute($params);
$pattern_data = $consumption_pattern->fetchAll();

// آمار کلی (نسخه سازگار)
$stats_sql = "
    SELECT 
        COUNT(DISTINCT شناسه_یکتای_کاربرد) as total_consumers,
        AVG(ABS(change_percent)) as avg_change,
        MAX(ABS(change_percent)) as max_change
    FROM (
        SELECT 
            c.شناسه_یکتای_کاربرد,
            CASE 
                WHEN p.previous_year_consumption > 0 
                THEN ROUND(((c.current_year_consumption - p.previous_year_consumption) / p.previous_year_consumption) * 100, 2)
                ELSE NULL
            END as change_percent
        FROM (
            SELECT 
                شناسه_یکتای_کاربرد,
                شهرستان,
                فراوده as product,
                MONTH(تاریخ_تایید_درخواست_توسط_پخش) as month,
                SUM(مقدار_تایید_شده_توسط_پخش) as current_year_consumption
            FROM fuel_requests 
            WHERE YEAR(تاریخ_تایید_درخواست_توسط_پخش) = ? 
            $where_sql_current
            GROUP BY شناسه_یکتای_کاربرد, شهرستان, فراوده, MONTH(تاریخ_تایید_درخواست_توسط_پخش)
        ) as c
        LEFT JOIN (
            SELECT 
                شناسه_یکتای_کاربرد,
                شهرستان,
                فراوده as product,
                MONTH(تاریخ_تایید_درخواست_توسط_پخش) as month,
                SUM(مقدار_تایید_شده_توسط_پخش) as previous_year_consumption
            FROM fuel_requests 
            WHERE YEAR(تاریخ_تایید_درخواست_توسط_پخش) = ? 
            $where_sql_previous
            GROUP BY شناسه_یکتای_کاربرد, شهرستان, فراوده, MONTH(تاریخ_تایید_درخواست_توسط_پخش)
        ) as p ON c.شناسه_یکتای_کاربرد = p.شناسه_یکتای_کاربرد 
            AND c.شهرستان = p.شهرستان 
            AND c.product = p.product 
            AND c.month = p.month
        HAVING change_percent IS NOT NULL 
            AND ABS(change_percent) >= ?
            " . ($change_direction == 'increase' ? " AND change_percent > 0" : "") . "
            " . ($change_direction == 'decrease' ? " AND change_percent < 0" : "") . "
    ) as subquery
";

$stats = $pdo->prepare($stats_sql);
$stats->execute($params);
$statistics = $stats->fetch();

// اگر آمار خالی بود، مقادیر پیش‌فرض تنظیم کنیم
if (!$statistics) {
    $statistics = [
        'total_consumers' => 0,
        'avg_change' => 0,
        'max_change' => 0
    ];
}

// دریافت لیست‌ها برای فیلتر
$cities = $pdo->query("SELECT DISTINCT شهرستان FROM fuel_requests WHERE شهرستان IS NOT NULL ORDER BY شهرستان")->fetchAll(PDO::FETCH_COLUMN);
$products = $pdo->query("SELECT DISTINCT فراوده FROM fuel_requests WHERE فراوده IS NOT NULL ORDER BY فراوده")->fetchAll(PDO::FETCH_COLUMN);

// دریافت سال‌های موجود - از تاریخ تایید پخش
$years_result = $pdo->query("SELECT DISTINCT YEAR(تاریخ_تایید_درخواست_توسط_پخش) as year FROM fuel_requests WHERE YEAR(تاریخ_تایید_درخواست_توسط_پخش) IS NOT NULL ORDER BY year DESC");
$years_gregorian = $years_result->fetchAll(PDO::FETCH_COLUMN, 0);

// تبدیل سال‌های میلادی به شمسی
$years = [];
foreach ($years_gregorian as $yr) {
    try {
        $carbon = \Carbon\Carbon::createFromDate($yr, 1, 1);
        $jalali = Jalalian::fromCarbon($carbon);
        $years[] = $jalali->getYear();
    } catch (Exception $e) {
        $years[] = $yr;
    }
}
$years = array_unique($years);
rsort($years);

// اگر سالی وجود ندارد، سال جاری شمسی و گذشته را اضافه کن
if (empty($years)) {
    $current_jalali = Jalalian::now();
    $years = [$current_jalali->getYear(), $current_jalali->getYear() - 1];
}
?>

<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>تحلیل تغییرات الگوی مصرف</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body { font-family: 'Tahoma', 'Arial', sans-serif; background-color: #f8f9fa; }
        .high-increase { background-color: #ffcccc !important; }
        .high-decrease { background-color: #ccffcc !important; }
        .stat-card { 
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); 
            color: white; 
            border-radius: 10px; 
            border: none;
        }
        .table th {
            background-color: #343a40;
            color: white;
        }
    </style>
</head>
<body>
    <div class="container-fluid mt-4">
        <div class="row">
            <div class="col-12">
                <div class="card shadow">
                    <div class="card-header bg-warning text-dark">
                        <h4 class="mb-0">تحلیل تغییرات الگوی مصرف</h4>
                    </div>
                    <div class="card-body">
                        <!-- فیلترها -->
                        <form method="GET" class="row g-3 mb-4">
                            <div class="col-md-2">
                                <label class="form-label">سال (شمسی)</label>
                                <select class="form-select" name="year">
                                    <?php 
                                    $current_jalali = Jalalian::now();
                                    $default_year = ($year > 1300 && $year < 1500) ? $year : $current_jalali->getYear();
                                    foreach($years as $yr): 
                                    ?>
                                        <option value="<?= $yr ?>" <?= $yr == $default_year ? 'selected' : '' ?>><?= $yr ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">حداقل درصد تغییر</label>
                                <input type="number" class="form-control" name="change_threshold" value="<?= $change_threshold ?>" step="5" min="0">
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">جهت تغییر</label>
                                <select class="form-select" name="change_direction">
                                    <option value="both" <?= $change_direction == 'both' ? 'selected' : '' ?>>هر دو</option>
                                    <option value="increase" <?= $change_direction == 'increase' ? 'selected' : '' ?>>افزایش</option>
                                    <option value="decrease" <?= $change_direction == 'decrease' ? 'selected' : '' ?>>کاهش</option>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">شهرستان</label>
                                <select class="form-select" name="city_filter">
                                    <option value="">همه شهرستان‌ها</option>
                                    <?php foreach($cities as $city): ?>
                                        <option value="<?= $city ?>" <?= $city == $city_filter ? 'selected' : '' ?>><?= $city ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">فراورده</label>
                                <select class="form-select" name="product_filter">
                                    <option value="">همه فراورده‌ها</option>
                                    <?php foreach($products as $product): ?>
                                        <option value="<?= $product ?>" <?= $product == $product_filter ? 'selected' : '' ?>><?= $product ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-12">
                                <button type="submit" class="btn btn-warning">اعمال فیلترها</button>
                                <a href="consumption_pattern.php" class="btn btn-secondary">پاک کردن فیلترها</a>
                            </div>
                        </form>

                        <!-- آمار کلی -->
                        <div class="row mb-4">
                            <div class="col-md-4">
                                <div class="stat-card p-3 text-center">
                                    <h3><?= $statistics['total_consumers'] ?? 0 ?></h3>
                                    <p class="mb-0">تعداد مصرف‌کنندگان با تغییر الگو</p>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="stat-card p-3 text-center">
                                    <h3><?= number_format($statistics['avg_change'] ?? 0, 1) ?>%</h3>
                                    <p class="mb-0">میانگین درصد تغییر</p>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="stat-card p-3 text-center">
                                    <h3><?= number_format($statistics['max_change'] ?? 0, 1) ?>%</h3>
                                    <p class="mb-0">بیشترین درصد تغییر</p>
                                </div>
                            </div>
                        </div>

                        <!-- اطلاعات فیلتر اعمال شده -->
                        <div class="alert alert-info">
                            <strong>فیلترهای اعمال شده:</strong><br>
                            سال: <?= $year ?> | حداقل تغییر: <?= $change_threshold ?>% | جهت تغییر: 
                            <?= $change_direction == 'both' ? 'هر دو' : ($change_direction == 'increase' ? 'افزایش' : 'کاهش') ?>
                            <?= $city_filter ? " | شهرستان: $city_filter" : "" ?>
                            <?= $product_filter ? " | فراورده: $product_filter" : "" ?>
                        </div>

                        <!-- جدول داده‌ها -->
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0">مصرف‌کنندگان با تغییر الگوی مصرف 
                                    <span class="badge bg-primary"><?= count($pattern_data) ?> مورد</span>
                                </h5>
                            </div>
                            <div class="card-body">
                                <?php if (empty($pattern_data)): ?>
                                    <div class="alert alert-warning text-center">
                                        <h5>هیچ داده‌ای با فیلترهای انتخابی یافت نشد</h5>
                                        <p class="mb-0">لطفاً فیلترها را تغییر دهید یا حداقل درصد تغییر را کاهش دهید.</p>
                                    </div>
                                <?php else: ?>
                                    <div class="table-responsive">
                                        <table class="table table-bordered table-hover">
                                            <thead class="table-dark">
                                                <tr>
                                                    <th>شناسه یکتای کاربرد</th>
                                                    <th>شهرستان</th>
                                                    <th>فراورده</th>
                                                    <th>ماه (شمسی)</th>
                                                    <th>مصرف سال جاری</th>
                                                    <th>مصرف سال گذشته</th>
                                                    <th>تغییر مقدار</th>
                                                    <th>درصد تغییر</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach($pattern_data as $row): ?>
                                                <tr class="<?= $row['change_percent'] > 0 ? 'high-increase' : 'high-decrease' ?>">
                                                    <td><code><?= $row['شناسه_یکتای_کاربرد'] ?></code></td>
                                                    <td><?= $row['شهرستان'] ?></td>
                                                    <td><?= $row['product'] ?></td>
                                                    <td>
                                                        <?php 
                                                        // تبدیل ماه میلادی به شمسی
                                                        try {
                                                            $carbon = \Carbon\Carbon::createFromDate($year_gregorian, $row['month'], 1);
                                                            $jalali = Jalalian::fromCarbon($carbon);
                                                            $month_names = ['فروردین', 'اردیبهشت', 'خرداد', 'تیر', 'مرداد', 'شهریور', 'مهر', 'آبان', 'آذر', 'دی', 'بهمن', 'اسفند'];
                                                            $shamsi_month = $jalali->getMonth();
                                                            echo '<span class="badge bg-secondary">' . $month_names[$shamsi_month - 1] . ' ' . $jalali->getYear() . '</span>';
                                                        } catch (Exception $e) {
                                                            echo '<span class="badge bg-secondary">ماه ' . $row['month'] . '</span>';
                                                        }
                                                        ?>
                                                    </td>
                                                    <td><?= number_format($row['current_year_consumption'], 2) ?></td>
                                                    <td><?= number_format($row['previous_year_consumption'] ?? 0, 2) ?></td>
                                                    <td>
                                                        <?php if ($row['change_amount'] !== null): ?>
                                                            <span class="<?= $row['change_amount'] > 0 ? 'text-danger' : 'text-success' ?>">
                                                                <?= number_format($row['change_amount'], 2) ?>
                                                            </span>
                                                        <?php else: ?>
                                                            <span class="text-muted">-</span>
                                                        <?php endif; ?>
                                                    </td>
                                                    <td class="<?= $row['change_percent'] > 0 ? 'text-danger fw-bold' : 'text-success fw-bold' ?>">
                                                        <?= $row['change_percent'] ?>%
                                                    </td>
                                                </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                    
                                    <!-- اطلاعات پایین جدول -->
                                    <div class="mt-3">
                                        <small class="text-muted">
                                            نمایش <?= count($pattern_data) ?> مورد از مواردی که تغییر الگوی مصرف بیش از <?= $change_threshold ?>% داشته‌اند.
                                        </small>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>