<?php
// fraud_detection.php
require_once 'config.php';
require_once 'vendor/autoload.php';

use Morilog\Jalali\Jalalian;
use Carbon\Carbon;

class FraudDetection {
    private $pdo;
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
        // غیرفعال کردن ONLY_FULL_GROUP_BY برای این session
        $this->pdo->exec("SET SESSION sql_mode = (SELECT REPLACE(@@sql_mode, 'ONLY_FULL_GROUP_BY', ''))");
    }
    
    // تابع تبدیل تاریخ شمسی - public شده
    public function getGregorianRange($shamsiYear, $period = 'yearly', $period_number = null) {
        try {
            $start_shamsi = null;
            $end_shamsi = null;

            switch($period) {
                case 'monthly':
                    if ($period_number && $period_number >= 1 && $period_number <= 12) {
                        $start_shamsi = new Jalalian($shamsiYear, $period_number, 1);
                        $end_shamsi = (new Jalalian($shamsiYear, $period_number, 1))->addMonths(1)->addDays(-1);
                    }
                    break;

                case 'quarterly':
                    if ($period_number && $period_number >= 1 && $period_number <= 4) {
                        $start_month = (($period_number - 1) * 3) + 1;
                        $end_month = $start_month + 2;
                        $start_shamsi = new Jalalian($shamsiYear, $start_month, 1);
                        $end_shamsi = (new Jalalian($shamsiYear, $end_month, 1))->addMonths(1)->addDays(-1);
                    }
                    break;

                case 'yearly':
                default:
                    $start_shamsi = new Jalalian($shamsiYear, 1, 1);
                    $end_shamsi = (new Jalalian($shamsiYear, 12, 1))->addMonths(1)->addDays(-1);
                    break;
            }

            if ($start_shamsi && $end_shamsi) {
                return [
                    'start' => $start_shamsi->toCarbon()->format('Y-m-d H:i:s'),
                    'end' => $end_shamsi->toCarbon()->setTime(23, 59, 59)->format('Y-m-d H:i:s')
                ];
            }
        } catch (Exception $e) {
            return null;
        }
        return null;
    }
    
    // تابع دریافت سال‌های موجود
    public function getAvailableShamsiYears() {
        $query = $this->pdo->query("
            SELECT DISTINCT تاریخ_تایید_درخواست_توسط_پخش 
            FROM fuel_requests 
            WHERE تاریخ_تایید_درخواست_توسط_پخش IS NOT NULL 
            ORDER BY تاریخ_تایید_درخواست_توسط_پخش DESC
        ");
        $dates = $query->fetchAll(PDO::FETCH_COLUMN);

        $years = [];
        foreach ($dates as $date) {
            try {
                $carbon = Carbon::parse($date);
                $jalali = Jalalian::fromCarbon($carbon);
                $shamsi_year = $jalali->getYear();
                if (!in_array($shamsi_year, $years)) {
                    $years[] = $shamsi_year;
                }
            } catch (Exception $e) {
                continue;
            }
        }

        rsort($years);
        return $years;
    }
    
    // 1. شناسایی کاربران با چند شناسه یکتا
    public function getUsersWithMultipleIdentifiers($filters = []) {
        $where_data = $this->buildWhereClause($filters);
        
         $query = "
        SELECT 
            کدملی_شناسه_ملی,
            شماره_درخواست,
            MIN(نام_متقاضی) as نام_متقاضی,  -- تغییر از ANY_VALUE به MIN
            COUNT(DISTINCT شناسه_یکتای_کاربرد) as identifier_count,
            GROUP_CONCAT(DISTINCT شناسه_یکتای_کاربرد ORDER BY شناسه_یکتای_کاربرد) as identifiers,
            SUM(مقدار_تایید_شده_توسط_پخش) as total_consumption,
            COUNT(*) as request_count
        FROM fuel_requests 
        {$where_data['where']}
        AND کدملی_شناسه_ملی IS NOT NULL 
        AND کدملی_شناسه_ملی != ''
        GROUP BY کدملی_شناسه_ملی
        HAVING COUNT(DISTINCT شناسه_یکتای_کاربرد) > 1
        ORDER BY identifier_count DESC, total_consumption DESC
    ";
        
        $stmt = $this->pdo->prepare($query);
        $stmt->execute($where_data['params']);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // 2. شناسایی تغییرات مصرف کاربران
    public function getConsumptionChanges($base_filters, $compare_filters, $comparison_type = 'corresponding') {
        $base_data = $this->getPeriodConsumption($base_filters, 'base');
        $compare_data = $this->getPeriodConsumption($compare_filters, 'compare');
        
        $changes = [];
        
        foreach ($base_data as $base) {
            $identifier = $base['کدملی_شناسه_ملی'];
            $base_consumption = (float)$base['total_consumption'];
            
            // پیدا کردن داده مقایسه‌ای
            $compare_consumption = 0;
            foreach ($compare_data as $compare) {
                if ($compare['کدملی_شناسه_ملی'] == $identifier) {
                    $compare_consumption = (float)$compare['total_consumption'];
                    break;
                }
            }
            
            if ($base_consumption > 0) {
                $change_percentage = (($compare_consumption - $base_consumption) / $base_consumption) * 100;
                $change_amount = $compare_consumption - $base_consumption;
                
                $changes[] = [
                    'کدملی_شناسه_ملی' => $identifier,
                    'نام_متقاضی' => $base['نام_متقاضی'],
                    'شهرستان' => $base['شهرستان'],
                    'فراوده' => $base['فراوده'],
                    'بخش_مصرف' => $base['بخش_مصرف'],
                    'base_consumption' => $base_consumption,
                    'compare_consumption' => $compare_consumption,
                    'change_amount' => $change_amount,
                    'change_percentage' => $change_percentage,
                    'trend' => $change_amount > 0 ? 'افزایش' : ($change_amount < 0 ? 'کاهش' : 'ثابت')
                ];
            }
        }
        
        return $changes;
    }
    
 private function getPeriodConsumption($filters, $period_name) {
    $where_conditions = [];
    $params = [];
    
    // فیلتر تاریخ
    if (!empty($filters['start_date']) && !empty($filters['end_date'])) {
        $where_conditions[] = "تاریخ_تایید_درخواست_توسط_پخش BETWEEN ? AND ?";
        $params[] = $filters['start_date'];
        $params[] = $filters['end_date'];
    }
    
    // فیلتر شهرستان
    if (!empty($filters['city']) && $filters['city'] !== 'all') {
        $where_conditions[] = "شهرستان = ?";
        $params[] = $filters['city'];
    }
    
    // فیلتر فرآورده
    if (!empty($filters['product']) && $filters['product'] !== 'all') {
        $where_conditions[] = "فراوده = ?";
        $params[] = $filters['product'];
    }
    
    // فیلتر بخش مصرف
    if (!empty($filters['usage_section']) && $filters['usage_section'] !== 'all') {
        $where_conditions[] = "بخش_مصرف = ?";
        $params[] = $filters['usage_section'];
    }
    
    $where_clause = "";
    if (!empty($where_conditions)) {
        $where_clause = "WHERE " . implode(" AND ", $where_conditions);
    }
    
    $query = "
        SELECT 
            کدملی_شناسه_ملی,
            MIN(نام_متقاضی) as نام_متقاضی,  -- تغییر به MIN
            MIN(شهرستان) as شهرستان,        -- تغییر به MIN
            MIN(فراوده) as فراوده,          -- تغییر به MIN
            MIN(بخش_مصرف) as بخش_مصرف,      -- تغییر به MIN
            SUM(مقدار_تایید_شده_توسط_پخش) as total_consumption,
            COUNT(*) as request_count
        FROM fuel_requests 
        {$where_clause}
        AND کدملی_شناسه_ملی IS NOT NULL 
        AND کدملی_شناسه_ملی != ''
        GROUP BY کدملی_شناسه_ملی
        HAVING total_consumption > 0
        ORDER BY total_consumption DESC
    ";
    
    $stmt = $this->pdo->prepare($query);
    $stmt->execute($params);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}
    
    // 3. شناسایی مصرف‌کنندگان با مصرف غیرعادی - نسخه اصلاح شده
   // 3. شناسایی مصرف‌کنندگان با مصرف غیرعادی
public function getAbnormalConsumers($filters, $threshold_percentage = 50) {
    $where_data = $this->buildWhereClause($filters);
    
    // ابتدا میانگین مصرف را محاسبه می‌کنیم
    $avg_query = "
        SELECT 
            فراوده,
            بخش_مصرف,
            AVG(مقدار_تایید_شده_توسط_پخش) as avg_consumption,
            STDDEV(مقدار_تایید_شده_توسط_پخش) as std_consumption
        FROM fuel_requests 
        {$where_data['where']}
        AND کدملی_شناسه_ملی IS NOT NULL
        AND مقدار_تایید_شده_توسط_پخش > 0
        GROUP BY فراوده, بخش_مصرف
    ";
    
    $stmt = $this->pdo->prepare($avg_query);
    $stmt->execute($where_data['params']);
    $averages = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if (empty($averages)) {
        return [];
    }
    
    // حالا مصرف‌کنندگان غیرعادی را پیدا می‌کنیم
    $abnormal_query = "
        SELECT 
            fr.کدملی_شناسه_ملی,
            fr.شماره_درخواست,
            MIN(fr.نام_متقاضی) as نام_متقاضی,
            MIN(fr.شهرستان) as شهرستان,
            fr.فراوده,
            fr.بخش_مصرف,
            MIN(fr.شناسه_یکتای_کاربرد) as شناسه_یکتای_کاربرد,
            SUM(fr.مقدار_تایید_شده_توسط_پخش) as total_consumption,
            COUNT(*) as request_count,
            AVG(stats.avg_consumption) as average_consumption,
            (SUM(fr.مقدار_تایید_شده_توسط_پخش) - AVG(stats.avg_consumption)) as difference_from_avg,
            ((SUM(fr.مقدار_تایید_شده_توسط_پخش) - AVG(stats.avg_consumption)) / AVG(stats.avg_consumption) * 100) as percentage_above_avg
        FROM fuel_requests fr
        INNER JOIN (
            SELECT 
                فراوده,
                بخش_مصرف,
                AVG(مقدار_تایید_شده_توسط_پخش) as avg_consumption,
                STDDEV(مقدار_تایید_شده_توسط_پخش) as std_consumption
            FROM fuel_requests 
            WHERE 1=1 
            AND کدملی_شناسه_ملی IS NOT NULL
            AND مقدار_تایید_شده_توسط_پخش > 0
            " . (!empty($where_data['where']) ? " AND " . substr($where_data['where'], 6) : "") . "
            GROUP BY فراوده, بخش_مصرف
        ) stats ON fr.فراوده = stats.فراوده AND fr.بخش_مصرف = stats.بخش_مصرف
        WHERE 1=1 
        AND fr.کدملی_شناسه_ملی IS NOT NULL 
        AND fr.کدملی_شناسه_ملی != ''
        AND fr.مقدار_تایید_شده_توسط_پخش > 0
        " . (!empty($where_data['where']) ? " AND " . $this->addTablePrefix($where_data['where'], 'fr') : "") . "
        GROUP BY fr.کدملی_شناسه_ملی, fr.فراوده, fr.بخش_مصرف
        HAVING total_consumption > (average_consumption + (average_consumption * ? / 100))
        ORDER BY percentage_above_avg DESC
    ";
    
    // پارامترها: ابتدا پارامترهای where_data، سپس threshold
    $params = array_merge($where_data['params'], $where_data['params'], [$threshold_percentage]);
    $stmt = $this->pdo->prepare($abnormal_query);
    $stmt->execute($params);
    
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// تابع کمکی برای اضافه کردن پیشوند جدول به شرایط WHERE
private function addTablePrefix($where_clause, $table_alias) {
    // حذف WHERE از ابتدا
    $conditions = substr($where_clause, 6);
    
    // اضافه کردن پیشوند fr. به ستون‌ها
    $columns = [
        'تاریخ_تایید_درخواست_توسط_پخش',
        'شهرستان',
        'فراوده',
        'بخش_مصرف',
        'کدملی_شناسه_ملی',
        'مقدار_تایید_شده_توسط_پخش'
    ];
    
    foreach ($columns as $column) {
        $conditions = str_replace($column, $table_alias . '.' . $column, $conditions);
    }
    
    return $conditions;
}
    
    private function buildWhereClause($filters) {
        $where_conditions = [];
        $params = [];
        
        // فیلتر تاریخ
        if (!empty($filters['start_date']) && !empty($filters['end_date'])) {
            $where_conditions[] = "تاریخ_تایید_درخواست_توسط_پخش BETWEEN ? AND ?";
            $params[] = $filters['start_date'];
            $params[] = $filters['end_date'];
        }
        
        // فیلتر شهرستان
        if (!empty($filters['city']) && $filters['city'] !== 'all') {
            $where_conditions[] = "شهرستان = ?";
            $params[] = $filters['city'];
        }
        
        // فیلتر فرآورده
        if (!empty($filters['product']) && $filters['product'] !== 'all') {
            $where_conditions[] = "فراوده = ?";
            $params[] = $filters['product'];
        }
        
        // فیلتر بخش مصرف
        if (!empty($filters['usage_section']) && $filters['usage_section'] !== 'all') {
            $where_conditions[] = "بخش_مصرف = ?";
            $params[] = $filters['usage_section'];
        }
        
        $where_clause = "";
        if (!empty($where_conditions)) {
            $where_clause = "WHERE " . implode(" AND ", $where_conditions);
        }
        
        return ['where' => $where_clause, 'params' => $params];
    }
    
    // 4. شناسایی متخلفین بر اساس افزایش ناگهانی مصرف
    public function getSuddenIncreaseViolators($base_filters, $compare_filters, $increase_threshold = 100) {
        $changes = $this->getConsumptionChanges($base_filters, $compare_filters);
        
        $violators = [];
        foreach ($changes as $change) {
            if ($change['change_percentage'] >= $increase_threshold && $change['change_amount'] > 0) {
                $violators[] = [
                    'کدملی_شناسه_ملی' => $change['کدملی_شناسه_ملی'],
                    'نام_متقاضی' => $change['نام_متقاضی'],
                    'شهرستان' => $change['شهرستان'],
                    'فراوده' => $change['فراوده'],
                    'بخش_مصرف' => $change['بخش_مصرف'],
                    'base_period_consumption' => $change['base_consumption'],
                    'compare_period_consumption' => $change['compare_consumption'],
                    'increase_amount' => $change['change_amount'],
                    'increase_percentage' => $change['change_percentage'],
                    'violation_type' => 'افزایش ناگهانی مصرف'
                ];
            }
        }
        
        usort($violators, function($a, $b) {
            return $b['increase_percentage'] <=> $a['increase_percentage'];
        });
        
        return $violators;
    }
    
    // 5. شناسایی کاربران بر اساس مصرف محصول و بخش
    public function getUsersByProductAndSection($filters) {
        $where_data = $this->buildWhereClause($filters);
        
        $query = "
        SELECT 
            کدملی_شناسه_ملی,
            شماره_درخواست,
            MIN(نام_متقاضی) as نام_متقاضی,  -- تغییر به MIN
            MIN(شهرستان) as شهرستان,        -- تغییر به MIN
            فراوده,
            بخش_مصرف,
            SUM(مقدار_تایید_شده_توسط_پخش) as total_consumption,
            COUNT(*) as request_count,
            COUNT(DISTINCT شناسه_یکتای_کاربرد) as identifier_count
        FROM fuel_requests 
        {$where_data['where']}
        AND کدملی_شناسه_ملی IS NOT NULL 
        AND کدملی_شناسه_ملی != ''
        GROUP BY کدملی_شناسه_ملی, فراوده, بخش_مصرف
        HAVING total_consumption > 0
        ORDER BY total_consumption DESC
    ";
        
        $stmt = $this->pdo->prepare($query);
        $stmt->execute($where_data['params']);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // 6. شناسایی بالاترین مصرف‌کنندگان
    public function getTopConsumers($filters, $limit = 50) {
        $where_data = $this->buildWhereClause($filters);
        
        $query = "
        SELECT 
            کدملی_شناسه_ملی,
            شماره_درخواست,
            MIN(نام_متقاضی) as نام_متقاضی,  -- تغییر به MIN
            MIN(شهرستان) as شهرستان,        -- تغییر به MIN
            MIN(فراوده) as فراوده,          -- تغییر به MIN
            MIN(بخش_مصرف) as بخش_مصرف,      -- تغییر به MIN
            SUM(مقدار_تایید_شده_توسط_پخش) as total_consumption,
            COUNT(*) as request_count,
            COUNT(DISTINCT شناسه_یکتای_کاربرد) as identifier_count
        FROM fuel_requests 
        {$where_data['where']}
        AND کدملی_شناسه_ملی IS NOT NULL 
        AND کدملی_شناسه_ملی != ''
        GROUP BY کدملی_شناسه_ملی
        ORDER BY total_consumption DESC
        LIMIT ?
    ";
        
        $params = array_merge($where_data['params'], [$limit]);
        $stmt = $this->pdo->prepare($query);
        //$stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}

// دریافت داده‌های موجود
$detection = new FraudDetection($pdo);

// دریافت سال‌های موجود
$available_years = $detection->getAvailableShamsiYears();
$cities = $pdo->query("SELECT DISTINCT شهرستان FROM fuel_requests WHERE شهرستان IS NOT NULL ORDER BY شهرستان")->fetchAll(PDO::FETCH_COLUMN);
$products = $pdo->query("SELECT DISTINCT فراوده FROM fuel_requests WHERE فراوده IS NOT NULL ORDER BY فراوده")->fetchAll(PDO::FETCH_COLUMN);
$usage_sections = $pdo->query("SELECT DISTINCT بخش_مصرف FROM fuel_requests WHERE بخش_مصرف IS NOT NULL ORDER BY بخش_مصرف")->fetchAll(PDO::FETCH_COLUMN);

// تنظیم فیلترهای پیش‌فرض
$current_year = $available_years[0] ?? 1400;
$prev_year = $current_year - 1;

// بررسی وجود سال قبلی در لیست
if (!in_array($prev_year, $available_years)) {
    $prev_year = $current_year;
}

$filters = [
    'base_year' => $_POST['base_year'] ?? $current_year,
    'base_period' => $_POST['base_period'] ?? 'yearly',
    'base_period_number' => $_POST['base_period_number'] ?? null,
    'compare_year' => $_POST['compare_year'] ?? $prev_year,
    'compare_period' => $_POST['compare_period'] ?? 'yearly',
    'compare_period_number' => $_POST['compare_period_number'] ?? null,
    'comparison_type' => $_POST['comparison_type'] ?? 'corresponding',
    'city' => $_POST['city'] ?? 'all',
    'product' => $_POST['product'] ?? 'all',
    'usage_section' => $_POST['usage_section'] ?? 'all',
    'increase_threshold' => $_POST['increase_threshold'] ?? 100,
    'abnormal_threshold' => $_POST['abnormal_threshold'] ?? 100
];

// محاسبه تاریخ‌های دوره پایه و مقایسه
$base_range = $detection->getGregorianRange(
    $filters['base_year'], 
    $filters['base_period'], 
    $filters['base_period_number']
);
$compare_range = $detection->getGregorianRange(
    $filters['compare_year'], 
    $filters['compare_period'], 
    $filters['compare_period_number']
);

// ساخت فیلترها برای هر دوره
$base_filters = [
    'start_date' => $base_range['start'] ?? null,
    'end_date' => $base_range['end'] ?? null,
    'city' => $filters['city'],
    'product' => $filters['product'],
    'usage_section' => $filters['usage_section']
];

$compare_filters = [
    'start_date' => $compare_range['start'] ?? null,
    'end_date' => $compare_range['end'] ?? null,
    'city' => $filters['city'],
    'product' => $filters['product'],
    'usage_section' => $filters['usage_section']
];

// تولید گزارشات
$reports = [];
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // 1. کاربران با چند شناسه
    $reports['multi_identifier'] = $detection->getUsersWithMultipleIdentifiers($base_filters);
    
    // 2. تغییرات مصرف
    $reports['consumption_changes'] = $detection->getConsumptionChanges(
        $base_filters, 
        $compare_filters, 
        $filters['comparison_type']
    );
    
    // 3. متخلفین افزایش ناگهانی
    $reports['sudden_increase'] = $detection->getSuddenIncreaseViolators(
        $base_filters, 
        $compare_filters, 
        $filters['increase_threshold']
    );
    
    // 4. مصرف‌کنندگان غیرعادی
    $reports['abnormal_consumers'] = $detection->getAbnormalConsumers(
        $base_filters, 
        $filters['abnormal_threshold']
    );
    
    // 5. بالاترین مصرف‌کنندگان
    $reports['top_consumers'] = $detection->getTopConsumers($base_filters, 20);
}
?>

<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>سیستم شناسایی متخلفین مصرف سوخت</title>
    <style>
        /* استایل‌ها همانند قبل */
        body {
            font-family: 'Tahoma', sans-serif;
            background-color: #f5f5f5;
            margin: 0;
            padding: 20px;
        }
        .container {
            max-width: 1400px;
            margin: 0 auto;
            background: white;
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
        }
        .filters {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        .filter-group {
            margin-bottom: 15px;
        }
        .filter-row {
            display: flex;
            gap: 15px;
            flex-wrap: wrap;
        }
        .filter-column {
            flex: 1;
            min-width: 200px;
        }
        label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
        }
        select, input {
            width: 100%;
            padding: 8px;
            border: 1px solid #ddd;
            border-radius: 4px;
        }
        .btn {
            background: #007bff;
            color: white;
            padding: 10px 20px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 16px;
            margin-top: 10px;
        }
        .btn:hover {
            background: #0056b3;
        }
        .btn-danger {
            background: #dc3545;
        }
        .btn-danger:hover {
            background: #c82333;
        }
        .report-section {
            margin: 30px 0;
            padding: 20px;
            border: 1px solid #ddd;
            border-radius: 8px;
        }
        .report-section h3 {
            color: #333;
            border-bottom: 2px solid #007bff;
            padding-bottom: 10px;
        }
        .data-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 15px;
            font-size: 14px;
        }
        .data-table th {
            background-color: #007bff;
            color: white;
            text-align: center;
            padding: 12px;
            position: sticky;
            top: 0;
        }
        .data-table td {
            border: 1px solid #ddd;
            padding: 8px;
            text-align: center;
        }
        .data-table tr:nth-child(even) {
            background-color: #f8f9fa;
        }
        .data-table tr:hover {
            background-color: #e9ecef;
        }
        .badge {
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 12px;
            font-weight: bold;
        }
        .badge-danger {
            background-color: #dc3545;
            color: white;
        }
        .badge-warning {
            background-color: #ffc107;
            color: #212529;
        }
        .badge-success {
            background-color: #28a745;
            color: white;
        }
        .badge-info {
            background-color: #17a2b8;
            color: white;
        }
        .summary-card {
            background: #f8f9fa;
            border: 1px solid #ddd;
            border-radius: 8px;
            padding: 15px;
            margin: 10px 0;
        }
        .summary-card h4 {
            margin-top: 0;
            color: #007bff;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>سیستم شناسایی متخلفین و تحلیل تغییرات مصرف سوخت</h1>
        
        <div class="filters">
            <form method="POST">
                <div class="filter-row">
                    <div class="filter-column">
                        <h3>دوره پایه</h3>
                        <div class="filter-group">
                            <label>سال دوره پایه:</label>
                            <select name="base_year">
                                <?php foreach ($available_years as $year): ?>
                                    <option value="<?= $year ?>" <?= $filters['base_year'] == $year ? 'selected' : '' ?>>
                                        <?= $year ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="filter-group">
                            <label>بازه زمانی دوره پایه:</label>
                            <select name="base_period" id="base_period">
                                <option value="yearly" <?= $filters['base_period'] == 'yearly' ? 'selected' : '' ?>>سالانه</option>
                                <option value="quarterly" <?= $filters['base_period'] == 'quarterly' ? 'selected' : '' ?>>فصلی</option>
                                <option value="monthly" <?= $filters['base_period'] == 'monthly' ? 'selected' : '' ?>>ماهانه</option>
                            </select>
                        </div>
                        
                        <div class="filter-group" id="base_period_number_group" style="display: <?= $filters['base_period'] != 'yearly' ? 'block' : 'none' ?>">
                            <label>شماره دوره پایه:</label>
                            <select name="base_period_number" id="base_period_number">
                                <!-- به صورت داینامیک پر می‌شود -->
                            </select>
                        </div>
                    </div>
                    
                    <div class="filter-column">
                        <h3>دوره مقایسه</h3>
                        <div class="filter-group">
                            <label>سال دوره مقایسه:</label>
                            <select name="compare_year">
                                <?php foreach ($available_years as $year): ?>
                                    <option value="<?= $year ?>" <?= $filters['compare_year'] == $year ? 'selected' : '' ?>>
                                        <?= $year ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="filter-group">
                            <label>بازه زمانی دوره مقایسه:</label>
                            <select name="compare_period" id="compare_period">
                                <option value="yearly" <?= $filters['compare_period'] == 'yearly' ? 'selected' : '' ?>>سالانه</option>
                                <option value="quarterly" <?= $filters['compare_period'] == 'quarterly' ? 'selected' : '' ?>>فصلی</option>
                                <option value="monthly" <?= $filters['compare_period'] == 'monthly' ? 'selected' : '' ?>>ماهانه</option>
                            </select>
                        </div>
                        
                        <div class="filter-group" id="compare_period_number_group" style="display: <?= $filters['compare_period'] != 'yearly' ? 'block' : 'none' ?>">
                            <label>شماره دوره مقایسه:</label>
                            <select name="compare_period_number" id="compare_period_number">
                                <!-- به صورت داینامیک پر می‌شود -->
                            </select>
                        </div>
                    </div>
                    
                    <div class="filter-column">
                        <h3>تنظیمات تحلیل</h3>
                        <div class="filter-group">
                            <label>نوع مقایسه:</label>
                            <select name="comparison_type">
                                <option value="corresponding" <?= $filters['comparison_type'] == 'corresponding' ? 'selected' : '' ?>>متناظر</option>
                                <option value="continuous" <?= $filters['comparison_type'] == 'continuous' ? 'selected' : '' ?>>پیوسته</option>
                            </select>
                        </div>
                        
                        <div class="filter-group">
                            <label>آستانه افزایش (%):</label>
                            <input type="number" name="increase_threshold" value="<?= $filters['increase_threshold'] ?>" min="0" max="1000" step="1">
                            <small>درصد افزایش برای شناسایی متخلفین</small>
                        </div>
                        
                        <div class="filter-group">
                            <label>آستانه غیرعادی (%):</label>
                            <input type="number" name="abnormal_threshold" value="<?= $filters['abnormal_threshold'] ?>" min="0" max="1000" step="1">
                            <small>درصد بالاتر از میانگین برای شناسایی مصرف غیرعادی</small>
                        </div>
                    </div>
                </div>
                
                <div class="filter-row" style="margin-top: 20px;">
                    <div class="filter-column">
                        <div class="filter-group">
                            <label>شهرستان:</label>
                            <select name="city">
                                <option value="all" <?= $filters['city'] == 'all' ? 'selected' : '' ?>>همه</option>
                                <?php foreach ($cities as $city): ?>
                                    <option value="<?= $city ?>" <?= $filters['city'] == $city ? 'selected' : '' ?>><?= $city ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    
                    <div class="filter-column">
                        <div class="filter-group">
                            <label>فرآورده:</label>
                            <select name="product">
                                <option value="all" <?= $filters['product'] == 'all' ? 'selected' : '' ?>>همه</option>
                                <?php foreach ($products as $product): ?>
                                    <option value="<?= $product ?>" <?= $filters['product'] == $product ? 'selected' : '' ?>><?= $product ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    
                    <div class="filter-column">
                        <div class="filter-group">
                            <label>بخش مصرف:</label>
                            <select name="usage_section">
                                <option value="all" <?= $filters['usage_section'] == 'all' ? 'selected' : '' ?>>همه</option>
                                <?php foreach ($usage_sections as $section): ?>
                                    <option value="<?= $section ?>" <?= $filters['usage_section'] == $section ? 'selected' : '' ?>><?= $section ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                </div>
                
                <button type="submit" class="btn">اجرای تحلیل و شناسایی متخلفین</button>
                <button type="button" class="btn btn-danger" onclick="exportToExcel()">خروجی اکسل</button>
            </form>
        </div>
        
        <?php if ($_SERVER['REQUEST_METHOD'] === 'POST' && !empty($reports)): ?>
            <!-- خلاصه گزارش -->
            <div class="summary-card">
                <h4>خلاصه نتایج تحلیل</h4>
                <p>تعداد کاربران با چند شناسه: <span class="badge badge-info"><?= count($reports['multi_identifier']) ?></span></p>
                <p>تعداد متخلفین افزایش ناگهانی: <span class="badge badge-danger"><?= count($reports['sudden_increase']) ?></span></p>
                <p>تعداد مصرف‌کنندگان غیرعادی: <span class="badge badge-warning"><?= count($reports['abnormal_consumers']) ?></span></p>
                <p>تعداد کل تغییرات مصرف: <span class="badge badge-info"><?= count($reports['consumption_changes']) ?></span></p>
            </div>
            
            <!-- 1. کاربران با چند شناسه -->
            <div class="report-section">
                <h3>۱- کاربران با چندین شناسه یکتا</h3>
                <?php if (!empty($reports['multi_identifier'])): ?>
                    <div class="summary-card">
                        <p>تعداد کاربران: <?= count($reports['multi_identifier']) ?> نفر</p>
                        <p>بیشترین تعداد شناسه: <?= max(array_column($reports['multi_identifier'], 'identifier_count')) ?></p>
                    </div>
                    <div style="overflow-x: auto;">
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>ردیف</th>
                                    <th>کدملی/شناسه ملی</th>
                                    <th>نام متقاضی</th>
                                    <th>تعداد شناسه</th>
                                    <th>شناسه‌های یکتا</th>
                                    <th>مصرف کل</th>
                                    <th>تعداد درخواست</th>
                                    <th>وضعیت</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($reports['multi_identifier'] as $index => $user): ?>
                                    <tr>
                                        <td><?= $index + 1 ?></td>
                                        <td><?= $user['کدملی_شناسه_ملی'] ?></td>
                                        <td><?= $user['نام_متقاضی'] ?? 'نامشخص' ?></td>
                                        <td><span class="badge badge-danger"><?= $user['identifier_count'] ?></span></td>
                                        <td title="<?= $user['identifiers'] ?>">
                                            <?= substr($user['identifiers'], 0, 50) ?>...
                                        </td>
                                        <td><?= number_format($user['total_consumption']) ?> لیتر</td>
                                        <td><?= $user['request_count'] ?></td>
                                        <td>
                                            <?php if ($user['identifier_count'] > 3): ?>
                                                <span class="badge badge-danger">متخلف پرخطر</span>
                                            <?php elseif ($user['identifier_count'] > 2): ?>
                                                <span class="badge badge-warning">متخلف متوسط</span>
                                            <?php else: ?>
                                                <span class="badge badge-info">متخلف کم‌خطر</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <p>کاربری با چندین شناسه یکتا یافت نشد.</p>
                <?php endif; ?>
            </div>
            
            <!-- 2. متخلفین افزایش ناگهانی -->
            <div class="report-section">
                <h3>۲- متخلفین با افزایش ناگهانی مصرف</h3>
                <?php if (!empty($reports['sudden_increase'])): ?>
                    <div class="summary-card">
                        <p>تعداد متخلفین: <?= count($reports['sudden_increase']) ?> نفر</p>
                        <p>بیشترین افزایش: <?= number_format(max(array_column($reports['sudden_increase'], 'increase_percentage')), 1) ?>%</p>
                        <p>میانگین افزایش: <?= number_format(array_sum(array_column($reports['sudden_increase'], 'increase_percentage')) / count($reports['sudden_increase']), 1) ?>%</p>
                    </div>
                    <div style="overflow-x: auto;">
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>ردیف</th>
                                    <th>کدملی</th>
                                    <th>نام</th>
                                    <th>شهرستان</th>
                                    <th>فرآورده</th>
                                    <th>بخش مصرف</th>
                                    <th>مصرف دوره پایه</th>
                                    <th>مصرف دوره مقایسه</th>
                                    <th>مقدار افزایش</th>
                                    <th>درصد افزایش</th>
                                    <th>نوع تخلف</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($reports['sudden_increase'] as $index => $violator): ?>
                                    <tr>
                                        <td><?= $index + 1 ?></td>
                                        <td><?= $violator['کدملی_شناسه_ملی'] ?></td>
                                        <td><?= $violator['نام_متقاضی'] ?></td>
                                        <td><?= $violator['شهرستان'] ?></td>
                                        <td><?= $violator['فراوده'] ?></td>
                                        <td><?= $violator['بخش_مصرف'] ?></td>
                                        <td><?= number_format($violator['base_period_consumption']) ?> لیتر</td>
                                        <td><?= number_format($violator['compare_period_consumption']) ?> لیتر</td>
                                        <td><?= number_format($violator['increase_amount']) ?> لیتر</td>
                                        <td>
                                            <span class="badge badge-danger">
                                                <?= number_format($violator['increase_percentage'], 1) ?>%
                                            </span>
                                        </td>
                                        <td><?= $violator['violation_type'] ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <p>هیچ متخلفی با افزایش ناگهانی مصرف یافت نشد.</p>
                <?php endif; ?>
            </div>
            
            <!-- 3. مصرف‌کنندگان غیرعادی -->
            <div class="report-section">
                <h3>۳- مصرف‌کنندگان با مصرف غیرعادی</h3>
                <?php if (!empty($reports['abnormal_consumers'])): ?>
                    <div class="summary-card">
                        <p>تعداد مصرف‌کنندگان غیرعادی: <?= count($reports['abnormal_consumers']) ?> نفر</p>
                    </div>
                    <div style="overflow-x: auto;">
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>ردیف</th>
                                    <th>کدملی</th>
                                    <th>نام</th>
                                    <th>شهرستان</th>
                                    <th>فرآورده</th>
                                    <th>بخش مصرف</th>
                                    <th>شناسه یکتا</th>
                                    <th>مصرف کل</th>
                                    <th>تعداد درخواست</th>
                                    <th>وضعیت</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($reports['abnormal_consumers'] as $index => $consumer): ?>
                                    <tr>
                                        <td><?= $index + 1 ?></td>
                                        <td><?= $consumer['کدملی_شناسه_ملی'] ?></td>
                                        <td><?= $consumer['نام_متقاضی'] ?></td>
                                        <td><?= $consumer['شهرستان'] ?></td>
                                        <td><?= $consumer['فراوده'] ?></td>
                                        <td><?= $consumer['بخش_مصرف'] ?></td>
                                         <td><?= $consumer['شماره_درخواست'] ?></td>
                                         <td><?= $consumer['شناسه_یکتای_کاربرد'] ?></td>
                                        <td><?= number_format($consumer['total_consumption']) ?> لیتر</td>
                                        <td><?= $consumer['request_count'] ?></td>
                                        <td><span class="badge badge-warning">مصرف غیرعادی</span></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <p>هیچ مصرف‌کننده غیرعادی یافت نشد.</p>
                <?php endif; ?>
            </div>
            
            <!-- 4. تغییرات مصرف کلی -->
            <div class="report-section">
                <h3>۴- تغییرات مصرف کلی کاربران</h3>
                <?php if (!empty($reports['consumption_changes'])): ?>
                    <div class="summary-card">
                        <?php
                        $increases = array_filter($reports['consumption_changes'], function($item) {
                            return $item['trend'] === 'افزایش';
                        });
                        $decreases = array_filter($reports['consumption_changes'], function($item) {
                            return $item['trend'] === 'کاهش';
                        });
                        $stable = array_filter($reports['consumption_changes'], function($item) {
                            return $item['trend'] === 'ثابت';
                        });
                        ?>
                        <p>تعداد افزایش مصرف: <span class="badge badge-danger"><?= count($increases) ?></span></p>
                        <p>تعداد کاهش مصرف: <span class="badge badge-success"><?= count($decreases) ?></span></p>
                        <p>تعداد بدون تغییر: <span class="badge badge-info"><?= count($stable) ?></span></p>
                    </div>
                    <div style="overflow-x: auto;">
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>ردیف</th>
                                    <th>کدملی</th>
                                    <th>نام</th>
                                    <th>شهرستان</th>
                                    <th>فرآورده</th>
                                    <th>بخش مصرف</th>
                                    <th>مصرف پایه</th>
                                    <th>مصرف مقایسه</th>
                                    <th>تغییر مقدار</th>
                                    <th>تغییر درصد</th>
                                    <th>روند</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($reports['consumption_changes'] as $index => $change): ?>
                                    <tr>
                                        <td><?= $index + 1 ?></td>
                                        <td><?= $change['کدملی_شناسه_ملی'] ?></td>
                                        <td><?= $change['نام_متقاضی'] ?></td>
                                        <td><?= $change['شهرستان'] ?></td>
                                        <td><?= $change['فراوده'] ?></td>
                                        <td><?= $change['بخش_مصرف'] ?></td>
                                        <td><?= number_format($change['base_consumption']) ?> لیتر</td>
                                        <td><?= number_format($change['compare_consumption']) ?> لیتر</td>
                                        <td>
                                            <?php if ($change['change_amount'] > 0): ?>
                                                <span style="color: #dc3545;">+<?= number_format($change['change_amount']) ?></span>
                                            <?php elseif ($change['change_amount'] < 0): ?>
                                                <span style="color: #28a745;"><?= number_format($change['change_amount']) ?></span>
                                            <?php else: ?>
                                                <?= number_format($change['change_amount']) ?>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($change['change_percentage'] > 0): ?>
                                                <span class="badge badge-danger">+<?= number_format($change['change_percentage'], 1) ?>%</span>
                                            <?php elseif ($change['change_percentage'] < 0): ?>
                                                <span class="badge badge-success"><?= number_format($change['change_percentage'], 1) ?>%</span>
                                            <?php else: ?>
                                                <span class="badge badge-info"><?= number_format($change['change_percentage'], 1) ?>%</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($change['trend'] === 'افزایش'): ?>
                                                <span class="badge badge-danger">افزایش</span>
                                            <?php elseif ($change['trend'] === 'کاهش'): ?>
                                                <span class="badge badge-success">کاهش</span>
                                            <?php else: ?>
                                                <span class="badge badge-info">ثابت</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <p>هیچ تغییری در مصرف یافت نشد.</p>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>

    <script>
        // مدیریت نمایش فیلدهای شماره دوره
        document.getElementById('base_period').addEventListener('change', function() {
            const periodNumberGroup = document.getElementById('base_period_number_group');
            const periodNumberSelect = document.getElementById('base_period_number');
            
            if (this.value === 'yearly') {
                periodNumberGroup.style.display = 'none';
            } else {
                periodNumberGroup.style.display = 'block';
                updatePeriodNumbers(periodNumberSelect, this.value);
            }
        });
        
        document.getElementById('compare_period').addEventListener('change', function() {
            const periodNumberGroup = document.getElementById('compare_period_number_group');
            const periodNumberSelect = document.getElementById('compare_period_number');
            
            if (this.value === 'yearly') {
                periodNumberGroup.style.display = 'none';
            } else {
                periodNumberGroup.style.display = 'block';
                updatePeriodNumbers(periodNumberSelect, this.value);
            }
        });
        
        // تابع برای به‌روزرسانی گزینه‌های شماره دوره
        function updatePeriodNumbers(selectElement, periodType) {
            selectElement.innerHTML = '';
            let options = [];
            
            if (periodType === 'monthly') {
                options = [
                    {value: 1, text: 'فروردین'},
                    {value: 2, text: 'اردیبهشت'},
                    {value: 3, text: 'خرداد'},
                    {value: 4, text: 'تیر'},
                    {value: 5, text: 'مرداد'},
                    {value: 6, text: 'شهریور'},
                    {value: 7, text: 'مهر'},
                    {value: 8, text: 'آبان'},
                    {value: 9, text: 'آذر'},
                    {value: 10, text: 'دی'},
                    {value: 11, text: 'بهمن'},
                    {value: 12, text: 'اسفند'}
                ];
            } else if (periodType === 'quarterly') {
                options = [
                    {value: 1, text: 'فصل اول (فروردین-خرداد)'},
                    {value: 2, text: 'فصل دوم (تیر-شهریور)'},
                    {value: 3, text: 'فصل سوم (مهر-آذر)'},
                    {value: 4, text: 'فصل چهارم (دی-اسفند)'}
                ];
            }
            
            options.forEach(option => {
                const optionElement = document.createElement('option');
                optionElement.value = option.value;
                optionElement.textContent = option.text;
                selectElement.appendChild(optionElement);
            });
        }
        
        // مقداردهی اولیه
        updatePeriodNumbers(document.getElementById('base_period_number'), document.getElementById('base_period').value);
        updatePeriodNumbers(document.getElementById('compare_period_number'), document.getElementById('compare_period').value);
        
        // تابع برای خروجی اکسل
        function exportToExcel() {
            // در اینجا می‌توانید کد export به اکسل را اضافه کنید
            alert('این قابلیت در نسخه فعلی پیاده‌سازی نشده است.');
        }
        
        // فیلتر خودکار جدول‌ها
        function filterTable(tableId, columnIndex, searchText) {
            const table = document.getElementById(tableId);
            const rows = table.getElementsByTagName('tr');
            
            for (let i = 1; i < rows.length; i++) {
                const cell = rows[i].getElementsByTagName('td')[columnIndex];
                if (cell) {
                    const cellText = cell.textContent || cell.innerText;
                    if (cellText.toLowerCase().indexOf(searchText.toLowerCase()) > -1) {
                        rows[i].style.display = '';
                    } else {
                        rows[i].style.display = 'none';
                    }
                }
            }
        }
    </script>
</body>
</html>