<?php
// regional_analysis.php
require_once 'config.php';
require_once 'vendor/autoload.php';

use Morilog\Jalali\Jalalian;
use Carbon\Carbon;

class RegionalAnalysis {
    private $pdo;
    private $column_mapping;

    public function __construct($pdo) {
        $this->pdo = $pdo;
    }

    // تابع تبدیل تاریخ شمسی به میلادی
    public function getGregorianRangeFromShamsi($shamsiYear, $period = 'yearly', $period_number = null) {
        try {
            $start_shamsi = null;
            $end_shamsi = null;

            switch($period) {
                case 'monthly':
                    if ($period_number && $period_number >= 1 && $period_number <= 12) {
                        $start_shamsi = new Jalalian($shamsiYear, $period_number, 1);
                        $end_shamsi = (new Jalalian($shamsiYear, $period_number, 1))->addMonths(1)->addDays(-1);
                    }
                    break;

                case 'quarterly':
                    if ($period_number && $period_number >= 1 && $period_number <= 4) {
                        $start_month = (($period_number - 1) * 3) + 1;
                        $end_month = $start_month + 2;
                        $start_shamsi = new Jalalian($shamsiYear, $start_month, 1);
                        $end_shamsi = (new Jalalian($shamsiYear, $end_month, 1))->addMonths(1)->addDays(-1);
                    }
                    break;

                case 'halfyearly':
                    if ($period_number && $period_number >= 1 && $period_number <= 2) {
                        $start_month = $period_number == 1 ? 1 : 7;
                        $end_month = $period_number == 1 ? 6 : 12;
                        $start_shamsi = new Jalalian($shamsiYear, $start_month, 1);
                        $end_shamsi = (new Jalalian($shamsiYear, $end_month, 1))->addMonths(1)->addDays(-1);
                    }
                    break;

                case 'yearly':
                default:
                    $start_shamsi = new Jalalian($shamsiYear, 1, 1);
                    $end_shamsi = (new Jalalian($shamsiYear, 12, 1))->addMonths(1)->addDays(-1);
                    break;
            }

            if ($start_shamsi && $end_shamsi) {
                $start_gregorian = $start_shamsi->toCarbon();
                $end_gregorian = $end_shamsi->toCarbon();
                $end_gregorian->setTime(23, 59, 59);
                
                return [
                    'start' => $start_gregorian->format('Y-m-d H:i:s'),
                    'end' => $end_gregorian->format('Y-m-d H:i:s'),
                    'shamsi_period' => $this->getShamsiPeriodName($period, $period_number, $shamsiYear)
                ];
            }
        } catch (Exception $e) {
            return null;
        }
        return null;
    }

    public function getShamsiPeriodName($period, $period_number, $year) {
        $month_names = [
            1 => 'فروردین', 2 => 'اردیبهشت', 3 => 'خرداد', 4 => 'تیر',
            5 => 'مرداد', 6 => 'شهریور', 7 => 'مهر', 8 => 'آبان',
            9 => 'آذر', 10 => 'دی', 11 => 'بهمن', 12 => 'اسفند'
        ];
        
        $quarter_names = [1 => 'اول', 2 => 'دوم', 3 => 'سوم', 4 => 'چهارم'];
        $halfyear_names = [1 => 'اول', 2 => 'دوم'];
        
        switch($period) {
            case 'monthly':
                return $month_names[$period_number] . ' ' . $year;
            case 'quarterly':
                return 'فصل ' . $quarter_names[$period_number] . ' ' . $year;
            case 'halfyearly':
                return 'نیمه ' . $halfyear_names[$period_number] . ' ' . $year;
            case 'yearly':
                return 'سال ' . $year;
            default:
                return 'سال ' . $year;
        }
    }

    public function getAvailableShamsiYears() {
        $query = $this->pdo->query("
            SELECT DISTINCT تاریخ_تایید_درخواست_توسط_پخش 
            FROM fuel_requests 
            WHERE تاریخ_تایید_درخواست_توسط_پخش IS NOT NULL 
            ORDER BY تاریخ_تایید_درخواست_توسط_پخش DESC
        ");
        $dates = $query->fetchAll(PDO::FETCH_COLUMN);

        $years = [];
        foreach ($dates as $date) {
            try {
                $carbon = Carbon::parse($date);
                $jalali = Jalalian::fromCarbon($carbon);
                $shamsi_year = $jalali->getYear();
                if (!in_array($shamsi_year, $years)) {
                    $years[] = $shamsi_year;
                }
            } catch (Exception $e) {
                continue;
            }
        }

        rsort($years);
        return $years;
    }

    private function buildWhereClause($filters) {
        $where_conditions = [];
        $params = [];

        // فیلتر سال‌ها
        if (!empty($filters['years'])) {
            $year_conditions = [];
            foreach ($filters['years'] as $year) {
                $start_date = $this->getGregorianRangeFromShamsi($year, 'yearly')['start'];
                $end_date = $this->getGregorianRangeFromShamsi($year, 'yearly')['end'];
                $year_conditions[] = "(تاریخ_تایید_درخواست_توسط_پخش BETWEEN ? AND ?)";
                $params[] = $start_date;
                $params[] = $end_date;
            }
            if (!empty($year_conditions)) {
                $where_conditions[] = "(" . implode(" OR ", $year_conditions) . ")";
            }
        }

        // فیلتر شهرستان
        if (!empty($filters['city']) && $filters['city'] !== 'all') {
            $where_conditions[] = "شهرستان = ?";
            $params[] = $filters['city'];
        }

        // فیلتر فرآورده
        if (!empty($filters['product']) && $filters['product'] !== 'all') {
            $where_conditions[] = "فراوده = ?";
            $params[] = $filters['product'];
        }

        // فیلتر بخش مصرف
        if (!empty($filters['usage_section']) && $filters['usage_section'] !== 'all') {
            $where_conditions[] = "بخش_مصرف = ?";
            $params[] = $filters['usage_section'];
        }

        $where_clause = "";
        if (!empty($where_conditions)) {
            $where_clause = "WHERE " . implode(" AND ", $where_conditions);
        }

        return ['where' => $where_clause, 'params' => $params];
    }

    // 1- مقایسه میزان مصرف کل فرآورده در کل ناحیه
    public function getRegionalTotalConsumption($filters) {
        $where_data = $this->buildWhereClause($filters);
        
        $query = "
            SELECT 
                YEAR(تاریخ_تایید_درخواست_توسط_پخش) as gregorian_year,
                MONTH(تاریخ_تایید_درخواست_توسط_پخش) as gregorian_month,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_consumption
            FROM fuel_requests 
            {$where_data['where']}
            GROUP BY YEAR(تاریخ_تایید_درخواست_توسط_پخش), MONTH(تاریخ_تایید_درخواست_توسط_پخش)
            ORDER BY gregorian_year, gregorian_month
        ";

        $stmt = $this->pdo->prepare($query);
        $stmt->execute($where_data['params']);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // 2- مقایسه هر شهرستان در فواصل زمانی مختلف
    public function getCityConsumptionComparison($filters) {
        $where_data = $this->buildWhereClause($filters);
        
        $query = "
            SELECT 
                شهرستان,
                YEAR(تاریخ_تایید_درخواست_توسط_پخش) as year,
                MONTH(تاریخ_تایید_درخواست_توسط_پخش) as month,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_consumption
            FROM fuel_requests 
            {$where_data['where']}
            GROUP BY شهرستان, YEAR(تاریخ_تایید_درخواست_توسط_پخش), MONTH(تاریخ_تایید_درخواست_توسط_پخش)
            ORDER BY شهرستان, year, month
        ";

        $stmt = $this->pdo->prepare($query);
        $stmt->execute($where_data['params']);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // 3- مقایسه مصرف کل ناحیه در هر نوع فراورده
    public function getRegionalProductConsumption($filters) {
        $where_data = $this->buildWhereClause($filters);
        
        $query = "
            SELECT 
                فراوده,
                YEAR(تاریخ_تایید_درخواست_توسط_پخش) as year,
                MONTH(تاریخ_تایید_درخواست_توسط_پخش) as month,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_consumption
            FROM fuel_requests 
            {$where_data['where']}
            GROUP BY فراوده, YEAR(تاریخ_تایید_درخواست_توسط_پخش), MONTH(تاریخ_تایید_درخواست_توسط_پخش)
            ORDER BY فراوده, year, month
        ";

        $stmt = $this->pdo->prepare($query);
        $stmt->execute($where_data['params']);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // 4- مقایسه مصرف هر شهرستان در هر نوع فرآورده
    public function getCityProductConsumption($filters) {
        $where_data = $this->buildWhereClause($filters);
        
        $query = "
            SELECT 
                شهرستان,
                فراوده,
                YEAR(تاریخ_تایید_درخواست_توسط_پخش) as year,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_consumption
            FROM fuel_requests 
            {$where_data['where']}
            GROUP BY شهرستان, فراوده, YEAR(تاریخ_تایید_درخواست_توسط_پخش)
            ORDER BY شهرستان, فراوده, year
        ";

        $stmt = $this->pdo->prepare($query);
        $stmt->execute($where_data['params']);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // 5- مقایسه مصرف کل ناحیه بر اساس نوع بخش مصرف + تفاوت مقادیر
    public function getRegionalUsageSectionConsumption($filters) {
        $where_data = $this->buildWhereClause($filters);
        
        $query = "
            SELECT 
                بخش_مصرف,
                YEAR(تاریخ_تایید_درخواست_توسط_پخش) as year,
                MONTH(تاریخ_تایید_درخواست_توسط_پخش) as month,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_distribution,
                SUM(مقدار_تایید_شده_توسط_متولی_اول) as total_first_responsible,
                SUM(مقدار_تایید_شده_توسط_پخش - مقدار_تایید_شده_توسط_متولی_اول) as difference
            FROM fuel_requests 
            {$where_data['where']}
            GROUP BY بخش_مصرف, YEAR(تاریخ_تایید_درخواست_توسط_پخش), MONTH(تاریخ_تایید_درخواست_توسط_پخش)
            ORDER BY بخش_مصرف, year, month
        ";

        $stmt = $this->pdo->prepare($query);
        $stmt->execute($where_data['params']);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // 6- مقایسه مصرف کل شهرستان بر اساس نوع بخش مصرف + تفاوت مقادیر
    public function getCityUsageSectionConsumption($filters) {
        $where_data = $this->buildWhereClause($filters);
        
        $query = "
            SELECT 
                شهرستان,
                بخش_مصرف,
                YEAR(تاریخ_تایید_درخواست_توسط_پخش) as year,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_distribution,
                SUM(مقدار_تایید_شده_توسط_متولی_اول) as total_first_responsible,
                SUM(مقدار_تایید_شده_توسط_پخش - مقدار_تایید_شده_توسط_متولی_اول) as difference
            FROM fuel_requests 
            {$where_data['where']}
            GROUP BY شهرستان, بخش_مصرف, YEAR(تاریخ_تایید_درخواست_توسط_پخش)
            ORDER BY شهرستان, بخش_مصرف, year
        ";

        $stmt = $this->pdo->prepare($query);
        $stmt->execute($where_data['params']);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // 7- مقایسه مصرف هر بخش مصرف از هر نوع فرآورده
    public function getUsageSectionProductConsumption($filters) {
        $where_data = $this->buildWhereClause($filters);
        
        $query = "
            SELECT 
                بخش_مصرف,
                فراوده,
                YEAR(تاریخ_تایید_درخواست_توسط_پخش) as year,
                MONTH(تاریخ_تایید_درخواست_توسط_پخش) as month,
                SUM(مقدار_تایید_شده_توسط_پخش) as total_consumption
            FROM fuel_requests 
            {$where_data['where']}
            GROUP BY بخش_مصرف, فراوده, YEAR(تاریخ_تایید_درخواست_توسط_پخش), MONTH(تاریخ_تایید_درخواست_توسط_پخش)
            ORDER BY بخش_مصرف, فراوده, year, month
        ";

        $stmt = $this->pdo->prepare($query);
        $stmt->execute($where_data['params']);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function generateAllReports($filters) {
        $reports = [];
        
        $reports['regional_total'] = $this->getRegionalTotalConsumption($filters);
        $reports['city_comparison'] = $this->getCityConsumptionComparison($filters);
        $reports['regional_product'] = $this->getRegionalProductConsumption($filters);
        $reports['city_product'] = $this->getCityProductConsumption($filters);
        $reports['regional_usage_section'] = $this->getRegionalUsageSectionConsumption($filters);
        $reports['city_usage_section'] = $this->getCityUsageSectionConsumption($filters);
        $reports['usage_section_product'] = $this->getUsageSectionProductConsumption($filters);
        
        return $reports;
    }

    // تابع برای سازماندهی داده‌ها بر اساس نوع مقایسه
    public function organizeDataForComparison($data, $comparison_type, $time_period) {
        $organized = [];
        
        foreach ($data as $row) {
            $year = $row['gregorian_year'] ?? $row['year'];
            $month = $row['gregorian_month'] ?? $row['month'];
            
            // تبدیل به تاریخ شمسی
            try {
                $carbon_date = Carbon::create($year, $month, 1);
                $jalali = Jalalian::fromCarbon($carbon_date);
                $shamsi_year = $jalali->getYear();
                $shamsi_month = $jalali->getMonth();
                
                if ($comparison_type === 'corresponding') {
                    // مقایسه ماه‌های متناظر
                    $key = $shamsi_month;
                    $label = $this->getShamsiPeriodName('monthly', $shamsi_month, '');
                } else {
                    // مقایسه پیوسته
                    $key = $shamsi_year . '-' . str_pad($shamsi_month, 2, '0', STR_PAD_LEFT);
                    $label = $this->getShamsiPeriodName('monthly', $shamsi_month, $shamsi_year);
                }
                
                if (!isset($organized[$key])) {
                    $organized[$key] = [
                        'label' => $label,
                        'value' => 0
                    ];
                }
                
                $organized[$key]['value'] += (float)($row['total_consumption'] ?? 0);
                
            } catch (Exception $e) {
                continue;
            }
        }

        return $organized;
    }
}

// دریافت داده‌های فیلتر
$analysis = new RegionalAnalysis($pdo);
$available_years = $analysis->getAvailableShamsiYears();
$cities = $pdo->query("SELECT DISTINCT شهرستان FROM fuel_requests WHERE شهرستان IS NOT NULL ORDER BY شهرستان")->fetchAll(PDO::FETCH_COLUMN);
$products = $pdo->query("SELECT DISTINCT فراوده FROM fuel_requests WHERE فراوده IS NOT NULL ORDER BY فراوده")->fetchAll(PDO::FETCH_COLUMN);
$usage_sections = $pdo->query("SELECT DISTINCT بخش_مصرف FROM fuel_requests WHERE بخش_مصرف IS NOT NULL ORDER BY بخش_مصرف")->fetchAll(PDO::FETCH_COLUMN);

// تنظیم فیلترهای پیش‌فرض
$filters = [
    'years' => $_POST['years'] ?? [$available_years[0] ?? 1400],
    'time_period' => $_POST['time_period'] ?? 'monthly',
    'comparison_type' => $_POST['comparison_type'] ?? 'continuous',
    'city' => $_POST['city'] ?? 'all',
    'product' => $_POST['product'] ?? 'all',
    'usage_section' => $_POST['usage_section'] ?? 'all'
];

// تولید گزارشات
$reports = [];
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $reports = $analysis->generateAllReports($filters);
}
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>تحلیل آماری مصرف سوخت</title>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        body {
            font-family: 'Tahoma', sans-serif;
            background-color: #f5f5f5;
            margin: 0;
            padding: 20px;
        }
        .container {
            max-width: 1400px;
            margin: 0 auto;
            background: white;
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
        }
        .filters {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        .filter-group {
            margin-bottom: 15px;
        }
        label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
        }
        select, input {
            width: 100%;
            padding: 8px;
            border: 1px solid #ddd;
            border-radius: 4px;
        }
        .chart-container {
            margin: 20px 0;
            padding: 15px;
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        canvas {
            max-height: 400px;
        }
        .btn {
            background: #007bff;
            color: white;
            padding: 10px 20px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 16px;
        }
        .btn:hover {
            background: #0056b3;
        }
        .report-section {
            margin: 30px 0;
            padding: 20px;
            border: 1px solid #ddd;
            border-radius: 8px;
        }
        .report-section h3 {
            color: #333;
            border-bottom: 2px solid #007bff;
            padding-bottom: 10px;
        }
        .data-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 15px;
        }
        .data-table th, .data-table td {
            border: 1px solid #ddd;
            padding: 8px;
            text-align: center;
        }
        .data-table th {
            background-color: #f8f9fa;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>سیستم تحلیل آماری مصرف سوخت</h1>
        
        <div class="filters">
            <form method="POST">
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">
                    <div class="filter-group">
                        <label>انتخاب سال‌ها:</label>
                        <select name="years[]" multiple style="height: 120px;">
                            <?php foreach ($available_years as $year): ?>
                                <option value="<?= $year ?>" 
                                    <?= in_array($year, $filters['years']) ? 'selected' : '' ?>>
                                    <?= $year ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="filter-group">
                        <label>بازه زمانی:</label>
                        <select name="time_period">
                            <option value="monthly" <?= $filters['time_period'] == 'monthly' ? 'selected' : '' ?>>ماهانه</option>
                            <option value="quarterly" <?= $filters['time_period'] == 'quarterly' ? 'selected' : '' ?>>فصلی</option>
                            <option value="halfyearly" <?= $filters['time_period'] == 'halfyearly' ? 'selected' : '' ?>>۶ ماهه</option>
                            <option value="yearly" <?= $filters['time_period'] == 'yearly' ? 'selected' : '' ?>>سالانه</option>
                        </select>
                    </div>
                    
                    <div class="filter-group">
                        <label>نوع مقایسه:</label>
                        <select name="comparison_type">
                            <option value="continuous" <?= $filters['comparison_type'] == 'continuous' ? 'selected' : '' ?>>پیوسته</option>
                            <option value="corresponding" <?= $filters['comparison_type'] == 'corresponding' ? 'selected' : '' ?>>متناظر</option>
                        </select>
                    </div>
                    
                    <div class="filter-group">
                        <label>شهرستان:</label>
                        <select name="city">
                            <option value="all" <?= $filters['city'] == 'all' ? 'selected' : '' ?>>همه</option>
                            <?php foreach ($cities as $city): ?>
                                <option value="<?= $city ?>" <?= $filters['city'] == $city ? 'selected' : '' ?>><?= $city ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="filter-group">
                        <label>فرآورده:</label>
                        <select name="product">
                            <option value="all" <?= $filters['product'] == 'all' ? 'selected' : '' ?>>همه</option>
                            <?php foreach ($products as $product): ?>
                                <option value="<?= $product ?>" <?= $filters['product'] == $product ? 'selected' : '' ?>><?= $product ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="filter-group">
                        <label>بخش مصرف:</label>
                        <select name="usage_section">
                            <option value="all" <?= $filters['usage_section'] == 'all' ? 'selected' : '' ?>>همه</option>
                            <?php foreach ($usage_sections as $section): ?>
                                <option value="<?= $section ?>" <?= $filters['usage_section'] == $section ? 'selected' : '' ?>><?= $section ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                
                <button type="submit" class="btn" style="margin-top: 20px;">تولید گزارشات</button>
            </form>
        </div>
        
        <?php if ($_SERVER['REQUEST_METHOD'] === 'POST' && !empty($reports)): ?>
            <!-- گزارش 1: مقایسه میزان مصرف کل فرآورده در کل ناحیه -->
            <div class="report-section">
                <h3>۱- مقایسه میزان مصرف کل فرآورده در کل ناحیه</h3>
                <div class="chart-container">
                    <canvas id="chart1"></canvas>
                </div>
            </div>

            <!-- گزارش 2: مقایسه هر شهرستان در فواصل زمانی مختلف -->
            <div class="report-section">
                <h3>۲- مقایسه هر شهرستان در فواصل زمانی مختلف</h3>
                <div class="chart-container">
                    <canvas id="chart2"></canvas>
                </div>
            </div>

            <!-- گزارش 3: مقایسه مصرف کل ناحیه در هر نوع فراورده -->
            <div class="report-section">
                <h3>۳- مقایسه مصرف کل ناحیه در هر نوع فراورده</h3>
                <div class="chart-container">
                    <canvas id="chart3"></canvas>
                </div>
            </div>

            <!-- گزارش 4: مقایسه مصرف هر شهرستان در هر نوع فرآورده -->
            <div class="report-section">
                <h3>۴- مقایسه مصرف هر شهرستان در هر نوع فرآورده</h3>
                <div class="chart-container">
                    <canvas id="chart4"></canvas>
                </div>
            </div>

            <!-- گزارش 5: مقایسه مصرف کل ناحیه بر اساس نوع بخش مصرف -->
            <div class="report-section">
                <h3>۵- مقایسه مصرف کل ناحیه بر اساس نوع بخش مصرف</h3>
                <div class="chart-container">
                    <canvas id="chart5"></canvas>
                </div>
            </div>

            <!-- گزارش 6: مقایسه مصرف کل شهرستان بر اساس نوع بخش مصرف -->
            <div class="report-section">
                <h3>۶- مقایسه مصرف کل شهرستان بر اساس نوع بخش مصرف</h3>
                <div class="chart-container">
                    <canvas id="chart6"></canvas>
                </div>
            </div>

            <!-- گزارش 7: مقایسه مصرف هر بخش مصرف از هر نوع فرآورده -->
            <div class="report-section">
                <h3>۷- مقایسه مصرف هر بخش مصرف از هر نوع فرآورده</h3>
                <div class="chart-container">
                    <canvas id="chart7"></canvas>
                </div>
            </div>

            <script>
// داده‌های PHP به JavaScript
const reports = <?= json_encode($reports) ?>;
const comparisonType = '<?= $filters['comparison_type'] ?>';
const timePeriod = '<?= $filters['time_period'] ?>';
const selectedYears = <?= json_encode($filters['years']) ?>;

// رنگ‌های مختلف برای نمودارها
const chartColors = [
    'rgba(54, 162, 235, 0.7)',
    'rgba(255, 99, 132, 0.7)',
    'rgba(75, 192, 192, 0.7)',
    'rgba(255, 159, 64, 0.7)',
    'rgba(153, 102, 255, 0.7)',
    'rgba(255, 205, 86, 0.7)',
    'rgba(201, 203, 207, 0.7)',
    'rgba(255, 99, 71, 0.7)',
    'rgba(147, 112, 219, 0.7)',
    'rgba(60, 179, 113, 0.7)'
];

// نام فصل‌ها
const seasonNames = {
    1: 'فصل اول (فروردین-خرداد)',
    2: 'فصل دوم (تیر-شهریور)',
    3: 'فصل سوم (مهر-آذر)',
    4: 'فصل چهارم (دی-اسفند)'
};

// نام نیم‌سال‌ها
const halfYearNames = {
    1: 'نیمه اول سال',
    2: 'نیمه دوم سال'
};

// تابع برای ایجاد نمودار میله‌ای
function createBarChart(canvasId, data, title) {
    const ctx = document.getElementById(canvasId).getContext('2d');
    
    // اگر داده‌ها شامل چند dataset باشد
    if (data.datasets) {
        return new Chart(ctx, {
            type: 'bar',
            data: {
                labels: data.labels,
                datasets: data.datasets
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    title: {
                        display: true,
                        text: title,
                        font: {
                            size: 16
                        }
                    },
                    legend: {
                        display: data.datasets.length > 1,
                        position: 'top',
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                return `${context.dataset.label}: ${context.parsed.y.toLocaleString('fa-IR')} لیتر`;
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        title: {
                            display: true,
                            text: 'مقدار مصرف (لیتر)'
                        },
                        ticks: {
                            callback: function(value) {
                                return value.toLocaleString('fa-IR');
                            }
                        }
                    },
                    x: {
                        title: {
                            display: true,
                            text: 'بازه زمانی'
                        }
                    }
                }
            }
        });
    } else {
        // اگر داده‌ها ساده باشد
        return new Chart(ctx, {
            type: 'bar',
            data: {
                labels: data.labels,
                datasets: [{
                    label: title,
                    data: data.values,
                    backgroundColor: chartColors,
                    borderColor: chartColors.map(color => color.replace('0.7', '1')),
                    borderWidth: 2
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    title: {
                        display: true,
                        text: title,
                        font: {
                            size: 16
                        }
                    },
                    legend: {
                        display: false
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                return `مصرف: ${context.parsed.y.toLocaleString('fa-IR')} لیتر`;
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        title: {
                            display: true,
                            text: 'مقدار مصرف (لیتر)'
                        },
                        ticks: {
                            callback: function(value) {
                                return value.toLocaleString('fa-IR');
                            }
                        }
                    },
                    x: {
                        title: {
                            display: true,
                            text: 'بازه زمانی'
                        }
                    }
                }
            }
        });
    }
}

// تابع برای تشخیص فصل از ماه
function getSeasonFromMonth(month) {
    if (month >= 1 && month <= 3) return 1;
    if (month >= 4 && month <= 6) return 2;
    if (month >= 7 && month <= 9) return 3;
    if (month >= 10 && month <= 12) return 4;
    return 1;
}

// تابع برای تشخیص نیم‌سال از ماه
function getHalfYearFromMonth(month) {
    return month <= 6 ? 1 : 2;
}

// 1- پردازش داده‌های مصرف کل منطقه
function processRegionalTotalData(data) {
    const periods = {};
    
    data.forEach(item => {
        const year = item.gregorian_year;
        const month = item.gregorian_month;
        let periodKey, periodLabel;

        if (timePeriod === 'monthly') {
            if (comparisonType === 'corresponding') {
                periodKey = `month_${month}`;
                periodLabel = `ماه ${month}`;
            } else {
                periodKey = `${year}_${month}`;
                periodLabel = `${year}-${month}`;
            }
        } else if (timePeriod === 'quarterly') {
            const season = getSeasonFromMonth(month);
            if (comparisonType === 'corresponding') {
                periodKey = `season_${season}`;
                periodLabel = seasonNames[season];
            } else {
                periodKey = `${year}_${season}`;
                periodLabel = `${seasonNames[season]} ${year}`;
            }
        } else if (timePeriod === 'halfyearly') {
            const halfYear = getHalfYearFromMonth(month);
            if (comparisonType === 'corresponding') {
                periodKey = `half_${halfYear}`;
                periodLabel = halfYearNames[halfYear];
            } else {
                periodKey = `${year}_${halfYear}`;
                periodLabel = `${halfYearNames[halfYear]} ${year}`;
            }
        } else { // yearly
            if (comparisonType === 'corresponding') {
                periodKey = 'year';
                periodLabel = 'سال';
            } else {
                periodKey = `${year}`;
                periodLabel = `سال ${year}`;
            }
        }

        if (!periods[periodKey]) {
            periods[periodKey] = {
                label: periodLabel,
                value: 0
            };
        }
        periods[periodKey].value += parseFloat(item.total_consumption) || 0;
    });

    // مرتب‌سازی دوره‌ها
    const sortedPeriods = Object.values(periods).sort((a, b) => {
        return a.label.localeCompare(b.label, 'fa-IR');
    });

    return {
        labels: sortedPeriods.map(p => p.label),
        values: sortedPeriods.map(p => p.value)
    };
}

// 2- پردازش داده‌های مقایسه شهرستان‌ها
function processCityComparisonData(data) {
    const cities = {};
    const allPeriods = new Set();
    
    data.forEach(item => {
        const year = item.year;
        const month = item.month;
        let periodKey, periodLabel;

        if (timePeriod === 'monthly') {
            if (comparisonType === 'corresponding') {
                periodKey = `month_${month}`;
                periodLabel = `ماه ${month}`;
            } else {
                periodKey = `${year}_${month}`;
                periodLabel = `${year}-${month}`;
            }
        } else if (timePeriod === 'quarterly') {
            const season = getSeasonFromMonth(month);
            if (comparisonType === 'corresponding') {
                periodKey = `season_${season}`;
                periodLabel = seasonNames[season];
            } else {
                periodKey = `${year}_${season}`;
                periodLabel = `${seasonNames[season]} ${year}`;
            }
        } else if (timePeriod === 'halfyearly') {
            const halfYear = getHalfYearFromMonth(month);
            if (comparisonType === 'corresponding') {
                periodKey = `half_${halfYear}`;
                periodLabel = halfYearNames[halfYear];
            } else {
                periodKey = `${year}_${halfYear}`;
                periodLabel = `${halfYearNames[halfYear]} ${year}`;
            }
        } else {
            if (comparisonType === 'corresponding') {
                periodKey = 'year';
                periodLabel = 'سال';
            } else {
                periodKey = `${year}`;
                periodLabel = `سال ${year}`;
            }
        }

        allPeriods.add(periodKey);

        if (!cities[item.شهرستان]) {
            cities[item.شهرستان] = {};
        }
        cities[item.شهرستان][periodKey] = (cities[item.شهرستان][periodKey] || 0) + parseFloat(item.total_consumption) || 0;
    });

    const sortedPeriods = Array.from(allPeriods).sort();
    const periodLabels = sortedPeriods.map(periodKey => {
        // استخراج label از periodKey
        if (periodKey.startsWith('month_')) {
            return `ماه ${periodKey.split('_')[1]}`;
        } else if (periodKey.startsWith('season_')) {
            return seasonNames[parseInt(periodKey.split('_')[1])];
        } else if (periodKey.startsWith('half_')) {
            return halfYearNames[parseInt(periodKey.split('_')[1])];
        } else if (periodKey.includes('_')) {
            const [year, period] = periodKey.split('_');
            if (timePeriod === 'quarterly') {
                return `${seasonNames[parseInt(period)]} ${year}`;
            } else if (timePeriod === 'halfyearly') {
                return `${halfYearNames[parseInt(period)]} ${year}`;
            } else {
                return `${year}-${period}`;
            }
        } else {
            return periodKey === 'year' ? 'سال' : `سال ${periodKey}`;
        }
    });

    const datasets = Object.keys(cities).map((city, index) => ({
        label: city,
        data: sortedPeriods.map(periodKey => cities[city][periodKey] || 0),
        backgroundColor: chartColors[index % chartColors.length],
        borderColor: chartColors[index % chartColors.length].replace('0.7', '1'),
        borderWidth: 2
    }));

    return {
        labels: periodLabels,
        datasets: datasets
    };
}

// 3- پردازش داده‌های مصرف بر اساس فرآورده
function processRegionalProductData(data) {
    const products = {};
    const allPeriods = new Set();
    
    data.forEach(item => {
        const year = item.year;
        const month = item.month;
        let periodKey, periodLabel;

        if (timePeriod === 'monthly') {
            periodKey = comparisonType === 'corresponding' ? `month_${month}` : `${year}_${month}`;
        } else if (timePeriod === 'quarterly') {
            const season = getSeasonFromMonth(month);
            periodKey = comparisonType === 'corresponding' ? `season_${season}` : `${year}_${season}`;
        } else if (timePeriod === 'halfyearly') {
            const halfYear = getHalfYearFromMonth(month);
            periodKey = comparisonType === 'corresponding' ? `half_${halfYear}` : `${year}_${halfYear}`;
        } else {
            periodKey = comparisonType === 'corresponding' ? 'year' : `${year}`;
        }

        allPeriods.add(periodKey);

        if (!products[item.فراوده]) {
            products[item.فراوده] = {};
        }
        products[item.فراوده][periodKey] = (products[item.فراوده][periodKey] || 0) + parseFloat(item.total_consumption) || 0;
    });

    const sortedPeriods = Array.from(allPeriods).sort();
    const periodLabels = getPeriodLabels(sortedPeriods);

    const datasets = Object.keys(products).map((product, index) => ({
        label: product,
        data: sortedPeriods.map(periodKey => products[product][periodKey] || 0),
        backgroundColor: chartColors[index % chartColors.length],
        borderColor: chartColors[index % chartColors.length].replace('0.7', '1'),
        borderWidth: 2
    }));

    return {
        labels: periodLabels,
        datasets: datasets
    };
}

// 4- پردازش داده‌های مصرف شهرستان بر اساس فرآورده
function processCityProductData(data) {
    const cityProducts = {};
    
    data.forEach(item => {
        const key = `${item.شهرستان} - ${item.فراوده}`;
        const year = item.year;
        
        if (!cityProducts[key]) {
            cityProducts[key] = {};
        }
        cityProducts[key][year] = (cityProducts[key][year] || 0) + parseFloat(item.total_consumption) || 0;
    });

    const years = [...new Set(data.map(item => item.year))].sort();
    const datasets = Object.keys(cityProducts).map((key, index) => ({
        label: key,
        data: years.map(year => cityProducts[key][year] || 0),
        backgroundColor: chartColors[index % chartColors.length],
        borderColor: chartColors[index % chartColors.length].replace('0.7', '1'),
        borderWidth: 2
    }));

    return {
        labels: years.map(year => `سال ${year}`),
        datasets: datasets
    };
}

// 5- پردازش داده‌های بخش مصرف منطقه
function processRegionalUsageSectionData(data) {
    const usageSections = {};
    const allPeriods = new Set();
    
    data.forEach(item => {
        const year = item.year;
        const month = item.month;
        let periodKey;

        if (timePeriod === 'monthly') {
            periodKey = comparisonType === 'corresponding' ? `month_${month}` : `${year}_${month}`;
        } else if (timePeriod === 'quarterly') {
            const season = getSeasonFromMonth(month);
            periodKey = comparisonType === 'corresponding' ? `season_${season}` : `${year}_${season}`;
        } else if (timePeriod === 'halfyearly') {
            const halfYear = getHalfYearFromMonth(month);
            periodKey = comparisonType === 'corresponding' ? `half_${halfYear}` : `${year}_${halfYear}`;
        } else {
            periodKey = comparisonType === 'corresponding' ? 'year' : `${year}`;
        }

        allPeriods.add(periodKey);

        if (!usageSections[item.بخش_مصرف]) {
            usageSections[item.بخش_مصرف] = {};
        }
        usageSections[item.بخش_مصرف][periodKey] = 
            (usageSections[item.بخش_مصرف][periodKey] || 0) + parseFloat(item.total_distribution) || 0;
    });

    const sortedPeriods = Array.from(allPeriods).sort();
    const periodLabels = getPeriodLabels(sortedPeriods);

    const datasets = Object.keys(usageSections).map((section, index) => ({
        label: section,
        data: sortedPeriods.map(periodKey => usageSections[section][periodKey] || 0),
        backgroundColor: chartColors[index % chartColors.length],
        borderColor: chartColors[index % chartColors.length].replace('0.7', '1'),
        borderWidth: 2
    }));

    return {
        labels: periodLabels,
        datasets: datasets
    };
}

// 6- پردازش داده‌های بخش مصرف شهرستان
function processCityUsageSectionData(data) {
    const citySections = {};
    
    data.forEach(item => {
        const key = `${item.شهرستان} - ${item.بخش_مصرف}`;
        const year = item.year;
        
        if (!citySections[key]) {
            citySections[key] = {};
        }
        citySections[key][year] = (citySections[key][year] || 0) + parseFloat(item.total_distribution) || 0;
    });

    const years = [...new Set(data.map(item => item.year))].sort();
    const datasets = Object.keys(citySections).map((key, index) => ({
        label: key,
        data: years.map(year => citySections[key][year] || 0),
        backgroundColor: chartColors[index % chartColors.length],
        borderColor: chartColors[index % chartColors.length].replace('0.7', '1'),
        borderWidth: 2
    }));

    return {
        labels: years.map(year => `سال ${year}`),
        datasets: datasets
    };
}

// 7- پردازش داده‌های بخش مصرف و فرآورده
function processUsageSectionProductData(data) {
    const sectionProducts = {};
    const allPeriods = new Set();
    
    data.forEach(item => {
        const year = item.year;
        const month = item.month;
        let periodKey;

        if (timePeriod === 'monthly') {
            periodKey = comparisonType === 'corresponding' ? `month_${month}` : `${year}_${month}`;
        } else if (timePeriod === 'quarterly') {
            const season = getSeasonFromMonth(month);
            periodKey = comparisonType === 'corresponding' ? `season_${season}` : `${year}_${season}`;
        } else if (timePeriod === 'halfyearly') {
            const halfYear = getHalfYearFromMonth(month);
            periodKey = comparisonType === 'corresponding' ? `half_${halfYear}` : `${year}_${halfYear}`;
        } else {
            periodKey = comparisonType === 'corresponding' ? 'year' : `${year}`;
        }

        allPeriods.add(periodKey);

        const key = `${item.بخش_مصرف} - ${item.فراوده}`;
        if (!sectionProducts[key]) {
            sectionProducts[key] = {};
        }
        sectionProducts[key][periodKey] = 
            (sectionProducts[key][periodKey] || 0) + parseFloat(item.total_consumption) || 0;
    });

    const sortedPeriods = Array.from(allPeriods).sort();
    const periodLabels = getPeriodLabels(sortedPeriods);

    const datasets = Object.keys(sectionProducts).map((key, index) => ({
        label: key,
        data: sortedPeriods.map(periodKey => sectionProducts[key][periodKey] || 0),
        backgroundColor: chartColors[index % chartColors.length],
        borderColor: chartColors[index % chartColors.length].replace('0.7', '1'),
        borderWidth: 2
    }));

    return {
        labels: periodLabels,
        datasets: datasets
    };
}

// تابع کمکی برای تولید label دوره‌ها
function getPeriodLabels(periodKeys) {
    return periodKeys.map(periodKey => {
        if (periodKey.startsWith('month_')) {
            return `ماه ${periodKey.split('_')[1]}`;
        } else if (periodKey.startsWith('season_')) {
            return seasonNames[parseInt(periodKey.split('_')[1])];
        } else if (periodKey.startsWith('half_')) {
            return halfYearNames[parseInt(periodKey.split('_')[1])];
        } else if (periodKey.includes('_')) {
            const [year, period] = periodKey.split('_');
            if (timePeriod === 'quarterly') {
                return `${seasonNames[parseInt(period)]} ${year}`;
            } else if (timePeriod === 'halfyearly') {
                return `${halfYearNames[parseInt(period)]} ${year}`;
            } else {
                return `${year}-${period}`;
            }
        } else {
            return periodKey === 'year' ? 'سال' : `سال ${periodKey}`;
        }
    });
}

// ایجاد تمام نمودارها پس از لود صفحه
document.addEventListener('DOMContentLoaded', function() {
    console.log('Data loaded:', reports);
    console.log('Filters:', { comparisonType, timePeriod, selectedYears });

    // نمودار 1: مصرف کل منطقه
    if (reports.regional_total && reports.regional_total.length > 0) {
        const chart1Data = processRegionalTotalData(reports.regional_total);
        createBarChart('chart1', chart1Data, '۱- مقایسه میزان مصرف کل فرآورده در کل ناحیه');
        console.log('Chart 1 Data:', chart1Data);
    }

    // نمودار 2: مقایسه شهرستان‌ها
    if (reports.city_comparison && reports.city_comparison.length > 0) {
        const chart2Data = processCityComparisonData(reports.city_comparison);
        createBarChart('chart2', chart2Data, '۲- مقایسه هر شهرستان در فواصل زمانی مختلف');
        console.log('Chart 2 Data:', chart2Data);
    }

    // نمودار 3: مصرف بر اساس فرآورده
    if (reports.regional_product && reports.regional_product.length > 0) {
        const chart3Data = processRegionalProductData(reports.regional_product);
        createBarChart('chart3', chart3Data, '۳- مقایسه مصرف کل ناحیه در هر نوع فراورده');
        console.log('Chart 3 Data:', chart3Data);
    }

    // نمودار 4: مصرف شهرستان بر اساس فرآورده
    if (reports.city_product && reports.city_product.length > 0) {
        const chart4Data = processCityProductData(reports.city_product);
        createBarChart('chart4', chart4Data, '۴- مقایسه مصرف هر شهرستان در هر نوع فرآورده');
        console.log('Chart 4 Data:', chart4Data);
    }

    // نمودار 5: بخش مصرف منطقه
    if (reports.regional_usage_section && reports.regional_usage_section.length > 0) {
        const chart5Data = processRegionalUsageSectionData(reports.regional_usage_section);
        createBarChart('chart5', chart5Data, '۵- مقایسه مصرف کل ناحیه بر اساس نوع بخش مصرف');
        console.log('Chart 5 Data:', chart5Data);
    }

    // نمودار 6: بخش مصرف شهرستان
    if (reports.city_usage_section && reports.city_usage_section.length > 0) {
        const chart6Data = processCityUsageSectionData(reports.city_usage_section);
        createBarChart('chart6', chart6Data, '۶- مقایسه مصرف کل شهرستان بر اساس نوع بخش مصرف');
        console.log('Chart 6 Data:', chart6Data);
    }

    // نمودار 7: بخش مصرف و فرآورده
    if (reports.usage_section_product && reports.usage_section_product.length > 0) {
        const chart7Data = processUsageSectionProductData(reports.usage_section_product);
        createBarChart('chart7', chart7Data, '۷- مقایسه مصرف هر بخش مصرف از هر نوع فرآورده');
        console.log('Chart 7 Data:', chart7Data);
    }
});

// تابع برای دانلود نمودارها
function downloadChart(chartId, filename) {
    const chartCanvas = document.getElementById(chartId);
    const link = document.createElement('a');
    link.download = filename + '.png';
    link.href = chartCanvas.toDataURL('image/png');
    link.click();
}
</script>
        <?php endif; ?>
    </div>
</body>
</html>